/****************************************************************************

    Module  : DockUtil.c

    Changes : 20/8/93 - Added use of RePaintSlot function to selectively
                        repaint a single slot instead of using invalidate
                        clientrect to paint whole window when only a single
                        slot changed.
              24/6/94 - rewrote code to position dock & abstracted code into
                        function SetDockPos()
			  3.0 Beta 2 : Changes to support multiple instances of FreeDock
              3.0 beta 5 : Added ProcessDroppedFiles function and FillOutSlot
                           functions to tidy up main code.
              3.0 beta 7 : Fixed bug in SetSubDockWinPos which caused FreeDock
              to loose the "always on top" state.
              Corrected bug in screen -> client coord convertion in 
              FindSlotHitInSubDock.
              Tidied Change Dir code into ChDir function 
              (thanks for spotting this Max).
*****************************************************************************/


#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>
#include <shellapi.h>
#include "freedock.h"
#include "dock_reg.h"

/*********************************************************
	Static (local) slot structure used in window enum
	function.
*********************************************************/
static SLOT_ENTRY TmpSlot;

/**********************************************************
	Global Data for the various system metrics we need
**********************************************************/
extern GLOBAL_METRICS gMetrics;
extern GLOBAL_OPTIONS gOptions;


SLOT_ENTRY *FindSlotHitInDock(DOCK_STRUCT *Dock, int XHit, int YHit)
{					  	

    if (GetDockOrientation(Dock->hwndDock) == DOCK_HORZ) {
   	    return ( &(Dock->Slot[XHit / gOptions.SlotWidth]) );
    } 
   	else {
       	return ( &(Dock->Slot[YHit / gOptions.SlotHeight]) );
    }
}

SLOT_ENTRY *FindSlotHitInSubDocks(DOCK_STRUCT *Dock, int XHit, int YHit)
{					  	
	DOCK_STRUCT *HitDock;
	RECT		 rect;
	POINT		 pt;
	HWND		 hwndUnderCursor;
	short int	 ClientX, ClientY;

    GetWindowRect(Dock->hwndDock, &rect);
    pt.x = rect.left + (short int)XHit + gMetrics.DlgFrameWidth;
    pt.y = rect.top  + (short int)YHit + gMetrics.DlgFrameHeight;

	hwndUnderCursor = WindowFromPoint( pt );

	if( IsDockWindow(hwndUnderCursor) ){
		HitDock = GetDockStruct( hwndUnderCursor );
		// Convent Coords of original dock in to system coords and then into
		// client coords relative to the dock in which the hit occured (they may both
		// be the same dock.) (pt already contains the system coords of the hit)
		GetWindowRect(hwndUnderCursor, &rect);
		ClientX = (short int)pt.x - (short int)rect.left;
		ClientY = (short int)pt.y - (short int)rect.top;

	    if (GetDockOrientation(HitDock->hwndDock) == DOCK_HORZ) {
    	    return ( &(HitDock->Slot[ClientX / gOptions.SlotWidth]) );
	    } 
    	else {
        	return ( &(HitDock->Slot[ClientY / gOptions.SlotHeight]) );
	    }
	}
	else{
//        MessageBox(Dock->hwndDock, "GetSlotHit returning NULL, this is bad.",
//                   "FreeDock Error", MB_OK | MB_ICONSTOP);
		return( (SLOT_ENTRY *)NULL );
	}
}


BOOL IsCursorOutOfDock(DOCK_STRUCT *Dock, int X, int Y){
	
	RECT	rect;
	POINT	pt;
	HWND	hwndUnderCursor;

	// manually convert client coords to screen coords to handle
	// correctly the event when the client coords are negative.
    GetWindowRect(Dock->hwndDock, &rect);
    pt.x = rect.left + (short int)X + gMetrics.DlgFrameWidth;
    pt.y = rect.top  + (short int)Y + gMetrics.DlgFrameHeight;

	hwndUnderCursor = WindowFromPoint( pt );

	if( !IsDockWindow(hwndUnderCursor) ){
		return( TRUE );
	}
	else{
		return( FALSE );
	}
}



/**************************************************************************
    ExecSlot.

    This function handles the execution of a program associated
    with a slot, it takes the Slot index so it can find the program
    to execute, it also takes the command line to pass, to enable
    this function to be used to execute a program as a result of
    a drag and drop action.
    In the case of drag&drop, the caller must create the command line
    before calling this function.
**************************************************************************/

void ExecSlot( SLOT_ENTRY *Slot, char *CmdLine ){

    int                 StartState;
    LPARAM              hStartAppInst;
    UINT                nItem;
    BOOL                bStartedApp = FALSE;
	DOCK_STRUCT			*Dock;

#ifdef WIN32
	DWORD				fdwCreate = 0L;
	STARTUPINFO         StartupInfo;
    PROCESS_INFORMATION ProcInfo;
    char    TmpBuffer[MAX_FPATH_LEN + MAX_CMDLINE_LEN + 1];
#endif

	Dock = Slot->Dock;

	/*************************************************************
        If the Pop-In on Execute option is selected, pop in 
        any child / parent docks.
		Check that this slot is not in a sub-dock, if so we 
		need to pop-in all the sub-docks to the root. Only
		close the sub-dock tree containing the executed slot,
		do not step back all the way to the root and close all
		children.
	*************************************************************/
	if( gOptions.PopInOnExec ){
		HWND TmpHwndDock;
	
		TmpHwndDock = Dock->hwndDock;
		// While this dock has a parent
		while( (GetParent(TmpHwndDock) != NULL) && (GetParent( GetParent(TmpHwndDock) ) != NULL ) ){
			// if it has a grand-parent (i.e. the parent is not the root dock).
			TmpHwndDock = GetParent(TmpHwndDock);
		}
		PopInChildren( TmpHwndDock );
	}

    switch (Slot->StartState) {

        case START_MINIMUM:
            StartState = SW_SHOWMINIMIZED;
            break;

        case START_MAXIMUM:
            StartState = SW_SHOWMAXIMIZED;
            break;

        case START_NORMAL:
        case START_STORE :
            StartState = SW_SHOWNORMAL;
            break;

        default: return;
    }

    if(Slot->StartOnTop){
        StartState = StartState | SW_SHOWMAXIMIZED;
    }


	/**********************************************
		Check if the working directory exists
		if not, return.
	**********************************************/
#ifdef WIN32
	if( !SetCurrentDirectory( Slot->RunTimeDir ) ){
#else
	if( !ChDir( Slot->RunTimeDir ) ){
#endif
        DockError( NULL, "The Runtime Directory for this application cannot be found.\r\r\n Cannot Start Application"); 
		return;
	}

    SetCursor(LoadCursor(NULL, IDC_WAIT));
    /**********************************************
        Need to start up the application in a
        diferent manner under NT so we can 
        keep track of it and find the window
        it creates inorder to re-position
        it or set it on top
    *********************************************/
#ifdef WIN32

    memset( &StartupInfo, '\0', sizeof(STARTUPINFO) );
    memset( &ProcInfo,    '\0', sizeof(PROCESS_INFORMATION) );
    StartupInfo.cb = sizeof(StartupInfo);

	/***********************************************************
		Setup the window Pos, size at startup time
		if required (only WIN32 supports this option)
		Can't specify that a window is always on top though,
		so we still use the EnumWindows function for this,
	***********************************************************/    
    if (Slot->StartState == START_STORE){
		StartupInfo.dwX		= Slot->WinX;
		StartupInfo.dwY		= Slot->WinY;
		StartupInfo.dwXSize	= Slot->WinWidth;
		StartupInfo.dwYSize	= Slot->WinHeight;
		StartupInfo.dwFlags = STARTF_USEPOSITION | STARTF_USESIZE;
	}
	else if (Slot->StartState == START_MINIMUM){
		StartupInfo.wShowWindow	= SW_SHOWMINIMIZED;
		StartupInfo.dwFlags = STARTF_USESHOWWINDOW;
	}
	else if (Slot->StartState == START_MAXIMUM){
		StartupInfo.wShowWindow	= SW_SHOWMAXIMIZED;
		StartupInfo.dwFlags = STARTF_USESHOWWINDOW;
	}

	/*****************************************************************
		Are we trying to start this app in separate address space ?
	*****************************************************************/
	if (Slot->StartSeparate == TRUE){
	 	fdwCreate = CREATE_SEPARATE_WOW_VDM;
	}
    // Create Full Command Line
    sprintf( TmpBuffer, "%s %s", Slot->AppName, CmdLine );
	// Start process.
    bStartedApp = CreateProcess( NULL, TmpBuffer,
                                 NULL,                 // default process security
                                 NULL,                 // default thread security
                                 FALSE,                // don't inherit handles
                                 fdwCreate,            // startup flags
                                 NULL,                 // Inherit the ENVIRONMENT
                                 Slot->RunTimeDir,     // Current Dir
                                 &StartupInfo,         // Process Startup Info
                                 &ProcInfo );          // Recieves Process data
    /** Save the process ID to pass into the EnumWindows func later if reqd **/
    hStartAppInst = (LPARAM)ProcInfo.dwProcessId;
    
#else
    hStartAppInst = (LPARAM)ShellExecute(GetDesktopWindow(),
        "open",
        Slot->AppName,
        CmdLine,
        Slot->RunTimeDir,
        StartState);

    if (hStartAppInst > 33){
        bStartedApp = TRUE;
    }
#endif

    /** No Sleep() in 16bit windows, and it does not seem to be reqd **/
#ifdef WIN32
    /** Have a kip to give the window a chance to appear **/
    Sleep(gOptions.SleepTime*500);
#endif

    /** Check for an error starting the application **/
    if( !bStartedApp ){
        DockError( Dock->hwndDock, "Error starting application.");
        SetCursor(LoadCursor(NULL, IDC_ARROW));
        return;
    }
    
	/** No need to try to reposition window in NT, it's done already **/
#ifdef WIN32
    if( Slot->StartOnTop ){               
#else
    if( (Slot->StartState == START_STORE) || 
        (Slot->StartOnTop)){               
#endif   
        /** Copy Slot into TmpSlot to pass to callback fuction **/
        memcpy( &TmpSlot, Slot, sizeof(SLOT_ENTRY) );

        gOptions.bFoundWindow = FALSE;	// Set Call back function success flag to fail

        do{
            nItem = IDCANCEL;
               EnumWindows( UtilSetupStartedWin, (LPARAM)hStartAppInst );
               if(!gOptions.bFoundWindow){
                    nItem = MessageBox(Dock->hwndDock, "Could not locate window for started application. Retry ?",
                            "Freedock Error", MB_RETRYCANCEL | MB_ICONSTOP);
               }
        }while( nItem != IDCANCEL );
    }

    SetCursor(LoadCursor(NULL, IDC_ARROW));

    return;
}


void SetRootDockPos( DOCK_STRUCT *Dock, DOCK_POS DockPos )
{
	// This fucntion correctly positions the dock on the desktop, 
	// allowing for the current orientation

	// Limit dock position to actual screen
	gOptions.RootDockLeft = max(gOptions.RootDockLeft, 0);
	gOptions.RootDockTop  = max(gOptions.RootDockTop, 0);

	gOptions.RootDockLeft = min(gOptions.RootDockLeft, (gMetrics.ScreenWidth - gOptions.SlotWidth));
	gOptions.RootDockTop  = min(gOptions.RootDockTop, (gMetrics.ScreenHeight - gOptions.SlotHeight));

    switch( DockPos ){

        case QX_HORZ:
            gOptions.RootOrientation = DOCK_HORZ;
            SetWindowPos(Dock->hwndDock,
                HWND_TOP,
                gOptions.RootDockLeft,
                gOptions.RootDockTop,
                Dock->SlotCount * gOptions.SlotWidth + 2,
                gOptions.SlotHeight + 2,
                SWP_NOACTIVATE | SWP_SHOWWINDOW);
            break;

        default :
            gOptions.RootOrientation = DOCK_VERT;
            SetWindowPos(Dock->hwndDock,
                HWND_TOP,
                gOptions.RootDockLeft,
                gOptions.RootDockTop,
                gOptions.SlotWidth + 2,
                Dock->SlotCount * gOptions.SlotHeight + 2,
                SWP_NOACTIVATE | SWP_SHOWWINDOW);
            break;

    }
}


void UtilLoadIcon( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot ){

    RECT	rect;
    HICON   IconHandle;
    static HBITMAP  hSlotButton    = NULL;
    static HBITMAP  hSubDockButton = NULL;

    // Load up the blank image if we don't already have it
    if( hSlotButton == NULL ){
        hSlotButton = LoadBitmap(gOptions.hAppInst, "SlotButton");
        hSubDockButton = LoadBitmap(gOptions.hAppInst, "SubDockButton");
    }
    
    // If the Slot parameter is NULL this means we need to delete
    // the internal bitmap handle.
    if( Slot == NULL ){
        DeleteObject( hSlotButton );
        DeleteObject( hSubDockButton );
        return;
    }
                                      
    rect.top = 0;
    rect.left = Slot->IconIndex;
    rect.bottom = gOptions.SlotHeight;
    rect.right = Slot->IconIndex + gOptions.SlotWidth;

    Slot->IconTotal = 0;
    
    switch( Slot->SlotType ){

        case SLOT_SPECIAL_TITLE :
#ifdef WIN32
            IconHandle = LoadIcon(gOptions.hAppInst, "FREEDOCK32");
#else
            IconHandle = LoadIcon(gOptions.hAppInst, "FREEDOCK16");
#endif
            break;

        case SLOT_SPECIAL_MAIL :
        	if( gOptions.bMailInBox ){
	            IconHandle = LoadIcon(gOptions.hAppInst, "MAILICON");
        	}
        	else{
            	IconHandle = LoadIcon(gOptions.hAppInst, "NOMAILICON");
            }
            break;

        case SLOT_SPECIAL_CLOCK :
            IconHandle = LoadIcon(gOptions.hAppInst, "CLOCKICON");
            break;

        case SLOT_SPECIAL_EXIT :
            IconHandle = LoadIcon(gOptions.hAppInst, "EXITICON");
            break;
    
        default:
            Slot->IconTotal  = (int)ExtractIcon(gOptions.hAppInst, Slot->IconFile, (UINT)-1);
            IconHandle = (HICON)ExtractIcon(gOptions.hAppInst, Slot->IconFile, Slot->IconPos);
            break;
    }

	FillRect( Dock->hdcIconCache, &rect, GetStockObject(LTGRAY_BRUSH) );
	/** Draw the blank slot first **/
	if( Slot->SlotType == SLOT_SPECIAL_SUB ){                                      
    	DrawBitmap(Dock->hdcIconCache, hSubDockButton, Slot->IconIndex, 0);
	}
	else{
    	DrawBitmap(Dock->hdcIconCache, hSlotButton, Slot->IconIndex, 0);
	}
    DrawIcon( Dock->hdcIconCache, Slot->IconIndex+3, 3, IconHandle );
}


/**********************************************************************
	Function to set all the variable fields in an emptied slot to
	their default values. Does not change ID or IconIndex fields
	since they are assigned at startup and cannot be changed.

	Also erases the Icon in the IconCache.
**********************************************************************/

void UtilEmptySlot( SLOT_ENTRY *Slot ){

    RECT	rect;
	DOCK_STRUCT *Dock;

	Dock = Slot->Dock;

    rect.top = 0;
    rect.left = Slot->IconIndex;
    rect.bottom = gMetrics.IconHeight;
    rect.right = Slot->IconIndex + gMetrics.IconWidth;
 
 	// Slot.ID must not be changed here, it must stay the same. 
	// Slot.Dock must not be changed since it points to the parent dock;
    Slot->SlotType		= SLOT_FREE;
    Slot->hwndSubDock	= NULL;
    Slot->AppName[0]	= '\0';
    Slot->CmdLine[0]	= '\0';
    Slot->RunTimeDir[0]	= '\0';
    Slot->Title[0]	    = '\0';
    Slot->StartState	= START_NORMAL;
    Slot->StartOnTop	= FALSE;
    Slot->StartSeparate	= FALSE;
    Slot->WinX			= DEF_STORE_X;
    Slot->WinY			= DEF_STORE_Y;
    Slot->WinWidth		= DEF_STORE_W;
    Slot->WinHeight		= DEF_STORE_H;
    Slot->IconFile[0]	= '\0';
    Slot->IconPos		= 0;
    Slot->IconTotal		= 0;

	// Delete the icon in the cache
	FillRect( Dock->hdcIconCache, &rect, GetStockObject(LTGRAY_BRUSH) );
}


BOOL CALLBACK UtilSetupStartedWin( HWND hwnd, LPARAM lParam ){


	/****************************************************
		Because Application Instances are not unique
		under WIN32 / Win NT we must locate the app
		just started in a diferent manner depending on
		the version of Windows in use
	****************************************************/

#ifdef WIN32
	{
    DWORD     ThreadId;
    DWORD     ProcessId = 1L;       // Must be non-NULL to recieve Process ID

    ThreadId = GetWindowThreadProcessId( hwnd, &ProcessId );
    if( ProcessId == (DWORD)lParam ){
        gOptions.bFoundWindow = TRUE;
    }
    }
#else
    if( GetWindowWord( hwnd, GWW_HINSTANCE ) == (HINSTANCE)lParam ){
        gOptions.bFoundWindow = TRUE;
    }
#endif

    if( gOptions.bFoundWindow ){

		// Window position is setup by CreateProcess if requried
        if(TmpSlot.StartState == START_STORE){
            /*****************************
            	Re-position the window 
            *****************************/
            SetWindowPos(hwnd, HWND_NOTOPMOST, TmpSlot.WinX, TmpSlot.WinY,
                    TmpSlot.WinWidth, TmpSlot.WinHeight, SWP_NOZORDER);
        }

        if( TmpSlot.StartOnTop ){
            /*****************************
                Put the window on top 
            *****************************/
            SetWindowPos(hwnd, HWND_TOPMOST, 0, 0, 0, 0, 
                         SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
        }
    }
    // returning FALSE keeps enumeration going, TRUE stops it
    return (!gOptions.bFoundWindow);
}


/*********************************************************************
	Swaps the contents of two slots
	(Only swaps non-variable fields)
	not  ID   : Gives location in dock
		 Dock : (pointer to parent) allows swapping between sub-docks
		 IconIndex : Same reason as above.
*********************************************************************/
void SwapSlots( SLOT_ENTRY *Slot1, SLOT_ENTRY *Slot2 ){

	int 		 Slot1_ID, Slot2_ID;
	int 		 Slot1_Index, Slot2_Index;
	int 		 Slot1_IconIndex, Slot2_IconIndex;
	DOCK_STRUCT *Slot1_Dock, *Slot2_Dock;
	SLOT_ENTRY   WorkingSlot;

	/****************************************
		Store original Slot Indexs for later
		These are important & cannot be moved.
	****************************************/
	Slot1_ID = Slot1->ID;
	Slot2_ID = Slot2->ID;
	Slot1_Index = Slot1->Index;
	Slot2_Index = Slot2->Index;
	Slot1_IconIndex = Slot1->IconIndex;
	Slot2_IconIndex = Slot2->IconIndex;
	Slot1_Dock = Slot1->Dock;
	Slot2_Dock = Slot2->Dock;

	/*********************************
		Copy Slot 1 to Working Slot
	*********************************/
	memcpy( &WorkingSlot, Slot1, sizeof(SLOT_ENTRY) );

	/*********************************
		Copy Slot 2 to  Slot1
	*********************************/
    memcpy( Slot1, Slot2, sizeof(SLOT_ENTRY) );

	/*********************************
		Copy Working Slot to Slot 2
	*********************************/
    memcpy( Slot2, &WorkingSlot, sizeof(SLOT_ENTRY) );

	/***************************************
		Now restore the correct values
		to the ID fields in the Slots
	***************************************/
	Slot1->ID = Slot1_ID;
	Slot2->ID = Slot2_ID;
	Slot1->Index = Slot1_Index;
	Slot2->Index = Slot2_Index;
	Slot1->IconIndex = Slot1_IconIndex;
	Slot2->IconIndex = Slot2_IconIndex;
	Slot1->Dock = Slot1_Dock;
	Slot2->Dock = Slot2_Dock;

	/*********************************************************
		Now check if either (or both) of the swapped docks
		was a sub-dock slot & setup the sub-dock's pointer
		to the parent slot correctly. Also need to re-register
		the new parent child relationship.
	*********************************************************/
	{
		DOCK_STRUCT *Dock;

		if( Slot1->SlotType == SLOT_SPECIAL_SUB ){
			Dock = GetDockStruct( Slot1->hwndSubDock );
			Dock->ParentSlot = Slot1;
			ReRegisterDock( Dock->hwndDock, Dock->ParentSlot->Dock->hwndDock );
		}
		if( Slot2->SlotType == SLOT_SPECIAL_SUB ){
			Dock = GetDockStruct( Slot2->hwndSubDock );
			Dock->ParentSlot = Slot2;
			ReRegisterDock( Dock->hwndDock, Dock->ParentSlot->Dock->hwndDock );
		}
	}

	// Force the re-loading of the icons in their new homes
	if(Slot1_Dock != NULL) UtilLoadIcon(Slot1->Dock, Slot1);
	if(Slot2_Dock != NULL) UtilLoadIcon(Slot2->Dock, Slot2);

	return;
}


void SetSubDockWinPos( DOCK_STRUCT *SubDock, BOOL bShow )
{
	RECT     rcParent, rcSlot, rcSubDock;
	DOCK_POS ParentOrientation; // SubDockOrientation;
	int      SlotIndex, SlotID;

	SlotIndex = SubDock->ParentSlot->Index;
	SlotID    = SubDock->ParentSlot->ID;

	// This function correctly positions the sub-docks relative to
	// their parent dock.
	// allowing for the current orientation

	// Get Real Coords of parent dock via slot pointer to parent
	GetWindowRect( GetParentDock(SubDock->hwndDock), &rcParent );

	// Get orientation of parent
	ParentOrientation = GetDockOrientation( GetParentDock(SubDock->hwndDock) );

	// Update the DockID since the parent Slot may have moved since
	// this sub-dock was last displayed.
	UpdateSubDockID( SubDock, SlotID );

	// Find pos of parent slot within dock (real coords)
	if(ParentOrientation == DOCK_HORZ){
		// Parent dock is horizontal
		rcSlot.left   = rcParent.left + (SlotIndex * gOptions.SlotWidth);
		rcSlot.top    = rcParent.top;
		rcSlot.right  = rcSlot.left + gOptions.SlotWidth;
		rcSlot.bottom = rcSlot.top + gOptions.SlotHeight;
	}
	else{
		// Parent dock is vertical
		rcSlot.left   = rcParent.left;
		rcSlot.top    = rcParent.top + (SlotIndex * gOptions.SlotHeight);
		rcSlot.right  = rcSlot.left + gOptions.SlotWidth;
		rcSlot.bottom = rcSlot.top + gOptions.SlotHeight;
	}

	// rcSub now contains the screen coordinates of the parent slot.

	// Check parent pos against screen size to determine which side to
	// position sub-dock on.
	if(ParentOrientation == DOCK_HORZ){
		// Orientation of the parent dock is Horizontal
		if( rcParent.top > (gMetrics.ScreenHeight/2) ){
			// Parent dock is in lower half of the screen, so pop the
			// child dock out above the parent dock
			rcSubDock.left   = rcSlot.left;
			rcSubDock.top    = rcSlot.top - (SubDock->SlotCount*gOptions.SlotHeight);
			rcSubDock.right  = rcSlot.right;
			rcSubDock.bottom = rcSlot.top;
		}
		else{
			// Parent dock is in upper half of the screen, so pop the
			// child dock out below the parent dock
			rcSubDock.left   = rcSlot.left;
			rcSubDock.top    = rcSlot.bottom;
			rcSubDock.right  = rcSlot.right;
			rcSubDock.bottom = rcSlot.bottom  + (SubDock->SlotCount*gOptions.SlotHeight);
		}
	}
	else{
		// Orientation of the parent dock is Vertical
		if( rcParent.left > (gMetrics.ScreenWidth/2) ){
			// Parent dock is in right half of the screen, so pop the
			// child dock out to the left of the parent dock
			rcSubDock.left   = rcSlot.left - (SubDock->SlotCount*gOptions.SlotWidth);
			rcSubDock.top    = rcSlot.top;
			rcSubDock.right  = rcSlot.left;
			rcSubDock.bottom = rcSlot.bottom;
		}
		else{
			// Parent dock is in left half of the screen, so pop the
			// child dock out to the right of the parent dock
			rcSubDock.left   = rcSlot.right;
			rcSubDock.top    = rcSlot.top;
			rcSubDock.right  = rcSlot.right  + (SubDock->SlotCount*gOptions.SlotWidth);;;
			rcSubDock.bottom = rcSlot.bottom;
		}
	}

	if(bShow){
		// Now show the sub dock window
		SetDockPoppedOut( SubDock->hwndDock );
    	SetWindowPos(SubDock->hwndDock,
//	                 HWND_NOTOPMOST,
                     NULL,
    	             rcSubDock.left,
        	         rcSubDock.top,
            	     rcSubDock.right - rcSubDock.left + 2,
                	 rcSubDock.bottom - rcSubDock.top + 2,
	                 SWP_NOACTIVATE | SWP_SHOWWINDOW | SWP_NOZORDER);
	}
	else {
		// Now Hide the sub dock window
		SetDockPoppedIn( SubDock->hwndDock );
    	SetWindowPos(SubDock->hwndDock,
//                   HWND_NOTOPMOST,
                     NULL,
    	             rcSubDock.left,
        	         rcSubDock.top,
            	     rcSubDock.right - rcSubDock.left + 2,
                	 rcSubDock.bottom - rcSubDock.top + 2,
	                 SWP_HIDEWINDOW | SWP_NOZORDER);
	}
}

/**************************************************************
    Process a drag and drop event arriving in a dock.
**************************************************************/
void ProcessDroppedFiles( DOCK_STRUCT *Dock, HANDLE hDrop){

    POINT          DropPoint;
	UINT		   nNumDropped;
    char           DroppedFileName[MAX_FPATH_LEN];
	char 		   CommandLine[MAX_CMDLINE_LEN];
    SLOT_ENTRY	  *SlotHit;
    UINT           FileIndex;
    int            SlotIndex, nDropSlot;

    /***********************************************************
       Find out if the drop point was in the client area.
       This must be the case, or an error has occurred.
    ***********************************************************/
	if( ! DragQueryPoint(hDrop, (LPPOINT)&DropPoint) ){
       	DockError( Dock->hwndDock, 
       	           "Error Occurred : ProcessDroppedFiles()\n(WM_DROPFILES, Drop point not in client area).");
		return;
	}

    /*******************************************
        Find out how many files were dropped
    *******************************************/
    nNumDropped = DragQueryFile(hDrop, (UINT)-1, (LPSTR)NULL, 0);

	if( nNumDropped < 1 ){
       	DockError( Dock->hwndDock, "Error Occurred : ProcessDroppedFiles()\n(WM_DROPFILES, DragQuery returned %d).", nNumDropped);
		return;
	}

    SlotHit = FindSlotHitInDock(Dock, DropPoint.x, DropPoint.y);

    switch ( SlotHit->SlotType) {

        case SLOT_USED:
            /***********************************************************
                Build up the command line and then call the ExecSlot
                function to execute the associated program
            ***********************************************************/
            sprintf(CommandLine, "%s ", SlotHit->CmdLine);
            for (FileIndex = 0; FileIndex < nNumDropped; FileIndex++) {
                DragQueryFile(hDrop, FileIndex, DroppedFileName, MAX_FPATH_LEN);
                /************************************************
                    Check if this file has an extension, if not
                    add a trailing '.'
                ************************************************/
                AddTrailingPoint( DroppedFileName );

				// Add Filename to Commandline if it will fit
				// Quote each filename incase it contains spaces.
				if( (strlen(CommandLine) + strlen(DroppedFileName)+1) < MAX_CMDLINE_LEN ) {
                   	 strcat(CommandLine, "\"");
                     strcat(CommandLine, DroppedFileName);
                     strcat(CommandLine, "\"");
                     strcat(CommandLine, " ");
				}
				else{
            	    DockError(Dock->hwndDock, 
            	              "Too many files dropped to fit on application command line, ignoring extras.");
				 	break;
				}
            }

            ExecSlot( SlotHit, CommandLine );

            break;

        case SLOT_FREE:
            nDropSlot = SlotHit->Index;
            SlotIndex = SlotHit->Index;
            for (FileIndex = 0; FileIndex < nNumDropped; FileIndex++) {
                DragQueryFile(hDrop, FileIndex, DroppedFileName, MAX_FPATH_LEN);
				// :-)
				FillOutSlot( Dock, &Dock->Slot[SlotIndex], DroppedFileName );

                RePaintSlot(Dock, &Dock->Slot[SlotIndex], FALSE);

                WriteSlotOptions(Dock, &Dock->Slot[SlotIndex]);    // Save the new slot

                SlotIndex++;
                while ((SlotIndex != nDropSlot) && (Dock->Slot[SlotIndex].SlotType != SLOT_FREE)) {
                    SlotIndex++;
                    if (SlotIndex >= Dock->SlotCount)
                        SlotIndex = 1;
                }
                if (SlotIndex == nDropSlot)
                    break;
            }
            break;

    }

    DragFinish(hDrop);
}


void FillOutSlot( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot, char *AppName ){

UINT           Status;
char          *TmpCharPtr;
char           TmpBuffer[MAX_FPATH_LEN];

    Slot->SlotType = SLOT_USED;
	Slot->StartState = START_NORMAL;
	Slot->WinX = DEF_STORE_X;
	Slot->WinY = DEF_STORE_Y;
	Slot->WinWidth = DEF_STORE_W;
	Slot->WinHeight = DEF_STORE_H;
	strcpy(Slot->AppName, AppName);
	strcpy(Slot->RunTimeDir, AppName);
	strcpy(Slot->IconFile, AppName);
	Slot->IconPos = 0;

	/************************************************
		Set Run Time Dir to be same path as filename
	************************************************/
	TmpCharPtr = strrchr(Slot->RunTimeDir, '\\');
	if (*(TmpCharPtr-1) == ':')
		*(TmpCharPtr+1) = '\0';
	else
    	*TmpCharPtr = '\0';

    /*****************************************************************
    	Check What type of file has been dropped, is it a .PIF, .COM
		or a .BAT ?
    *****************************************************************/
    if (!stricmp(&AppName[strlen(AppName) - 4], ".PIF") ||
    	!stricmp(&AppName[strlen(AppName) - 4], ".COM") ||
        !stricmp(&AppName[strlen(AppName) - 4], ".BAT")) {


			Status = (UINT)FindExecutable("progman.exe",
	                                      NULL,
		                                  Slot->IconFile);
	        if( Status <= 31 ){
        	     DockError(Dock->hwndDock, "Program Manager not found, cannot select default Icon.");
            }
            Slot->IconPos = 1;
            // Load the Icon into the slot's own bitmap
            UtilLoadIcon( Dock, Slot );
	}
    /******************************************************************
    	Is is a .EXE, if so is it a Windows EXE or a DOS EXE ?
    ******************************************************************/
    else if (!stricmp(&AppName[strlen(AppName) - 4], ".EXE")) {
    	// Load the Icon into the slot's own bitmap
        UtilLoadIcon( Dock, Slot );
        if (Slot->IconTotal == 0) {

	    	Status = (UINT)FindExecutable("progman.exe",
		                                  NULL,
			                              Slot->IconFile);
    		if( Status <= 31 ){
        		DockError(Dock->hwndDock, "Program Manager not found, cannot select default Icon.");
           	}

            Slot->IconPos = 1;
            // Load the Icon into the slot's own bitmap
            UtilLoadIcon( Dock, Slot );

    	}
	}
    /******************************************************************
      It must be a document file, try to find an association for it
    ******************************************************************/
    else {
        // if the document has a space in it, it must be quoted
        if( strchr( Slot->AppName, ' ' ) ){
            sprintf( "\"%s\"", Slot->AppName );
        }
        else{
		    strcpy(Slot->CmdLine, Slot->AppName);
        }
        // Read the executable name in,
        // then call SearchPath to get entire path for application.
        // Erase the AppName, since that is what we are searching for
        Status = (UINT)FindExecutable(Slot->CmdLine,
     	  			                  Slot->RunTimeDir,
                                      Slot->AppName);
                        
        // Now call find executable to ensure we have a complete
        // path to the application
        if( FindExecutable(Slot->AppName, Slot->RunTimeDir, TmpBuffer) > (HINSTANCE)32 ){
        	strcpy( Slot->AppName, TmpBuffer);
        }
                                
        // Incase this filename has no extension
        AddTrailingPoint( Slot->AppName );

        if ( (Status > 32) && (Slot->AppName[strlen(Slot->AppName)-1] != '.')) {
        	strcpy(Slot->IconFile, Slot->AppName);
            Slot->IconPos = 0;
            // Load the Icon into the slot's own bitmap
            UtilLoadIcon( Dock, Slot );
        } 
        else {
            DockError(Dock->hwndDock, 
                      "%s\nis not an executable or an associated file", 
                      Slot->CmdLine);
            Slot->SlotType = SLOT_FREE;
       }
	}

}

/******************************************************
    Function to add a trailing point to a filename
    if it has no extension.
    Thanks to Max Shesterikoff for spotting the problem
    this function solves and providing the function.
******************************************************/

void AddTrailingPoint( char *path ){

    char *p = strrchr( path, '.' );

    if(!p || !(p > strrchr( path, '\\' ))){
        strcat( path, "." );
    }
}

/*********************************************************
    Adjust the Security privileges to allow a shutdown
    or reboot command (WIN32 ONLY)
*********************************************************/
#ifdef WIN32
void GetShutdownPrivilege (void){
    
HANDLE hToken;              /* handle to process token */ 
TOKEN_PRIVILEGES tkp;       /* ptr. to token structure */ 
 
/* 
 * Get the current process token handle 
 * so we can get shutdown privilege. 
 */ 
 
OpenProcessToken(GetCurrentProcess(), 
        TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken);
 
/* Get the LUID for shutdown privilege. */ 
LookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, 
        &tkp.Privileges[0].Luid); 
 
tkp.PrivilegeCount = 1;  /* one privilege to set    */ 
tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED; 
 
/* Get shutdown privilege for this process. */ 
 
AdjustTokenPrivileges(hToken, FALSE, &tkp, 0, 
    (PTOKEN_PRIVILEGES) NULL, 0); 
}
#endif

void DockError( HWND hwndParent, char *Format, ... ){

    char    TmpBuffer[1024];
    va_list vaArgList;

    va_start( vaArgList, Format );
    _vsnprintf( TmpBuffer, 1024, Format, vaArgList );

    MessageBox(hwndParent, TmpBuffer, "FreeDock Error", MB_OK | MB_ICONSTOP);

    va_end( vaArgList );
}

void DockInfo( HWND hwndParent, char *Format, ... ){

    char    TmpBuffer[1024];
    va_list vaArgList;

    va_start( vaArgList, Format );
    _vsnprintf( TmpBuffer, 1024, Format, vaArgList );

    MessageBox(hwndParent, TmpBuffer, "FreeDock Information", MB_OK | MB_ICONINFORMATION);

    va_end( vaArgList );
}

BOOL ChDir( const char *dir ){
    if( dir && dir[0] != '\0' ){

        if( _chdir( dir ) ){
            return FALSE;
        }
        if( dir[1] == ':' ){
            _chdrive( toupper( dir[0] ) - 'A' + 1);
        }
    return TRUE;
    }
 return FALSE;
}
