/****************************************************************************

    Module  : DrawBmp.c

              This module contains all the functions related to drawing
              the slots onto the Dock.

*****************************************************************************/

/****************************************************************************

    FUNCTION: DrawBitmap(HDC hdc, HBITMAP hBitmap, short xStart, short yStart)

    VERSION : v1.0

    PURPOSE : To draw the bitmap references by hBitmap, at the
              coordinated specified by (xStart,yStart), in the device
              context specified by hdc.

    PARAMETERS:  hdc - Handle to the device context in which the bitmap
                       is to be drawn.
                 hBitmap - Handle for the bitmap to be drawn.
                 xStart - X coordinate (upper left corner) of the bitmap
                 yStart - Y coordinate (lower right corner) of the bitmap

    COMMENTS:

****************************************************************************/


#include <windows.h>            /* include standard windows header */
#include "freedock.h"
#include "dock_reg.h"
/**********************************************************
    Global Data for the various system metrics we need
**********************************************************/
extern GLOBAL_METRICS gMetrics;
extern GLOBAL_OPTIONS gOptions;

void DrawBitmap(HDC hdc, HBITMAP hBitmap, int xStart, int yStart)
{
    BITMAP          bm;         /* Bitmap structure, used to find dimensions
                                 * of bitmap to be drawn */
    HDC             hdcMem;     /* Handle to a Memory device context, ie an
                                 * area of memory into which the bitmap can
                                 * be fetched, before being drawn */
    POINT           ptSize;     /* a point struct to contain the bitmap size*/
    POINT           ptOrg;      /* a point struct to contain the origin of
                                 * the bitmap in the hdcMem */

    hdcMem = CreateCompatibleDC(hdc);   /* create memory storage for the
                                         * bitmap */
    SelectObject(hdcMem, hBitmap);      /* Select the bitmap into the memory
                                         * device context */

    SetMapMode(hdcMem, GetMapMode(hdc));        /* Set the mapping mode of
                                                 * the hdcMem to be the same
                                                 * as the destination device
                                                 * context */

    /*
     * load the bitmap into a Bitmap struct so the dimensions can be
     * determined
     */
    GetObject(hBitmap, sizeof(BITMAP), (LPSTR) & bm);

    ptSize.x = bm.bmWidth;   /* store the Width and height of the bitmap */
    ptSize.y = bm.bmHeight;
    DPtoLP(hdc, &ptSize, 1); /* convert Device Points into Logical Points */

    ptOrg.x = 0;               /* set the origin of the source bitmap */
    ptOrg.y = 0;
    DPtoLP(hdcMem, &ptOrg, 1); /* convert Device Points into Logical Points */

    /* Blit the source bitmap into the destination context */
    BitBlt( hdc, xStart, yStart, ptSize.x, ptSize.y, hdcMem,
            ptOrg.x, ptOrg.y, SRCCOPY);

    DeleteDC(hdcMem);          /* delete the memory context & free the
                                * memory used */
}


/***************************************************************
    PaintSlot paints an individual slot, the parameters
    supplied are the slot to paint (nSlot) and the hdc for
    the dock window.

***************************************************************/

void PaintSlot(HDC hdc, DOCK_STRUCT *Dock, SLOT_ENTRY *Slot)
{
    if (Slot->Index < 0 ) return;
    if (Slot->Index >= Dock->SlotCount) return;

    if (Slot->SlotType != SLOT_FREE) {
        if ( GetDockOrientation(Dock->hwndDock) == DOCK_HORZ) {
            UtilDrawIcon( hdc, Dock, Slot, Slot->Index*gOptions.SlotWidth,
                          0, TRUE );
        } 
        else {
            UtilDrawIcon( hdc, Dock, Slot, 0,
                          Slot->Index*gOptions.SlotHeight, TRUE );
        }
    }
}


/*************************************************************
    RePaint slot invalidates the client area of a slot within
    the dock, forcing windows to send a WM_PAINT message for
    that area-> Therefore forcing a slot to be re-drawn
*************************************************************/

void RePaintSlot(DOCK_STRUCT *Dock, SLOT_ENTRY *Slot, BOOL bErase)
{
    RECT            rect;

    if (Slot == NULL) return;
    if (Slot->Index < 0 ) return;
    if (Slot->Index >= Dock->SlotCount) return;

    if (GetDockOrientation(Dock->hwndDock) == DOCK_HORZ) {
        rect.top = 0;
        rect.left = (Slot->Index * gOptions.SlotWidth);
        rect.bottom = gOptions.SlotHeight;
        rect.right = (Slot->Index * gOptions.SlotWidth) + gOptions.SlotWidth;
    } 
    else {
        rect.top = (Slot->Index * gOptions.SlotHeight);
        rect.left = 0;
        rect.bottom = (Slot->Index * gOptions.SlotHeight) +
                      gOptions.SlotHeight;
        rect.right = gOptions.SlotWidth;
    }

    InvalidateRect(Dock->hwndDock, &rect, bErase);
}


/*********************************************************************
    BlankSlot function is used in the draging of a slot around the
    dock. Used in DockWinProc (Dockwin.c) in the processing of a
    WM_MOUSEMOVE message when a slot is being dragged to a new
    position in the dock. The function draws a blank area in the
    dock when a slot is dragged from its starting position.
*********************************************************************/

void BlankSlot( HDC hdc, DOCK_STRUCT *Dock, SLOT_ENTRY *Slot ){

    RECT    rect;

    if (Slot->Index < 0 ) return;
    if (Slot->Index >= Dock->SlotCount) return;

    if (GetDockOrientation(Dock->hwndDock) == DOCK_HORZ) {
        rect.top = 0;
        rect.left = (Slot->Index * gOptions.SlotWidth);
        rect.bottom = gOptions.SlotHeight;
        rect.right = (Slot->Index * gOptions.SlotWidth) + gOptions.SlotWidth;
    } 
    else {
        rect.top = (Slot->Index * gOptions.SlotHeight);
        rect.left = 0;
        rect.bottom = (Slot->Index * gOptions.SlotHeight) +
                      gOptions.SlotHeight;
        rect.right = gOptions.SlotWidth;
    }

    FillRect( hdc, &rect, GetStockObject(LTGRAY_BRUSH) );
}


/****************************************************************
    Function to draw a button/Blank slot with the Icon on it.
    This function is used by the PaintSlot() funtion to do the
    actual drawing of the slot (icon & button).
    When this function is first called it loads the bitmap for
    the Blank Slot Button, this is released again by calling
    the function with the hdcDest parameter set to NULL
    X & Y are the X,Y Coordinates of the location of the slot
    to be drawn within the dock.

    The bScale flag indicates if the icon is to be drawn
    to the current scale of the dock, or at 1:1
****************************************************************/

void UtilDrawIcon( HDC hdcDest, DOCK_STRUCT *Dock, SLOT_ENTRY *Slot,
                   int X, int Y, BOOL bScale ){

  if (!bScale){
    /** Now copy the relevant Icon from the IconCache **/
    BitBlt( hdcDest, X, Y, DEF_SLOT_WIDTH, DEF_SLOT_HEIGHT, 
            Dock->hdcIconCache, Slot->IconIndex, 0, SRCCOPY);
  }
  else{  
    /** Now copy & Scale the relevant Icon from the IconCache **/
    StretchBlt( hdcDest, X, Y, gOptions.SlotWidth, gOptions.SlotHeight,
                Dock->hdcIconCache, Slot->IconIndex, 0, 
                DEF_SLOT_WIDTH, DEF_SLOT_HEIGHT, SRCCOPY );
  }
} 
