/**************************************************************************

    Module  : FreeDock.h

**************************************************************************/

#define VersionText "v3.0"    // Current version of FreeDock


#ifdef  WIN32
#define OS_Bits "(32 Bit)"     // WIN32 (NT or Windows 95, or WIN32s system)
#else
#define OS_Bits "(16 Bit)"     // Windows 3.x system
#endif

#define STRICT

#define FREEDOCK_SEMAPHORE "FreeDockInstanceCount"

#define MAX_DOCKS  10    // Maximum number of instances of FreeDock allowed
                        // 10 for no good reason

/*****************************************
    Define the various constants used
*****************************************/

#define DEFAULT_DOCK_SIZE   10  // defualt number of slots (if not in .ini)
#define MAX_FPATH_LEN      128  // Max length of a path
#define TMP_BUFF_LEN        32  // Length of temp string buffer in main
#define MAX_CMDLINE_LEN   1024  // Max len of command line
#define DEF_SLOT_WIDTH      38  // 32x32 icon + border of 3 on each side
#define DEF_SLOT_HEIGHT     38  // ditto (the size of the blank button BitMap)
#define MAX_SLOT_SCALE      76  // Max size for slots when scaling the dock 
                                // (double normal size) 2:1
#define MIN_SLOT_SCALE       8  // Min size for Slots when scaling the dock 
                                // (probably too small anyway)    1:4ish
#define DEF_SLOT_SCALE      38  // Default size for the slots in the dock,
                                // this is 1:1

#define DEF_MAX_SLOTCOUNT   32  // This is the default size of array which 
                                // is malloc'd to hold the slots in the
                                // Dock, it can be altered by specifying 
                                // a smaller/larger value in the .ini file
                                // as required.

#define DEF_STORE_X          0  // Initial non-fatal values for
#define DEF_STORE_Y          0  // stored window position, incase
#define DEF_STORE_W        400  // user selects "stored" without
#define DEF_STORE_H        200  // then grabbing a window

#define MAIL_TIMER           1  // Timer ID for the mail checking slot

#define TITLE_TIMER          2  // Timer ID for the pop-up Titles
#define DEF_TITLE_TIMEOUT 1000  // Default Timer value (1 second)

#define MAX_DOCK_ID_LEN    256  // Max length of a dock's id string 
                                // (used to retrieve slots from .INI file

#define MAX_SUB_DOCKS       32  // Maximum number of sub-docks allowed.

typedef enum  _SlotType{ SLOT_FREE,           //  empty
                         SLOT_USED,           //  in use
                         SLOT_SPECIAL_TITLE,  //  (Title Icon)
                         SLOT_SPECIAL_EXIT,   //  (Exit Windows)
                         SLOT_SPECIAL_CLOCK,  //  (Clock slot)
                         SLOT_SPECIAL_MAIL,   //  (Mail Check)
                         SLOT_SPECIAL_SUB     //  (Has a sub-dock)
}SLOT_TYPE;

typedef enum  _StartType{ START_MINIMUM,      // "Start Minimised"
                          START_MAXIMUM,      // "Start Maximised"
                          START_NORMAL,       // "Start Normal"
                          START_STORE         // "Start In Stored Postion"
}START_TYPE;

typedef enum _DockPos{ DOCK_HORZ = 0,         // Horizontal
                       DOCK_VERT = 1          // Vertical
}DOCK_POS ;

typedef enum _DockType{ ROOT_DOCK,            // Dock is the root Dock
                        SUB_DOCK              // Dock is a sub-Dock
}DOCK_TYPE;

typedef enum _ShutdownType{ FD_LOGOFF,           // Log off
                            FD_SHUTDOWN,         // Shutdown
                            FD_REBOOT,           // Shutdown & Reboot
                            FD_LOCKUP            // Lock Work Station
}SHUTDOWN_TYPE;

/*************************************
    Predeclare typedefs to avoid
    compiler complaints.
*************************************/
struct _SLOT_ENTRY;

/******************************************************************
    Data structure to hold all global system metrics used
******************************************************************/
typedef struct _GlobalMetrics{
        int          ScreenWidth;       // Width of Screen
        int          ScreenHeight;      // Height of screen
        int          IconWidth;         // Width of Icons
        int          IconHeight;        // Height of Icons
        int          DlgTitleHeight;    // Height of the title bar
        int          DlgFrameWidth;     // Width of dialog frame
        int          DlgFrameHeight;    // Height of dialog frame
        int          DockFrameWidth;    // Width of Dock Window Frame
        int          DockFrameHeight;   // Height of Dock Window Frame
}GLOBAL_METRICS;

/******************************************************************
    Data structure to hold all global Dock Options used
    Global means : applies to root dock and all sub-docks
******************************************************************/
typedef struct _GlobalOptions{
    char     szAppName[MAX_FPATH_LEN];   // Name of the Application
    int      RootDockID;                 // Root Dock ID
    DOCK_POS RootOrientation;            // Orientation of Root Dock
    LONG     RootDockLeft;
    LONG     RootDockTop;
    BOOL     PopInOnExec;                // Pop-in Docks when executing slot ?
    BOOL     ClockActive;                // is Clock Built-in Active ?
    BOOL     WinExitActive;              // is Exit Windows Built-in Active ?
    WORD     TitleTimer;                 // Timeout value for title timer.
    BOOL     TitlesActive;               // Are pop-up titles active ?
    SHUTDOWN_TYPE ShutdownAction;        // Action to take on win exit (WIN 32)
    BOOL     WinExitConfirm;             // Ask user to confirm Exit windows ?
    BOOL     MailActive;                 // is the Mail Built-in Active ?
    BOOL     MaxView;                    // is MaxView option active ?
    BOOL     AlwaysOnTop;                // is Dock set to always on top ?
    BOOL     SingleClickStart;           // start apps with a single click ?
    BOOL     HideDlgOnGrab;              // hide App Options dialog when
                                         // grabbing a window ?
    BOOL     DockLocked;                 // Are the slots locked in the dock?
    char     MailPath[MAX_FPATH_LEN];    // Path to Mail Box
    int      MailFreq;                   // Frequency to check mail box
                                         // (minutes)
    BOOL     MailSound;                  // Beep when mail arrives ?
    long     MailBoxSize;                // Last known size of mail box
    long     MailBoxDateTime;            // Last known date stamp of mail box
    BOOL     bMailInBox;                 // Is there mail in the mailbox ?
    int      SlotWidth;                  // Slot Width (Used in scaling dock)
    int      SlotHeight;                 // Slot Height(Used in scaling dock)
    int      SlotScale;                  // Slot Scale (same as Width&Height)
                                         // Width & Height used in code
                                         // to improve understanding.
    int      MaxDockSize;                // Maximum Size of the dock
                                         // (# slots malloced at start)
    int      SleepTime;                  // Period to sleep before enumerating
                                         // windows (NT Only)
    BOOL     FixedTitle;                 // Is the window title fixed or does it
                                         // contain the instance No.
    BOOL     bFoundWindow;               // Used when enumerating windows.
    HANDLE   hAppInst;                   // handle to application instance
}GLOBAL_OPTIONS;

/********************************************************************
    Data structure to hold main dock options
********************************************************************/

typedef struct _DOCK_STRUCT{
    HWND                hwndDock;
    DOCK_TYPE           DockType;
    char                DockID[MAX_DOCK_ID_LEN];
    struct _SLOT_ENTRY *ParentSlot;
    int                 SlotCount;
    HBITMAP             hbmIconCache;     // Hndl to Icon Cache Bitmap
    HBITMAP             hbmOldIconCache;  // Hndl to Original Icon Cache BMP
    HDC                 hdcIconCache;     // Hndl to Icon Cache DC
    struct _SLOT_ENTRY *Slot;             // Pointer to Slot array for dock.
}DOCK_STRUCT;


/******************************************************************
    Data structure to hold all data relating to a single slot 
******************************************************************/

typedef struct _SLOT_ENTRY{
    SLOT_TYPE    SlotType;                   // type of the slot
    DOCK_STRUCT *Dock;                       // Pointer to the parent dock
    HWND         hwndSubDock;                // Pointer to attached sub-dock
                                             // (if any) otherwise NULL
    int          ID;                         // Slot ID Number
    int          Index;                      // Slot's index in slot array
    char         Title[MAX_FPATH_LEN];       // App title / name
    char         AppName[MAX_FPATH_LEN];     // App pathname
    char         CmdLine[MAX_CMDLINE_LEN];   // App command line
    char         RunTimeDir[MAX_FPATH_LEN];  // App Runtime dir
    START_TYPE   StartState;                 // App Start State
    BOOL         StartOnTop;                 // App started on top ?
    BOOL         StartSeparate;              // App start in separate mem.
    int          WinX;                       // App window x position
    int          WinY;                       // App window y position
    int          WinWidth;                   // App window width
    int          WinHeight;                  // App window height
    char         IconFile[MAX_FPATH_LEN];    // App icon file name
    int          IconPos;                    // Position in file of app icon
    int          IconTotal;                  // Total # Icons in icon file
    int          IconIndex;                  // Index into the icon Cache
}SLOT_ENTRY;

/**************************************************************************
 *  Declare function prototypes for all of the procs associated with a    *
 *  specific window.                                                      *
 **************************************************************************/

/******************
    DrawBmp.c
******************/
void DrawBitmap(HDC hdc, HBITMAP hBitmap, int xStart, int yStart);
void PaintSlot(HDC hdc, DOCK_STRUCT *DockOptions, SLOT_ENTRY *Slot);
void RePaintSlot(DOCK_STRUCT *DockOptions, SLOT_ENTRY *Slot, BOOL bErase);
void BlankSlot( HDC hdc, DOCK_STRUCT *DockOptions, SLOT_ENTRY *Slot);
void UtilDrawIcon( HDC hdcDest, DOCK_STRUCT *DockOptions, SLOT_ENTRY *Slot,
                   int X, int Y, BOOL bScale );

/********************
    DockUtil.c
********************/
SLOT_ENTRY *FindSlotHitInDock(DOCK_STRUCT *Dock, int XHit, int YHit);
SLOT_ENTRY *FindSlotHitInSubDocks(DOCK_STRUCT *Dock, int XHit, int YHit);
BOOL IsCursorOutOfDock(DOCK_STRUCT *Dock, int X, int Y);
BOOL CALLBACK UtilSetupStartedWin( HWND hwnd, LPARAM lParam );
void ExecSlot( SLOT_ENTRY *Slot, char *CmdLine );
void SetRootDockPos( DOCK_STRUCT *Dock, DOCK_POS DockPos );
void UtilLoadIcon( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot );
void UtilEmptySlot( SLOT_ENTRY *Slot );
void SwapSlots( SLOT_ENTRY *Slot1, SLOT_ENTRY *Slot2 );
void SetSubDockWinPos( DOCK_STRUCT *Dock, BOOL bShow );
void ProcessDroppedFiles( DOCK_STRUCT *Dock, HANDLE hDrop);
void FillOutSlot( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot, char *AppName );
void AddTrailingPoint( char *path );
void GetShutdownPrivilege (void);
void DockError( HWND hwndParent, char *Format, ... );
void DockInfo( HWND hwndParent, char *Format, ... );
BOOL ChDir( const char *dir );

/****************************
    Dialogs.c
****************************/
BOOL FAR PASCAL AppOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                  LPARAM lParam);
BOOL FAR PASCAL MainOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                   LPARAM lParam);
BOOL FAR PASCAL SubDockOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                      LPARAM lParam);
BOOL FAR PASCAL WinExitOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                      LPARAM lParam);
BOOL FAR PASCAL MailOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                   LPARAM lParam);
BOOL FAR PASCAL ClockOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                    LPARAM lParam);
BOOL FAR PASCAL AboutDockDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                 LPARAM lParam);
BOOL FAR PASCAL ChooseIconDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                  LPARAM lParam);
BOOL FAR PASCAL PreviewerDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam,
                                 LPARAM lParam);
BOOL FAR PASCAL BrowseIconFile( HWND hwnd, char *FileName );
BOOL FAR PASCAL BrowseFileName( HWND hwnd, char *FileName );

/****************************
    INI_File.c
****************************/
void ReadGlobalOptions( void );
void ReadDockOptions( DOCK_STRUCT *Dock );
void ReadSlotOptions( DOCK_STRUCT *Dock );
void WriteGlobalOptions( void );
void WriteDockOptions( DOCK_STRUCT *Dock );
void WriteSlotOptions( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot );
void DeleteSlotOptions( DOCK_STRUCT *Dock, SLOT_ENTRY *Slot );
void ConvertINIFile(void);

/****************************
    DockWin.c
****************************/
long FAR PASCAL DockWinProc(HWND, UINT, WPARAM, LPARAM);

/***************************************************************
 * Definitions for the Dialog boxes                            *
 * These values are included from the resource header file     *
 ***************************************************************/
#include "resource.h"
