/****************************************************************************

    Module  : Titles.c

    Changes : 3.0 Beta 6 : Created this module
              3.0 beta 7 : Fixed bug in clipping od popped up title window,
              which caused title window to be incorrectly positioned when it
              was clipped for the bottom of the screen
*****************************************************************************/

#include<windows.h>
#include<string.h>
#include"freedock.h"
#include"dock_reg.h"
#include"titles.h"

extern GLOBAL_METRICS gMetrics;
extern GLOBAL_OPTIONS gOptions;

// Exported Global Variables
BOOL  bTitleShown;   // Global "is title window on screen variable"

// Private variables
static HWND   hwndTitleWindow, hwndRoot = NULL;
static BOOL   bWindowCreated, bDisableTitles = FALSE;
static BOOL   bTimerActive = FALSE;
static char  TitleText[MAX_TITLE_LEN] = {0,};

// Private Functions
void ShowTitle( int x, int y, char *Title );
void HideTitle( void );
BOOL ShowTitleUnderCursor( void );
void StartTitleTimer();
void StopTitleTimer();

// Predefine Title Window procedure 
long FAR PASCAL TitleWinProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam);
// Predefine Timer call back procedure
VOID CALLBACK TimerProc( HWND hwnd, UINT msg, UINT id, DWORD dwTime );

void InitTitleWindow( HANDLE hInstance, HWND hwndRoot ){

    WNDCLASS wcTitleClass = {0,};

    bWindowCreated = FALSE;
    bTitleShown     = FALSE;
    bDisableTitles  = FALSE;

    /**********************************************
     * define and register the title window class *
     * Specify all non-zero fields                *
     **********************************************/
    wcTitleClass.lpfnWndProc = (WNDPROC) TitleWinProc;
    wcTitleClass.hInstance = hInstance;
    wcTitleClass.hbrBackground = GetStockObject(LTGRAY_BRUSH);
    wcTitleClass.hCursor = LoadCursor(NULL, IDC_ARROW);
    wcTitleClass.lpszClassName = TitleClassName;

    if(!RegisterClass(&wcTitleClass)){
        DockError( NULL, "Error Registering Title Window Class" );
        return;
    }

    hwndTitleWindow = CreateWindow(
                            TitleClassName,       /* window class name    */
                            NULL,                 /* window caption (none)*/
                            WS_VISIBLE | WS_POPUP | WS_BORDER,  /* style  */
                            0,                    /* initial x position   */
                            0,                    /* initial y position   */
                            1,                    /* initial x size       */
                            1,                    /* initial y size       */
                            hwndRoot,             /* parent window handle */
                            NULL,                 /* window menu handle   */
                            hInstance,            /* program inst handle  */
                            NULL                  /* creation parameters  */
                            );

    if( hwndTitleWindow == NULL ){    
              DockError( NULL, "Error, creating Title Window" ); 
              return;
    }

    bWindowCreated = TRUE;
}


void DestroyTitleWindow( void ){
    
    if( bWindowCreated ){
        StopTitleTimer();
        DestroyWindow( hwndTitleWindow );
    }
}

void ShowTitle( int x, int y, char *Title ){

#ifdef WIN32
    SIZE sExtent;
#else
    DWORD dwExtent;
#endif
    WORD  wHeight, wWidth;    
    HDC   hdcTitleWindow;
    RECT  rect;

    if( bDisableTitles ) return;

    memset( TitleText, '\0', MAX_TITLE_LEN );
    strncpy( TitleText, Title, MAX_TITLE_LEN - 1 );
    
    // Compute size of text in window    
    hdcTitleWindow = GetDC( hwndTitleWindow );
#ifdef WIN32
    GetTextExtentPoint32( hdcTitleWindow, TitleText, strlen( TitleText ), &sExtent );
    wHeight = (WORD)sExtent.cy;
    wWidth  = (WORD)sExtent.cx;
#else
    dwExtent = GetTextExtent( hdcTitleWindow, TitleText, strlen( TitleText ) );
    wHeight = HIWORD( dwExtent );
    wWidth  = LOWORD( dwExtent );
#endif
    ReleaseDC( hwndTitleWindow, hdcTitleWindow );

    if( (wWidth == 0) || (wHeight == 0) ){
        // Return silently and don't display non-existant title.
//        DockError( NULL, "Error, GetTextExtent Returned 0 for height or width." ); 
        HideTitle();        // Hide any currently shown title
        return;       
    }
    // Correct for window border
    wWidth  += 4;
    wHeight += 2;
    
    // Move out from under cursor
    x += 10;
    y += 10;

    // Limit x & y to the screen
    if ( (x + wWidth) >= gMetrics.ScreenWidth ){
        x = gMetrics.ScreenWidth - wWidth;
    }
    if ( (y + wHeight) >= gMetrics.ScreenHeight ){
        y = gMetrics.ScreenHeight - wHeight;
    }

    SetWindowPos(hwndTitleWindow,
                 HWND_NOTOPMOST,
                 x,
                 y,
                 wWidth,
                 wHeight,
                 SWP_NOACTIVATE );
    GetClientRect( hwndTitleWindow, &rect );
    InvalidateRect( hwndTitleWindow, &rect, TRUE );
    ShowWindow( hwndTitleWindow, SW_SHOWNA );
    bTitleShown = TRUE;
}

void HideTitle( void ){
    
    if( bTitleShown ){
        ShowWindow( hwndTitleWindow, SW_HIDE );
        bTitleShown = FALSE;
    }
}

void DisableTitles( void ){
    StopTitleTimer();
    HideTitle();
    bDisableTitles = TRUE;
}

void EnableTitles (void){
    bDisableTitles = FALSE;
    if( !gOptions.TitlesActive ) return;
    StartTitleTimer();
}


long FAR PASCAL TitleWinProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam){

    HDC         hdcWindow;
    PAINTSTRUCT ps;

    switch (message) {

        case WM_CREATE : 
             return 0;

        case WM_PAINT : 
             hdcWindow = BeginPaint( hwnd, &ps );
             TextOut( hdcWindow, 0, 0, TitleText, strlen( TitleText ) );
             EndPaint( hwnd, &ps );
             return 0;

        default :
            break;
    }

   return DefWindowProc(hwnd, message, wParam, lParam);
}

void StartTitleTimer(){
    if( bDisableTitles ) return;
    if( !bTimerActive ){
        SetTimer(hwndTitleWindow, TITLE_TIMER, gOptions.TitleTimer, (TIMERPROC)TimerProc);
        bTimerActive = TRUE;
    }
}

void StopTitleTimer(){
    if( bDisableTitles ) return;
    if( bTimerActive ){
        KillTimer( hwndTitleWindow, TITLE_TIMER );
        bTimerActive = FALSE;
    }
}

void ResetTitleTimer(){
    if( bDisableTitles ) return;
    if( bTimerActive ){
        StopTitleTimer();
        StartTitleTimer();
    }
}

// Timer procedure, caller each time a timer event occurs
VOID CALLBACK TimerProc( HWND hwnd, UINT msg, UINT id, DWORD dwTime ){
     ShowTitleUnderCursor( );
}


BOOL ShowTitleUnderCursor( void ){

       SLOT_ENTRY *SlotUnderCursor;
static SLOT_ENTRY *LastSlotUnderCursor = NULL;

    POINT       pCursorPos;
    int         x, y, x_c, y_c ;
    HWND        hwndUnderCursor;

    if( bDisableTitles ) return FALSE;

    GetCursorPos( &pCursorPos );
    // Save Screen coords
    x = (int)pCursorPos.x;
    y = (int)pCursorPos.y;

    // Find Window under cursor
    hwndUnderCursor = WindowFromPoint( pCursorPos );
    
    if( IsDockWindow( hwndUnderCursor ) ){
        ScreenToClient( hwndUnderCursor, &pCursorPos );
        // save client coords
        x_c = (int)pCursorPos.x;
        y_c = (int)pCursorPos.y;
        SlotUnderCursor = FindSlotHitInSubDocks( GetDockStruct(hwndUnderCursor), x_c, y_c);

        if( SlotUnderCursor == LastSlotUnderCursor ){
            return (bTitleShown);
        }
        LastSlotUnderCursor = SlotUnderCursor;

        if( (SlotUnderCursor != NULL) ){
            switch( SlotUnderCursor->SlotType ){
                case SLOT_FREE :
                     ShowTitle( x, y, "Empty Slot" );
                     break;
                case SLOT_USED :
                case SLOT_SPECIAL_TITLE :
                case SLOT_SPECIAL_EXIT :
                case SLOT_SPECIAL_CLOCK :
                case SLOT_SPECIAL_MAIL :
                case SLOT_SPECIAL_SUB :
                     ShowTitle( x, y, SlotUnderCursor->Title );
                     break;
                default:
                     ShowTitle( x, y, "Unknown Slot Type" );
                     break;
            }
        }
        else if (SlotUnderCursor == NULL){
            // out of dock so delete title
            if( bTitleShown ){
                HideTitle();
            }
        }
    }
    else{
        //Cursor not in a dock window
        if( bTitleShown ){
            LastSlotUnderCursor = NULL;
            HideTitle();
        }
    }
    return (bTitleShown);
}
