/***************************************************************************
 * Comment.Cmd 1.09 (14-Feb-1996 18:25:52)
 *
 * Copyright 1996 Christopher J. Madsen
 *
 * Comment.Cmd is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Comment.Cmd is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Christopher J. Madsen        <ac608@yfn.ysu.edu>
 * 3222 Darby Ln.
 * Denton, TX  76207-1305
 *
 * Display or change file comments
 ***************************************************************************/
'@echo off'

IF RxFuncQuery('SysLoadFuncs') THEN DO
  CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
  CALL SysLoadFuncs
END

PARSE SOURCE . . commandPath
commandName = filespec('n',commandPath)

versionStr = 'Comment.Cmd version 1.09 (14-Feb-1996)'

/***************************************************************************
 * Parse options:
 ***************************************************************************/
PARSE ARG commandLine

longOpts = '?:HELP: A:ALL-FILES: D:DELETE: F:FORCE: Q:QUIET: S:SUBJECT:'
PARSE VALUE rxGetOpt(commandLine,'',longOpts) WITH resultCode commandLine

IF resultCode <> 0 THEN DO
  SAY commandName': ' commandLine
  SIGNAL Usage
END

allFiles    = 0                 /* Do not list files without comments */
deleting    = 0                 /* We are not deleting comments */
eaHeader    = 'DFFF00000100FDFF'x
eaDesc      = 'comments'        /* The EA name to use in messages */
eaName      = '.comments'       /* The EA we are manipulating */
forceUpdate = 0                 /* Don't replace existing comments */
verbose     = 1                 /* Display progress messages */

globalVars = 'allFiles eaHdrLen eaHeader eaDesc eaName forceUpdate verbose'

DO FOREVER
  PARSE VAR commandLine argument commandLine
  SELECT
    WHEN argument = '--' THEN LEAVE
    WHEN argument = '-A' THEN
      allFiles = 1              /* List all files */
    WHEN argument = '-D' THEN
      deleting = 1              /* Delete comments from files */
    WHEN argument = '-F' THEN
      forceUpdate = 1           /* Replace existing comments */
    WHEN argument = '-Q' THEN
      verbose = 0               /* Don't display progress messages */
    WHEN argument = '-S' THEN DO
      /* Work with the subject attribute */
      eaHeader = 'FDFF'x
      eaDesc = 'subjects'
      eaName = '.subject'
    END /* when --subject */
    WHEN argument = '--LICENSE' THEN
      SIGNAL License
    WHEN argument = '--VERSION' THEN DO
      SAY versionStr
      EXIT
    END /* when --version */
    OTHERWISE
      IF argument <> '-?' THEN
        SAY commandName': Unrecognized option `'argument"'"
      SIGNAL Usage
  END /* select */
END /* do while more options */

eaHdrLen = LENGTH(eaHeader)     /* Remember length of EA header */

/***************************************************************************
 * Parse filenames and perform action:
 ***************************************************************************/
fileSpec = nextArg()

IF fileSpec = '' THEN SIGNAL Usage

IF deleting THEN DO
  IF verbose THEN do
    CALL CHAROUT ,,
      'Are you sure you want to delete the' eaDesc 'from those files (Y/N)? '
    key = SysGetKey()
    SAY
    IF key <> 'Y' & key <> 'y' THEN EXIT 1
    SAY 'Deleting file' eaDesc'...'
  END /* if verbose */
  DO WHILE fileSpec <> ''
    IF SysFileTree(fileSpec, 'file', 'O') THEN SIGNAL ERROR
    CALL deleteComments
    fileSpec = nextArg()
  END /* DO */
END /* if deleting */
ELSE DO
  comment = commandLine

  IF SysFileTree(fileSpec, 'file', 'O') THEN SIGNAL ERROR

  IF file.0 = 0 THEN DO
    IF verbose THEN SAY 'No files.'
    EXIT
  END

  IF comment = '' THEN
    CALL displayComments
  ELSE
    CALL setComments comment
END /* else change or list comments */

EXIT /* main program */

/***************************************************************************
 * Delete file comments
 *
 * The stem variable FILE should contain a list of complete pathnames.
 * File.0 should contain the number of files.
 ***************************************************************************/
deleteComments: PROCEDURE EXPOSE file. (globalVars)
  DO i = 1 TO file.0
    fileName = filespec('n',file.i)
    IF LENGTH(fileName) < 12 THEN fileName = RIGHT(fileName,12)
    oldComment = readComment(file.i)
    IF oldComment <> '' THEN DO
      IF SysPutEA(file.i,eaName,'') = 0 & verbose THEN
          SAY fileName': Deleted "'oldComment'"'
      ELSE
        SAY fileName': Error'
    END /* if file has comment */
  END /* for i */
  RETURN
/* end deleteComments */

/***************************************************************************
 * Display file comments
 *
 * The stem variable FILE should contain a list of complete pathnames.
 * File.0 should contain the number of files.
 ***************************************************************************/
displayComments: PROCEDURE EXPOSE file. (globalVars)
  DO i = 1 TO file.0
    comment = readComment(file.i)
    IF (comment <> '') | allFiles THEN DO
      fileName = filespec('n',file.i)
      IF LENGTH(fileName) < 12 THEN
        fileName = RIGHT(fileName,12)
      SAY fileName':' comment
    END /* if */
  END /* for i */
  RETURN
/* end displayComments */

/***************************************************************************
 * Set file comments
 *
 * The stem variable FILE should contain a list of complete pathnames.
 * File.0 should contain the number of files.
 ***************************************************************************/
setComments: PROCEDURE EXPOSE file. (globalVars)
  PARSE ARG comment

  fileComment = eaHeader||D2C(LENGTH(comment))||'00'x||comment
  IF verbose THEN
    SAY 'Setting file' eaDesc 'to "'comment'"...'
  DO i = 1 TO file.0
    fileName = filespec('n',file.i)
    IF LENGTH(fileName) < 12 THEN fileName = RIGHT(fileName,12)
    oldComment = readComment(file.i)
    IF forceUpdate | oldComment = '' THEN DO
      IF SysPutEA(file.i,eaName, fileComment) = 0 & verbose THEN DO
        IF oldComment = '' THEN
          SAY fileName': OK'
        ELSE
          SAY fileName': OK (Replaced "'oldComment'")'
      END /* if verbose */
      ELSE
        SAY fileName': Error'
    END /* if we should write comment to this file */
    ELSE
      IF verbose THEN SAY fileName': Did not replace "'oldComment'"'
  END /* for i */
  RETURN
/* end setComments */

/***************************************************************************
 * Read file comment
 *
 * Input:
 *   pathname of file to get comment from
 * Output:
 *   The comment.  (Null string if no comment)
 ***************************************************************************/
readComment: PROCEDURE EXPOSE (globalVars)
  PARSE ARG fileName
  IF SysGetEA(fileName, eaName, 'comment') <> 0 THEN
    RETURN ''
  IF comment = '' THEN
    RETURN ''
  IF LEFT(comment,eaHdrLen) = eaHeader THEN
    RETURN SUBSTR(comment,eaHdrLen+3)
  ELSE
    RETURN '<Multi-line comment>'
/* end readComment */

/***************************************************************************
 * Read an argument from the command line:
 *
 * Input:
 *   The command line must be stored in commandLine
 * Output:
 *   Returns the argument.  If it was enclosed in quotation marks, the
 *     quotes are removed.
 *   The retrieved argument is removed from commandLine.
 ***************************************************************************/
nextArg: PROCEDURE EXPOSE commandLine
  IF POS(LEFT(commandLine,1),'''"') > 0 THEN DO
    /* Get a quoted argument */
    quoteChar = LEFT(commandLine,1)
    PARSE VAR commandLine (quoteChar) argument (quoteChar) commandLine
  END /* if this is a quoted argument */
  ELSE
    PARSE VAR commandLine argument commandLine
  commandLine = STRIP(commandLine,'L') /* Strip leading blanks */
  RETURN argument
/* end nextArg */

/***************************************************************************
 * Signal Handlers
 ***************************************************************************/
Error:
  SAY 'Abnormal exit!'
  EXIT 1

License:
  SAY
  SAY versionStr
  /* Read license text from top of command file */
  CALL LINEIN commandPath, 1, 0
  DO  3; CALL LINEIN commandPath;                     END
  DO 19; CALL LINEOUT, SUBSTR(LINEIN(commandPath),4); END
  EXIT

Usage:
  SAY
  SAY versionStr
  SAY
  SAY 'Usage: comment [options] fileSpec [comment]'
  SAY '       comment -d fileSpec...'
  SAY 'Options:'
  SAY '    -a, --all-files List all files (even those without comments)'
  SAY '    -d, --delete    Remove comments from specified files'
  SAY '    -f, --force     Replace existing comments'
  SAY '    -q, --quiet     Do not display progress messages',
    'or ask for confirmation'
  SAY '    -s, --subject   Work with the subject field',
    'instead of the comments field'
  SAY '    -?, --help      Display this help and exit'
  SAY '        --license   Display license information and exit'
  SAY '        --version   Display version information and exit'
  EXIT
