/* r0comm.c
   Ring 0 side of communications
*/

#define INCL_DOSERRORS
#define INCL_NOCOMMON
#define INCL_NOPMAPI
#define _Seg16
#define _Far16
#define _System
#include <os2.h>
#include "os216.h"

#include <string.h>
#include <dos.h>
#include "fsd.h"
#include "r0devhlp.h"
#include "..\r0r3shar.h"
#include "r0struct.h"
#include "r0comm.h"
#include "fsh.h"
#include "r0global.h"
#include "..\banner.h"
#include "log.h"

/* Local prototypes */
static void DetachCP(void);

/* The list of mounted volumes */
static MOUNTSTRUC MountList[MAX_MOUNTSTRUCS];

/******************************************************************************
**
** FS_INIT - Initialize the IFS
**
** Parameters
** ---------
** char far *szParm                    pointer to command line parameters
** unsigned long pDevHlp               pointer to DevHlp entry point
** unsigned long far *pMiniFSD         pointer to data passed between the
**                                     mini-FSD and the IFS
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_INIT(char far *szParm,
                                             void (*pDevHlp)(void),
                                             unsigned long far *pMiniFSD)
{
  SEL GDTSels[2];
  int rc;
  static char alloc1err[]="Error allocating lock handles\r\n";
  static char alloc2err[]="Error allocating GDT selectors\r\n";

  /* Put up banner */
  DosPutMessage(h_stdout, sizeof(banner)-1, banner);

  /* Run C runtime startup code */
  Startup();

  /* If there were any command line parameters, save them */
  if (szParm != NULL) {
    strncpy(CmdLineSave, szParm, sizeof(CmdLineSave));
  }

  /* Save the DevHelp entry point */
  DevHelp = pDevHlp;

  /* Allocate virtual memory for the two lock handles */
  rc = VMAlloc(VMAPA_NOPHYSADDR, 2 * sizeof(HLOCK),
               VMAF_SWAPPABLE | VMAF_GLOBALSPACE, &Lock1);
  if (rc != NO_ERROR) {
    DosPutMessage(1, sizeof(alloc1err)-1, alloc1err);
    return rc;
  }
  Lock2 = Lock1 + sizeof(HLOCK);

  /* Allocate 2 GDT selectors to communicate with the control program */
  rc = AllocGDTSelector(2, GDTSels);
  if (rc != NO_ERROR) {
    DosPutMessage(1, sizeof(alloc2err)-1, alloc2err);
    return rc;
  }

  /* Set the pointers to the control program communication buffer */
  CPData.OpData = (OPDATA *)MK_FP(GDTSels[0], 0);
  CPData.Buf = (void *)MK_FP(GDTSels[1], 0);

  /* Say the control program has never been attached */
  /* Probably unnecessary */
  CPAttached = 0;
  AttachedEver = 0;

  return NO_ERROR;
}

/******************************************************************************
**
** FS_SHUTDOWN - Prepare for a system shutdown
**
** Parameters
** ---------
** unsigned short usType               flavor of call
**   values:
**     SD_BEGIN                  begining the shutdown process
**     SD_COMPLETE               finishing the shutdown process
** unsigned long ulReserved            reserved
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_SHUTDOWN(unsigned short usType,
                                                 unsigned long ulReserved)
{
  return NO_ERROR;
}

/******************************************************************************
**
** FS_FSCTL - Extended IFS control
**
** Parameters
** ---------
** union argdat far *pArgdat
** unsigned short iArgType             argument type
**   values:
**     FSCTL_ARG_FILEINSTANCE    filehandle directed
**     FSCTL_ARG_CURDIR          pathname directed
**     FSCTL_ARG_NULL            FSD directed
** unsigned short func                 function code
**   values:
**     FSCTL_FUNC_NONE           NULL function
**     FSCTL_FUNC_NEW_INFO       return error code information
**     FSCTL_FUNC_EASIZE         return max EA size and max EA list size
** char far *pParm                     UNVERIFIED pointer to parameter area
** unsigned short lenParm              size of area pointed to by pParm
** unsigned short far *plenParmOut     length of parameters passed in pParm
** char far *pData                     UNVERIFIED pointer to information area
** unsigned short lenData              size of area pointed to by pData
** unsigned short far *plenDataOut     length of parameters passed in pData
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FSCTL(union argdat far *pArgdat,
                                              unsigned short iArgType,
                                              unsigned short func,
                                              char far *pParm,
                                              unsigned short lenParm,
                                              unsigned short far *plenParmOut,
                                              char far *pData,
                                              unsigned short lenData,
                                              unsigned short far *plenDataOut)
{
  int rc;
  char far *p;
  unsigned short oldVPB;

  switch (func) {
    case FSCTL_FUNC_NONE:
      return NO_ERROR;

    case FSCTL_FUNC_NEW_INFO:
    case FSCTL_FUNC_EASIZE:
      return ERROR_NOT_SUPPORTED;

    case FSCTL_FUNC_GETLOGDATA:
      /* Wait on the log semaphore */
      if( (rc=FSH_SEMWAIT(&logbuf_sem, TO_INFINITE)) == ERROR_INTERRUPT )
        return rc;

      /* Check the user buffer and length field */
      if( (rc=FSH_PROBEBUF(PB_OPWRITE, pData, lenData)) != NO_ERROR )
        return rc;
      if( (rc=FSH_PROBEBUF(PB_OPWRITE, plenDataOut, sizeof(unsigned short))) != NO_ERROR )
        return rc;

      /* If no data present, set the semaphore */
      if( logbuf_head == logbuf_tail ) {
        *plenDataOut = 0;
        if((rc=FSH_SEMSET(&logbuf_sem)) == ERROR_INTERRUPT)
          return rc;
        return NO_ERROR;
      }

      /* Copy data */
      p = pData;
      for( ; logbuf_head!=logbuf_tail && p<pData+lenData; logbuf_head=(logbuf_head+1)%LOGBUF_SIZE )
        *p++ = logbuf[logbuf_head];
      *plenDataOut = p-pData;
      if ((rc = FSH_SEMSET(&logbuf_sem)) == ERROR_INTERRUPT)
        return rc;

      return NO_ERROR;

    case FSCTL_FUNC_READSECTORS:
      /* Verify parameter and data areas */
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(RWSTRUC));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, &((RWSTRUC *)pParm)->n_sectors,
                       sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, pData,
                        ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE);
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenParmOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenDataOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      /* Verify buffer size */
      if( lenData < ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE ) {
        *plenDataOut = ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE;
        return ERROR_BUFFER_OVERFLOW;
      }

      /* Read sectors */
      rc = FSH_DOVOLIO(DVIO_OPREAD,
                       DVIO_ALLFAIL|DVIO_ALLRETRY|DVIO_ALLABORT,
                       ((RWSTRUC *)pParm)->hVPB,
                       pData,
                       &((RWSTRUC *)pParm)->n_sectors,
                       ((RWSTRUC *)pParm)->starting_sector);
      *plenParmOut = sizeof(unsigned short);
      *plenDataOut = ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE;
      return rc;

    case FSCTL_FUNC_WRITESECTORS:
      /* Verify parameter and data areas */
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(RWSTRUC));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, &((RWSTRUC *)pParm)->n_sectors,
                       sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPREAD, pData,
                        ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE);
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenParmOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenDataOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      /* Verify buffer size */
      if( lenData < ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE ) {
        *plenDataOut = ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE;
        return ERROR_BUFFER_OVERFLOW;
      }

      /* Write sectors */
      rc = FSH_DOVOLIO(DVIO_OPWRITE,
                       DVIO_ALLFAIL|DVIO_ALLRETRY|DVIO_ALLABORT,
                       ((RWSTRUC *)pParm)->hVPB,
                       pData,
                       &((RWSTRUC *)pParm)->n_sectors,
                       ((RWSTRUC *)pParm)->starting_sector);
      *plenParmOut = sizeof(unsigned short);
      *plenDataOut = ((RWSTRUC *)pParm)->n_sectors * SECTOR_SIZE;
      return rc;

    case FSCTL_FUNC_FINDDUPHVPB:
      /* Verify parameter area */
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenParmOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;

      /* Call FS helper */
      rc = FSH_FINDDUPHVPB(*(unsigned short *)pParm, &oldVPB);
      *(unsigned short *)pParm = oldVPB;
      *plenParmOut = sizeof(unsigned short);
      return rc;

    case FSCTL_FUNC_GET_MTSTAT: {
      PMTSTATSTRUC mts=(PMTSTATSTRUC)pParm;
      unsigned short i;

      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, pParm, sizeof(MTSTATSTRUC));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenParmOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;

      /* Search the mount list for the given hVPB */
      i=0;
      while(i<MAX_MOUNTSTRUCS && *(unsigned short *)pParm != MountList[i].hVPB)
	i++;
      if(i==MAX_MOUNTSTRUCS)
	return ERROR_INVALID_PARAMETER;
#ifdef DEBUG
      printk("Returning mount struc %u, hVPB=%u, status=%u\n", i, MountList[i].hVPB, MountList[i].status);
#endif
      mts->drive = MountList[i].drive;
      mts->status = MountList[i].status;
      mts->hVPB = MountList[i].hVPB;
      *plenParmOut = sizeof(MTSTATSTRUC);
      return NO_ERROR;
    }

    case FSCTL_FUNC_SET_MTSTAT: {
      PMTSTATSTRUC mts=(PMTSTATSTRUC)pParm;
      unsigned short i;

      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(MTSTATSTRUC));
      if( rc != NO_ERROR )
        return rc;

      i=0;
      while(i<MAX_MOUNTSTRUCS && MountList[i].hVPB!=mts->hVPB)
	i++;
      if(i==MAX_MOUNTSTRUCS) {
	/* Find a free mount struc */
	i=0;
	while(i<MAX_MOUNTSTRUCS && MountList[i].status!=MTSTAT_FREE)
	  i++;
	if(i==MAX_MOUNTSTRUCS)
	  return ERROR_BUFFER_OVERFLOW;
      }
#ifdef DEBUG
      printk("Setting mount struc %u, hVPB=%u, status=%u\n", i, mts->hVPB, mts->status);
#endif
      MountList[i].hVPB = mts->hVPB;
      MountList[i].drive = mts->drive;
      MountList[i].status = mts->status;
      return NO_ERROR;
    }

    case FSCTL_FUNC_WILDMATCH:
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(WILDSTRUC));
      if( rc != NO_ERROR )
        return rc;

      return FSH_WILDMATCH(((PWILDSTRUC)pParm)->pattern,
                           ((PWILDSTRUC)pParm)->text);

    case FSCTL_FUNC_UPPERCASE:
      rc = FSH_PROBEBUF(PB_OPREAD, pData, lenData);
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, lenData);
      if( rc != NO_ERROR )
	return rc;

      return FSH_UPPERCASE(pData, lenData, pData);

    case FSCTL_FUNC_SETVOLUME:
      /* Verify parameter area */
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
#ifdef DEBUG
      printk("Calling FSH_SETVOLUME, hVPB = %u\n", *(unsigned short *)pParm);
#endif
      rc = FSH_SETVOLUME(*(unsigned short *)pParm, 1);
#ifdef DEBUG
      if(rc!=NO_ERROR)
        printk("FSH_SETVOLUME failed, rc = %d\n", rc);
#endif
      return rc;

    case FSCTL_FUNC_GETVOLPARM: {
      unsigned short hVPB;
      struct vpfsi *pvpfsi;
      struct vpfsd *pvpfsd;
      PVOLPARMSTRUC volparm=(PVOLPARMSTRUC)pData;

      /* Verify parameter area */
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, sizeof(VOLPARMSTRUC));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, plenDataOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      if(lenData < sizeof(VOLPARMSTRUC))
	return ERROR_BUFFER_OVERFLOW;

      hVPB = *(unsigned short *)pParm;
      rc=FSH_GETVOLPARM(hVPB, &pvpfsi, &pvpfsd);
      if(rc!=NO_ERROR)
	return rc;
      volparm->vpfsi = *pvpfsi;
      volparm->vpfsd = *pvpfsd;
      *plenDataOut = sizeof(VOLPARMSTRUC);
      return NO_ERROR;
    }

    case FSCTL_FUNC_SETVOLPARM: {
      unsigned short hVPB;
      struct vpfsi *pvpfsi;
      struct vpfsd *pvpfsd;
      PVOLPARMSTRUC volparm=(PVOLPARMSTRUC)pData;

      /* Verify parameter area */
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPREAD, pData, sizeof(VOLPARMSTRUC));
      if( rc != NO_ERROR )
        return rc;

      hVPB = *(unsigned short *)pParm;
      rc=FSH_GETVOLPARM(hVPB, &pvpfsi, &pvpfsd);
      if(rc!=NO_ERROR)
	return rc;
      *pvpfsi = volparm->vpfsi;
      *pvpfsd = volparm->vpfsd;
      return NO_ERROR;
    }

    case FSCTL_FUNC_GET_HVPB_FROM_DRV: {
      unsigned short i;
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPWRITE, pParm, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;
      rc = FSH_PROBEBUF(PB_OPREAD, plenParmOut, sizeof(unsigned short));
      if( rc != NO_ERROR )
        return rc;

      i=0;
      while(i<MAX_MOUNTSTRUCS && (MountList[i].drive!=*(unsigned short *)pParm
				  || MountList[i].status!=MTSTAT_MOUNTED))
	i++;
      if(i==MAX_MOUNTSTRUCS)
	return ERROR_INVALID_PARAMETER;
      *(unsigned short *)pParm = MountList[i].hVPB;
      *plenParmOut=sizeof(unsigned short);
      return NO_ERROR;
    }

    case FSCTL_FUNC_INIT:
      /* If control program has never been attached, ignore these checks */
      if (AttachedEver) {
        /* Make sure the control program isn't already attached */
        if (CPAttached)
          return 0xef00;

        /* Make sure that it is OK to attach at this time */
        rc = FSH_SEMWAIT(&CPData.BufLock, 0);
        if (rc != NO_ERROR)
          return 0xef01;
      }

      /* Verify size of and addressability of the pParm area */
      if (lenParm != sizeof(INITSTRUC))
        return 0xef02;
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, lenParm);
      if (rc != NO_ERROR)
        return rc;

      /* Lock down the operation buffer */
      rc = VMLock(((INITSTRUC *)pParm)->OpData, sizeof(OPDATA), VMLPL_NOPAGELIST,
                  Lock1, VMLAF_WRITE | VMLAF_LONG, NULL);
      if (rc != NO_ERROR)
        return rc;

      /* Map the GDT selector we got at init time to the memory we just locked */
      rc = LinToGDTSelector(FP_SEG(CPData.OpData),
                            ((INITSTRUC *)pParm)->OpData, sizeof(OPDATA));
      if (rc != NO_ERROR)
        return rc;

      /* Lock down the data buffer */
      rc = VMLock(((INITSTRUC *)pParm)->Buf, 65536, VMLPL_NOPAGELIST,
                  Lock2, VMLAF_WRITE | VMLAF_LONG, NULL);
      if (rc != NO_ERROR)
        return rc;

      /* Map the GDT selector we got at init time to the memory we just locked */
      rc = LinToGDTSelector(FP_SEG(CPData.Buf), ((INITSTRUC *)pParm)->Buf,
                            65536);
      if (rc != NO_ERROR)
        return rc;

      /* The operation data area is now OK to use */
      rc = FSH_SEMCLEAR(&CPData.BufLock);
      if (rc != NO_ERROR)
        return rc;

      /* Save control program's process ID */
      CPPID = ((INITSTRUC *)pParm)->CPPID;

      /* The control program is now attached */
      CPAttached = AttachedEver = 1;

#ifdef DEBUG
      printk("Attached to control program, PID = %lu\n", (unsigned long)CPPID);
#endif

      /* FALL THROUGH */

    case FSCTL_FUNC_NEXT:
      if (!CPAttached)
        return 0xef03;

      /* Set the semaphore that indicates the control program is ready for
         another operation */
      rc = FSH_SEMSET(&CPData.CmdReady);
      if (rc == ERROR_INTERRUPT) {
        DetachCP();
        return rc;
      }

      /* Clear the semaphore that indicates the control program just returned */
      rc = FSH_SEMCLEAR(&CPData.CmdComplete);
      if (rc == ERROR_INTERRUPT) {
        DetachCP();
        return rc;
      }

      /* Wait for the semaphore that indicates another operation is ready */
      rc = FSH_SEMWAIT(&CPData.CmdReady, -1);
      if (rc == ERROR_INTERRUPT)
        DetachCP();

      return rc;

    case FSCTL_FUNC_KILL:
      DetachCP();
      return NO_ERROR;

    default:
      return ERROR_NOT_SUPPORTED;
  }
}

/******************************************************************************
**
** FS_EXIT - Notify FSD that a process is ending
**
** Parameters
** ---------
** unsigned short uid                   user ID of process
** unsigned short pid                   process ID of process
** unsigned short pdb                   DOS mode process ID of process
**
******************************************************************************/

#pragma argsused
void far pascal _export _loadds FS_EXIT(unsigned short uid, unsigned short pid,
                                        unsigned short pdb)
{
  /* If the control program is exiting, detach ourselves from it */
  if (CPAttached && pid == CPPID)
    DetachCP();
}

/******************************************************************************
**
** DetachCP - Detach the control program from the IFS
**
******************************************************************************/

static void DetachCP(void) {
#ifdef DEBUG
  printk("Detaching from CP...");
#endif
  CPAttached = 0; /* We leave AttachedEver set */
  VMUnlock(Lock1);
  VMUnlock(Lock2);
  FSH_SEMCLEAR(&CPData.CmdComplete);
  FSH_SEMCLEAR(&CPData.BufLock);
#ifdef DEBUG
  printk("done\n");
#endif
}

/******************************************************************************
**
** PreSetup - Stuff that needs to be done before we setup the buffers for a
**            call
**
******************************************************************************/

int PreSetup(void) {
  int rc;

  /* If control program isn't attached, immediately return */
  if (!CPAttached)
    return (ERROR_NOT_READY);

  /* Make sure this thread can change the buffers */
  rc = FSH_SEMREQUEST(&CPData.BufLock, MAXCPRDYWAIT);
  if (rc == ERROR_SEM_TIMEOUT)
    return (ERROR_NOT_READY);
  else if (rc != NO_ERROR)
    return rc;

  /* And make sure the control program isn't changing them */
  rc = FSH_SEMWAIT(&CPData.CmdComplete, MAXCPRDYWAIT);
  if (rc == ERROR_SEM_TIMEOUT) {
    FSH_SEMCLEAR(&CPData.BufLock);
    return (ERROR_NOT_READY);
  }
  else if (rc != NO_ERROR) {
    FSH_SEMCLEAR(&CPData.BufLock);
    return rc;
  }

  return NO_ERROR;
}


/******************************************************************************
**
** PostSetup - Stuff that needs to be done after we setup the buffers for a
**             call
**
******************************************************************************/

int PostSetup(void) {
  int rc;

  /* Set sem that we'll wait on */
  rc = FSH_SEMSET(&CPData.CmdComplete);
  if (rc != NO_ERROR) {
    FSH_SEMCLEAR(&CPData.CmdComplete);
    FSH_SEMCLEAR(&CPData.BufLock);
    return rc;
  }

  /* Tell the control program to do the operation */
  rc = FSH_SEMCLEAR(&CPData.CmdReady);

  /* Wait until the CP is done */
  rc = FSH_SEMWAIT(&CPData.CmdComplete, MAXCPRESWAIT);
  if (!CPAttached || rc == ERROR_SEM_TIMEOUT) {
    /* If we got a timeout, abort */
    FSH_SEMCLEAR(&CPData.BufLock);
    return (ERROR_NOT_READY);
  }
  else if (rc != NO_ERROR) {
    /* Some other error occurred, abort */
    FSH_SEMCLEAR(&CPData.BufLock);
    return rc;
  }

  return NO_ERROR;
}
