					/* Functions to access the nodes */
					/* of the B-trees */
#include "../defs.h"
#include "../Lock/lock.h"
#include "../mountrec.h"
#include "../DevSupport/readwrite.h"
#include "btree.h"
#include "local_btree.h"
#include "../errors.h"

			      /* Used to implement the clock algorithm for 
                               * cache replacement
			       */
static int AdvanceHand(BT_HEADER *head, int hand);

			      /* Does the conversion between a BTree node and 
                               * a sector number */
static int NodeToSector(MOUNT_RECORD *mnt, BT_HEADER *head, 
			      LONGWORD node, LONGWORD *sect);

			      /* Read in an extent record containing the 
                               * extent descriptor that locates block
			       */
static int GetBTExtentRecord(MOUNT_RECORD *mnt,
			     BT_HEADER *head, 
			     WORD block, 
			     EXTENT_RECORD ext,
			     WORD *foundblock);


			      /* Read a BTree node in buffer
			       */
int  BT_ReadNode(MOUNT_RECORD *mnt, BT_HEADER *head, 
		 LONGWORD node, BYTE* buffer) {
int err;
LONGWORD sect;
				
  CHKERR(NodeToSector(mnt, head, node,&sect))	
			      /* First I must find out the sector number  */
  err = readsects(mnt, sect, 1, (char*)buffer);
  if(err < 0)
    return(err);

  return(0);
}

			      /* Write a BTree node from buffer
			       */
int  BT_WriteNode(MOUNT_RECORD *mnt, BT_HEADER *head, 
		  LONGWORD node,  BYTE *buffer) {
int err;
LONGWORD sect;
				
  CHKERR(NodeToSector(mnt, head, node, &sect));	
			      /* First I must find out the sector number  */

  if (node==0) {
    err = writesectsunbuf(mnt, sect, 1, (char*)buffer);
  } else {
    err = writesects(mnt, sect, 1, (char*)buffer);
  }
  
  if(err!=1) { 
    return BT_WRITE_NODE_FAILED;
  }

  return(0);
}



			      /* Converts the node number into a sector 
                               * number. Assumes that the node size is equal 
                               * to the sector size
			       */
static int NodeToSector(MOUNT_RECORD *mnt, BT_HEADER *head, 
			LONGWORD node, LONGWORD *sect) {
					
  int node_block, node_in_block, i, pack;
  WORD curr_block;
  int found,err;
  EXTENT_RECORD  ext_rec;
  LONGWORD block;

			      /* Converts the node number to allocation 
                               * blocks  */
  pack = (mnt->vib.AlBlkSiz / BT_NODESIZE); /* nodes in a block */
  node_block = node / pack;   /* block in file */
  node_in_block = node % pack; /* sector in block */


  CHKERR(GetBTExtentRecord(mnt, head, node_block, ext_rec, &curr_block));
			      /* Reads the next extent record */
       
  for(i=0,found=0;i<3 && !found; i++) {
    if(ext_rec[i].length == 0)		/* Last extent passed */
      return (BT_NODENOTFOUND);
    if( (curr_block + ext_rec[i].length) > node_block ) {
      found = 1;			/* We have found the extent */
      break;				/* Be sure we not increment i */
    }
    curr_block += ext_rec[i].length;
  }

  if(i == 3)
    return(BT_NODENOTFOUND);

  block = node_block - curr_block + ext_rec[i].first;
  *sect = mnt->vib.AlBlSt + block * pack + node_in_block;

  return(0);
}
	
 

			      /* Reads a extent record for the Btree. Has 
                               * similar meaning as the global function 
                               * GetExtentRecord but does not necessarily 
                               * read the extents tree
			       */
static int GetBTExtentRecord(MOUNT_RECORD *mnt,
			     BT_HEADER *head, 
			     WORD block, 
			     EXTENT_RECORD ext,
			     WORD *foundblock) {

  int left,mid,right;
  EXTENT_CACHE_ENTRY *p_cache;
  int free_slot, read_at;
  int old_hand, curr_hand, founddf, i;
  LONGWORD foundid;
  int err;				
			      /* Binary search for the last block less or 
                               * equal to BLOCK  */

  /* 0 <= i < left =>  cache[i].block <= BLOCK */
  /* right < i <= cache_size -1   => cache[i].block > BLOCK */
  left  = 0;				
  right = head->cache_size - 1;
					/* While the length of the middle */
					/* zone is greater than 0*/
  while(right - left  + 1 > 0)  {
    
    mid = (left + right) >> 1;
    if( head->cache[mid].block <= block)
      left  = mid + 1;
    else
      right = mid - 1;
  }

  /* Now the cache[right] is the one I need */
  /* I know that right >= 0 because head->cache_size >= 1 and the first*/
  /* entry contains the entry for the smallest block in file (0) */
  
  p_cache = & head->cache[right];

  if( p_cache->block + p_cache->length > block)  {
					/* I found the needed block */
    bcopy(& p_cache->ext, ext, sizeof(EXTENT_RECORD));
    *foundblock = p_cache->block;
					/* Mark it referenced */
    p_cache->flags |= EXT_REFERENCED;
    return 0;
  }

					/* Need to read a new extent */

  if(head->BTfileid != CATALOG_FILE_ID)	/* For this file I don't go to read 
                                         * the extents tree  */
    return(BT_MANYEXTENTS);

					/* First make space */
  free_slot = 0; 
  if( head->cache_size < EXTENT_CACHE_SIZE) {
					/* There is enough space */
    free_slot = head->cache_size ++;
  } else {
					/* Throw one out */

    curr_hand = old_hand = head->cache_hand; /* Old position to avoid 
                                              * looping  */

    do  {
      curr_hand = AdvanceHand(head, curr_hand); /* go to next item (skip 0)*/

      if(head->cache[curr_hand].flags & EXT_REFERENCED)
	head->cache[curr_hand].flags &= (~EXT_REFERENCED);
      else {
	head->cache[curr_hand].flags &= (~EXT_REFERENCED);
	free_slot = curr_hand;
      }
    } while(!free_slot && curr_hand != old_hand);

    if(!free_slot) 			/* Loop detected */
      free_slot = AdvanceHand(head, old_hand); /* FIFO type */

  }

  /* Now right is the slot before the desired one and the free_slot is a 
   * free slot; move something to make space right after RIGHT  */

  if(right >= free_slot) {    /* Move chunk (FREE+1---RIGHT) one slot back  */
    bcopy(& head->cache[free_slot+1],
	  & head->cache[free_slot],
          sizeof(EXTENT_CACHE_ENTRY) * (right - free_slot));
    read_at = right;
  }
  else	{				/* Move chunk (RIGHT+1 --- FREE-1 */
					/* one slot forward*/
    bcopy(& head->cache[right+1],
	  & head->cache[right+2],
          sizeof(EXTENT_CACHE_ENTRY) * (free_slot - right -1));
    read_at = right +1;
  }

  p_cache = & head->cache[read_at];
  head->cache_hand = AdvanceHand(head, read_at);
  p_cache->flags |= EXT_REFERENCED;

/* Read the needed extent form the Extents tree in the slot P_CACHE */

  err = GetExtentRecord(mnt, head->BTfileid, 1, block,
			p_cache->ext, &foundid, &founddf, & p_cache->block);

  if(err && err != BT_FOUNDLESS)
    return (err);

  for(p_cache->length = 0,i=0;i<3;i++)
    p_cache->length += p_cache->ext[i].length;

  if(head->BTfileid != foundid || founddf != 1 ||
     p_cache->block + p_cache->length <= block)
    return(BT_BADTREE);

					/* Return the read extent */
  bcopy(p_cache->ext, ext, sizeof(EXTENT_RECORD));
  *foundblock = p_cache->block;
  return 0;
}


			      /* Advance the clock hand skipping the first 
                               * entry in the cache because it cannot be 
                               * discarded  */
static int AdvanceHand(BT_HEADER *head, int hand)   {

  if(++hand >= head->cache_size)
    return 1;
  else
    return(hand);
}


 







