/* dirsearch.c
   Directory search functions.
*/

#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#define INCL_NOCOMMON
#define INCL_NOPMAPI
#include <os2.h>

#include <string.h>
#include <stdlib.h>
#ifdef DEBUG
#include <stdio.h>
#endif

#include "fsd.h"
#include "../r0r3shar.h"
#include "dirsearch.h"
#include "names.h"
#include "times.h"
#include "r3global.h"
#include "fsctl.h"

#include "MacFS/defs.h"
#include "MacFS/mountrec.h"
#include "MacFS/LibFuncs/resolve.h"
#include "MacFS/LibFuncs/catalog.h"

/******************************************************************************
**
** FS_FINDFIRST - Begin a new directory search
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename mask
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short attr                  attribute mask
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short *pcMatch              maximum number of entries to return
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
** unsigned long *oError                offset where error occurred in GEA
**                                      list (EA levels only)
**
******************************************************************************/

#pragma argsused
short int FS_FINDFIRST(struct vpfsd *pvpfsd,
		       struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                       char *pName, unsigned short iCurDirEnd,
                       unsigned short attr, struct fsfsi *pfsfsi,
                       struct fsfsd *pfsfsd, char *pData,
                       unsigned short cbData, unsigned short *pcMatch,
                       unsigned short level, unsigned short flags,
                       unsigned long *oError)
{
  CATALOG_RECORD *cr;
  LONGWORD parid;
  char dirname[CCHMAXPATH], name[CCHMAXPATH];
  int rc;
  
  cr = (CATALOG_RECORD *)malloc(sizeof(CATALOG_RECORD));
  pfsfsd->pattern = (char *)malloc(CCHMAXPATH);
  pfsfsd->cr_list = NULL;
  pfsfsd->this_cr = NULL;
  if(os2_to_mac_name(pName+2, name)) {
    free(pfsfsd->pattern);
    free(cr);
    pfsfsd->pattern = NULL;
    return ERROR_PATH_NOT_FOUND;
  }
  if( macresolvepnwithnonexistinglastelement(pvpfsd->mr,
					     name,
					     &parid,
					     dirname,
					     cr,
					     pfsfsd->pattern) 
     || cr->cdrType != CDR_DIR ) {
    free(pfsfsd->pattern);
    free(cr);
    pfsfsd->pattern = NULL;
    return ERROR_PATH_NOT_FOUND;
  }
  rc=mdirentries(pvpfsd->mr, cr->U.cdrDirRec.dirDirID, &(pfsfsd->cr_list));
  if( rc < 0 ) {
    free(pfsfsd->pattern);
    free(cr);
    pfsfsd->pattern = NULL;
    return ERROR_PATH_NOT_FOUND; /* Should perhaps be something else */
  }
  pfsfsd->this_cr = cr;
  pfsfsd->cr_entries = (unsigned short)rc;
  if(!strcmp(pfsfsd->pattern, "*"))
    pfsfsd->cur_catrec = CR_DOT;    /* Start by returning . directory */
  else
    pfsfsd->cur_catrec = CR_START;
  pfsfsd->attr = attr;
  rc = do_findnext(pvpfsd, pfsfsd, pData, cbData, 
		   pcMatch, level, flags);
  if( rc==ERROR_NO_MORE_FILES )
    return ERROR_PATH_NOT_FOUND;
  else
    return rc;

  /* Note that pData doesn't contain an EAOP for level==FIL_QUERYEASIZE,
     because the control program takes care of that */
}


/******************************************************************************
**
** FS_FINDFROMNAME - Restart directory search
**
** Parameters
** ----------
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short *pcMatch              maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned long position               position in directory to restart search from
** char *pName                          pointer to filename to restart search from
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
** unsigned long *oError                offset where error occurred in GEA
**                                      list (EA levels only)
**
******************************************************************************/

#pragma argsused
short int FS_FINDFROMNAME(struct vpfsd *pvpfsd,
			  struct fsfsi *pfsfsi, struct fsfsd *pfsfsd,
                          char *pData, unsigned short cbData,
                          unsigned short *pcMatch, unsigned short level,
                          unsigned long position, char *pName,
                          unsigned short flags, unsigned long *oError)
{
  pfsfsd->cur_catrec = position;
  return do_findnext(pvpfsd, pfsfsd, pData, cbData, 
		     pcMatch, level, flags);
}

/******************************************************************************
**
** FS_FINDNEXT - Continue directory search
**
** Parameters
** ----------
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short *pcMatch              maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned short flag                  indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
** unsigned long *oError                offset where error occurred in GEA
**                                      list (EA levels only)
**
******************************************************************************/

#pragma argsused
short int FS_FINDNEXT(struct vpfsd *pvpfsd,
		      struct fsfsi *pfsfsi, struct fsfsd *pfsfsd,
                      char *pData, unsigned short cbData, 
                      unsigned short *pcMatch, unsigned short level,
                      unsigned short flag, unsigned long *oError)
{
  return do_findnext(pvpfsd, pfsfsd, pData, cbData, 
		     pcMatch, level, flag);
}

/******************************************************************************
**
** FS_FINDCLOSE - End a directory search
**
** Parameters
** ----------
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
**
******************************************************************************/

#pragma argsused
short int FS_FINDCLOSE(struct vpfsd *pvpfsd,
		       struct fsfsi *pfsfsi, struct fsfsd *pfsfsd)
{
  if( pfsfsd->cr_list )
    free(pfsfsd->cr_list);
  pfsfsd->cr_list = NULL;
  if( pfsfsd->pattern )
    free(pfsfsd->pattern);
  pfsfsd->pattern = NULL;
  if( pfsfsd->this_cr )
    free(pfsfsd->this_cr);
  pfsfsd->this_cr = NULL;
  return NO_ERROR;
}

/*****************************************************************************
** do_findnext
******************************************************************************/

unsigned short do_findnext(struct vpfsd *pvpfsd, struct fsfsd *pfsfsd,
			   char *buf, unsigned short buflen,
			   unsigned short *entries, unsigned short level,
			   unsigned short flag)
{
  unsigned short max_entries=*entries;
  char *p=buf, *buf_end = buf+buflen;
  char tempbuf[sizeof(FOUNDFILECOMMON)+CCHMAXPATH+30];

  *entries=0;
  for(; p<buf_end && *entries<max_entries 
      && pfsfsd->cur_catrec<pfsfsd->cr_entries;
      pfsfsd->cur_catrec++ ) {
    if( pfsfsd->cur_catrec == CR_DOT || pfsfsd->cur_catrec == CR_DOTDOT ) {
      int len = catrec_to_filebuf(tempbuf, pfsfsd->this_cr, level);
      int extra = flag==FF_GETPOS ? sizeof(long) : 0;

      if(!len)
	continue;
      if( pfsfsd->attr & 0xff00 & ~MUST_HAVE_DIRECTORY )
	continue; /* if "must have" attr other than directory are set */
      if( !(pfsfsd->attr&FILE_DIRECTORY) && pfsfsd->attr )
	continue; /* if directory attr is not "may have" */

      if( pfsfsd->cur_catrec==CR_DOT ) {
	tempbuf[len++] = 1;
	strcpy(tempbuf+len, ".");
	len += 2;
      }
      else {
      	tempbuf[len++] = 2;
	strcpy(tempbuf+len, "..");
	len += 3;
      }
      if( p+len+extra > buf_end )
	break;
      if( flag==FF_GETPOS ) {
	*(long *)p = pfsfsd->cur_catrec+1;
	p += sizeof(long);
      }
      memcpy(p, tempbuf, len);
      p += len;
      (*entries)++;
    }
    else {
      unsigned char os2_name[CCHMAXPATH];
      if(mac_to_os2_name(pfsfsd->cr_list[pfsfsd->cur_catrec].name, os2_name))
	continue;
      if(wildcard_match(pfsfsd->pattern, os2_name)==NO_ERROR
	 && attrib_match(pfsfsd->attr, &pfsfsd->cr_list[pfsfsd->cur_catrec])) {
	int len = catrec_to_filebuf(tempbuf, 
				    &pfsfsd->cr_list[pfsfsd->cur_catrec].cr,
                                    level);
	int extra = flag==FF_GETPOS ? sizeof(long) : 0;
	if(!len)
	  continue;

	/* Transfer filename including terminating zero */
	tempbuf[len++] = strlen(os2_name);
	strcpy(tempbuf+len, os2_name);
	len += strlen(tempbuf+len)+1;

	if( p+len+extra > buf_end )
	  break;
	if( flag==FF_GETPOS ) {
	  *(long *)p = pfsfsd->cur_catrec+1;
	  p += sizeof(long);
	}
	memcpy(p, tempbuf, len);
	p += len;
	(*entries)++;
      }
    }
  }
  if( *entries )
    return NO_ERROR;
  else
    return ERROR_NO_MORE_FILES;
}

unsigned short catrec_to_filebuf(char *buf, CATALOG_RECORD *cr, 
                                 unsigned short level)
{
  PFOUNDFILECOMMON p = (PFOUNDFILECOMMON)buf;
  char *pc = buf+sizeof(FOUNDFILECOMMON);

  switch(cr->cdrType) {
  case CDR_DIR: 
    p->fdateCreation = mac_to_os2_date(cr->U.cdrDirRec.dirCrDat);
    p->ftimeCreation = mac_to_os2_time(cr->U.cdrDirRec.dirCrDat);
    p->fdateLastWrite = mac_to_os2_date(cr->U.cdrDirRec.dirMdDat);
    p->ftimeLastWrite = mac_to_os2_time(cr->U.cdrDirRec.dirMdDat);
    p->fdateLastAccess = mac_to_os2_date(0);
    p->ftimeLastAccess = mac_to_os2_time(0);
    p->cbFile = 0;
    p->cbFileAlloc = 0;
    p->attrFile = FILE_DIRECTORY;
    break;

  case CDR_FILE:
    p->fdateCreation = mac_to_os2_date(cr->U.cdrFilRec.filCrDat);
    p->ftimeCreation = mac_to_os2_time(cr->U.cdrFilRec.filCrDat);
    p->fdateLastWrite = mac_to_os2_date(cr->U.cdrFilRec.filMdDat);
    p->ftimeLastWrite = mac_to_os2_time(cr->U.cdrFilRec.filMdDat);
    p->fdateLastAccess = mac_to_os2_date(0);
    p->ftimeLastAccess = mac_to_os2_time(0);
    p->cbFile = cr->U.cdrFilRec.filLgLen;
    p->cbFileAlloc = cr->U.cdrFilRec.filPyLen;
    p->attrFile = FILE_NORMAL;
    break;

  default:
    return 0;
  }

  switch(level) {
  case FIL_STANDARD:
    break;
  case FIL_QUERYEASIZE:
    *(ULONG *)pc = 0;  /* EA size is zero */
    pc += sizeof(ULONG);
    break;
  case FIL_QUERYEASFROMLIST:
    *(ULONG *)pc = 4; /* Size of FEA list */
    pc += sizeof(ULONG);
    break;
  default:
#ifdef DEBUG
    printf("do_findnext: Unsupported level = %hd\n", level);
#endif
    return 0;
  }
  return pc-buf;
}

unsigned short wildcard_match(char *pattern, char *text)
{
  WILDSTRUC w;
  unsigned long len_parm, len_data;

  strcpy(w.pattern, pattern);
  strcpy(w.text, text);

  /* Uppercase pattern and text */
  if(do_FSCtl(w.pattern, sizeof(w.pattern), &len_data,
	      NULL, 0, NULL, FSCTL_FUNC_UPPERCASE) != NO_ERROR) {
#ifdef DEBUG
    printf("FSCTL_FUNC_UPPERCASE failed!\n");
#endif
    return 0;
  }
  if(do_FSCtl(w.text, sizeof(w.text), &len_data,
	      NULL, 0, NULL, FSCTL_FUNC_UPPERCASE) != NO_ERROR) {
#ifdef DEBUG
    printf("FSCTL_FUNC_UPPERCASE failed!\n");
#endif
    return 0;
  }

  return do_FSCtl(NULL, 0, NULL, &w, sizeof(WILDSTRUC), &len_parm,
		  FSCTL_FUNC_WILDMATCH);
}

unsigned attrib_match(unsigned short attr, CATRECANDNAME *crn)
{
  /* Test "must have" attributes */
  if(attr&MUST_HAVE_HIDDEN&0xff00 || attr&MUST_HAVE_SYSTEM&0xff00)
    return 0;

  if(attr&MUST_HAVE_DIRECTORY&0xff00 && crn->cr.cdrType!=CDR_DIR)
    return 0;

  if(!(attr&FILE_DIRECTORY) && crn->cr.cdrType!=CDR_FILE)
    return 0;

  if(!(attr&0x40) && !is_short_filename(crn->name))
    return 0;

  return 1;
}
