/* file.c
   File functions.
*/

#define INCL_DOSFILEMGR
#define INCL_DOSERRORS
#include <os2.h>

#include <stdlib.h>
#include <string.h>
#ifdef DEBUG
#include <stdio.h>
#endif

#include "fsd.h"
#include "names.h"
#include "times.h"
#include "dirsearch.h"
#include "diskfunc.h"

#include "MacFS/defs.h"
#include "MacFS/errors.h"
#include "MacFS/mountrec.h"
#include "MacFS/LibFuncs/resolve.h"
#include "MacFS/LibFuncs/catalog.h"
#include "MacFS/LibFuncs/macreadwrite.h"
#include "MacFS/btree/btree.h"

/******************************************************************************
**
** FS_CHGFILEPTR - Change current location in file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** long offset                          signed offset
** unsigned short type                  indicates seek type
**   values:
**     CFP_RELBEGIN     move pointer relative to begining of file
**     CFP_RELCUR       move pointer relative to current position in file
**     CFP_RELEND       move pointer relative to end of file
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_CHGFILEPTR(struct vpfsd *pvpfsd, 
			struct sffsi *psffsi, struct sffsd *psffsd,
                        long offset, unsigned short type, 
                        unsigned short IOflag)
{
  switch(type) {
  case CFP_RELBEGIN:
    psffsi->sfi_position = offset;
    return NO_ERROR;
  case CFP_RELCUR:
    psffsi->sfi_position += offset;
    return NO_ERROR;
  case CFP_RELEND:
    psffsi->sfi_position = psffsi->sfi_size+offset;
    return NO_ERROR;
  default:
    return ERROR_NOT_SUPPORTED;
  }
}


/******************************************************************************
**
** FS_CLOSE - Close an open file
**
** Parameters
** ----------
** unsigned short type                  indicates close type
**   values:
**     FS_CL_ORDINARY   this is not the final close of the file
**     FS_CL_FORPROC    this is the final close of the file for the process
**     FS_CL_FORSYS     this is the final close of the file for the whole system
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int FS_CLOSE(struct vpfsd *pvpfsd, 
		   unsigned short type, unsigned short IOflag,
                   struct sffsi *psffsi, struct sffsd *psffsd)
{
  int dirty=0;

  if(psffsi->sfi_mode & OPEN_FLAGS_DASD)
    return NO_ERROR;

  /* Do timestamping if necessary */
  if(psffsi->sfi_tstamp & ST_SCREAT && psffsi->sfi_tstamp & ST_PCREAT) {
    dirty=1;
    if(psffsd->cr->cdrType==CDR_FILE)
      psffsd->cr->U.cdrFilRec.filCrDat = os2_to_mac_time_int(psffsi->sfi_cdate,
							   psffsi->sfi_ctime);
    else
      psffsd->cr->U.cdrDirRec.dirCrDat = os2_to_mac_time_int(psffsi->sfi_cdate,
							  psffsi->sfi_ctime);
  }
  if(psffsi->sfi_tstamp & ST_SWRITE && psffsi->sfi_tstamp & ST_PWRITE) {
    dirty=1;
    if(psffsd->cr->cdrType==CDR_FILE)
      psffsd->cr->U.cdrFilRec.filMdDat = os2_to_mac_time_int(psffsi->sfi_mdate,
							   psffsi->sfi_mtime);
    else
      psffsd->cr->U.cdrDirRec.dirMdDat = os2_to_mac_time_int(psffsi->sfi_mdate,
							  psffsi->sfi_mtime);
  }
  psffsi->sfi_tstamp=0;
  if(dirty)
    if(PutCatalogRecord(pvpfsd->mr, psffsd->parid, psffsd->name, psffsd->cr))
      return ERROR_WRITE_FAULT;

  /* The documentation doesn't say if this should be done for all types of close */
  free(psffsd->cr);
  free(psffsd->name);
  return NO_ERROR;
}


/******************************************************************************
**
** FS_COMMIT - Commit a file to disk
**
** Parameters
** ----------
** unsigned short type                  indicates commit type
**   values:
**     FS_COMMIT_ONE    commit this one file
**     FS_COMMIT_ALL    commit all files
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int FS_COMMIT(struct vpfsd *pvpfsd,
		    unsigned short type, unsigned short IOflag,
                    struct sffsi *psffsi, struct sffsd *psffsd)
{
  int dirty=0;

  if(psffsi->sfi_mode & OPEN_FLAGS_DASD)
    return NO_ERROR;

  /* Do timestamping if necessary */
  if(psffsi->sfi_tstamp & ST_SCREAT && psffsi->sfi_tstamp & ST_PCREAT) {
    dirty=1;
    if(psffsd->cr->cdrType==CDR_FILE)
      psffsd->cr->U.cdrFilRec.filCrDat = os2_to_mac_time_int(psffsi->sfi_cdate,
							   psffsi->sfi_ctime);
    else
      psffsd->cr->U.cdrDirRec.dirCrDat = os2_to_mac_time_int(psffsi->sfi_cdate,
							  psffsi->sfi_ctime);
  }
  if(psffsi->sfi_tstamp & ST_SWRITE && psffsi->sfi_tstamp & ST_PWRITE) {
    dirty=1;
    if(psffsd->cr->cdrType==CDR_FILE)
      psffsd->cr->U.cdrFilRec.filMdDat = os2_to_mac_time_int(psffsi->sfi_mdate,
							   psffsi->sfi_mtime);
    else
      psffsd->cr->U.cdrDirRec.dirMdDat = os2_to_mac_time_int(psffsi->sfi_mdate,
							  psffsi->sfi_mtime);
  }
  psffsi->sfi_tstamp=0;
  if(dirty)
    if(PutCatalogRecord(pvpfsd->mr, psffsd->parid, psffsd->name, psffsd->cr))
      return ERROR_WRITE_FAULT;
  return NO_ERROR;
}


/******************************************************************************
**
** FS_COPY - Copy a file
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     DCPY_EXISTING    if destination file exists, replace it
**     DCPY_APPEND      source file should be appended to destination file
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pSrc                           pointer to source filename
** unsigned short iSrcCurDirEnd         offset to the end of the current directory in pSrc
** char *pDst                           pointer to destination filename
** unsigned short iDstCurDirEnd         offset to the end of the current directory in pDst
** unsigned short nameType              0x40 = destination is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int FS_COPY(struct vpfsd *pvpfsd,
		  unsigned short flag, struct cdfsi *pcdfsi,
                  struct cdfsd *pcdfsd, char *pSrc, 
                  unsigned short iSrcCurDirEnd, char *pDst,
                  unsigned short iDstCurDirEnd, unsigned short nameType)
{
  return ERROR_CANNOT_COPY;
}


/******************************************************************************
**
** FS_DELETE - Delete a file
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pFile                          pointer to filename to delete
** unsigned short iCurDirEnd            offset to the end of the current directory in pFile
**
******************************************************************************/

#pragma argsused
short int FS_DELETE(struct vpfsd *pvpfsd,
		    struct cdfsi *pcdfsi, struct cdfsd *pcdfsd, char *pFile, 
                    unsigned short iCurDirEnd)
{
  char macname[CCHMAXPATH], name[MAXFNLENGTH], parname[MAXFNLENGTH];
  CATALOG_RECORD cr, parcr;
  LONGWORD parid, parparid;

  if(os2_to_mac_name(pFile+2, macname))
    return ERROR_ACCESS_DENIED;
  if(macresolvepathname(pvpfsd->mr, macname, &cr, &parid, name))
    return ERROR_ACCESS_DENIED;
  if(cr.cdrType != CDR_FILE)
    return ERROR_ACCESS_DENIED;
  if(macresolvepnwithnonexistinglastelement(pvpfsd->mr, macname, &parparid,
					    parname, &parcr, name))
    return ERROR_ACCESS_DENIED;
  if(mdeletefile(pvpfsd->mr, &parcr, name, &cr))
    return ERROR_WRITE_FAULT;
  return NO_ERROR;
}


/******************************************************************************
**
** FS_FILEATTRIBUTE - Get/Set DOS file attributes
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     FA_RETRIEVE      retrieve attribute
**     FA_SET           set attribute
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short *pAttr                pointer to the attribute
**
******************************************************************************/

#pragma argsused
short int FS_FILEATTRIBUTE(struct vpfsd *pvpfsd, unsigned short flag, struct cdfsi *pcdfsi,
                           struct cdfsd *pcdfsd, char *pName,
                           unsigned short iCurDirEnd, unsigned short *pAttr)
{
  char path[CCHMAXPATH], name[MAXFNLENGTH];
  CATALOG_RECORD cr;
  LONGWORD parid;

  if( flag == FA_SET )
    return ERROR_NOT_SUPPORTED;

  if(os2_to_mac_name(pName+2, path))
    return ERROR_ACCESS_DENIED;

  /* This should be made to use relative pathnames */
  if( macresolvepathname(pvpfsd->mr, path, &cr, &parid, name) )
    return ERROR_FILE_NOT_FOUND;
  
  if( cr.cdrType == CDR_DIR )
    *pAttr = FILE_DIRECTORY;
  else
    *pAttr = FILE_NORMAL;
  return NO_ERROR;
}


/******************************************************************************
**
** FS_FILEINFO - Get/Set file information
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     FI_RETRIEVE      retrieve information
**     FI_SET           set information
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** unsigned short level                 level of information to get/set
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** unsigned long *oError                offset where error occurred in GEA/FEA
**                                      list (EA calls only)
**
******************************************************************************/

#pragma argsused
short int FS_FILEINFO(struct vpfsd *pvpfsd,
		      unsigned short flag, struct sffsi *psffsi,
                      struct sffsd *psffsd, unsigned short level, char *pData, 
                      unsigned short cbData, unsigned short IOflag, 
                      unsigned long *oError)
{
  if(level==FIL_STANDARD || level==FIL_QUERYEASIZE) {
    if(flag==FI_RETRIEVE) {
      if((level==FIL_STANDARD && cbData<sizeof(FILESTATUS))
	 || (level==FIL_QUERYEASIZE && cbData<sizeof(FILESTATUS2)))
	return ERROR_BUFFER_OVERFLOW;
      catrec_to_filebuf(pData, psffsd->cr, level);
      return NO_ERROR;
    }
    else {
      PFILESTATUS p = (PFILESTATUS)pData;
      int dirty=0;

      if(cbData<sizeof(FILESTATUS))
	return ERROR_INSUFFICIENT_BUFFER;
      if(*(unsigned short *)(&p->fdateCreation) 
	 || *(unsigned short *)(&p->ftimeCreation)) {
	psffsi->sfi_cdate=*(unsigned short *)(&p->fdateCreation);
	psffsi->sfi_ctime=*(unsigned short *)(&p->ftimeCreation);
	psffsd->cr->U.cdrFilRec.filCrDat=os2_to_mac_time(p->fdateCreation,
							 p->ftimeCreation);
	psffsi->sfi_tstamp |= ST_PCREAT;
	psffsi->sfi_tstamp &= ~ST_SCREAT;
	dirty=1;
      }
      if(*(unsigned short *)(&p->fdateLastWrite) 
	 || *(unsigned short *)(&p->ftimeLastWrite)) {
	psffsi->sfi_mdate=*(unsigned short *)(&p->fdateLastWrite);
	psffsi->sfi_mtime=*(unsigned short *)(&p->ftimeLastWrite);
	psffsd->cr->U.cdrFilRec.filMdDat=os2_to_mac_time(p->fdateLastWrite,
							 p->ftimeLastWrite);
	psffsi->sfi_tstamp |= ST_PWRITE;
	psffsi->sfi_tstamp &= ~ST_SWRITE;
	dirty=1;
      }
      if(dirty) {
	if(PutCatalogRecord(pvpfsd->mr, psffsd->parid, 
			    psffsd->name, psffsd->cr))
	  return ERROR_ACCESS_DENIED;
      }
      return NO_ERROR;
    }
  }
  else if(level==FIL_QUERYEASFROMLIST || level==FIL_QUERYALLEAS) {
#ifdef DEBUG
    printf("FS_FILEINFO: level %hd\n", level);
#endif
    if(flag==FI_RETRIEVE) {
      ((PFEALIST)pData)->cbList=4;
      *oError=0;
      return NO_ERROR;
    }
    else
      return ERROR_NOT_SUPPORTED;
  }
  return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_MOVE - Move/rename a file
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pSrc                           pointer to source filename
** unsigned short iSrcCurDirEnd         offset to the end of the current directory in pSrc
** char *pDst                           pointer to destination filename
** unsigned short iDstCurDirEnd         offset to the end of the current directory in pDst
** unsigned short flags                 0x40 = destination is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int FS_MOVE(struct vpfsd *pvpfsd,
		  struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                  char *pSrc, unsigned short iSrcCurDirEnd,
                  char *pDst, unsigned short iDstCurDirEnd,
                  unsigned short flags)
{
  /* This should be implemented as MacFS has a move function */
  return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_NEWSIZE - Change size of file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** unsigned long len                    new length of file
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_NEWSIZE(struct vpfsd *pvpfsd,
		     struct sffsi *psffsi, struct sffsd *psffsd,
                     unsigned long len, unsigned short IOflag)
{
  if(psffsi->sfi_mode & OPEN_FLAGS_DASD)
    return ERROR_CANNOT_MAKE;

  if(len>psffsd->cr->U.cdrFilRec.filPyLen) {
    /* Attempt to grow the file */
    int rc = mgrowfile(pvpfsd->mr, psffsd->cr, 
		       len/pvpfsd->mr->vib.AlBlkSiz+1);
    if(rc==MGROWNOTENOUGHSPACE)
      return ERROR_DISK_FULL;
    else if(rc)
      return ERROR_CANNOT_MAKE;
  }
  psffsd->cr->U.cdrFilRec.filLgLen = psffsi->sfi_size = len;
  
  if(PutCatalogRecord(pvpfsd->mr, psffsd->parid, psffsd->name, psffsd->cr))
    return ERROR_WRITE_FAULT;
  return NO_ERROR;
}


/******************************************************************************
**
** FS_OPENCREATE - Open or create a new file
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** unsigned long openmode               sharing and access mode
** unsigned short openflag              action to take when file exists/doesn't exist
** unsigned short *pAction              returns the action that the IFS took
** unsigned short attr                  OS/2 file attributes
** char *pEABuf                         pointer to EAs to attach to new file
** unsigned short *pfgenFlag            flags returned by the IFS
**   values:
**     FOC_NEEDEAS      indicates there are critical EAs associated with the file
** unsigned long *oError                offset where error occurred in FEA list
**
******************************************************************************/

#pragma argsused
short int FS_OPENCREATE(struct vpfsd *pvpfsd,
			struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                        char *pName, unsigned short iCurDirEnd,
                        struct sffsi *psffsi, struct sffsd *psffsd,
                        unsigned long openmode, unsigned short openflag,
                        unsigned short *pAction, unsigned short attr,
                        char *pEABuf, unsigned short *pfgenFlag,
                        unsigned long *oError)
{
  char path[CCHMAXPATH], name[MAXFNLENGTH];
  CATALOG_RECORD cr;
  LONGWORD parid;
  int rc;

#ifdef DEBUG
  printf("FS_OPENCREATE: file = '%s'\n", pName);
#endif

  if(openmode & OPEN_FLAGS_DASD) {
    psffsi->sfi_size = pvpfsd->mr->vib.NmAlBlks*pvpfsd->mr->vib.AlBlkSiz
      + pvpfsd->mr->vib.AlBlSt*BYTESPERSECT;
    psffsi->sfi_position=0;
    return NO_ERROR;
  }

  if(os2_to_mac_name(pName+2, path))
    return ERROR_ACCESS_DENIED;
  /* This should use relative pathnames */
  rc = macresolvepathname(pvpfsd->mr, path, &cr, &parid, name);
  if(rc==0) {
    /* File exists */
    if(cr.cdrType!=CDR_FILE)
      return ERROR_ACCESS_DENIED;
    if((openflag&0x0f) == OPEN_ACTION_OPEN_IF_EXISTS) {
      *pAction=FILE_EXISTED;
      psffsi->sfi_size=cr.U.cdrFilRec.filLgLen;
      psffsi->sfi_ctime=mac_to_os2_time_int(cr.U.cdrFilRec.filCrDat);
      psffsi->sfi_cdate=mac_to_os2_date_int(cr.U.cdrFilRec.filCrDat);
      psffsi->sfi_mtime=mac_to_os2_time_int(cr.U.cdrFilRec.filMdDat);
      psffsi->sfi_mdate=mac_to_os2_date_int(cr.U.cdrFilRec.filMdDat);
      psffsi->sfi_atime=psffsi->sfi_adate=0;

      psffsd->cr=(CATALOG_RECORD *)malloc(sizeof(CATALOG_RECORD));
      *psffsd->cr=cr;
      psffsd->name=(char *)malloc(MAXFNLENGTH);
      strcpy(psffsd->name, name);
      psffsd->parid=parid;
      return NO_ERROR;
    }
    else if((openflag&0x0f) == OPEN_ACTION_REPLACE_IF_EXISTS) {
      CATALOG_RECORD parcr;
      LONGWORD parparid;
      char parname[MAXFNLENGTH];

      *pAction=FILE_TRUNCATED;
      if(macresolvepnwithnonexistinglastelement(pvpfsd->mr, path, &parparid,
					      parname, &parcr, name))
	return ERROR_ACCESS_DENIED;
      if(mcreatefile(pvpfsd->mr, &parcr, name, &cr))
	return ERROR_ACCESS_DENIED;
      
      /* Grow file if necessary */
      if(psffsi->sfi_size) {
	if(psffsi->sfi_size > cr.U.cdrFilRec.filPyLen) {
	  int rc = mgrowfile(pvpfsd->mr, &cr, psffsi->sfi_size/pvpfsd->mr->vib.AlBlkSiz+1);
	  if(rc==MGROWNOTENOUGHSPACE)
	    return ERROR_DISK_FULL;
	  else if(rc)
	    return ERROR_CANNOT_MAKE;
	}
	cr.U.cdrFilRec.filLgLen = psffsi->sfi_size;
	if(PutCatalogRecord(pvpfsd->mr, parid, name, &cr))
	  return ERROR_WRITE_FAULT;
      }

      psffsi->sfi_ctime=mac_to_os2_time_int(cr.U.cdrFilRec.filCrDat);
      psffsi->sfi_cdate=mac_to_os2_date_int(cr.U.cdrFilRec.filCrDat);
      psffsi->sfi_mtime=mac_to_os2_time_int(cr.U.cdrFilRec.filMdDat);
      psffsi->sfi_mdate=mac_to_os2_date_int(cr.U.cdrFilRec.filMdDat);
      psffsi->sfi_atime=psffsi->sfi_adate=0;
      psffsi->sfi_tstamp |= ST_PCREAT | ST_PWRITE;
      
      psffsd->cr=(CATALOG_RECORD *)malloc(sizeof(CATALOG_RECORD));
      *psffsd->cr=cr;
      psffsd->name=(char *)malloc(MAXFNLENGTH);
      strcpy(psffsd->name, name);
      psffsd->parid=parid;
      return NO_ERROR;
    }
    else if((openflag&0x0f) == OPEN_ACTION_FAIL_IF_EXISTS)
      return ERROR_OPEN_FAILED;
    else
      return ERROR_ACCESS_DENIED;
  }
  else {
    /* File doesn't exist */
    if((openflag&0xf0) == OPEN_ACTION_CREATE_IF_NEW) {
      CATALOG_RECORD parcr;
      LONGWORD parparid;
      char parname[MAXFNLENGTH];

      if(macresolvepnwithnonexistinglastelement(pvpfsd->mr, path, &parparid, 
					      parname, &parcr, name))
	return ERROR_ACCESS_DENIED;
      if(mcreatefile(pvpfsd->mr, &parcr, name, &cr))
	return ERROR_ACCESS_DENIED;

      /* Grow file if necessary */
      if(psffsi->sfi_size) {
	if(psffsi->sfi_size > cr.U.cdrFilRec.filPyLen) {
	  int rc = mgrowfile(pvpfsd->mr, &cr, psffsi->sfi_size/pvpfsd->mr->vib.AlBlkSiz+1);
	  if(rc==MGROWNOTENOUGHSPACE)
	    return ERROR_DISK_FULL;
	  else if(rc)
	    return ERROR_CANNOT_MAKE;
	}
	cr.U.cdrFilRec.filLgLen = psffsi->sfi_size;
	if(PutCatalogRecord(pvpfsd->mr, parcr.U.cdrDirRec.dirDirID, name, &cr))
	  return ERROR_WRITE_FAULT;
      }

      psffsi->sfi_ctime=mac_to_os2_time_int(cr.U.cdrFilRec.filCrDat);
      psffsi->sfi_cdate=mac_to_os2_date_int(cr.U.cdrFilRec.filCrDat);
      psffsi->sfi_mtime=mac_to_os2_time_int(cr.U.cdrFilRec.filMdDat);
      psffsi->sfi_mdate=mac_to_os2_date_int(cr.U.cdrFilRec.filMdDat);
      psffsi->sfi_atime=psffsi->sfi_adate=0;
      psffsi->sfi_tstamp |= ST_PCREAT | ST_PWRITE;
      
      psffsd->cr=(CATALOG_RECORD *)malloc(sizeof(CATALOG_RECORD));
      *psffsd->cr=cr;
      psffsd->name=(char *)malloc(MAXFNLENGTH);
      strcpy(psffsd->name, name);
      psffsd->parid=parcr.U.cdrDirRec.dirDirID;
      *pAction=FILE_CREATED;
      return NO_ERROR;
    }
    else if((openflag&0xf0) == OPEN_ACTION_FAIL_IF_NEW)
      return ERROR_FILE_NOT_FOUND;
    else
      return ERROR_ACCESS_DENIED;
  }
}


/******************************************************************************
**
** FS_PATHINFO - get/set file information by filename
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     PI_RETRIEVE      retrieve information
**     PI_SET           set information
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short level                 level of information to get/set
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned long *oError                offset where error occurred in GEA/FEA
**                                      list (EA calls only)
**
******************************************************************************/

#pragma argsused
short int FS_PATHINFO(struct vpfsd *pvpfsd,
		      unsigned short flag, struct cdfsi *pcdfsi,
                      struct cdfsd *pcdfsd, char *pName, 
                      unsigned short iCurDirEnd, unsigned short level, 
                      char *pData, unsigned short cbData, 
                      unsigned long *oError)
{
  char macname[CCHMAXPATH], name[MAXFNLENGTH];
  CATALOG_RECORD cr;
  LONGWORD parid;

  if(os2_to_mac_name(pName+2, macname))
    return ERROR_ACCESS_DENIED;
  if(macresolvepathname(pvpfsd->mr, macname, &cr, &parid, name))
    return ERROR_PATH_NOT_FOUND;

  if(level==FIL_STANDARD || level==FIL_QUERYEASIZE) {
    if(flag==FI_RETRIEVE) {
      if((level==FIL_STANDARD && cbData<sizeof(FILESTATUS))
	 || (level==FIL_QUERYEASIZE && cbData<sizeof(FILESTATUS2)))
	return ERROR_BUFFER_OVERFLOW;
      catrec_to_filebuf(pData, &cr, level);
      return NO_ERROR;
    }
    else {
      PFILESTATUS p = (PFILESTATUS)pData;
      int dirty=0;

      if(cbData<sizeof(FILESTATUS))
	return ERROR_INSUFFICIENT_BUFFER;
      if(*(unsigned short *)(&p->fdateCreation) 
	 || *(unsigned short *)(&p->ftimeCreation)) {
	cr.U.cdrFilRec.filCrDat=os2_to_mac_time(p->fdateCreation,
						p->ftimeCreation);
	dirty=1;
      }
      if(*(unsigned short *)(&p->fdateLastWrite) 
	 || *(unsigned short *)(&p->ftimeLastWrite)) {
	cr.U.cdrFilRec.filMdDat=os2_to_mac_time(p->fdateLastWrite,
						p->ftimeLastWrite);
	dirty=1;
      }
      if(dirty) {
	if(PutCatalogRecord(pvpfsd->mr, parid, name, &cr))
	  return ERROR_ACCESS_DENIED;
      }
      return NO_ERROR;
    }
  }
  else if(level==FIL_QUERYEASFROMLIST || level==FIL_QUERYALLEAS) {
    if(flag==FI_RETRIEVE) {
      ((PFEALIST)pData)->cbList=4;
      *oError=0;
      return NO_ERROR;
    }
    else
      return ERROR_NOT_SUPPORTED;
  }
  return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_READ - read data from a file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** char *pData                          pointer to buffer
** unsigned short *pLen                 length of buffer
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_READ(struct vpfsd *pvpfsd,
		  struct sffsi *psffsi, struct sffsd *psffsd,
                  char *pData, unsigned short *pLen, unsigned short IOflag)
{
  LONGWORD bytes_read;
  char *buf;

  if(psffsi->sfi_position >= psffsi->sfi_size) {
    *pLen=0;
    return NO_ERROR;
  }

  if(psffsi->sfi_mode & OPEN_FLAGS_DASD) {
    buf=(char *)malloc(*pLen+SECTOR_SIZE*3);
    if(read_sector(psffsi->sfi_position/SECTOR_SIZE, *pLen/SECTOR_SIZE+3,
		pvpfsd->mr->sr.hVPB, buf) != NO_ERROR) {
      free(buf);
      return ERROR_READ_FAULT;
    }
    memcpy(pData, buf+psffsi->sfi_position%SECTOR_SIZE, *pLen);
    free(buf);
    psffsi->sfi_position += *pLen;
  }
  else {
    /* We need to use another buffer because the pData buffer is inaccessible
       by the ring 0 piece */
    buf = (char *)malloc(*pLen);
    if(mread(pvpfsd->mr, psffsd->cr, psffsi->sfi_position, *pLen, buf, &bytes_read)) {
      *pLen=0;
      free(buf);
      return ERROR_READ_FAULT;
    }
    memcpy(pData, buf, bytes_read);
    free(buf);
    psffsi->sfi_position += bytes_read;
    *pLen = bytes_read;
  }
  return NO_ERROR;
}


/******************************************************************************
**
** FS_WRITE - write data to a file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** char *pData                          pointer to buffer
** unsigned short *pLen                 length of buffer
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_WRITE(struct vpfsd *pvpfsd,
		   struct sffsi *psffsi, struct sffsd *psffsd,
                   char *pData, unsigned short *pLen, unsigned short IOflag)
{
  LONGWORD bytes_written;
  char *buf;
  int rc;

  /* If positioned beyond end of file, disallow write */
  if(psffsi->sfi_position > psffsi->sfi_size) {
    *pLen=0;
    return NO_ERROR;
  }

  if(psffsi->sfi_mode & OPEN_FLAGS_DASD) {
    int start_offset = psffsi->sfi_position%SECTOR_SIZE;
    int end_offset = (start_offset+*pLen)%SECTOR_SIZE;
    buf=(char *)malloc(*pLen+SECTOR_SIZE*3);
    /* Read first partial sector */
    if(start_offset)
      if(read_sector(psffsi->sfi_position/SECTOR_SIZE, 1, 
		     pvpfsd->mr->sr.hVPB, buf) != NO_ERROR) {
	free(buf);
	return ERROR_READ_FAULT;
      }
    /* Read last partial sector */
    if(end_offset)
      if(read_sector((psffsi->sfi_position+*pLen)/SECTOR_SIZE, 1, pvpfsd->mr->sr.hVPB, 
		     buf+SECTOR_SIZE*((start_offset+*pLen)/SECTOR_SIZE))
	 != NO_ERROR) {
	free(buf);
	return ERROR_READ_FAULT;
      }
    
    memcpy(buf+start_offset, pData, *pLen);
    if(write_sector(psffsi->sfi_position/SECTOR_SIZE, (start_offset+*pLen)/SECTOR_SIZE,
		    pvpfsd->mr->sr.hVPB, buf) != NO_ERROR) {
      free(buf);
      return ERROR_WRITE_FAULT;
    }
    free(buf);
    psffsi->sfi_position += *pLen;
  }
  else {
    /* We allocate a temporary buffer because pData is inaccessible by ring 0 piece */
    buf = (char *)malloc(*pLen);
    memcpy(buf, pData, *pLen);
    rc=mwrite(pvpfsd->mr, psffsd->cr, psffsi->sfi_position, *pLen, buf, &bytes_written);
    free(buf);
    if(rc) {
      *pLen=0;
      return ERROR_WRITE_FAULT;
    }
    psffsi->sfi_position += bytes_written;
    if(psffsi->sfi_position > psffsi->sfi_size)
      psffsi->sfi_size=psffsi->sfi_position;
    *pLen = bytes_written;
    psffsi->sfi_tstamp |= ST_SWRITE | ST_PWRITE;
  }
  return NO_ERROR;
}
