/* names.c
   Filename translation functions.
*/

#include <string.h>
#include <ctype.h>
#ifdef DEBUG
#include <stdio.h>
#endif

#include "names.h"
#include "trans.h"

int filename_translation = FN_TR_CP850;
unsigned char quote_char = '%';

static int mac_to_os2_name_cp850(const unsigned char *, unsigned char *);
static int os2_to_mac_name_cp850(const unsigned char *, unsigned char *);
static unsigned char to_hex(int);

/* Translate Macintosh filename to OS/2 filename. Nonzero return value means filename
   is invalid. */
int mac_to_os2_name(const unsigned char *src, unsigned char *dest)
{
#ifdef _DEBUG
  int rc = mac_to_os2_name_cp850(src, dest);
  printf("Mac name '%s' translated to '%s'\n", src, dest);
  return rc;
#else
  return mac_to_os2_name_cp850(src, dest);
#endif
}

/* Translate OS/2 filename to Macintosh filename. Nonzero return value means filename
   is invalid. */
int os2_to_mac_name(const unsigned char *src, unsigned char *dest)
{
#ifdef _DEBUG
  int rc = os2_to_mac_name_cp850(src, dest);
  printf("OS/2 name '%s' translated to '%s'\n", src, dest);
  return rc;
#else
  return os2_to_mac_name_cp850(src, dest);
#endif
}

static int mac_to_os2_name_cp850(const unsigned char *src, unsigned char *dest)
{
  int i, j;
  char *p=dest;

  for(i=j=0; i<strlen(src); i++) {
    unsigned char c=src[i];
    if(c==MAC_PATH_SEPARATOR)
      *p++ = OS2_PATH_SEPARATOR;
    else if(c>=32 && c<=127 && !os2_invalid(c) && c!=quote_char)
      *p++ = c;
    else if(c>=128 && mac_to_cp850(c))
      *p++ = mac_to_cp850(c);
    else {
      *p++ = quote_char;
      *p++ = to_hex(c/16);
      *p++ = to_hex(c%16);
    }
  }
  *p=0;
  return 0;
}

static int os2_to_mac_name_cp850(const unsigned char *src, unsigned char *dest)
{
  static int table_generated;
  int i, j;
  unsigned char *p=dest;

  /* Generate reverse mapping first time */
  if(!table_generated) {
    int i;
    for(i=0; i<128; i++)
      if(mac_to_cp850_map[i])
	cp850_to_mac_map[mac_to_cp850_map[i]-128]=i+128;
    table_generated=1;
  }

  for(i=j=0; i<strlen(src) && p-dest<MAX_MAC_FN_LEN; i++) {
    unsigned char c=src[i];
    if(c==OS2_PATH_SEPARATOR)
      *p++ = MAC_PATH_SEPARATOR;
    else if(c==quote_char) {
      if(strlen(src+i+1)<2)
	return 1;
      if(isxdigit(src[i+1]) && isxdigit(src[i+2])) {
	int j, n=0;
	for(j=1; j<=2; j++) {
	  unsigned char ch=src[i+j];
	  n <<= 4;
	  if(isalpha(ch))
	    n+=toupper(ch)-'A'+10;
	  else
	    n+=ch-'0';
	}
	if(n==quote_char)
	  *p++ = quote_char;
	else if(n>=128 && cp850_to_mac(n))
	  *p++ = cp850_to_mac(n);
	else
	  return 1;
      }
      else
	return 1;
    }
    else if(c>=32 && c<=127)
      *p++ = c;
    else if(c>=128)
      if(cp850_to_mac(c))
	*p++ = cp850_to_mac(c);
      else
	return 1;
    else
      return 1;
  }
  *p=0;
  if(i==strlen(src))
    return 0;
  else
    return 1;
}

static unsigned char to_hex(int i)
{
  if(i<10)
    return i+'0';
  else
    return i-10+'a';
}

/* Determine if filename is FAT 8.3 format */
unsigned is_short_filename(char *name)
{
  const char FAT_special_chars[]="$%'-_@{}~`!#()";
  int i, j, k;

  /* Check name */
  for( i=j=k=0; i<strlen(name) && name[i]!='.'; i++ ) {
    if(!isalnum(name[i]) && !strchr(FAT_special_chars, name[i]) && name[i]!=' ')
      return 0;

    if(name[i]==' ')
      k++;
    else if(k>0) {
      j+=k;
      k=0;
      j++;
    }
    else
      j++;
  }
  if(j>8)
    return 0;

  if(i==strlen(name))
    return 1;
  i++;

  /* Check extension */
  for( j=k=0; i<strlen(name); i++ ) {
    if(!isalnum(name[i]) && !strchr(FAT_special_chars, name[i]) && name[i]!=' ')
      return 0;

    if(name[i]==' ')
      k++;
    else if(k>0) {
      j+=k;
      k=0;
      j++;
    }
    else
      j++;
  }
  if(j>3)
    return 0;

  return 1;
}
