/* vol.c
   Volume management.
*/

#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#define INCL_NOCOMMON
#define INCL_NOPMAPI
#include <os2.h>

#include <stdlib.h>
#include <string.h>
#ifdef DEBUG
#include <stdio.h>
#endif

#include "fsd.h"
#include "../r0r3shar.h"
#include "diskfunc.h"
#include "diskstruc.h"
#include "vol.h"
#include "cache.h"
#include "fsctl.h"
#include "names.h"

#include "MacFS/mountrec.h"
#include "MacFS/DevSupport/readwrite.h"
#include "MacFS/Volume/volinfo.h"
#include "MacFS/Volume/volbmap.h"
#include "MacFS/btree/btree.h"

/******************************************************************************
** Name of IFS
******************************************************************************/

const char FS_NAME[] = "HFS";

const unsigned long FSID = 0x32534648;  /* 'HFS2', used in FS_FSINFO */


/*-----------------------------------------------------------------------------
--
-- Volume management
--
-----------------------------------------------------------------------------*/


/******************************************************************************
**
** FS_MOUNT - Examine a volume to determine if the IFS knows its format
**
** Parameters
** ----------
** unsigned short flag                  indicates function to perform
**   values:
**     MOUNT_MOUNT              mount or accept the volume
**     MOUNT_VOL_REMOVED        volume has been removed
**     MOUNT_RELEASE            release all resources associated with the volume
**     MOUNT_ACCEPT             accept the volume in preparation for formatting
** struct vpfsi far *pvpfsi             pointer to FSD independant volume parameters
** struct vpfsd far *pvpfsd             pointer to FSD dependant volume parameters
** unsigned short hVPB                  volume handle
** char far *pBoot                      pointer to sector 0 data
**
******************************************************************************/

short int FS_MOUNT(unsigned short flag, struct vpfsi *pvpfsi,
                   struct vpfsd *pvpfsd, unsigned short hVPB, 
                   unsigned char *pBoot)
{
  EXTENDED_BOOT *boot = (EXTENDED_BOOT *)pBoot;
  unsigned short oldVPB;
  unsigned long len_parm;
  int status;
  unsigned char label[CCHMAXPATH], maclabel[VOLNAMELEN];

#ifdef DEBUG
  printf("Mount, flag = %hu, hVPB = %hu\n", flag, hVPB);
#endif
  switch(flag) {
    case MOUNT_MOUNT:
      /* The cache uses hVPB as part of the key, so we flush it here. */
      flush_cache();
      /* Check for our homemade HFS signature */
      if( strncmp(boot->Boot_System_ID, "HFS     ", 8) )
        return ERROR_VOLUME_NOT_MOUNTED;

      /* Check for duplicate VPB */
      oldVPB = hVPB;
#ifdef DEBUG
      printf("Checking for dup VPB...");
#endif
      if( do_FSCtl(NULL, 0, NULL,
                   &oldVPB, sizeof(unsigned short), &len_parm,
                   FSCTL_FUNC_FINDDUPHVPB)
	 == NO_ERROR ) {
#ifdef DEBUG
        printf("yes\n");
#endif
	/* We should read the vol info and bitmap again here for the old VPB,
	   as the volume may have been written while it was removed. */

	/* Mark the VPB as duplicate */
	set_vol_status(hVPB, pvpfsi->vpi_drive, MTSTAT_DUPLICATE);
	/* If necessary, mark the old VPB as mounted */
	if(get_mount_status(oldVPB)==MTSTAT_REMOVED)
	  set_vol_status(oldVPB, pvpfsi->vpi_drive, MTSTAT_MOUNTED);
#ifdef DEBUG
        printf("Found dup VPB = %hu\n", oldVPB);
#endif
        return NO_ERROR;
      }
 
#ifdef DEBUG
      printf("no\n");
#endif
      /* Set up new mount record */
      pvpfsd->mr = (MOUNT_RECORD *)malloc(sizeof(MOUNT_RECORD));
      if(initreadwrite(hVPB, pvpfsd->mr)) {
#ifdef DEBUG
        printf("initreadwrite failed!\n");
#endif
      }
      if(readvolinfo(pvpfsd->mr)) {
#ifdef DEBUG
        printf("readvolinfo failed!\n");
#endif
      }
      if(readvolbmap(pvpfsd->mr)) {
#ifdef DEBUG
        printf("readvolbmap failed!\n");
#endif
      }
      if(BTreeMount(pvpfsd->mr)) {
#ifdef DEBUG
        printf("BTreeMount failed!\n");
#endif
      }

      /* Fill vpfsi fields */
      /* Use creation timestamp as ID */
      pvpfsi->vpi_vid = pvpfsd->mr->vib.CrDate;
      strncpy(maclabel, pvpfsd->mr->vib.VN, pvpfsd->mr->vib.VNlen);
      maclabel[pvpfsd->mr->vib.VNlen]=0;
      mac_to_os2_name(maclabel, label);
      strncpy(pvpfsi->vpi_text, label, 12);
      pvpfsi->vpi_text[11]=0;

      set_vol_status(hVPB, pvpfsi->vpi_drive, MTSTAT_MOUNTED);
#ifdef DEBUG
      printf("Got HFS volume\n");
#endif
      return NO_ERROR;

    case MOUNT_VOL_REMOVED:
      flush_cache();
      set_vol_status(hVPB, pvpfsi->vpi_drive, MTSTAT_REMOVED);
      return NO_ERROR;

    case MOUNT_RELEASE:
      status = get_mount_status(hVPB);
      if(status==MTSTAT_MOUNTED || status==MTSTAT_REMOVED) {
	/* We should sync here, but this does not seem to work. For the
	   moment, the user will have to run sync.exe manually. */
	/*
        BTreeSync(pvpfsd->mr);
	writevolbmap(pvpfsd->mr);
	writevolinfo(pvpfsd->mr);
	*/
	flush_cache();
        closereadwrite(pvpfsd->mr);
	free(pvpfsd->mr);
      }
      else if(status!=MTSTAT_DUPLICATE) {
#ifdef DEBUG
	printf("Trying to unmount non-mounted volume!\n");
#endif
      }
      set_vol_status(hVPB, 0, MTSTAT_FREE);
      return NO_ERROR;

    case MOUNT_ACCEPT:
      return ERROR_NOT_SUPPORTED;

    default:
      return ERROR_NOT_SUPPORTED;
  }
}

/******************************************************************************
**
** FS_FSINFO - Get/Set file system information
**
** Parameters
** ----------
** unsigned short flag                  indicates function to perform
**   values:
**     INFO_RETREIVE                    retrieve information
**     INFO_SET                         set information
** unsigned short hVPB                  VPB handle
** struct vpfsi *pvpfsi                 pointer to FSD independent volume parameters
** struct vpfsd *pvpfsd                 pointer to FSD dependant volume parameters
** char far *pData                      pointer to data buffer
** unsigned short cbData                length of data buffer
** unsigned short level                 type of information to return
**
******************************************************************************/

#pragma argsused
short int FS_FSINFO(unsigned short flag, unsigned short hVPB,
                    struct vpfsi *vpfsi, struct vpfsd *vpfsd, 
                    char *pData, unsigned short cbData, unsigned short level)
{
  if( flag==INFO_RETRIEVE ) {
    switch(level) {
      case FSIL_ALLOC:
        if( cbData < sizeof(FSALLOCATE) )
          return ERROR_BUFFER_OVERFLOW;
        ((PFSALLOCATE)pData)->idFileSystem = FSID;
        ((PFSALLOCATE)pData)->cSectorUnit = vpfsd->mr->vib.AlBlkSiz/SECTOR_SIZE;
        ((PFSALLOCATE)pData)->cUnit = vpfsd->mr->vib.NmAlBlks;
        ((PFSALLOCATE)pData)->cUnitAvail = vpfsd->mr->vib.FreeBks;
        ((PFSALLOCATE)pData)->cbSector = SECTOR_SIZE;
        break;
      case FSIL_VOLSER:
        if( cbData < sizeof(FSINFO) )
          return ERROR_BUFFER_OVERFLOW;
        ((PFSINFO2)pData)->ulVSN = vpfsi->vpi_vid;
	strcpy(((PFSINFO2)pData)->vol.szVolLabel, vpfsi->vpi_text);
	((PFSINFO2)pData)->vol.cch = strlen(vpfsi->vpi_text);
        break;
      default:
        return ERROR_INVALID_LEVEL;
    }
  }
  else if( flag==INFO_SET ) {
    if( level != FSIL_VOLSER )
      return ERROR_INVALID_LEVEL;
    if( vpfsi->vpi_vid != ((PFSINFO2)pData)->ulVSN )
      return ERROR_CANNOT_MAKE;  /* Cannot change serial number */
    strcpy(vpfsi->vpi_text, ((PFSINFO2)pData)->vol.szVolLabel);
    /* Update VIB */
    vpfsd->mr->vib.VNlen = strlen(vpfsi->vpi_text);
    strcpy(vpfsd->mr->vib.VN, vpfsi->vpi_text);
  }
  else 
    return ERROR_NOT_SUPPORTED;

  return NO_ERROR;
}


/******************************************************************************
**
** FS_FLUSHBUF - Flush buffers for a specified volume
**
** Parameters
** ----------
** struct vpfsd *pvpfsd                 pointer to FSD dependant volume parameters
** unsigned short flag                  indicates whether to discard or retain cache
**   values:
**     FLUSH_RETAIN     retain cached information
**     FLUSH_DISCARD    discard cached information
**
******************************************************************************/

#pragma argsused
short int FS_FLUSHBUF(struct vpfsd *vpfsd, unsigned short flag)
{
  if(syncmountrecord(vpfsd->mr)) {
#ifdef DEBUG
    printf("FS_FLUSHBUF: syncmountrecord() failed!\n");
#endif
    return ERROR_WRITE_FAULT;
  }
  if(flag==FLUSH_DISCARD)
    flush_cache();
  return NO_ERROR;
}

/******************************************************************************
**
** FS_ATTACH - Attach or detach a drive or device
**
** Parameters
** ----------
** unsigned short flag                  indicates attaching/detaching
**   values:
**     FSA_ATTACH               attach drive/device
**     FSA_DETACH               detach drive/device
**     FSA_ATTACH_INFO          return info on attached drive/device
** char *pDev                           drive or device that is being attached/detached
** struct vpfsd *pvpfsd                 pointer to FSD dependant volume parameters
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** void *pParm                          pointer to FSD dependant attachment info
** unsigned short far *pLen             length of area pointed to by pParm
**
******************************************************************************/

#pragma argsused
short int FSD_ATTACH(unsigned short flag, char *pDev, struct vpfsd *pvpfsd,
                    struct cdfsd *pcdfsd, void *pParm,
                    unsigned short *pLen)
{
  return ERROR_NOT_SUPPORTED;
}

/******************************************************************************
** Mount status functions
******************************************************************************/

unsigned short get_mount_status(unsigned short hVPB)
{
  unsigned long lenparam=sizeof(unsigned short);
  MTSTATSTRUC mts;
  unsigned short *param = (unsigned short *)&mts;

  *param = hVPB;
  if( do_FSCtl(NULL, 0, NULL, param, sizeof(MTSTATSTRUC), &lenparam,
	       FSCTL_FUNC_GET_MTSTAT) != NO_ERROR ) {
#ifdef DEBUG
    printf("FSCTL_FUNC_GET_MTSTAT failed!\n");
#endif
    return (unsigned short)MTSTAT_ERROR;
  }
  return mts.status;
}

void set_vol_status(unsigned short hVPB, unsigned short drive, 
		    unsigned short status)
{
  unsigned long lenparam=sizeof(MTSTATSTRUC);
  MTSTATSTRUC mts;

  mts.drive = drive;
  mts.status = status;
  mts.hVPB = hVPB;
  if( do_FSCtl(NULL, 0, NULL, &mts, sizeof(MTSTATSTRUC), &lenparam,
	       FSCTL_FUNC_SET_MTSTAT) != NO_ERROR ) {
#ifdef DEBUG
    printf("FSCTL_FUNC_SET_MTSTAT failed!\n");
#endif
  }
}
