/*
Graphical display of the TCP/IP connection showing:
    Momentary and overall data recieve rate
    Percent utilization (active time/total time)
    Momentary and overall data send rate

Requires: TCP/IP and VREXX

Parameters: Single, numeric value representing the amount
of time (seconds) to wait between TCP/IP poll operations.
Missing or invalid(character) values result in default wait
of 10 seconds.

John Ross Porter                               28 June 1996
joropo@notes.cc.bellcore.com
*/
PARSE ARG sltm .

/* Validate parameters */
IF ( DATATYPE( sltm ) = "CHAR" ) THEN DO
    sltm = 10
    SAY "Default sleep time set (10 sec.)"
    END

IF ( sltm < 1 ) THEN DO
    sltm = 10
    SAY "Default sleep time set (10 sec.)"
    END

/* Variable/ counter initializations */    
ttl_time = 0
idl_time = 0
atv_time = 0

first_bytrec = 0
first_bytsnt = 0

last_bytrec = 0
last_bytsnt = 0

/* initialize VREXX */
IF RxFuncQuery( 'VInit' ) THEN DO                       
    call RxFuncAdd 'VInit', 'VREXX', 'VINIT'
    END

initcode = VInit()
IF initcode = 'ERROR' THEN SIGNAL CLEANUP
 
SIGNAL ON HALT    NAME CLEANUP
SIGNAL ON ERROR   NAME BUMMED
SIGNAL ON SYNTAX  NAME BUMMED
SIGNAL ON FAILURE NAME CLEANUP
 
/* check whether RxFuncs are loaded, if not, load them */    
IF RxFuncQuery( 'SysLoadFuncs' ) THEN DO                       
    CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
    CALL SysLoadFuncs                                        
    END                                                      

/* Define a queue to get poll results from */
/* qname = rxqueue( "Create" ) */
/* oldq  = rxqueue( "Set", qname ) */

/* Initialize VREXX window */
CALL Init_Vwin

/* Start of polling loop */
DO forever

    /* wait */
    CALL syssleep sltm

    /* Ask for current byte counts */              
    "@NETSTAT -n | RXQUEUE " /* qname */

    /* pick out the poll data we want */
    CALL RawData

    /* How long since last here? */
    now = TIME( "Reset" )

    /* Time online (polling) */
    ttl_time  = ttl_time + now
   
    IF ( bytrec = '' ) THEN
        LEAVE

    IF ( now > 0 ) THEN DO /* have enough to start drawing */
        /* rate since last poll and since polling started */
        recv_rate = (bytrec - last_bytrec)  / now
        send_rate = (bytsnt - last_bytsnt)  / now
        rovr_rate = (bytrec - first_bytrec) / ttl_time
        sovr_rate = (bytsnt - first_bytsnt) / ttl_time

        /* anything happen since we last looked? */
        IF ( recv_rate = 0 & send_rate = 0 ) THEN DO
            idl_time = idl_time + now
            END
        ELSE DO
            atv_time = atv_time + now
            END

        /* update display */
        CALL VUpdate
        END  /* Do */
    ELSE DO /* first time around polling loop */
        /* set base for overall rate calculations */
        first_bytrec = bytrec
        first_bytsnt = bytsnt
        END  /* Do */

    /* new base for momentary rate calculations */
    last_bytrec = bytrec
    last_bytsnt = bytsnt
    END /* forever */

/* done */
SIGNAL CLEANUP

/*====================================================================
In case of trouble 
    (very useful/necessary while developing!)
====================================================================*/
BUMMED:
TRACE OFF
SAY CONDITION('C') "signaled:" CONDITION('D') "at line" sigl

/*====================================================================
MUST execute the following code NO MATTER WHAT 
to gracefully shutdown VREXX
====================================================================*/
CLEANUP:
/* some final statistics */
IF ( last_bytrec > 0 ) THEN DO
    SAY "recieved:" FORMAT( last_bytrec, 9) "bytes",
        "    Rate:" FORMAT( rovr_rate, 6, 1) "bytes/sec"
    SAY "    sent:" FORMAT( last_bytsnt, 9) "bytes",
        "    Rate:" FORMAT( sovr_rate, 6, 1) "bytes/sec"
    SAY "    time:" FORMAT( ttl_time, 4, 2) "seconds",
        " Utilization:" FORMAT( 100 * atv_time / ttl_time, 3, 2)"%"
    END
ELSE
    SAY "No TCP/IP Connection present"

/* delete queue  */
/* CALL RXQUEUE "delete", qname */

/* pause to keep final display until user dismisses */
SAY "Enter to complete"
PULL ans

/* terminate VREXX */
CALL VExit
 
EXIT
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/*====================================================================
initialize data structures for VREXX interface
    define initial window
    position, within window, of the three graphs
    label graphs
    initialize graph data
====================================================================*/
Init_Vwin:

BackGround = 'BLACK'

win_pos.top    = 100
win_pos.bottom = 75
win_pos.left   = 0
win_pos.right  = 25
wid = VOpenWindow( 'TCP/IP Utilization', BackGround, win_pos)

np = 50
M_ofst = 1
X_ofst = 2
W_ofst = 3

rpos.M_ofst = 0
rpos.X_ofst = 10
rpos.W_ofst = 384

upos.M_ofst = 100
upos.X_ofst = 404
upos.W_ofst = 192

spos.M_ofst = 0
spos.X_ofst = 606
spos.W_ofst = 384

Ymin = 10
Ymax = 900

recv.0 = "GREEN"
rovr.0 = "CYAN"
util.0 = "YELLOW"
send.0 = "RED"
sovr.0 = "PINK"

CALL VForeColor wid, rovr.0
CALL Vsay wid, rpos.X_ofst, Ymax + 1, "Recv"

CALL VForeColor wid, util.0
CALL Vsay wid, upos.X_ofst, Ymax + 1, "Util"

CALL VForeColor wid, Sovr.0
CALL Vsay wid, spos.X_ofst, Ymax + 1, "Send"

pdelt = (Ymax - Ymin) / np
DO i = 1 TO np
    recv.i = rpos.X_ofst
    rovr.i = rpos.X_ofst
    util.i = upos.X_ofst
    send.i = spos.X_ofst
    sovr.i = spos.X_ofst
    Y.i = Ymin + (i - 1) * pdelt
    END     

RETURN

/*====================================================================
(re)draw the graphs based upon current data
====================================================================*/
VUpdate:
/* erase previous 'recieve rate' lines */
CALL VForeColor  wid, BackGround
CALL VDraw       wid, 'LINE', recv, Y, np
CALL VDraw       wid, 'LINE', rovr, Y, np

/* new 'momentary rate' line */
CALL Rate_Append recv_rate,  "recv", "rpos"
CALL VForeColor  wid,         recv.0
CALL VDraw       wid, 'LINE', recv, Y, np

/* new 'overall rate' line */
CALL Rate_Append rovr_rate,  "rovr", "rpos"
CALL VForeColor  wid,         rovr.0
CALL VDraw       wid, 'LINE', rovr, Y, np

/* erase previous 'utilization' line */
CALL VForeColor  wid, BackGround
CALL VDraw       wid, 'LINE', util, Y, np

/* new 'utilization' line */
CALL Rate_Append 100 * (atv_time / ttl_time), "util", "upos"
CALL VForeColor  wid,                          util.0
CALL VDraw       wid, 'LINE',                  util, Y, np

/* erase previous 'send rate' lines */
CALL VForeColor  wid, BackGround
CALL VDraw       wid, 'LINE', send, Y, np
CALL VDraw       wid, 'LINE', sovr, Y, np

/* new 'momentary' rate line */
CALL Rate_Append send_rate,  "send", "spos"
CALL VForeColor  wid,         send.0
CALL VDraw       wid, 'LINE', send, Y, np

/* new 'overall' rate line */
CALL Rate_Append sovr_rate,  "sovr", "spos"
CALL VForeColor  wid,         sovr.0
CALL VDraw       wid, 'LINE', sovr, Y, np

RETURN

/*====================================================================
Very Busy Routine
    called five times for each poll interval
    updates and rescales if necessary the 'rate' data arrays
====================================================================*/
Rate_Append: 

R = ARG(1) /* rate value */
A = ARG(2) /* 'name' of rate array */
P = ARG(3) /* 'name' of chart position parameters */

INTERPRET "Rm = "P".M_ofst" /* current rate maximum */
INTERPRET "Z = "P".X_ofst"  /* graph 'zero' position */
INTERPRET "W = "P".W_ofst"  /* graph width or 'max rate' offset */

/* have to work 'real hard' if new maximum rate present */
IF ( R > Rm ) THEN DO
    /* rate-rescale constants */
    K1 = Rm / R
    K2 = Z * (R - Rm) / R

    /* erase previous maximum rate display */
    CALL VForeColor wid, BackGround
    CALL VSay       wid, Z + W - 120, Ymax + 1, FORMAT(Rm, ,0)
    /* save new maximum */
    Rm = R
    INTERPRET P".M_ofst = "R

    /* display new maximum rate value */
    INTERPRET "clr = "A".0"
    CALL VForeColor wid, clr
    CALL VSay       wid, Z + W - 120, Ymax + 1, FORMAT(R, ,0)

    readj = 1 /* flag on */
    END  /* Do */
ELSE
    readj = 0 /* flag off - no readjustment necessary - whew */

/* scroll the values in the rate array */
DO i = np TO 2 BY -1
    j = i - 1
    IF ( readj = 1 ) THEN /* new maximum */
        INTERPRET A".i = "A".j * K1 + K2"
    ELSE /* just copy */
        INTERPRET A".i = "A".j"
    END

/*put 'newest' value into rate array */
IF ( Rm > 0 ) THEN
    INTERPRET A".1 = "Z + ((R * W) / Rm)
ELSE
    INTERPRET A".1 = "Z

RETURN

/*====================================================================
remove results of NETSTAT command from queue 
and parse out the values we want.
            Sample Queue Data
----------------------------------------------------------------------
Interface 10: Serial Interface ppp0
physical address    000000000000      MTU 1500

speed 115200 bits/sec
unicast packets received 7102
broadcast packets received 0
total bytes received 3446636
unicast packets sent 6446
broadcast packets sent 0
total bytes sent 280569
packets discarded on transmission 0
packets discarded on reception 0
received packets in error 0
errors trying to send 0
packets received in unsupported protocols 0
----------------------------------------------------------------------
RXQUEUE translates lowercase to uppercase
====================================================================*/
RawData: 

bytrec = ''
bytsnt = ''
st = 1

DO WHILE (queued() > 0)

    PULL li

    SELECT
       WHEN ( st = 1 ) THEN DO
          IF ( pos('TOTAL BYTES RECEIVED', li) > 0 ) THEN DO
             st = 2
             PARSE var li . . . bytrec .
             END  /* Do */
          END  /* Do */
       WHEN ( st = 2 ) THEN DO
          IF ( pos('TOTAL BYTES SENT', li) > 0 ) THEN DO
             st = 3
             PARSE var li . . . bytsnt .
             END  /* Do */
          END  /* Do */
       OTHERWISE
       END  /* select */
    END /* do */

RETURN

