/*----------------------------------------------------------------------*\
    zipdel.cmd	  -- delete files that were unpacked from a zip file

    Wonkoo Kim, December 25, 1995

    This script requires Info-Zip's unzip program in PATH.

    History:
	April 25, 1995	    The first version.
	August 11, 1995     Fixed bug when file name had special symbols.
	August 18, 1995     Fixed bug: compute with size of actual files
			    instead of file size info from .zip contents
	August 25, 1995     Added delete option for Read-Only files.
	December 25, 1995   Now more reliably deletes empty dirs.
\*----------------------------------------------------------------------*/

'@echo off'

Call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
Call SysLoadFuncs

parse arg args

subdir = ''
force = 0
junk = 0
quiet = 0
readonly = 0

cmdargs = ''
do i=1 to words(args)
    opt = word(args, i)

    select
    when left(opt, 2) = '-?' then
	leave i
    when left(opt, 2) = '-f' then
	force = 1
    when left(opt, 2) = '-j' then
	junk = 1
    when left(opt, 2) = '-d' then
	do
	    subdir = substr(opt,3)
	    if subdir = '' then do
		i = i + 1
		subdir = word(args, i)
	    end
	end
    when left(opt, 2) = '-q' then
	do
	    quiet = quiet + 1
	    if substr(opt,3) = 'q' then quiet = quiet + 1
	end
    when left(opt, 2) = '-r' then
	readonly = 1
    otherwise
	if cmdargs = '' then
	    cmdargs = opt
	else
	    cmdargs = cmdargs||' '||opt
    end
end

if cmdargs = '' | left(opt, 2) = '-?' then do
    say
    say 'Usage: zipdel [-option] [zipfile]'
    say
    say 'Options:'
    say '   -?          This help screen'
    say '   -d subdir   files to be deleted are under "subdir" tree'
    say '   -f          force to delete even if size and time stamp are different'
    say '   -j          Junk (don''t use) directory info of zip file'
    say '   -q[q]       Quiet mode (default mode is verbose); -qq means quieter'
    say '   -r          Delete Read-Only files as well'
    say
    say 'Description:'
    say '   ZIPDEL will delete any file that has the same time stamp and size'
    say '   and that is under the same subdir as the one contained in zip file.'
    say '   If -j option is used, ZIPDEL will look at current dir for all files.'
    say '   by disregarding subdir info of files in zip file.'
    say '   The zip file is safe (untouched) from this operation.'
    say
    say 'WARNING:'
    say "   This program checks file's time stamp only up to minutes, not seconds,"
    say '   thus can be fooled by such slight time stamp differences.'
    say
    say 'Wonkoo Kim (wkim+@pitt.edu)'
    exit
end

if (subdir \= '') & (right(subdir,1) \= ':') then
do
    subdir = translate(subdir,'\','/')
    subdir = strip(subdir, 't', '\')||'\'
end

tmpfile = SysTempFileName('zipdel.???')

do i=1 to words(cmdargs)
    zipfile = word(cmdargs, i)
    if exists(zipfile) then
    do
	if quiet < 2 then do
	    say
	    say 'Source Archive:' zipfile
	end

	'unzip -Z' zipfile '>' tmpfile
	tmpfile2 = SysTempFileName('zipdel.???')

	deleted_files = 0
	deleted_bytes = 0
	notdel_files = 0
	notdel_bytes = 0
	not_found = 0
	nf_bytes = 0
	rmdirs = 0
	nrmdirs = 0
	prevpath = ''
	do while lines(tmpfile)
	    list = linein( tmpfile)
	    file = translate( subword(list, 9), '\', '/')
	    path = strip( filespec("path", file), 'T', '\')
	    if (junk) then do
		file = filespec("name", file)
		path = ''
	    end
	    if left(list, 1) = '-' then
	    do
		size = word(list, 4)
		date = right( word(list, 7), 9, '0')
		time = word(list, 8)
		stamp = date||' '||time
		file = subdir||file
		if exists(file) then
		do
		    fbytes = fsize(file)
		    fstamp = left(fasctime(file),9)||right(fbytes, 9)
		    if ( ( left(fasctime(file),length(stamp)) = stamp &,
			   fbytes = size ) | force) then
		    do
			if readonly = 1 then 'attrib -r "'||file||'"'
			'del "'||file||'"'
			if (rc = 0) then do
			    if quiet=0 then say ' Deleted: ' fstamp file
			    deleted_files = deleted_files + 1
			    deleted_bytes = deleted_bytes + fbytes
			end
			else do
			    if quiet=0 then say ' ** Not Deleted:' fstamp file
			    notdel_files = notdel_files + 1
			    notdel_bytes = notdel_bytes + fbytes
			end
		    end
		    else do
			if quiet=0 then say ' ** Not Deleted:' fstamp file
			notdel_files = notdel_files + 1
			notdel_bytes = notdel_bytes + fbytes
		    end
		end
		else do
		    if quiet = 0 then say ' ** Not Found:' file
		    not_found = not_found + 1
		    nf_bytes = nf_bytes + size
		end
	    end
	    /*
	     * save a list of subdir names (to delete empty dirs)
	     */
	    if path \= prevpath then do
		rc = lineout(tmpfile2, subdir||path)
		if rc \= 0 then do
		    say '**ERROR: lineout()' tmpfile2
		    exit 1
		end
		pathtoken = translate(path, ' ', '\')
		do j = words(pathtoken) to 1 by -1
		    k = wordindex(pathtoken, j) - 1
		    if k > 0 then k = k - 1
		    if left(path, k) == left(prevpath, k) then
			leave j
		    rc = lineout(tmpfile2, subdir||left(path,k))
		end
		prevpath = path
	    end
	end
	call stream tmpfile, 'C', 'CLOSE'

	if exists(tmpfile2) then do
	    call stream tmpfile2, 'C', 'CLOSE'
	    'sort /r <' tmpfile2 '>' tmpfile

	    if quiet < 2 & \junk then do
		say
		say ' Purging Empty Directories ...'
		end
	    path = ''
	    do while lines(tmpfile)
		input = linein(tmpfile)
		if input \= '' & input \= path then do
		    path = input
		    rc = SysRmDir(path)
		    if rc = 0 then do
			say '  Removed:  ' path||'\'
			rmdirs = rmdirs + 1
		    end
		    if rc \= 0 & rc \= 3 & quiet < 2 then do
			say '  Non-empty:' path
			nrmdirs = nrmdirs + 1
		    end
		end
	    end
	    call stream tmpfile, 'C', 'CLOSE'
	    'del' tmpfile2
	end
	'del' tmpfile

	if quiet < 2 then do
	    len = max(length(deleted_bytes), length(notdel_bytes), length(nf_bytes))
	    say
	    say ' Result:'
	    if (deleted_files > 0) then
		say '  Deleted:    ' format(deleted_files,4) 'files (',
		    || right(commas(deleted_bytes),len+2) 'bytes)'
	    if (notdel_files > 0) then
		say '  Not Deleted:' format(notdel_files,4) 'files (',
		    || right(commas(notdel_bytes),len+2) 'bytes)'
	    if (not_found > 0) then
		say '  Not Found:  ' format(not_found,4) 'files (',
		    || right(commas(nf_bytes),len+2) 'bytes)'
	    if (rmdirs > 0) then
		say '  Removed:    ' format(rmdirs,4) 'dirs.'
	    if (nrmdirs > 0) then
		say '  Not Removed:' format(nrmdirs,4) 'dirs.'
	end
    end
end

exit 0

/*----------------------------------------------------------------------*/
exists: procedure
    arg fname
/*----------------------------------------------------------------------*/
/* check if a file exists.
*/
    return stream(fname, 'C', 'QUERY EXISTS') \= ''

/*----------------------------------------------------------------------*/
fsize: procedure
    arg fname
/*----------------------------------------------------------------------*/
/* get file size */
    return stream(fname, 'C', 'QUERY SIZE')

/*----------------------------------------------------------------------*/
fasctime: procedure
    arg fname
/*----------------------------------------------------------------------*/
/* convert datetime into more readable date string without time part.
 * input argument format is based on getftime().
 */
    datetime = space( stream(fname, 'C', 'QUERY DATETIME'), 1)
    parse var datetime month'-'day'-'year time
    month.01 = 'Jan'
    month.02 = 'Feb'
    month.03 = 'Mar'
    month.04 = 'Apr'
    month.05 = 'May'
    month.06 = 'Jun'
    month.07 = 'Jul'
    month.08 = 'Aug'
    month.09 = 'Sep'
    month.10 = 'Oct'
    month.11 = 'Nov'
    month.12 = 'Dec'
    return day'-'month.month'-'year time

/*----------------------------------------------------------------------*/
commas: procedure
    arg number
/*----------------------------------------------------------------------*/
/* insert thousand commas in a positive number string */
    str = number
    p = length(trunc(number))
    do while p > 3
	p = p - 3
	str = insert(',', str, p)
    end
    return str

