/*	mnu.c

A keyboard/video handler to create point-and-shoot menus.
Reads a menu definition text file, then hightlights the
menu items as defined.  Returns a DOS errorlevel to indicate
which item was selected.

MNU uses the Zortech disp_ package for video IO.

Written by Francis X. Guidry
Released to the public domain

*/

#include <stdio.h>
#include <disp.h>
#include <bios.h>


/*	The double linked list menu item	*/

struct menuitem {
	int number;
	char hotkey;
	int row;
	int col;
	int length;
	unsigned attr;
	char *prompt;
	struct menuitem *prev;
	struct menuitem *next;
};

struct menuitem *menuhead;


/* Default help line location */

int helprow = 24;
int helpcol = 0;
int helplen = 79;

/* Scan codes for PC cursor keys */

#define HOME  71
#define END   79
#define UP    72
#define DOWN  80
#define LEFT  75
#define RIGHT 77
#define BACK  15

/* -----------------------------------------------------------
	Test command line.  Drive the remaining processing.
   ----------------------------------------------------------- */

main(int argc, char** argv)
{
	int start = 1;
	
	if(argc < 2)
		errexit("Must specify menu definition file");
	if(argc == 3)
		start = atoi(argv[2]);
	read_def(argv[1]);
	return proc_mnu(start);
}

/* -----------------------------------------------------------
	Exit due to an error condition.  Because errorlevel is
	being used to control selection, return errorlevel 0.
   ----------------------------------------------------------- */

errexit(char *s)
{
	puts(s);
	exit(0);
}


/* -----------------------------------------------------------
	Read the menu definition file and build the list of
	menu items.
   ----------------------------------------------------------- */

read_def(char *s)
{
	FILE *f;
	char buff[100];
	struct menuitem* m = 0;
	struct menuitem* set_item();
	
	f = fopen(s,"r");
	if(!f)
		errexit("Unable to open menu definition file");
	
	while(fgets(buff,99,f)) {
		if(*buff == '!')
			set_prompt(buff + 1);
		else if(*buff == ':')
			m = set_item(m,buff + 1);
		else if(*buff == '?')
			add_prompt(m,buff + 1);
	}
}


/* -----------------------------------------------------------
	Highlight selections, display prompts, and accept keys.
   ----------------------------------------------------------- */

proc_mnu(int start)
{
	struct menuitem* m = menuhead;
	int ch, flag;
	struct menuitem* check_left();
	struct menuitem* check_right();
	struct menuitem* check_start();

	m = check_start(start,m);
	disp_open();
	disp_hidecursor();
	highlight(m);
	
	for(;;) {
		ch = bioskey(0);
		if(ch & 0xff) {
			ch &= 0xff;
			if(ch == 27) {
				flag = 0;
				break;
			} else if(ch == '\r') {
				flag = m->number;
				break;
			} else if(isalpha(ch)) {
				flag = check_hot(ch,m);
				if(flag)
					break;
			} else if(ch == '\t') {
				lowlight(m);
				m = m->next;
				highlight(m);
			}
		} else {
			ch = ch >> 8;
			if(ch == HOME) {
				lowlight(m);
				m = menuhead;
				highlight(m);
			} else if(ch == END) {
				lowlight(m);
				m = menuhead->prev;
				highlight(m);
			} else if(ch == UP || ch == BACK) {
				lowlight(m);
				m = m->prev;
				highlight(m);
			} else if(ch == DOWN) {
				lowlight(m);
				m = m->next;
				highlight(m);
			} else if(ch == LEFT) {
				m = check_left(m);
			} else if(ch == RIGHT) {
				m = check_right(m);
			}
		}
	}
	
	disp_showcursor();
	disp_close();
	return flag;
}


/* -----------------------------------------------------------
	Set the row, col, and length of the prompt.
   ----------------------------------------------------------- */

set_prompt(char *s)
{
	sscanf(s,"%d %d %d",&helprow,&helpcol,&helplen);
}


/* -----------------------------------------------------------
	Create a new item and set it's values.
   ----------------------------------------------------------- */

struct menuitem *
set_item(struct menuitem *m,char *s)
{
	struct menuitem *n = (struct menuitem*) malloc(sizeof(struct menuitem));
	if(!n)
		errexit("Out of memory");
	
	if(!m) {
		menuhead = n->prev = n->next = n;
		n->number = 1;
	}
	else {
		n->prev = m;
		n->next = m->next;
		m->next = n->next->prev = n;
		n->number = m->number + 1;
	}
	n->prompt = 0;
	
	while(*s == ' ')
		s++;
	sscanf(s,"%c %d %d %d %d",&(n->hotkey),&(n->row),&(n->col),&(n->length));
	if(n->hotkey >= 'a')
		n->hotkey -= 32;
	
	return n;
}


/* -----------------------------------------------------------
	Add the prompt string to a menu item.
   ----------------------------------------------------------- */

add_prompt(struct menuitem *m,char *s)
{
	int l;
	
	if(!m)
		return;
	
	l = strlen(s);
	l--;
	if(*(s + l) == '\n')
		*(s + l) = '\0';
	else l++;
	
	m->prompt = (char *) malloc(l+1);
		if(!m->prompt)
		errexit("Out of memory in prompt");

	while(*s == ' ')
		s++;
	strcpy(m->prompt,s);
}


/* -----------------------------------------------------------
	Display this menu item with reverse attribute.
	Display the prompt if present, centered or truncated.
   ----------------------------------------------------------- */

highlight(struct menuitem* m)
{
	char buff[100];
	int i, l, o;
	unsigned attr;

	m->attr = disp_peekw(m->row,m->col) >> 8;
	for(i = 0; i < m->length; i++)
		buff[i] = disp_peekw(m->row,m->col + i) & 0xff;
	buff[i] = '\0';
	disp_move(m->row,m->col);
	disp_setattr(m->attr);
	disp_startstand();
	disp_printf("%s",buff);
	disp_flush();
	disp_endstand();

	if(m->prompt) {
		l = strlen(m->prompt);
		if(l > helplen) {
			*(m->prompt + helplen) = '\0';
			o = 0;
		} else {
			o = (helplen/2) - (l/2);
		}
		attr = disp_peekw(helprow,helpcol) >> 8;
		disp_setattr(attr);
		disp_move(helprow,helpcol + o);
		disp_printf("%s",m->prompt);
		disp_flush();
	}
}


/* -----------------------------------------------------------
	Display this menu item with normal attribute
   ----------------------------------------------------------- */

lowlight(struct menuitem* m)
{
	char buff[100];
	int i;
	unsigned attr;

	for(i = 0; i < m->length; i++)
		buff[i] = disp_peekw(m->row,m->col + i) & 0xff;
	buff[i] = '\0';
	disp_move(m->row,m->col);
	disp_setattr(m->attr);
	disp_printf("%s",buff);
	disp_flush();

	if(m->prompt) {
		attr = disp_peekw(helprow,helpcol) >> 8;
		disp_setattr(attr);
		for(i = 0; i < helplen; i++) {
			disp_move(helprow,helpcol + i);
			disp_putc(' ');
		}
	}
}


/* -----------------------------------------------------------
	Look for a hot key, starting with this menu item.
   ----------------------------------------------------------- */

check_hot(int ch,struct menuitem* m)
{
	struct menuitem* n = m;
	
	if(ch > 'Z')
		ch -= 32;
	
	do {
		if(n->hotkey == ch) {
			lowlight(m);
			highlight(n);
			return n->number;
		} else n = n->next;
	} while(n != m);
	
	return 0;
}


/* -----------------------------------------------------------
	Look for a menu item on this row, to the left
   ----------------------------------------------------------- */

struct menuitem *
check_left(struct menuitem* m)
{
	struct menuitem* n = m->prev;
	
	do {
		if(n->row == m->row && n->col < m->col) {
			lowlight(m);
			highlight(n);
			return n;
		} else n = n->prev;
	} while(n != m);
	
	lowlight(m);
	highlight(m->prev);
	return m->prev;
}


/* -----------------------------------------------------------
	Look for a menu item on this row, to the right
   ----------------------------------------------------------- */

struct menuitem *
check_right(struct menuitem* m)
{
	struct menuitem* n = m->next;
	
	do {
		if(n->row == m->row && n->col > m->col) {
			lowlight(m);
			highlight(n);
			return n;
		} else n = n->next;
	} while(n != m);
	
	lowlight(m);
	highlight(m->next);
	return m->next;
}


/* -----------------------------------------------------------
	Look for a menu item with this number.
   ----------------------------------------------------------- */

struct menuitem *
check_start(int start,struct menuitem* m)
{
	struct menuitem* n = m;
	
	do {
		if(n->number == start) {
			return n;
		} else n = n->next;
	} while(n != m);
	
	return m;
}


