/* backupfile.c -- make Emacs style backup file names
   Copyright (C) 1990, 1997 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* David MacKenzie <djm@gnu.ai.mit.edu>.
   Some algorithms adapted from GNU Emacs.  */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include "backupfile.h"

#if HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
# ifndef strrchr
#  define strrchr rindex
# endif
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
# define NLENGTH(Direct) (strlen((Direct)->d_name))
#else
# define dirent direct
# define NLENGTH(Direct) ((Direct)->d_namlen)
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#if CLOSEDIR_VOID
/* Fake a return value.  */
# define CLOSEDIR(d) (closedir (d), 0)
#else
# define CLOSEDIR(d) closedir (d)
#endif

#if STDC_HEADERS
# include <stdlib.h>
#else
char *malloc ();
#endif

#if defined (STDC_HEADERS) || (!defined (isascii) && !defined (HAVE_ISASCII))
# define IN_CTYPE_DOMAIN(Char) 1
#else
# define IN_CTYPE_DOMAIN(Char) isascii(Char)
#endif

#define ISDIGIT(Char) (IN_CTYPE_DOMAIN ((unsigned char) (Char)) \
		       && isdigit ((unsigned char) (Char)))

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef _POSIX_VERSION
/* POSIX does not require that the d_ino field be present, and some
   systems do not provide it.  */
# define REAL_DIR_ENTRY(Dp) 1
#else
# define REAL_DIR_ENTRY(Dp) ((Dp)->d_ino != 0)
#endif

/* Which type of backup file names are generated.  */
enum backup_type backup_type = none;

/* The extension added to file names to produce a simple (as opposed
   to numbered) backup file name.  */
char *simple_backup_suffix = "~";

#ifdef MSDOS
/* Support long file names when available (Windows 9X).  */
# ifdef __DJGPP__
#  include <fcntl.h>
# endif
static int lfn = 0;
#endif

static char *concat ();
char *find_backup_file_name ();
static char *make_version_name ();
static int max_backup_version ();
static int version_number ();

/* Return the name of the new backup file for file FILE,
   allocated with malloc.  Return 0 if out of memory.
   FILE must not end with a '/' unless it is the root directory.
   Do not call this function if backup_type == none.  */

char *
find_backup_file_name (file)
     const char *file;
{
  char *copy;
  char *base;
  const char *dir;
  char *base_versions;
  int highest_backup;

#ifdef __DJGPP__
  /* Support long file names when FILE resides on a filesystem
     which makes them available.  */
  lfn = _use_lfn (file);
#endif

  if (backup_type == simple)
    return concat (file, simple_backup_suffix);

  copy = malloc (strlen (file) + 2);
  if (copy == 0)
    return 0;
  strcpy (copy, file);

  base = strrchr (copy, '/');

  /* Make sure DIR is not empty for root directories.  */
  if (base == copy
#if MSDOS
      /* "d:" and "d:/" are NOT the same.  */
      || (base == copy + 2 && base[-1] == ':')
      /* "d:foo" is NOT relative to ".".  */
      || (base == 0 && copy[1] == ':' && (base = copy + 1))
#endif
      )
    {
      /* Leave the trailing slash alone, make a room for a NULL
	 character after the slash.  */
      strcpy (base + 2, file + (base - copy) + 1);
      base++;
    }
  if (base == 0)
    {
      base = copy;
      dir = ".";
    }
  else
    {
      *base++ = '\0';
      dir = copy;
    }

  base_versions = concat (base, ".~");
  if (base_versions == 0)
    {
      free (copy);
      return 0;
    }

  highest_backup = max_backup_version (base_versions, dir);
  free (copy);
  free (base_versions);

  if (backup_type == numbered_existing && highest_backup == 0)
    return concat (file, simple_backup_suffix);

  return make_version_name (file, highest_backup + 1);
}

/* Return the number of the highest-numbered backup file for file
   FILE in directory DIR.  If there are no numbered backups
   of FILE in DIR, or an error occurs reading DIR, return 0.
   FILE should already have ".~" appended to it.  */

static int
max_backup_version (file, dir)
     const char *file;
     const char *dir;
{
  DIR *dirp;
  struct dirent *dp;
  int highest_version;
  int this_version;
  size_t file_name_length;

  dirp = opendir (dir);
  if (!dirp)
    return 0;

  highest_version = 0;
  file_name_length = strlen (file);

  while ((dp = readdir (dirp)) != 0)
    {
      if (!REAL_DIR_ENTRY (dp)
#if !defined(MSDOS) || defined(__DJGPP__)
	  /* When long file names aren't supported, examine all files.  */
	  || (NLENGTH (dp) <= file_name_length
	  && lfn
	      )
#endif
	  )
	continue;

      this_version = version_number (file, dp->d_name, file_name_length);
      if (this_version > highest_version)
	highest_version = this_version;
    }
  if (CLOSEDIR (dirp))
    return 0;
  return highest_version;
}

/* Return a string, allocated with malloc, containing
   "FILE.~VERSION~".  Return 0 if out of memory.  */

static char *
make_version_name (file, version)
     const char *file;
     int version;
{
  char *backup_name;

  backup_name = malloc (strlen (file) + 16);
  if (backup_name == 0)
    return 0;
  sprintf (backup_name, "%s.~%d~", file, version);
#ifdef MSDOS
  /* If long filenames aren't supported on the filesystem where
     FILE belongs, squeeze the backup suffix into 8+3 namespace.  */
  if (!lfn)
    {
      char *short_name;

      /* Let `concat' do the hard job (see below) when we
	 cannot use long filenames.  */
      sprintf (backup_name, ".~%d~", version);
      short_name = concat (file, backup_name);
      free (backup_name);
      backup_name = short_name;
    }
#endif
  return backup_name;
}

/* If BACKUP is a numbered backup of BASE, return its version number;
   otherwise return 0.  BASE_LENGTH is the length of BASE.
   BASE should already have ".~" appended to it.  */

static int
version_number (base, backup, base_length)
     const char *base;
     const char *backup;
     int base_length;
{
  int version;
  const char *p;

  version = 0;
#ifdef MSDOS
  if (!lfn)
    {
      /* When long filenames aren't supported, the numbered backup
	 suffix moves into the name; see `concat'.  We want filenames
	 like e.g. "foo.bar" and "foo1234~" to match here.
	 The call to `strncmp' is of course superflous after this,
	 but letting the non-LFN systems suffer leaves the code cleaner.  */
      const char *pbase = base, *pbackup = backup;
      int len;

      for (len = base_length; len; len--)
	if (*pbase++ != *pbackup++)
	  {
	    base_length -= len;
	    break;
	  }
      /* Avoid lossage when BASE is already a numbered backup.  */
      if (!len && pbase > base + 2 && pbase[-1] == '~' && ISDIGIT (pbase[-2]))
	{
	  base_length--;
	  for (pbase = pbase - 2; pbase >= base && ISDIGIT (*pbase); pbase--)
	    base_length--;
	}
    }
#endif
  if (!strncmp (base, backup, base_length) && ISDIGIT (backup[base_length]))
    {
      for (p = &backup[base_length]; ISDIGIT (*p); ++p)
	version = version * 10 + *p - '0';
      if (p[0] != '~' || p[1])
	version = 0;
    }
  return version;
}

/* Return the newly-allocated concatenation of STR1 and STR2.
   If out of memory, return 0.  */

static char *
concat (str1, str2)
     const char *str1;
     const char *str2;
{
  char *newstr;
  int str1_length = strlen (str1);

  newstr = malloc (str1_length + strlen (str2) + 1);
  if (newstr == 0)
    return 0;
  strcpy (newstr, str1);
  strcpy (newstr + str1_length, str2);
#ifdef MSDOS
  if (!lfn)
    {
      /* Without long filenames, we have to live with the darn 8+3
	 basename restriction.  The only way to have numbered
	 backups in that case is to eat some characters off the end
	 of the original name.  */
      extern char *basename (const char *);
      char *base = basename (newstr);
      int baselen = strlen (base);
      char *dot;
      char *str2_start = newstr + str1_length;
      int s2_len = strlen (str2_start);
      int extlen;
      int str2_starts_with_dot;

      /* Kludgy feature: if `STR1' already has an extension, and
	 `STR2' begins with ".~", we remove the `~'.  This is so
	 we could salvage at least 1 character of the original
	 extension for the first 9 backups (e.g., "foo.c" + ".~7~"
	 is ``concatenated'' into "foo.c7~", not "foo.~7~").  */
      if (memchr (base, '.', baselen - s2_len)
	  && (*str2_start == '.' && str2_start[1] == '~'))
	{
	  memmove (str2_start + 1, str2_start + 2, s2_len - 1);
	  s2_len--;
	  baselen--;
	}

      /* Weed out all but the first dot in `base'.  */
      dot = memchr (base, '.', baselen);
      if (dot)
	{
	  char *s = dot + 1;
	  char *d = s;
	  int dots_before_str2 = 0;

	  do {
	    while (*s == '.')
	      {
		if (s < str2_start)
		  dots_before_str2++;
		else
		  s2_len--;
		s++;
		baselen--;
	      }
	  } while ((*d++ = *s++));

	  str2_start -= dots_before_str2;
	  extlen = baselen - (dot - base) - 1; /* 1 for the dot itself */
	}
      else
	{
	  dot = base + baselen;
	  extlen = 0;
	}

      /* Make sure `STR2' will not be truncated by the OS.  If `STR2' is
	 in the extension, it must be in the first 3 characters after the
	 dot; if it's before the dot, it must be one of the first 8
	 characters of the basename.  */
      str2_starts_with_dot = str2_start[0] == '.';
      if (extlen > 3)
	{
	  char *new_str2_start;

	  s2_len = strlen (str2_start);
	  if (s2_len <= 3 + str2_starts_with_dot)
	    new_str2_start = dot + !str2_starts_with_dot + 3 - s2_len;
	  else
	    {
	      new_str2_start = dot;
	      if (str2_start == new_str2_start)
		{
		  /* The case of foo -> foo.~10~
		     The original file has no extension, but the backup
		     extension is longer than 3 characters.  We need to
		     move the extension into the name (make it foo10~).  */
		  if (*++str2_start == '~')
		    str2_start++;
		  s2_len -= str2_start - dot;
		}
	      dot += s2_len;
	    }
	  memmove (new_str2_start, str2_start, s2_len + 1);
	  str2_start = new_str2_start;
	}
      /* If the name part (before the dot) is longer than 8 characters,
	 we cannot be sure it won't clash with an existing file.  So we
	 must move `STR2' into the name.  */
      if (dot - base > 8)
	{
	  if (str2_starts_with_dot)
	    {
	      str2_start++;
	      s2_len--;
	    }
	  memmove (base + 8 - s2_len, str2_start, s2_len + 1);
	}
    }
#endif /* MSDOS */
  return newstr;
}
