/* Project SWORD
   V2.0

   SubSystem : Mathematical toolbox
   ErrNos    : 4300..4399
   File      : Src/ToolBox/Math/Eval.CC
   Author    : Eric NICOLAS
   Overview  : Math expression evaluator
   UpDate    : Dec 02, 1995

** Copyright (C) 1993,1995 The SWORD Group
**
** This file is distributed under the terms listed in the document
** "copying.en". A copy of "copying.en" should accompany this file.
** if not, a copy should be available from where this file was obtained.
** This file may not be distributed without a verbatim copy of "copying.en".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <math.h>
#include <ctype.h>

#include "Common/Common.H"
#include "ToolBox/Math/complex.H"
#include "ToolBox/Math/Eval.H"

// --- Defines

// Valeurs des lexmes non-caractres

#define NB    0
#define ID    1
#define MC    2
#define FDF   3
#define OP1   4
#define OP2   5
#define RTC   6

// Constantes diverses
#define RIEN  NULL
#define T_Tab 100       // Taille de la table des symboles
#define T_Pil 20        // Taille de la pile de calcul
#define T_Id  40        // Taille maxi des identificateurs
#define NbOpe 24

// --- Types

// Entre de la table des symboles

struct TEntree
{ char   *Nom;
  char    Lex;
  char    Exe;
  complex Val;
};

// --- Global vars

char    SepDec = '.';
char    SepId  = ',';

// --- Local vars

static char *Messages[9] =
       { "The stack is full",
         "The stack is empty",
         "The symbol table is full",
         "Syntax error",
         "Math error : 1/0",
         "Math error : 0^0",
         "Math error : LN(0)",
         "Function needs real argument",
         "Unknown variable"
       };

static char *Oper[NbOpe] =
       { "SQRN","MOD","OPP","SIN","COS","ASIN","ACOS","TAN","ATAN","SINH",
         "COSH","ASINH","ACOSH","TANH","ATANH","LOG","ALOG","LN","EXP",
         "ABS","INV","ARG","SQRT","CONJ" };

static TEntree   TableSymb[T_Tab];
static complex   Pile[T_Pil];
static int       Sommet;
static char     *AdrBuf;
static int       FinTable;
static char     *TampLex;
static void     *ValLex;
static char      Symbole;
static double    PassReal;

// ===== Fonctions locales

// --- Gestion de la pile

static void VerifieSommet(void)
{ if (Sommet+1>T_Pil) Error=erStackFull;
}

static void VerifieNbArg(char n)
{ if (Sommet<n) Error=erStackEmpty;
}

// --- Table des symboles

static TEntree *Chercher(char *Symb)
{ int i;
  for(i=FinTable-1;i>=0;i--)
    if (!strcmp(TableSymb[i].Nom,Symb)) return &TableSymb[i];
  return NULL;
}

static TEntree *Inserer(char *Symb, char Lex, char Exe, complex Val)
{ if (FinTable>T_Tab)
  { Error=erSymbolTableFull;
    return NULL;
  }
  TableSymb[FinTable].Nom=strdup(Symb);
  TableSymb[FinTable].Lex=Lex;
  TableSymb[FinTable].Exe=Exe;
  TableSymb[FinTable].Val=Val;
  FinTable++;
  return &TableSymb[FinTable-1];
}

// --- Calculs

static void Modulo(void)
{ if ((imag(Pile[Sommet-1])) || (imag(Pile[Sommet])))
  { Error=erArgNotReal;
    return;
  }
  if (!real(Pile[Sommet]))
  { Error=erMath_DivideByZero;
    return;
  }
  Pile[Sommet-1]=complex((long)real(Pile[Sommet-1]) %
                         (long)real(Pile[Sommet]),0);
}

static void Somme(void)
{ VerifieNbArg(2);
  Pile[Sommet-1]=Pile[Sommet-1]+Pile[Sommet];
  Sommet--;
}

static void Difference(void)
{ VerifieNbArg(2);
  Pile[Sommet-1]=Pile[Sommet-1]-Pile[Sommet];
  Sommet--;
}

static void Produit(void)
{ VerifieNbArg(2);
  Pile[Sommet-1]=Pile[Sommet-1]*Pile[Sommet];
  Sommet--;
}

static void Division(void)
{ if (Pile[Sommet]==NUL)
  { Error=erMath_DivideByZero;
    return;
  }
  VerifieNbArg(2);
  Pile[Sommet-1]=Pile[Sommet-1]/Pile[Sommet];
  Sommet--;
}

static void Inverse(void)
{ if (Pile[Sommet]==NUL)
  { Error=erMath_DivideByZero;
    return;
  }
  Pile[Sommet]=complex(1,0)/Pile[Sommet];
}

static void Oppose(void)
{ Pile[Sommet]=-Pile[Sommet];
}

static void Absolue(void)
{ Pile[Sommet]=abs(Pile[Sommet]);
}

static void Expon(void)
{ Pile[Sommet]=exp(Pile[Sommet]);
}

static void Arg(void)
{ Pile[Sommet]=complex(arg(Pile[Sommet]),0);
}

static void Puissance(void)
{ VerifieNbArg(2);
  Pile[Sommet-1]=pow(Pile[Sommet-1],Pile[Sommet]);
  Sommet--;
}

static void LogNeper(void)
{ Pile[Sommet]=log(Pile[Sommet]);
}

static void Log(void)
{ Pile[Sommet]=log10(Pile[Sommet]);
}

static void ALog(void)
{ Pile[Sommet]=exp(Pile[Sommet]*log(complex(10,0)));
}

static void CosH(void)
{ Pile[Sommet]=cosh(Pile[Sommet]);
}

static void SinH(void)
{ Pile[Sommet]=sinh(Pile[Sommet]);
}

static void TanH(void)
{ Pile[Sommet]=tanh(Pile[Sommet]);
}

static void Cosinus(void)
{ Pile[Sommet]=cos(Pile[Sommet]);
}

static void Sinus(void)
{ Pile[Sommet]=sin(Pile[Sommet]);
}

static void Tan(void)
{ Pile[Sommet]=tan(Pile[Sommet]);
}

static void ArcTangente(void)
{ //Pile[Sommet]=atan(Pile[Sommet]);
}

static void ArcCosinus(void)
{ //Pile[Sommet]=acos(Pile[Sommet]);
}

static void ArcSinus(void)
{ //Pile[Sommet]=asin(Pile[Sommet]);
}

/*
Procedure ATanH;
Var m,r : Real;
Begin
  VerifieSommet;
  With Pile[Sommet] do
  Begin
    m:=Sqr(-Im)+Sqr(1-Re);
    Pile[Sommet+1].Re:=(-Sqr(-Im)-Sqr(Re)+1)/m;
    Pile[Sommet+1].Im:=(2*Im)/m;
    r:=Arg1(Sommet+1);
    m:=LN(Sqrt(Sqr(Pile[Sommet+1].Re)+Sqr(Pile[Sommet+1].Im)));
    Re:=m/2;
    Im:=r/2;
  End;
End;

Procedure ACosH;
Var m,r : Real;
Begin
  VerifieSommet;
  With Pile[Sommet] do
  Begin
    Pile[Sommet+1].Re:=Sqr(Re)-Sqr(Im)-1;
    Pile[Sommet+1].Im:=2*Re*Im;
    m:=Sqrt(Sqrt(Sqr(Pile[Sommet+1].Re)+Sqr(Pile[Sommet+1].Im)));
    r:=Arg1(Sommet+1);
    Pile[Sommet+1].Re:=Re+m*COS(r/2);
    Pile[Sommet+1].Im:=Im+m*SIN(r/2);
    m:=Sqrt(Sqr(Pile[Sommet+1].Re)+Sqr(Pile[Sommet+1].Im));
    r:=Arg1(Sommet+1);
    Re:=LN(m);
    Im:=r;
  End;
End;

Procedure ASinH;
Var m,r : Real;
Begin
  VerifieSommet;
  With Pile[Sommet] do
  Begin
    Pile[Sommet+1].Re:=-Sqr(-Im)+Sqr(Re)+1;
    Pile[Sommet+1].Im:=2*Re*Im;
    m:=Sqrt(Sqrt(Sqr(Pile[Sommet+1].Re)+Sqr(Pile[Sommet+1].Im)));
    r:=Arg1(Sommet+1);
    Pile[Sommet+1].Re:=-Re+m*COS(r/2);
    Pile[Sommet+1].Im:=-Im+m*SIN(r/2);
    m:=Sqrt(Sqr(Pile[Sommet+1].Re)+Sqr(Pile[Sommet+1].Im));
    r:=Arg1(Sommet+1);
    Re:=-LN(m);
    Im:=-r;
  End;
End;
*/

static void Racine2(void)
{ Pile[Sommet]=sqrt(Pile[Sommet]);
}

static void RacineN(void)
{ if (imag(Pile[Sommet]))
  { Error=erArgNotReal;
    return;
  }
  Pile[Sommet-1]=pow(Pile[Sommet-1],complex(1,0)/Pile[Sommet]);
}

static void Conj(void)
{ Pile[Sommet]=conj(Pile[Sommet]);
}

// --- Analyseur

static char GetChar(void)
{ return (*(AdrBuf++));
}

static void UnGetChar(void)
{ AdrBuf--;
}

static double PrendsDecimal(char c)
{ double r=0,puis=1E-1;
  int   dec=FALSE;
  while(isdigit(c) || (c==SepDec))
  { if (c==SepDec)
    { if (dec) { Error=erSyntaxError; return 0; }
      dec=TRUE;
    }
    else
      if (dec)
      { r=r+(c-48)*puis;
        puis/=10;
      }
      else
        r=r*10+(c-48);
    c=GetChar();
  }
  UnGetChar();
  return r;
}

static double PrendsReel(char c)
{ double m,e=0,s=1;
  m=PrendsDecimal(c);
  if (Error) return 0;
  c=toupper(GetChar());
  if (c=='E')
  { c=GetChar();
    if (c=='-') { s=-1; c=GetChar(); }
    if (c=='+') c=GetChar();
    e=PrendsDecimal(c);
    if (Error) return 0;
  }
  else UnGetChar();
  return m*exp(s*e*log(10));
}

static char AnalLex(void)
{ char c;
  while(1)
  { if (Error) return 0;
    c=toupper(GetChar());
    if isdigit(c)
      { PassReal=PrendsReel(c);
        ValLex=&PassReal;
        return NB;
      }
    if isalpha(c)
      { TampLex=(char*)malloc(T_Id);
        char *P=TampLex;
        while(isalnum(c))
        { if (P-TampLex>T_Id)
          { Error=erSyntaxError;
            return 0;
          }
          *P=c; P++;
          c=toupper(GetChar());
        }
        *P=0;
        UnGetChar();
        if ((ValLex=Chercher(TampLex))==NULL)
          ValLex=Inserer(TampLex,ID,0,NUL);
        free(TampLex);
        if (ValLex==NULL) return 0;
        return ((TEntree*)ValLex)->Lex;
      }
    if ((c==0) || (c==27)) return FDF;
    if ((c==9) || (c==' ')) ;
    else return c;
  }
}

static void Expr(void);      // Dfinition Forward

static void Emettre(char Lex, void *Val)
{ if (Error) return;
  switch(Lex)
  { case '+' : Somme();                              break;
    case '*' : Produit();                            break;
    case '-' : Difference();                         break;
    case '/' : Division();                           break;
    case '^' : Puissance();                          break;
    case ID  : printf("%s\n",((TEntree*)(Val))->Nom);     break;
    case OP2 : VerifieNbArg(2);
               switch(((TEntree*)(Val))->Exe)
               { case 0 : RacineN(); break;
                 case 1 : Modulo();  break;
               }
               Sommet--;
               break;
    case OP1 : VerifieNbArg(1);
               switch(((TEntree*)(Val))->Exe)
               { case 2  : Oppose(); break;
                 case 3  : Sinus();  break;
                 case 4  : Cosinus(); break;
                 case 5  : ArcSinus(); break;
                 case 6  : ArcCosinus(); break;
                 case 7  : Tan();        break;
                 case 8  : ArcTangente();break;
                 case 9  : SinH();       break;
                 case 10 : CosH();       break;
//                 case 11 : ASinH();      break;
//                 case 12 : ACosH();      break;
                 case 13 : TanH();       break;
//                 case 14 : ATanH();      break;
                 case 15 : Log();        break;
                 case 16 : ALog();       break;
                 case 17 : LogNeper();   break;
                 case 18 : Expon();      break;
                 case 19 : Absolue();    break;
                 case 20 : Inverse();    break;
                 case 21 : Arg();        break;
                 case 22 : Racine2();    break;
                 case 23 : Conj();       break;
               }
               break;
    case NB  : VerifieSommet;
               Pile[Sommet+1]=complex(*((double*)Val),0);
               Sommet++;
               break;
    case RTC : VerifieNbArg(2);
               if ((imag(Pile[Sommet])) || (imag(Pile[Sommet-1])))
               { Error=erArgNotReal;
                 return;
               }
               Pile[Sommet-1]=complex(real(Pile[Sommet-1]),real(Pile[Sommet]));
               Sommet--;
               break;
    default :  printf("Lexme %d   ValLex %d\n",(int)Lex, *((int*)Val));
               break;
  }
}

static void Accepter(char Lex)
{ if (Error) return;
  if (Symbole==Lex) Symbole=AnalLex();
               else Error=erSyntaxError;
}

static void Puiss(void)
{ void *V;
  if (Error) return;
  switch(Symbole)
  { case '-' : Accepter('-');
               Puiss();
               Oppose();
               break;
    case '(' : Accepter('(');
               Expr();
               if (Symbole==')') Accepter(')');
               else
                 if (Symbole==SepId)
                 { Accepter(SepId);
                   Expr();
                   Accepter(')');
                   Emettre(RTC,NULL);
                 }
                 else
                 { Error=erSyntaxError;
                   return;
                 }
               break;
    case NB  : Emettre(NB,ValLex);
               Accepter(NB);
               break;
    case ID  : V=ValLex;
               Accepter(ID);
               if (Symbole=='=')
               { Accepter('=');
                 Expr();
                 ((TEntree*)V)->Val=Pile[Sommet];
                 Sommet--;
               }
               else
               { VerifieSommet();
                 Pile[Sommet+1]=((TEntree*)V)->Val;
                 Sommet++;
               }
               break;
    case OP1 : V=ValLex;
               Accepter(OP1);
               Accepter('(');
               Expr();
               Accepter(')');
               Emettre(OP1,V);
               break;
    default  : Error=erSyntaxError;
               break;
  }
}

static void Facteur(void)
{ if (Error) return;
  Puiss();
  while(1)
  { if (Error) return;
    if (Symbole=='^')
    { Accepter(Symbole);
      Puiss();
      Emettre('^',NULL);
    }
    else return;
  }
}

static void Terme(void)
{ if (Error) return;
  Facteur();
  while(1)
  { if (Error) return;
    if ((Symbole=='*') || (Symbole=='/') || (Symbole==OP2))
    { char  T=Symbole;
      void *V=ValLex;
      Accepter(Symbole);
      Facteur();
      Emettre(T,V);
    } else return;
  }
}

static void Expr(void)
{ if (Error) return;
  Terme();
  while(1)
  { if (Error) return;
    if ((Symbole=='+') || (Symbole=='-'))
    { char T=Symbole;
      Accepter(Symbole);
      Terme();
      Emettre(T,NULL);
    } else return;
  }
}

static boolean ErrorMessages(char *Message)
{ int MessageNo;
  switch(Error)
  { case erStackFull           : MessageNo=0; break;
    case erStackEmpty          : MessageNo=1; break;
    case erSymbolTableFull     : MessageNo=2; break;
    case erSyntaxError         : MessageNo=3; break;
    case erMath_DivideByZero   : MessageNo=4; break;
    case erMath_ZeroPowerZero  : MessageNo=5; break;
    case erMath_LogZero        : MessageNo=6; break;
    case erArgNotReal          : MessageNo=7; break;
    case erUnknownVariable     : MessageNo=8; break;
    default:
      return FALSE;
  }
  strcpy(Message,Messages[MessageNo]);
  return TRUE;
}

// ---- Functions exportees

void Evaluate(char *Command)
{ AdrBuf=Command;
  ValLex=NULL;
  Sommet=0;
  Symbole=AnalLex();
  while(Symbole!=FDF)
  { Expr();
    if (Symbole!=FDF) Accepter(';');
    if (Error) return;
  }
}

void Cree_Var(char *Nom, complex Val)
{ strupr(Nom);
  Inserer(Nom,ID,0,Val);
}

void Change_Var(char *Nom, complex Val)
{ TEntree *P;
  strupr(Nom);
  if ((P=Chercher(Nom))==NULL) P=Inserer(Nom,ID,0,Val);
  P->Val=Val;
}

void Demande_Var(char *Nom, complex &Val)
{ TEntree *P;
  strupr(Nom);
  P=Chercher(Nom);
  if (P==NULL) Error=erUnknownVariable;
          else Val=P->Val;
}

void InitEval(void)
{ AddErrorHandler(ErrorMessages);
}

void DoneEval(void)
{ }

void BeginEvaluate(void)
{ int i;
  FinTable=0;
  for(i=0;i<NbOpe;i++)
    if (i<=1) Inserer(Oper[i],OP2,i,NUL);
         else Inserer(Oper[i],OP1,i,NUL);
  Inserer("I",ID,0,cI);
  Inserer("J",ID,0,cJ);
  Inserer("PI",ID,0,cPi);
  Inserer("E",ID,0,cE);
}

void EndEvaluate(void)
{ int i;
  for(i=0;i<FinTable;i++)
    free(TableSymb[i].Nom);
}