// Associated include file : ToolBox/Math/Matrix.H

#include <math.h>
#include "Common/Common.H"
#include "ToolBox/Math/Complex.H"
#include "ToolBox/Math/Vector.H"
#include "ToolBox/Math/Matrix.H"

#define Epsilon 1E-6

int     MatrixDefaultSizeI=1;
int     MatrixDefaultSizeJ=1;

static TMatrix tempMatrix;
static TVector tempVector;

TMatrix::TMatrix(int eSizeI, int eSizeJ)
{ SizeI=eSizeI;
  SizeJ=eSizeJ;
  Tabl=new complex[SizeI*SizeJ];
}

TMatrix::TMatrix(TMatrix& M)
{ int i;
  SizeI=M.SizeI;
  SizeJ=M.SizeJ;
  Tabl=new complex[SizeI*SizeJ];
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]=M.Tabl[i];
}

TMatrix::~TMatrix()
{ delete Tabl;
}

void TMatrix::Reset(void)
{ int i;
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]=0;
}

TMatrix TMatrix::operator=(TMatrix& M)
{ int i;
  if ((SizeI!=M.SizeI)&&(SizeJ!=M.SizeJ))
  { // Ajust sizes
    delete Tabl;
    SizeI=M.SizeI;
    SizeJ=M.SizeJ;
    Tabl=new complex[SizeI*SizeJ];
  }
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]=M.Tabl[i];
  return (*this);
}

TMatrix TMatrix::operator+(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) Error=erMatrix_MixingSizes;
  if (M.SizeJ!=SizeJ) Error=erMatrix_MixingSizes;
  if (Error) return (*this);
  tempMatrix=*this;
  for(i=0;i<SizeI*SizeJ;i++) tempMatrix.Tabl[i]+=M.Tabl[i];
  return tempMatrix;
}

TMatrix TMatrix::operator+=(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) Error=erMatrix_MixingSizes;
  if (M.SizeJ!=SizeJ) Error=erMatrix_MixingSizes;
  if (Error) return (*this);
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]+=M.Tabl[i];
  return (*this);
}

TMatrix TMatrix::operator-(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) Error=erMatrix_MixingSizes;
  if (M.SizeJ!=SizeJ) Error=erMatrix_MixingSizes;
  if (Error) return (*this);
  tempMatrix=*this;
  for(i=0;i<SizeI*SizeJ;i++) tempMatrix.Tabl[i]-=M.Tabl[i];
  return tempMatrix;
}

TMatrix TMatrix::operator-=(TMatrix& M)
{ int i;
  if (M.SizeI!=SizeI) Error=erMatrix_MixingSizes;
  if (M.SizeJ!=SizeJ) Error=erMatrix_MixingSizes;
  if (Error) return (*this);
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]-=M.Tabl[i];
  return (*this);
}

TMatrix TMatrix::operator*(TMatrix& M)
{ int     i,j,k;
  complex C;
  if (SizeJ!=M.SizeI) Error=erMatrix_MixingSizes;
  if (Error) return (*this);
  tempMatrix=TMatrix(SizeI,M.SizeJ);
  for(i=0;i<SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
    { C=0;
      for(k=0;k<SizeJ;k++) C+=operator()(i,k)*M(k,j);
      tempMatrix(i,j)=C;
    }
  return tempMatrix;
}

TMatrix TMatrix::operator*=(TMatrix& M)
{ int     i,j,k;
  complex C;
  if (SizeI!=M.SizeI) Error=erMatrix_MixingSizes;
  if (SizeJ!=M.SizeJ) Error=erMatrix_MixingSizes;
  if (Error) return (*this);
  tempMatrix=*this;
  for(i=0;i<SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
    { C=0;
      for(k=0;k<SizeJ;k++) C+=operator()(i,k)*M(k,j);
      tempMatrix(i,j)=C;
    }
  *this=tempMatrix;
  return (*this);
}

TMatrix T(TMatrix& M)
{ int i,j;
  if (M.SizeI!=M.SizeJ) Error=erMatrix_Rectangular;
  if (Error) return M;
  tempMatrix=M;
  for(i=0;i<M.SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
      tempMatrix(j,i)=conj(M(i,j));
  return tempMatrix;
}

boolean Pivot(TMatrix& M, TMatrix *I, TVector *V, int ReferenceRow)
{ // Search for one row where ReferenceRow is non-zero the switch rows
  // in order to make diagonal ReferenceRow element non-zero.
  // Apply the same operations on matrix I
  int     NewRow;
  complex C;
  for(NewRow=ReferenceRow+1;NewRow<M.SizeI;NewRow++)
  { if (abs(M(NewRow,ReferenceRow))>Epsilon)
    { M.ExchangeRows(NewRow,ReferenceRow);
      if (I!=NULL) I->ExchangeRows(NewRow,ReferenceRow);
      if (V!=NULL)
      { C=(*V)(NewRow);
        (*V)(NewRow)=(*V)(ReferenceRow);
        (*V)(ReferenceRow)=C;
      }
      return FALSE;
    }
  }
  return TRUE;
}

complex Det(TMatrix& M)
{ if (M.SizeI!=M.SizeJ) Error=erMatrix_Rectangular;
  if (Error) return 0;
  complex PartialDet=1;
  complex DiagCoef,Multiplier;
  int     Row,ReferenceRow=0;
  tempMatrix=M;
  // Make the matrix upper-triangular
  for(ReferenceRow=0;ReferenceRow<tempMatrix.SizeI-1;ReferenceRow++)
  { // If Diagonal element is 0, then switch rows
    if (abs(tempMatrix(ReferenceRow,ReferenceRow))<Epsilon)
    { if (Pivot(tempMatrix,NULL,NULL,ReferenceRow))
      { Error=erMatrix_Singularity;
        return 0;
      }
      PartialDet=-PartialDet;
    }
    DiagCoef=tempMatrix(ReferenceRow,ReferenceRow);
    for(Row=ReferenceRow+1;Row<tempMatrix.SizeI;Row++)
    { // Make the ReferenceRow Element of this Row zero
      if (abs(tempMatrix(Row,ReferenceRow))>Epsilon)
      { Multiplier=-tempMatrix(Row,ReferenceRow)/DiagCoef;
        tempMatrix.MultAddRow(ReferenceRow,Row,Multiplier);
      }
    }
    // Multiply the diagonal term into PartialDeter
    PartialDet*=DiagCoef;
  }
  return PartialDet*tempMatrix(tempMatrix.SizeI-1,tempMatrix.SizeI-1);
}

TMatrix Inv(TMatrix& M)
{ if (M.SizeI!=M.SizeJ) Error=erMatrix_Rectangular;
  if (Error) return M;
  int i,j;
  // Create the result
  TMatrix InvMatrix=M;
  // Check for trivial case : SizeI==SizeJ==1
  if (M.SizeI==1)
    if (abs(M.Tabl[0])<Epsilon)
    { // Exception : Singular Matrix
      Error=erMatrix_Singularity;
      return M;
    }
    else
    { InvMatrix.Tabl[0]=1/M.Tabl[0];
      return InvMatrix;
    }
  // Make the inverse to be Identity Matrix
  for(i=0;i<M.SizeI;i++)
    for(j=0;j<M.SizeJ;j++)
      if (i==j) InvMatrix(i,j)=1; else InvMatrix(i,j)=0;
  // Copy the original matrix in order to proceed operations on it
  tempMatrix=M;
  // Datas for computing
  int     ReferenceRow,Row;
  complex Divisor;
  complex Multiplier;
  for(ReferenceRow=0;ReferenceRow<tempMatrix.SizeI;ReferenceRow++)
  { // Make the diagonal element non-zero
    if (abs(tempMatrix(ReferenceRow,ReferenceRow))<Epsilon)
      if (Pivot(tempMatrix,&InvMatrix,NULL,ReferenceRow))
      { // Exception : Singular Matrix
        Error=erMatrix_Singularity;
        return M;
      }
    Divisor=tempMatrix(ReferenceRow,ReferenceRow);
    tempMatrix.DivideRow(ReferenceRow,Divisor);
    InvMatrix.DivideRow(ReferenceRow,Divisor);
    //
    for(Row=0;Row<M.SizeI;Row++)
    { if (Row!=ReferenceRow)
        if (abs(tempMatrix(Row,ReferenceRow))>Epsilon)
        { // Make the ReferenceRow element of this Row zero
          Multiplier=-tempMatrix(Row,ReferenceRow);
          tempMatrix.MultAddRow(ReferenceRow,Row,Multiplier);
          InvMatrix.MultAddRow(ReferenceRow,Row,Multiplier);
        }
    }
  }
  return InvMatrix;
}

// ------

TVector TMatrix::operator*(TVector& V)
{ if (SizeJ!=V.size()) Error=erMatrix_VectorSize;
  if (Error) return V;
  tempVector=TVector(SizeI);
  complex C;
  int     i,k;
  for(i=0;i<SizeI;i++)
  { C=0;
    for(k=0;k<SizeJ;k++)
      C+=operator()(i,k)*V(k);
    tempVector(i)=C;
  }
  return tempVector;
}

TVector TMatrix::SolveSystem(TVector& Constant)
{ if (SizeJ!=Constant.size()) Error=erMatrix_VectorSize;
  if (SizeI!=SizeJ)         Error=erMatrix_Rectangular;
  if (Error) return Constant;
  // Create the solution vector
  TVector Sol(SizeJ);
  // Solve the trivial case
  if (SizeI==1)
  { if (abs(Tabl[0])<Epsilon) erMatrix_Singularity;
    if (Error) return Constant;
    Sol(0)=Constant(0)/Tabl[0];
    return Sol;
  }
  // Copy the coefficient matrix and the constant vector in order to perform
  // transformations on it
  TMatrix Coefs=*this;
  TVector Const=Constant;
  // Makes the coefficient matrix upper-triangular and perform the same
  // operations on the constant vector
  int     ReferenceRow,Row,Term;
  complex DiagCoef,Multiplier,C;
  for(ReferenceRow=0;ReferenceRow<SizeI-1;ReferenceRow++)
  { // Check if the diagonal element is zero
    if (abs(Coefs(ReferenceRow,ReferenceRow))<Epsilon)
      if (Pivot(Coefs,NULL,&Const,ReferenceRow))
      { // Exception : Singular Matrix
        Error=erMatrix_Singularity;
        return Const;
      }
    // Make other coefficients of this ReferenceRow column zero
    DiagCoef=Coefs(ReferenceRow,ReferenceRow);
    for(Row=ReferenceRow+1;Row<SizeI;Row++)
    { if (abs(Coefs(Row,ReferenceRow))>Epsilon)
      { Multiplier=-Coefs(Row,ReferenceRow)/DiagCoef;
        Coefs.MultAddRow(ReferenceRow,Row,Multiplier);
        Const(Row)+=Multiplier*Const(ReferenceRow);
      }
    }
  }
  if (abs(Coefs(SizeI-1,SizeJ-1))<Epsilon)
  { Error=erMatrix_Singularity;
    return Const;
  }
  // Apply Backwards substitution to the upper triangular Coefs Matrix and
  // Constant vector.
  for(Term=SizeI-1;Term>=0;Term--)
  { C=0;
    for(Row=Term+1;Row<SizeI;Row++)
      C+=Coefs(Term,Row)*Sol(Row);
    Sol(Term)=(Const(Term)-C)/Coefs(Term,Term);
  }
  return Sol;
}

// ------
         
TMatrix operator*(TMatrix& M, complex  C)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]*=C;
  return tempMatrix;
}

TMatrix operator*(complex  C, TMatrix& M)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]*=C;
  return tempMatrix;
}

TMatrix TMatrix::operator*=(complex  C)
{ int i;
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]*=C;
  return (*this);
}

TMatrix operator/(TMatrix& M, complex  C)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]/=C;
  return tempMatrix;
}

TMatrix operator/(complex  C, TMatrix& M)
{ int i;
  tempMatrix=M;
  for(i=0;i<tempMatrix.SizeI*tempMatrix.SizeJ;i++) tempMatrix.Tabl[i]=C/M.Tabl[i];
  return tempMatrix;
}

TMatrix TMatrix::operator/=(complex  C)
{ int i;
  for(i=0;i<SizeI*SizeJ;i++) Tabl[i]/=C;
  return (*this);
}

void TMatrix::SetColumn(int J, TVector& V)
{ if (V.size()!=SizeI) Error=erMatrix_VectorSize;
  if ((J<0)||(J>=SizeJ)) Error=erMatrix_OutOfSize;
  if (Error) return;
  int i;
  for(i=0;i<SizeI;i++)
    operator()(i,J)=V(i);
}

void TMatrix::SetRow(int I, TVector& V)
{ if (V.size()!=SizeJ) Error=erMatrix_VectorSize;
  if ((I<0)||(I>=SizeI)) Error=erMatrix_OutOfSize;
  if (Error) return;
  int j;
  for(j=0;j<SizeJ;j++)
    operator()(I,j)=V(j);
}

TVector TMatrix::Column(int J)
{ if (J<0)      { Error=erMatrix_OutOfSize; J=0;       }
  if (J>=SizeJ) { Error=erMatrix_OutOfSize; J=SizeJ-1; }
  int i;
  tempVector=TVector(SizeI);
  for(i=0;i<SizeI;i++) tempVector(i)=operator()(i,J);
  return tempVector;
}

TVector TMatrix::Row(int I)
{ if (I<0)      { Error=erMatrix_OutOfSize; I=0;       }
  if (I>=SizeI) { Error=erMatrix_OutOfSize; I=SizeI-1; }
  int j;
  tempVector=TVector(SizeJ);
  for(j=0;j<SizeJ;j++) tempVector(j)=operator()(I,j);
  return tempVector;
}

void TMatrix::ExchangeRows(int I1, int I2)
{ if (I1<0)      { Error=erMatrix_OutOfSize; I1=0;       }
  if (I1>=SizeI) { Error=erMatrix_OutOfSize; I1=SizeI-1; }
  if (I2<0)      { Error=erMatrix_OutOfSize; I2=0;       }
  if (I2>=SizeI) { Error=erMatrix_OutOfSize; I2=SizeI-1; }
  int      j;
  complex  C;
  complex *P1,*P2;
  P1=Tabl+I1*SizeJ;
  P2=Tabl+I2*SizeJ;
  for(j=0;j<SizeJ;j++)
  { C=*P1; *P1=*P2; *P2=C;
    P1++;
    P2++;
  }
}

void TMatrix::ExchangeColumns(int J1, int J2)
{ if (J1<0)      { Error=erMatrix_OutOfSize; J1=0;       }
  if (J1>=SizeJ) { Error=erMatrix_OutOfSize; J1=SizeJ-1; }
  if (J2<0)      { Error=erMatrix_OutOfSize; J2=0;       }
  if (J2>=SizeJ) { Error=erMatrix_OutOfSize; J2=SizeJ-1; }
  int      i;
  complex  C;
  complex *P1,*P2;
  P1=Tabl+J1;
  P2=Tabl+J2;
  for(i=0;i<SizeI;i++)
  { C=*P1; *P1=*P2; *P2=C;
    P1+=SizeJ;
    P2+=SizeJ;
  }
}

void TMatrix::MultAddRow(int I1, int I2, complex C)
{ if (I1<0)      { Error=erMatrix_OutOfSize; I1=0;       }
  if (I1>=SizeI) { Error=erMatrix_OutOfSize; I1=SizeI-1; }
  if (I2<0)      { Error=erMatrix_OutOfSize; I2=0;       }
  if (I2>=SizeI) { Error=erMatrix_OutOfSize; I2=SizeI-1; }
  int      j;
  complex *P1,*P2;
  P1=Tabl+I1*SizeJ;
  P2=Tabl+I2*SizeJ;
  for(j=0;j<SizeJ;j++)
  { *P2+=*P1*C;
    P1++;
    P2++;
  }
}

void TMatrix::MultAddColumn(int J1, int J2, complex C)
{ if (J1<0)      { Error=erMatrix_OutOfSize; J1=0;       }
  if (J1>=SizeJ) { Error=erMatrix_OutOfSize; J1=SizeJ-1; }
  if (J2<0)      { Error=erMatrix_OutOfSize; J2=0;       }
  if (J2>=SizeJ) { Error=erMatrix_OutOfSize; J2=SizeJ-1; }
  int      i;
  complex *P1,*P2;
  P1=Tabl+J1;
  P2=Tabl+J2;
  for(i=0;i<SizeI;i++)
  { *P2+=*P1*C;
    P1+=SizeJ;
    P2+=SizeJ;
  }
}

void TMatrix::DivideRow(int I, complex C)
{ if (I<0)      { Error=erMatrix_OutOfSize; I=0;       }
  if (I>=SizeI) { Error=erMatrix_OutOfSize; I=SizeI-1; }
  int      j;
  complex *P=Tabl+I*SizeJ;
  for(j=0;j<SizeJ;j++)
  { *P/=C;
    P++;
  }
}

void TMatrix::DivideColumn(int J, complex C)
{ if (J<0)      { Error=erMatrix_OutOfSize; J=0;       }
  if (J>=SizeJ) { Error=erMatrix_OutOfSize; J=SizeJ-1; }
  int      i;
  complex *P=Tabl+J;
  for(i=0;i<SizeI;i++)
  { *P/=C;
    P+=SizeI;
  }
}
