***********************************************
* Module    : OS11.MOD
* Programmer: Tony Papadimitriou (FREEWARE)
* Purpose   : Various OS-called routines
* Language  : MC68HC11 (ASM11 v1.61+)
* History   : 98.04.30 Version 1.00    Original
*           : 99.02.03 Version 1.01    Detached from FRAME.ASM
*           : 99.02.11 Version 1.02    Adapted to v1.56 (local labels)
*           : 99.02.14 Version 1.03    Minor fixes, added Error printing routines
*           : 99.02.15 Version 1.04    Added LCD-related routines
***********************************************
$ifdef DEBUG
  $message DEBUG Mode Assembly (do NOT burn device)
$endif

* The following conditional allows this file to be assembled separately *
$ifmain
$listoff
$ifdef MCU711E9
          $message  Target: MCU711E9
                    $include  711E9.INC           -- 68HC711E9 specific equates
$else
          $message  Target: MCU811E2
                    $include  811E2.INC           -- 68HC811E2 specific equates (default)
$endif
                    $include  COMMON.INC          -- Common Equates
$liston
$endif

$PAGE
**********************************************************************
*                        OS Error Codes                              *
*            (Any number from 0 to 255 may be defined)               *
**********************************************************************
errNone             equ       0                   No Error
errOther            equ       1                   Other error
errUnused           equ       2                   Opcode unused
errBadParm          equ       3                   Bad parameter
errFailure          equ       4                   Operation failed
errOutOfRange       equ       5                   Value out of range
errNotFound         equ       6                   Not Found (Search, etc.)

$PAGE
                    $ROM
$S19FLUSH                     -- Force on a new S19 record
**********************************************************************
* SWI routines table (OpCodes $00 to $FF may be used for 8-bit mode) *
*       (Opcodes $0000 to $7FFF may be used for 16-bit mode)         *
* They must be listed in the table in sequence from lowest to highest*
* If a number must be left unused in between, point it to Op_Unused  *
* When calling with OS or OSW, use the fXXX name, eg.  OS fPrint     *
* You may change the order of the calls as long as you move both EQU *
* and related DW lines together.  They are separated with blank lines*
* for easy identification.                                           *
**********************************************************************
?OSCommands         equ       *

fError              equ       *-?OSCommands/2     Print error message on the SCI
                    dw        ?Error

fErrorJump          equ       *-?OSCommands/2     fError and jump to address
                    dw        ?ErrorJump

fUsrErrorJump       equ       *-?OSCommands/2     Like fErrorJump with User error
                    dw        ?UsrErrorJump

fDelayMS            equ       *-?OSCommands/2     Delay variable number of ms
                    dw        ?DelayMS

fDelay10MS          equ       *-?OSCommands/2     Delay 10ms
                    dw        ?Delay10MS

fPos                equ       *-?OSCommands/2     Get character position
                    dw        ?Pos

fCompare            equ       *-?OSCommands/2     Compare strings at X and Y
                    dw        ?Compare

fCopyMem            equ       *-?OSCommands/2     Copy a block of memory
                    dw        ?CopyMem

fSetBAUD            equ       *-?OSCommands/2     Set SCI bps ("baud") rate
                    dw        ?SetBAUD

fPrint              equ       *-?OSCommands/2     Print the following FCS string
                    dw        ?Print

fBin2Dec            equ       *-?OSCommands/2     Binary to Decimal conversion
                    dw        ?Bin2Dec

fBin2BCD            equ       *-?OSCommands/2     Binary to BCD conversion (actually any base)
                    dw        ?Bin2BCD

fPutChar            equ       *-?OSCommands/2     Write RegA character to the SCI
                    dw        ?PutChar

fWrite              equ       *-?OSCommands/2     Write a "Pascal" string to SCI
                    dw        ?Write

fWriteln            equ       *-?OSCommands/2     fWrite followed by CR,LF pair
                    dw        ?Writeln

fNewLine            equ       *-?OSCommands/2     Send a CR,LF pair to the SCI
                    dw        ?NewLine

fWriteZ             equ       *-?OSCommands/2     Write an ASCIIZ string to SCI
                    dw        ?WriteZ

fWritelnZ           equ       *-?OSCommands/2     fWriteZ followed by CR,LF pair
                    dw        ?WritelnZ

fUpString           equ       *-?OSCommands/2     Convert ASCIIZ string to uppercase
                    dw        ?UpString

fUpPascalStr        equ       *-?OSCommands/2     Convert Pascal-type string pointed to by X to uppercase
                    dw        ?UpPascalStr

fUpChar             equ       *-?OSCommands/2     Convert char in RegA to uppercase
                    dw        ?UpChar

fWriteEE            equ       *-?OSCommands/2     Write one EEPROM byte
                    dw        ?WriteEE

fEraseEE            equ       *-?OSCommands/2     Erase one EEPROM byte
                    dw        ?EraseEE

fBulkEraseEE        equ       *-?OSCommands/2     Bulk Erase EEPROM
                    dw        ?BulkEraseEE

fSearchTable        equ       *-?OSCommands/2     Search any data table
                    dw        ?SearchTable

fConvertAD          equ       *-?OSCommands/2     Convert A/D value to volts
                    dw        ?ConvertAD

fLength             equ       *-?OSCommands/2     Get ASCIIZ string length
                    dw        ?Length

fInsert             equ       *-?OSCommands/2     Insert char within ASCIIZ string
                    dw        ?Insert

fDelete             equ       *-?OSCommands/2     Delete char within ASCIIZ string
                    dw        ?Delete

fKickCOP            equ       *-?OSCommands/2     Kick the COP timer to prevent timeout resets
                    dw        KickCOP             Not a local label, called either way

fHexToASCII         equ       *-?OSCommands/2     Convert a hex number to its ASCII equivalent
                    dw        ?HexToASCII

fNoLeadChar         equ       *-?OSCommands/2     Convert leading RegB chars to RegA char
                    dw        ?NoLeadChar

;fCRC               equ       *-?OSCommands/2     Calculate CRC (disabled)
;                   dw        ?CRC

fLoadS19            equ       *-?OSCommands/2     Load an S19 file via the SCI
                    dw        ?LoadS19

fHexToBin           equ       *-?OSCommands/2     Convert a hex digit (0-F) in RegA to its binary equivalent
                    dw        ?HexToBin

fGarble             equ       *-?OSCommands/2     Garble buffer at X with seed in RegA
                    dw        ?Garble

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
$ifdef LCD
          $message  LCD OS calls enabled
fLCDInit            equ       *-?OSCommands/2     Initialize the LCD display
                    dw        ?LCD.Init

fLCDCls             equ       *-?OSCommands/2     Clear the LCD display
                    dw        ?LCD.Cls

fLCDOn              equ       *-?OSCommands/2     Enable the LCD display
                    dw        ?DispOn

fDirLeft            equ       *-?OSCommands/2     Set write direction towards left (eg. numbers)
                    dw        ?DirLeft

fDirRght            equ       *-?OSCommands/2     Set write direction towards right (normal)
                    dw        ?DirRght

fSetAddr            equ       *-?OSCommands/2     Set LCD address
                    dw        ?SetDDRAM

fClrEOL             equ       *-?OSCommands/2     Clear to the end of current LCD line
                    dw        ?ClrEOL

fLCDPutChar         equ       *-?OSCommands/2     Put a character to the LCD display
                    dw        ?LCD.PutChar

$endif
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;fCallName          equ       *-?OSCommands/2     Keep adding more functions
;                   dw        ?Routine_Name       ..like with these two lines
;
;                   dw        ?Op_Unused          This is one line only (no EQU needed)
;
;fCallName          equ       *-?OSCommands/2     Another two line entry
;                   dw        ?Routine_Name       ..for another call
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

?OS.Entries         equ       *-?OSCommands/2     Number of opcodes defined

$if ?OS.Entries > $FF
?OS_16BIT           ;Use (16 bit) OSW instruction, rather than (8-bit) OS
$message OS calls are 16-bit (be sure to use OSW only)
$endif

$PAGE
**********************************************************************
*                       OS CALLS COME HERE                           *
**********************************************************************
* Call format: OS CallName (OSW CallName for 16-bit opcodes)         *
*            : Parameters are passed in registers depending on call  *
*            : Results either in memory or regs depending on call    *
*            : No registers (other than results) are destroyed       *
*            : All OS calls are re-entrant, you can use them with    *
*            : other OS calls (just be careful of stack usage).      *
* Returns    : No Carry if no error, B is unaffected                 *
*            : Carry if error, B holds error code                    *
* Note(s)    : Each subcall should get parameters from Reg??,y and   *
*            : return results in Reg??_,y and error code in B_,y     *
*            : User need not save the Y index as it is restored by   *
*            : the dispatcher.                                       *
*            : Each function should be coded as a subroutine ending  *
*            : with an RTS instruction, not ending with RTI.         *
*            : Functions need not save any of the registers.         *
*            : Functions should either clear or set the Carry flag   *
*            : on exit to indicate success or failure, respectively. *
*            : On failure (Carry Set), functions should also set the *
*            : value of the B register with an appropriate error     *
*            : code.                                                 *
*            : Interrupts will be allowed during the call if the I   *
*            : bit before the call was clear, otherwise they won't.  *
* Gen Reg Use: Generally, registers used should have a similar role  *
*            : in each routine.  However, no specific restrictions   *
*            : are imposed.                                          *
*            : X is used as a destination memory pointer             *
*            :   (if only one pointer is used, it is X, not Y)       *
*            : Y is used as a source memory pointer                  *
*            : D (or just B if dealing with 8-bit quantities) is     *
*            :   used as a counter.                                  *
*            : This arrangement may be violated when necessary, for  *
*            : example, if there is a need for one pointer and two   *
*            : counters, X will be the pointer, D and Y the primary  *
*            : and secondary counters, respectively.                 *
* Total Stack: 11 (9 for caller + 2 for local storage without subs)  *
* Note       : Changed from previous versions to reduce stack used   *
*            : and not pre-load D (each sub must explicitly load D). *
*            : Also, removed pre-loading of CCR.  Subs may use the   *
*            : stacked CCR instead.                                  *
**********************************************************************
* OS Command Dispatcher.  Based on function, it calls appropriate subroutine.
SWI_Handler         tsy                           point Y to SWI stack frame
                    brset     CCR_,y,I.,?SWI_Continue if I bit was set before OS call, don't allow ints
                    cli                           Enable interrupts
?SWI_Continue       ldx       PC_,y               adjust PC for opcode
$ifdef ?OS_16BIT
                    ldd       ,x                  OSW type calls
                    inx                           skip 2nd opcode byte
$else
                    clra                          we're gonna use D (for OS calls)
                    ldb       ,x                  get OS opcode byte
$endif
                    inx                           skip 1st opcode byte
                    stx       PC_,y               and save new PC to return to
                    cpd       #?OS.Entries        is it a valid OpCode?
                    bhs       ?SWI_Ret
                    lsld                          multiply by 2 (the size of table entries)
                    addd      #?OSCommands
                    xgdx                          put result in X
                    ldx       ,x                  get jmp vector
                    jsr       ,x                  Perform function
                    tsy                           Re-load Y in case it was destroyed
;now, we should prepare the stack frame to be returned with appropriate values
                    bcs       ?SWI_Error
                    bclr      CCR_,y,C.           clear Carry flag
                    bra       AnRTI
?SWI_Error          bset      CCR_,y,C.           set Carry flag
                    stb       B_,y                save error code for user
AnRTI               rti                           A globally accessed RTI
* The following section handles invalid System Opcodes *
?SWI_Ret            ldx       PC_,y               adjust PC to start of opcode
                    dex:2                         ..back 1 byte for operand and 1 for the OS opcode
$ifdef ?OS_16BIT
                    dex                           ..one more for 16-bit operands
$endif
                    stx       PC_,y               and save new PC to return to
                    ldx       ILLOP_VECTOR
                    cpx       #ERASED_STATE       Check if not initialized (for EPROM, ie., $FFFF)
                    beq       ?SWI_Ret.2
                    jmp       ,x                  Invalid System Call
?SWI_Ret.2          jmp       Halt_MCU            In case ILLOP handler not installed

$PAGE
**********************************************************************
*                 Operating System routines expanded                 *
**********************************************************************

?Op_Unused          ldb       #errUnused          Dummy call for missing opcodes
                    sec                           ..use your own non-zero error codes
AnRTS               rts                           Also, provides a globally accessed RTS

$ifdef fErrorJump
  $ifndef fError
    $error To use fErrorJump you must enable fError
  $else
; Purpose: Print error message on the SCI and jump to address in X
; Input  : B holds error code
;        : X holds address to jump to after printing error
; Output : None
?ErrorJump          ldx       X_,y                get address to jump to
                    stx       PC_,y               and use it as return PC
          * fall through to fError (do not change fError's location)
  $endif
$endif

$ifdef fError
  $ifndef fWritelnZ
    $error To use fError you must enable fWritelnZ
  $else
; Purpose: Print error message on the SCI
; Input  : B holds error code
; Output : None
; Note(s): Uses fPrint, fWritelnZ
?Error              ldb       B_,y                get error code
                    beq       ?Error.Exit         not an error
                    decb                          make error zero-based
                    clra                          we'll be using D
                    lsld                          multiply by size of table entries
                    cmpd      #?Error.Size        is it within our table size
                    bhs       ?Error.Exit         invalid error code, get out
                    addd      #?Error.Table       point to beginning of table
                    xgdx                          put pointer in X
                    ldx       ,x                  get pointer to error msg
                    os        fPrint              print standard error header
                    fcs       CR,LF,'OS ERROR('
                    os        fWritelnZ
?Error.Exit         clc
                    rts
?Error.Table        dw        ?Error.1
                    dw        ?Error.2
                    dw        ?Error.3
                    dw        ?Error.4
                    dw        ?Error.5
                    dw        ?Error.6
?Error.Size         equ       *-?Error.Table/2
?Error.1            fcs       '1): Other'
?Error.2            fcs       '2): Invalid Call'
?Error.3            fcs       '3): Bad Parameter'
?Error.4            fcs       '4): Failed'
?Error.5            fcs       '5): Out Of Range'
?Error.6            fcs       '6): Not found'
  $endif
$endif

$ifdef fUsrErrorJump
  $ifndef fWritelnZ
    $error To use fUsrErrorJump you must enable fWritelnZ
  $else
; Purpose: Print user error message on the SCI
; Input  : Y -> user error message
;        : X holds address to jump to after printing error
; Output : None
; Note(s): Uses fPrint, fWritelnZ
?UsrErrorJump       os        fPrint              print standard error header
                    fcs       CR,LF,'ERROR: '
                    ldx       Y_,y                get pointer to error msg
                    os        fWritelnZ
                    ldx       X_,y                set user address
                    stx       PC_,y               ..to return to
                    clc
                    rts
  $endif
$endif

$ifdef fDelayMS
; Purpose: Delay ~ RegX milliseconds (not counting caller's overhead)
?DelayMS            ldx       X_,y                Get milliseconds to delay
?DelayMS.01         pshx
$ifdef DEBUG
                    ldx       #1                  for Simulator use minimum delay
$else
                    ldx       #333                constant for 1 millisecond
$endif
                    dex
                    bne       *-1
                    pulx
                    dex
                    bne       ?DelayMS.01
                    clc                           No errors in any case
                    rts
$endif

$ifdef fDelay10MS
; Purpose: Delay 10 msec (20000 cycles @500nsec/cycle assuming 2MHz E clock)
; Note(s): All involved registers are preserved so it can be called internally
;          This routine executes in 19999 cycles from entry (including BSR or
;          JSR instruction) to completion of RTS
?Delay10MS          pshx                          4 cycles
$ifdef DEBUG
                    ldx       #1                  for Simulator use minimum delay
$else
                    ldx       #3329               3 cycles
$endif
                    dex                           3 cycles
                    bne       *-1                 3 cycles
                    pulx                          5 cycles
                    clc                           2 cycles
                    rts                           5 cycles
$endif

$ifdef fPos
; Purpose: Get the position of a character starting at X
; Input  : X points to beginning position in search buffer (0,CR,LF terminated)
;        : A holds character to search for
;        : B holds number of matches to perform
; Output : IF FOUND: B holds offset from X to found character (ready for ABX)
;        :           Carry is Clear
;        : IF NOT FOUND: B is unaffected and Carry is Set
; Note(s): Target character can also be one of the terminators
?Pos                lda       A_,y                and get target character in A
                    ldx       X_,y                and load X
                    ldb       B_,y
                    beq       ?Pos.NotFound.Out
                    pshb
                    clrb                          Initialize B counter
?Pos.Loop           cmpa      ,x                  have we found the character?
                    beq       ?Pos.Found
                    psha                          save for tests
                    tst       ,x
                    beq       ?Pos.NotFound       A NULL means end of string
                    lda       #CR
                    cmpa      ,x
                    beq       ?Pos.NotFound       A CR means end of string
                    lda       #LF
                    cmpa      ,x
                    beq       ?Pos.NotFound       A LF means end of string
                    pula                          restore after tests
                    incb
                    inx
                    bra       ?Pos.Loop
?Pos.NotFound       pula
                    pulb
?Pos.NotFound.Out   sec
                    rts
?Pos.Found          inx
                    incb
                    dec       B_,y
                    bne       ?Pos.Loop
                    decb
                    stb       B_,y
                    pulb
                    clc
                    rts
$endif

$ifdef fCompare
; Purpose: Compare buffers pointed by X and Y for equality (case sensitive)
; Input  : B holds number of bytes to compare
;        : Y points to start of first memory
;        : X points to start of second memory
; Output : Zero flag set if equal (use BEQ), clear if not (use BNE)
; Note(s): All registers except Zero and Carry flags are preserved
;        : X and Y can point to either buffer (order not important)
?Compare            pshy
                    ldb       B_,y                load parameters
                    beq       ?Compare.OK         zero length always OK
                    ldx       X_,y
                    ldy       Y_,y
?Compare.Loop       lda       ,x                  get character at X
                    cmpa      ,y                  is the same as the one at Y?
                    bne       ?Compare.No         No, get out then
                    decb                          are we done with all characters?
                    beq       ?Compare.OK         Yes, and the all compare OK
                    inx                           No, let's go check next ones
                    iny
                    bra       ?Compare.Loop
?Compare.No         pulx
                    bclr      CCR_,x,Z.           clear the returned zero flag
                    clc
                    rts
?Compare.OK         pulx
                    bset      CCR_,x,Z.           set the returned zero flag
                    clc
                    rts
$endif


$ifdef fCopyMem
; Purpose: Copy (move) bytes from source to destination memory
; Input  : D holds number of bytes to copy
;        : Y points to start of source memory
;        : X points to start of destination memory
; Note(s): Correctly handles copies (even partially) to EEPROM
?CopyMem            pshy
                    lda       A_,y                load parameters
                    ldb       B_,y
                    ldx       X_,y
                    ldy       Y_,y
                    cpd       #0                  any bytes to copy?
                    beq       ?CopyMem.NoError    no, get out of here
?CopyMem.Loop       pshd                          save counter between loops
$ifdef fWriteEE
                    cpx       #EEPROM             check destination for EEPROM
                    blo       ?CopyMem.RAM
                    cpx       #EEPROM_END
                    bhi       ?CopyMem.RAM
     *?CopyMem.EEPROM
                    lda       ,y                  get a byte from source
                    jsr       ?WriteEE.Local      write EEPROM byte A to ,X
                    bcs       ?CopyMem.Error      on error...
                    bra       ?CopyMem.LoopEnd    no error, continue
$endif
?CopyMem.RAM        lda       ,y                  get byte from source
                    sta       ,x                  put byte to destination
?CopyMem.LoopEnd    inx                           bump up both pointers
                    iny
                    puld                          get loop counter
                    decd                          and count down
                    bne       ?CopyMem.Loop
                    bra       ?CopyMem.NoError
?CopyMem.Error      puld
                    puly
                    ldb       #errFailure
                    sec
                    rts
?CopyMem.NoError    puly
                    clc
                    rts
$endif

$ifdef fSetBAUD
; Purpose: Set SCI BAUD rate to one of the following values (in A_,y)
;          3(00), 12(00), 24(00), 48(00), 96(00), 125(000)
;          (use number outside parentheses for corresponding baud rate)
; Input  : RegA = Baud Rate
;        : RegB = Length of buffer (required only for interrupt mode)
;        : RegX = Pointer to buffer to use (required only for interrupt mode)
;        : Carry = Set when we want interrupt driven mode
;        : Carry = Clear when we want polling mode
; Note(s): Assumes 8MHz XTAL (2MHz E-Clock)
?SetBAUD            lda       A_,y
                    cmpa      #3                  is it 300bps?
                    bne       ?SetBAUD.1200
                    lda       #$35
                    bra       ?SetBAUD.SetBaud
?SetBAUD.1200       cmpa      #12                 is it 1200bps?
                    bne       ?SetBAUD.2400
                    lda       #$33
                    bra       ?SetBAUD.SetBaud
?SetBAUD.2400       cmpa      #24                 is it 2400bps?
                    bne       ?SetBAUD.4800
                    lda       #$32
                    bra       ?SetBAUD.SetBaud
?SetBAUD.4800       cmpa      #48                 is it 4800bps?
                    bne       ?SetBAUD.9600
                    lda       #$31
                    bra       ?SetBAUD.SetBaud
?SetBAUD.9600       cmpa      #96                 is it 9600bps?
                    bne       ?SetBAUD.125000
                    lda       #$30
                    bra       ?SetBAUD.SetBaud
?SetBAUD.125000     cmpa      #125                is it 125000bps?
                    bne       ?SetBAUD.Error
                    clra
?SetBAUD.SetBaud    clr       SCCR2               disable SCI while changing BAUD
                    sta       BAUD
                    clr       SCCR1
                    brset     CCR_,y,C.,?SetBAUD.WithInt
                    lda       #$0C                No interrupt mode
                    sta       SCCR2
                    bra       ?SetBAUD.Exit       exit without errors
?SetBAUD.WithInt    lda       #$2C                Interrupt mode
                    sta       SCCR2
* Clear the buffer to be used for transmission/receipt of data
                    ldx       X_,y                get pointer to buffer
                    ldb       B_,y                get length of buffer
                    clr       ,x                  zero length in buffer[1]
?SetBAUD.Loop       inx
                    clr       ,x                  clear buffer and advance pointer
                    decb
                    bne       ?SetBAUD.Loop
?SetBAUD.Exit       clc                           no error exit
                    rts
?SetBAUD.Error      ldb       #errBadParm         return error code in B
                    sec
                    rts
$endif

$ifdef fPrint
; Purpose: Print ASCIIZ string following OS instruction (with FCS)
; Parms  : None
?Print              ldx       PC_,y               get address following OS
                    pshy
                    ldy       #REGS
?Print.Loop         lda       ,x
                    beq       ?Print.Exit
                    brclr     [SCSR,y,$80,*       wait for ready
                    sta       SCDR
                    inx
                    bra       ?Print.Loop
?Print.Exit         inx                           point after NUL
                    puly
                    stx       PC_,y               ..to use as return address
                    clc                           no errors
                    rts
$endif

$ifdef fBin2Dec
; Purpose: Convert a binary number to decimal
; Input  : D=Unsigned binary number from 0 to 65535 ($FFFF)
; Output : D=First two ASCII digits of result (first char is blank)
;        : X=Second two ASCII digits of result
;        : Y=Third two ASCII digits of result
?Bin2Dec            lda       A_,y
                    ldb       B_,y
                    ldx       #'00'
                    pshx:3                        initialize stack result
                    pshy                          original Y
                    tsy                           and put it's pointer in Y
                    iny:2
?Bin2Dec.01.0       subd      #10000              figure out tens of thousands
                    bcs       ?Bin2Dec.01
                    inc       1,y                 increment tens of thousands
                    bra       ?Bin2Dec.01.0
?Bin2Dec.01         addd      #10000
?Bin2Dec.01.1       subd      #1000               figure out thousands
                    bcs       ?Bin2Dec.02
                    inc       2,y                 increment thousands
                    bra       ?Bin2Dec.01.1
?Bin2Dec.02         addd      #1000
?Bin2Dec.02.1       subd      #100
                    bcs       ?Bin2Dec.03
                    inc       3,y                 increment hundreds
                    bra       ?Bin2Dec.02.1
?Bin2Dec.03         addd      #100
?Bin2Dec.03.1       subd      #10
                    bcs       ?Bin2Dec.04
                    inc       4,y                 increment tens
                    bra       ?Bin2Dec.03.1
?Bin2Dec.04         addd      #10+'0'
                    stb       5,y                 finally, save units
                    puly                          get original Y
                    puld
                    sta       A_,y
                    stb       B_,y
                    puld
                    std       X_,y
                    puld
                    std       Y_,y
?Bin2Dec.Exit       clc
                    rts
$endif

$ifdef fPutChar
; Purpose: Write RegA character to the SCI
; Input  : A holds character
?PutChar            lda       A_,y
?PutChar.Local      brclr     [SCSR,y,$80,*
                    sta       SCDR
                    clc
                    rts
$endif

$ifdef fWrite
; Purpose: Write (send) a string to the SCI
; Input  : X->buffer of Pascal-like string, ie., LengthByte,Char1,Char2,...
; Note(s): buffer should not be used for incoming chars, for it will be
;          messed up.
?Write              ldx       X_,y
                    ldy       #REGS
                    ldb       ,x                  get length of string
                    beq       ?Write.NoError      if 0, nothing to send
                    inx                           point to data
?Write.Loop         lda       ,x
                    bsr       ?PutChar.Local
                    inx
                    decb
                    bne       ?Write.Loop
                    bra       ?Write.NoError
?Write.Error        ldb       #errFailure
                    sec
                    rts
?Write.NoError      clc
                    rts
$endif

$ifdef fWriteln
; Purpose: Writeln (send) a string to the SCI followed by a CR,LF pair
; Input  : X->buffer of Pascal-like string, ie., LengthByte,Char1,Char2,...
?Writeln            bsr       ?Write              do regular fWrite
                    bcs       ?Writeln.Exit       on failure, exit
; Purpose: Advance a line sending a CR,LF pair to the SCI
?NewLine            ldy       #REGS
                    brclr     [SCSR,y,$80,*       wait for ready
                    lda       #CR                 send a CR
                    sta       SCDR
                    brclr     [SCSR,y,$80,*       wait for ready
                    lda       #LF                 send a LF
                    sta       SCDR
                    clc
?Writeln.Exit       rts
$endif

$ifdef fWriteZ
; Purpose: Write (send) a string to the SCI
; Input  : X->ASCIIZ string, ie., Char1,Char2,...,0
; Note(s): buffer should not be used for incoming chars, for it will be
;          messed up.
?WriteZ             ldx       X_,y
                    ldy       #REGS
?WriteZ.Loop        lda       ,x
                    beq       ?WriteZ.NoError
                    brclr     [SCSR,y,$80,*
                    sta       SCDR
                    inx
                    bne       ?WriteZ.Loop
?WriteZ.Error       ldb       #errFailure
                    sec
                    rts
?WriteZ.NoError     clc
                    rts
$endif

$ifdef fWritelnZ
  $ifndef fWriteZ
    $error To use fWritelnZ you must enable fWriteZ
  $else
; Purpose: Writeln (send) a string to the SCI followed by a CR,LF pair
; Input  : X->ASCIIZ string, ie., Char1,Char2,...,0
?WritelnZ           bsr       ?WriteZ             do regular fWriteZ
                    bcs       ?Writeln.Exit       on failure, exit
                    bra       ?NewLine
  $endif
$endif

$ifdef fUpString
; Purpose: Convert ASCIIZ string pointed to by X to uppercase
; Input  : X_,y->string
; Output : X_,y->STRING
?UpString           ldx       X_,y                point X to string
?UpString.Loop      lda       ,x
                    beq       ?UpString.Exit
                    jsr       Upcase              convert to uppercase
                    sta       ,x
                    cpx       #RAM_END            do not run wild outside of RAM
                    bhs       ?UpString.Exit
                    inx
                    bne       ?UpString.Loop
?UpString.Exit      clc
                    rts
$endif

$ifdef fUpPascalStr
; Purpose: Convert Pascal-type string pointed to by X to uppercase
; Input  : X_,y->string
; Output : X_,y->STRING
?UpPascalStr        ldx       X_,y                point X to string
                    ldb       ,x                  get length in B
                    beq       ?UpPascalStr.Exit   get out now, if length is 0
                    clra                          zero high byte
                    xgdy                          copy to loop counter (Y)
                    inx                           point to first data byte
?UpPascalStr.Loop   lda       ,x
                    jsr       Upcase              convert to uppercase
                    sta       ,x
                    inx
                    dey
                    bne       ?UpPascalStr.Loop
?UpPascalStr.Exit   clc
                    rts
$endif

$ifdef fUpChar
; Purpose: Convert ASCII character in A to uppercase
; Input  : RegA->char
; Output : RegA->CHAR
?UpChar             lda       A_,y
                    jsr       Upcase
                    sta       A_,y
                    clc
                    rts
$endif

$ifdef fWriteEE
; Purpose: Program an internal EEPROM location
; Input  : A_,y->value to be programmed
;        : X_,y->address to program
?WriteEE            lda       A_,y
                    ldx       X_,y
?WriteEE.Local      equ       *                   local call entry point
                    ldb       #$02                EELAT = 1
                    stb       PPROG               Set EELAT bit
                    sta       ,x                  save data to address
                    ldb       #$03                EELAT = 1 and EPGM = 1
                    stb       PPROG               Turn on programming voltage
                    jsr       ?Delay10MS          Delay 10 msec
                    clr       PPROG               Turn off programming voltage
                    cmpa      ,x                  Compare with value written
                    beq       ?WriteEE.Exit       if equal, get out, no errors
                    ldb       #errFailure
                    sec
                    rts
?WriteEE.Exit       clc
                    rts
$endif

$ifdef fEraseEE
; Purpose: Byte Erase an internal EEPROM location
; Input  : X_,y->address to erase
?EraseEE            ldx       X_,y
                    ldb       #$16                BYTE = 1, ERASE = 1, EELAT = 1
                    stb       PPROG
                    stb       ,x                  any data to this address will do
                    ldb       #$17                BYTE = 1, ERASE = 1, EELAT = 1, EPGM = 1
                    stb       PPROG               Turn on programming voltage
                    jsr       ?Delay10MS          Delay 10 msec
                    clr       PPROG               Turn off programming voltage
                    lda       ,x                  Read back the byte for check
                    cmpa      #[ERASED_STATE      is it erased ($FF)?
                    beq       ?EraseEE.Exit       if equal, get out, no errors
                    ldb       #errFailure
                    sec
                    rts
?EraseEE.Exit       clc
                    rts
$endif

$ifdef fBulkEraseEE
; Purpose: Bulk Erase EEPROM
?BulkEraseEE        ldb       #$06                EELAT = 1
                    stb       PPROG
                    sta       EEPROM              any EEPROM address will do
                    ldb       #$07                EELAT = 1 and EPGM = 1
                    stb       PPROG               Turn on programming voltage
                    jsr       ?Delay10MS          Delay 10 msec
                    clr       PPROG               Turn off programming voltage
                    ldx       #EEPROM             Check if all EEPROM is erased
                    ldd       #ERASED_STATE       Value for erased words
?BulkEraseEE.Check  cmpd      ,x                  Is word at ,X erased?
                    bne       ?BulkEraseEE.Error  No, get out with error
                    inx:2                         Yes, get ready for next word
                    cmpx      #EEPROM_END         Are we at the end of EEPROM?
                    blo       ?BulkEraseEE.Check  No, let's do it again
                    clc                           Yes, no errors encountered
                    rts
?BulkEraseEE.Error  ldb       #errFailure
                    sec
                    rts
$endif

$ifdef fBin2BCD
; Purpose: Convert a binary word value to BCD or other base
; Input  : D holds word value
;        : X points to output buffer
;        : Y holds length of buffer
;        : First byte of buffer holds conversion base
; Note(s): Although user's stack frame is used for temporary storage,
;        : it is left intact.
?Bin2BCD            equ       *
          ; save original stack frame
                    ldx       Y_,y                Save stack frame's Y
                    pshx
                    ldx       X_,y                Save stack frame's X (pointer to buffer)
                    pshx
          ; check conversion base for validity
                    clra                          Get conversion base in D
                    ldb       ,x
                    cmpb      #2                  Check base to be 2 to 36
                    blo       ?Bin2BCD.Error
                    cmpb      #36
                    bhi       ?Bin2BCD.Error
          ; make buffer pointer point to end of buffer (last character)
                    xgdx                          Now X holds base, D holds buffer pointer
                    addd      Y_,y                add length of buffer
                    decd                          minus 1 to point to end of buffer
                    std       X_,y                and save in temp space

                    lda       A_,y                Load D register with word
                    ldb       B_,y                ..to convert
                    pshx                          push needed for first pass of next loop

?Bin2BCD.01         pulx
                    pshx
                    idiv                          X should be holding base here
                    jsr       ToDigit

          ; save current digit to buffer and adjust buffer pointer backwards
                    pshx
                    ldx       X_,y                Point to current buffer character
                    stb       ,x                  Save remainder in output buffer
                    dex                           decrement pointer toward start of buffer
                    stx       X_,y                and save it
                    pulx

                    xgdx                          and use integer result as input for
                    dec       Y_+1,y              decrement [low byte] of Y counter
                    bne       ?Bin2BCD.01         2nd.last digit? if no, loop back
?Bin2BCD.Exit       pulx:2                        adjust stack and restore stack frame's X
                    stx       X_,y
                    pulx                          restore stack frame's Y
                    stx       Y_,y
                    clc
                    rts
?Bin2BCD.Error      pulx:2                        dummy PULLs to adjust stack
                    ldb       #errBadParm
                    sec
                    rts
$endif

$ifdef fSearchTable
; Purpose: Search a data table for some value, and return its index
;        : Table must be terminated with zero (byte or word) depending
;        : on target size
; Input  : B = number of bytes per entry (eg. 5)
;        : X points to beginning of table
;        : Y holds value to search for (if byte, lower 8-bits only)
;        : Carry Clear = Byte Search
;        : Carry Set   = Word Search
; Output : IF FOUND:     X points to matched table entry
;        :               B is unaffected
;        :               Carry Clear
;        : IF NOT FOUND: X is unaffected
;        :               B holds error code (errNotFound)
;        :               Carry Set
?SearchTable        ldb       B_,y                get size of entries in B
                    ldx       X_,y                get actual X value
                    brclr     CCR_,y,C.,?SearchTable.Byte go to the byte search part
?SearchTable.Loop   ldd       ,x                  get table value
                    beq       ?SearchTable.Fail   end-of-table, get out
                    cmpd      Y_,y                compare with word target
                    beq       ?SearchTable.Found  if equal, we found it!
                    ldb       B_,y                reload B with size of entries
                    abx                           point to next entry
                    bra       ?SearchTable.Loop   ..and try again
?SearchTable.Found  stx       X_,y                save return result
;                   clc                           not really needed after BEQ true
                    rts
?SearchTable.Fail   ldb       #errNotFound
                    sec
                    rts
?SearchTable.Byte   lda       ,x                  get table value
                    beq       ?SearchTable.Fail   end-of-table, get out
                    cmpa      Y_+1,y              compare with byte target
                    beq       ?SearchTable.Found  if equal, we found it!
                    abx                           point to next entry
                    bra       ?SearchTable.Byte   ..and try again
$endif

$ifdef fConvertAD
; Purpose: Convert an A/D reading (based on 5Volt Vrh-Vrl) to an ASCII
;        : string with the equivalent decimal number (10digits)
; Input  : X points to beginning of output buffer
;        : A holds A/D reading
; Output : Buffer pointed by X is filled with result
;        : Carry Clear
; Note(s): No registers are affected
?ConvertAD          ldx       X_,y                get actual X value
                    lda       A_,y                get actual A value
                    pshx
                    ldb       #9                  number of digits
                    pshb                          temporary byte for counter
                    tsy                           point to new stack frame
                    ldb       #5                  Scale to 5Volts
?Convert.Loop       mul
                    adda      #'0'
                    sta       1,x
                    inx
                    tba
                    ldb       #10
                    dec       ,y
                    bne       ?Convert.Loop
                    pulb                          kill temp
                    pulx
                    lda       1,x
                    sta       ,x
                    lda       #'.'
                    sta       1,x
                    clc                           never an error from here
                    rts
$endif

$ifdef fLength
; Routine: Length
; Purpose: Get the length of an ASCIIZ string
; Input  : X points to string
; Output : B holds length if Carry Clear, error "errOutOfRange" if Carry Set
; Note(s): Maximum string length is 255 characters
?Length             clrb
                    ldx       X_,y
?Length.Loop        tst       ,x
                    beq       ?Length.Exit
                    incb
                    beq       ?Length.Error
                    inx
                    bra       ?Length.Loop
?Length.Error       ldb       #errOutOfRange
                    sec
                    rts
?Length.Exit        stb       B_,y
                    clc
                    rts
$endif

$ifdef fInsert
; Routine: Insert
; Purpose: Insert a character within an ASCIIZ string
; Input  : X points to string
;        : A holds character
;        : B holds position before which to insert
; Output : None
?Insert             lda       A_,y
                    ldb       B_,y
                    beq       ?Insert.Exit        Nothing to do, get out
                    ldx       X_,y
                    decb                          Make zero-based
                    abx
                    lda       ,x                  get first character
?Insert.Loop        beq       ?Insert.End         are we done moving string up?
                    ldb       1,x                 get next character
                    sta       1,x
                    inx                           go upstring
                    tba                           move next char to A
                    bra       ?Insert.Loop        repeat
?Insert.End         sta       1,x                 save trailing zero
                    ldx       X_,y                now, add the character
                    lda       A_,y
                    ldb       B_,y
                    decb
                    abx
                    sta       ,x
?Insert.Exit        clc
                    rts
$endif

$ifdef fDelete
; Routine: Delete
; Purpose: Delete a character within an ASCIIZ string
; Input  : X points to string
;        : B holds position to delete
; Output : None
?Delete             ldb       B_,y
                    beq       ?Delete.Exit        Nothing to do, get out
                    ldx       X_,y
                    decb                          Make zero-based
                    abx
?Delete.Loop        tst       ,x                  check if at end-of-string
                    beq       ?Delete.Exit        are we done?
                    lda       1,x                 get next character
                    sta       ,x                  move down one place
                    inx                           go upstring
                    bra       ?Delete.Loop        repeat
?Delete.Exit        clc
                    rts
$endif

$ifdef fKickCOP
; Routine: KickCOP
; Purpose: Kick the COP timer (Call this to prevent COP timeout resets)
; Input  : None
; Output : None
; Note(s): Call either way: "OS fKickCOP" or "JSR KickCOP" (faster)
KickCOP             pshd                          save registers to be used
                    tpa                           keep copy of CCR
                    sei                           no IRQs during sequence
                    ldb       #$55
                    stb       COPRST
                    comb
                    stb       COPRST
                    tap                           restore CCR (re-enable IRQs?)
                    puld                          restore used registers
                    clc                           never an error from here
                    rts
$endif

$ifdef fHexToASCII
; Routine: HexToASCII
; Purpose: Convert a hex number in lower A to ASCII equivalent in D
; Input  : A holds binary number (higher nibble is ignored)
; Output : D holds ASCII equivalent of number
?HexToASCII         lda       A_,y
                    anda      #$0F                kill higher nibble just in case
                    daa                           convert to BCD
                    tab                           make a copy in B
                    lsra:4                        transfer A to lower half
                    andb      #$0F                clear upper half of B
                    addd      #'00'               finally, convert to ASCII
                    sta       A_,y                save result for caller
                    stb       B_,y
?HexToASCII.Exit    clc                           never an error from this routine
                    rts
$endif

$ifdef fNoLeadChar
; Routine: NoLeadChar
; Purpose: Convert leading RegB chars to RegA char
; Input  : A  Character to use for filling
;        : B  Character to be replaced
;        : X->Buffer area to change leading zeros to char in RegA
; Output : X->Buffer with leading RegB characters changed to RegA character
; Note(s): Terminates at first non-matching (RegB) character
?NoLeadChar         ldx       X_,y
                    lda       A_,y
                    ldb       B_,y
?NoLeadChar.Loop    cmpb      ,x
                    bne       ?NoLeadChar.Exit
                    sta       ,x                  replace with RegA character
                    inx
                    bra       ?NoLeadChar.Loop
?NoLeadChar.Exit    clc
                    rts
$endif

$ifdef fCRC
; Routine: CRC
; Purpose: Calculate the CRC of a buffer area
; Input  : X -> Pointer to data buffer
;        : Y -> Number of bytes in buffer
;        : A -> Method to use: 1=CRC-16, 2=CCITT
; Output : D -> 16-bit CRC value
; Note(s): Uses 1024 bytes of static table data.
; Stack  : Stack variables used (accessed by nnn,Y) after an TSY
;          Table     Table address              6,Y word
;          Length    Number of bytes to process 4,Y word
;          Address   Pointer to data buffer     2,Y word
;          CRC       Running CRC                0,Y word

?CRC                ldx       Y_,y                are there any bytes to process?
                    beq       ?CRC.Exit           no, get out

; --- setup table to use ---

                    ldx       #?CRC.Tab1          assume method 1
                    cmpa      #1                  is it method 1?
                    beq       ?CRC.Cont
                    ldx       #?CRC.Tab2          assume method 2
                    cmpa      #2                  is it method 2?
                    beq       ?CRC.Cont

?CRC.Error          ldb       #errOutOfRange      invalid method specified, exit with error
                    sec
                    rts

; --- create local variables ---

?CRC.Cont           pshx                          save table address to use
                    ldx       Y_,y                Get caller's Y
                    pshx                          Save counter on stack at offset 4
                    ldx       X_,y                Get caller's X
                    pshx                          Save pointer on stack at offset 2
                    clrx                          Start with a zero initial CRC in stack
                    pshx                          ..put it on stack at offset 0
                    tsy                           use Y for stack access

* add byte loop
?CRC.Loop           ldx       2,y                 get next byte pointer
                    ldb       ,x                  get next byte to be included in CRC
                    inx
                    stx       2,y                 save pointer to next byte
                    eorb      1,y
                    ldx       6,y                 point to CRC Table
                    abx:2                         get index into CRC table (add twice the byte value)
                    lda       0,X                 get upper CRC byte
                    ldb       ,y                  get old upper CRC byte
                    eorb      1,X                 XOR with lower table byte
                    std       ,y                  save new CRC
                    ldx       4,y                 get counter
                    dex
                    stx       4,y                 save decremented counter
                    bne       ?CRC.Loop           --> next byte
                    pulx:4                        De-allocate local storage (4 words)
?CRC.Exit           xgab                          compensate for wrong stacked D order
                    tsy                           we must reload Y accounting for push
                    std       D_+4,y              ..and return address to the SWI handler
                    clc
                    rts

* Table for polynomial = $14003 (CRC-16)
?CRC.Tab1           fdb       $0000,$C0C1,$C181,$0140,$C301,$03C0,$0280,$C241
                    fdb       $C601,$06C0,$0780,$C741,$0500,$C5C1,$C481,$0440
                    fdb       $CC01,$0CC0,$0D80,$CD41,$0F00,$CFC1,$CE81,$0E40
                    fdb       $0A00,$CAC1,$CB81,$0B40,$C901,$09C0,$0880,$C841
                    fdb       $D801,$18C0,$1980,$D941,$1B00,$DBC1,$DA81,$1A40
                    fdb       $1E00,$DEC1,$DF81,$1F40,$DD01,$1DC0,$1C80,$DC41
                    fdb       $1400,$D4C1,$D581,$1540,$D701,$17C0,$1680,$D641
                    fdb       $D201,$12C0,$1380,$D341,$1100,$D1C1,$D081,$1040
                    fdb       $F001,$30C0,$3180,$F141,$3300,$F3C1,$F281,$3240
                    fdb       $3600,$F6C1,$F781,$3740,$F501,$35C0,$3480,$F441
                    fdb       $3C00,$FCC1,$FD81,$3D40,$FF01,$3FC0,$3E80,$FE41
                    fdb       $FA01,$3AC0,$3B80,$FB41,$3900,$F9C1,$F881,$3840
                    fdb       $2800,$E8C1,$E981,$2940,$EB01,$2BC0,$2A80,$EA41
                    fdb       $EE01,$2EC0,$2F80,$EF41,$2D00,$EDC1,$EC81,$2C40
                    fdb       $E401,$24C0,$2580,$E541,$2700,$E7C1,$E681,$2640
                    fdb       $2200,$E2C1,$E381,$2340,$E101,$21C0,$2080,$E041
                    fdb       $A001,$60C0,$6180,$A141,$6300,$A3C1,$A281,$6240
                    fdb       $6600,$A6C1,$A781,$6740,$A501,$65C0,$6480,$A441
                    fdb       $6C00,$ACC1,$AD81,$6D40,$AF01,$6FC0,$6E80,$AE41
                    fdb       $AA01,$6AC0,$6B80,$AB41,$6900,$A9C1,$A881,$6840
                    fdb       $7800,$B8C1,$B981,$7940,$BB01,$7BC0,$7A80,$BA41
                    fdb       $BE01,$7EC0,$7F80,$BF41,$7D00,$BDC1,$BC81,$7C40
                    fdb       $B401,$74C0,$7580,$B541,$7700,$B7C1,$B681,$7640
                    fdb       $7200,$B2C1,$B381,$7340,$B101,$71C0,$7080,$B041
                    fdb       $5000,$90C1,$9181,$5140,$9301,$53C0,$5280,$9241
                    fdb       $9601,$56C0,$5780,$9741,$5500,$95C1,$9481,$5440
                    fdb       $9C01,$5CC0,$5D80,$9D41,$5F00,$9FC1,$9E81,$5E40
                    fdb       $5A00,$9AC1,$9B81,$5B40,$9901,$59C0,$5880,$9841
                    fdb       $8801,$48C0,$4980,$8941,$4B00,$8BC1,$8A81,$4A40
                    fdb       $4E00,$8EC1,$8F81,$4F40,$8D01,$4DC0,$4C80,$8C41
                    fdb       $4400,$84C1,$8581,$4540,$8701,$47C0,$4680,$8641
                    fdb       $8201,$42C0,$4380,$8341,$4100,$81C1,$8081,$4040

* Table for polynomial = $10811 (CCITT)
?CRC.Tab2           fdb       $0000,$1189,$2312,$329B,$4624,$57AD,$6536,$74BF
                    fdb       $8C48,$9DC1,$AF5A,$BED3,$CA6C,$DBE5,$E97E,$F8F7
                    fdb       $1081,$0108,$3393,$221A,$56A5,$472C,$75B7,$643E
                    fdb       $9CC9,$8D40,$BFDB,$AE52,$DAED,$CB64,$F9FF,$E876
                    fdb       $2102,$308B,$0210,$1399,$6726,$76AF,$4434,$55BD
                    fdb       $AD4A,$BCC3,$8E58,$9FD1,$EB6E,$FAE7,$C87C,$D9F5
                    fdb       $3183,$200A,$1291,$0318,$77A7,$662E,$54B5,$453C
                    fdb       $BDCB,$AC42,$9ED9,$8F50,$FBEF,$EA66,$D8FD,$C974
                    fdb       $4204,$538D,$6116,$709F,$0420,$15A9,$2732,$36BB
                    fdb       $CE4C,$DFC5,$ED5E,$FCD7,$8868,$99E1,$AB7A,$BAF3
                    fdb       $5285,$430C,$7197,$601E,$14A1,$0528,$37B3,$263A
                    fdb       $DECD,$CF44,$FDDF,$EC56,$98E9,$8960,$BBFB,$AA72
                    fdb       $6306,$728F,$4014,$519D,$2522,$34AB,$0630,$17B9
                    fdb       $EF4E,$FEC7,$CC5C,$DDD5,$A96A,$B8E3,$8A78,$9BF1
                    fdb       $7387,$620E,$5095,$411C,$35A3,$242A,$16B1,$0738
                    fdb       $FFCF,$EE46,$DCDD,$CD54,$B9EB,$A862,$9AF9,$8B70
                    fdb       $8408,$9581,$A71A,$B693,$C22C,$D3A5,$E13E,$F0B7
                    fdb       $0840,$19C9,$2B52,$3ADB,$4E64,$5FED,$6D76,$7CFF
                    fdb       $9489,$8500,$B79B,$A612,$D2AD,$C324,$F1BF,$E036
                    fdb       $18C1,$0948,$3BD3,$2A5A,$5EE5,$4F6C,$7DF7,$6C7E
                    fdb       $A50A,$B483,$8618,$9791,$E32E,$F2A7,$C03C,$D1B5
                    fdb       $2942,$38CB,$0A50,$1BD9,$6F66,$7EEF,$4C74,$5DFD
                    fdb       $B58B,$A402,$9699,$8710,$F3AF,$E226,$D0BD,$C134
                    fdb       $39C3,$284A,$1AD1,$0B58,$7FE7,$6E6E,$5CF5,$4D7C
                    fdb       $C60C,$D785,$E51E,$F497,$8028,$91A1,$A33A,$B2B3
                    fdb       $4A44,$5BCD,$6956,$78DF,$0C60,$1DE9,$2F72,$3EFB
                    fdb       $D68D,$C704,$F59F,$E416,$90A9,$8120,$B3BB,$A232
                    fdb       $5AC5,$4B4C,$79D7,$685E,$1CE1,$0D68,$3FF3,$2E7A
                    fdb       $E70E,$F687,$C41C,$D595,$A12A,$B0A3,$8238,$93B1
                    fdb       $6B46,$7ACF,$4854,$59DD,$2D62,$3CEB,$0E70,$1FF9
                    fdb       $F78F,$E606,$D49D,$C514,$B1AB,$A022,$92B9,$8330
                    fdb       $7BC7,$6A4E,$58D5,$495C,$3DE3,$2C6A,$1EF1,$0F78
$endif

$ifdef fLoadS19
; Routine: LoadS19
; Purpose: Load an S19 file through the SCI port
; Input  : X -> Load Offset for S19 addresses (X holds two's complement offset)
; Output : X -> Execution address from S9 record (possibly zero)
; Stack  : Stack variables used (accessed by nnn,X) after an TSX
;          Load Offset Offset to adjust by      6,X  word
;          Temp        Temporary usage          5,X  byte
;          RecType     S-record type field      4,X  byte
;          Length      S-record length field    3,X  byte
;          Address     S-record address field   1,X  word
;          CRC         S-record running CRC     0,X  byte
?LoadS19            ldx       X_,y
                    pshx                          Save two's complement offset
                    clrx
                    pshx:3                        Reserve three words for variables (init to 0)
                    tsx                           Get base for variables
?LoadS19.New        jsr       ?LoadS19.ReadByte   Get first character
                    bcs       ?LoadS19.OK         if EOL, get out
                    cmpa      #'S'                Probable S record
                    bne       ?LoadS19.EOL        No, ignore rest of line
                    jsr       ?LoadS19.ReadByte   Get next character
                    bcs       ?LoadS19.Error      if EOL, get out with error
                    cmpa      #'0'                Header record?
                    bne       ?LoadS19.S9
                    bsr       ?LoadS19.ToEOL      Yes, ignore rest of line
                    bra       ?LoadS19.New        go read another line
?LoadS19.S9         cmpa      #'9'                Terminator record
                    beq       ?LoadS19.OK
                    cmpa      #'1'                Code/data record
                    beq       ?LoadS19.OK
                    bra       ?LoadS19.EOL        Not a valid record type, ignore line
?LoadS19.OK         sta       4,x                 Save this type

          ;Get length of Record Bytes (including 16-bit address and 8-bit CRC)
                    jsr       ?LoadS19.ReadHex    Get next 2 characters in binary
                    bcs       ?LoadS19.Error      if something wrong, get out with error
                    bsr       ?LoadS19.UpdateCRC
                    suba      #3                  adjust for 2-byte address and 1-byte CRC
                    sta       3,X                 save Length of record

          ;Now, get the load address
                    jsr       ?LoadS19.ReadHex
                    bcs       ?LoadS19.Error
                    bsr       ?LoadS19.UpdateCRC
                    sta       1,X                 Save MSB
                    jsr       ?LoadS19.ReadHex
                    bcs       ?LoadS19.Error
                    bsr       ?LoadS19.UpdateCRC
                    sta       2,X                 Save LSB
                    ldd       6,X                 Get offset
                    addd      1,X                 Add actual address
                    std       1,X                 And re-save actual address

          ;Now, get the code/data bytes
                    tst       3,X                 Check Length of zero
?LoadS19.Loop       beq       ?LoadS19.CRC        Empty code/data section of record
                    bsr       ?LoadS19.ReadHex
                    bcs       ?LoadS19.Error      if something wrong, get out with error
                    bsr       ?LoadS19.UpdateCRC
                    bsr       ?LoadS19.StoreByte  save byte and advance pointer
                    dec       3,X                 One less byte to read
                    bra       ?LoadS19.Loop

?LoadS19.CRC        bsr       ?LoadS19.ReadHex    Get CRC byte
                    bcs       ?LoadS19.Error      if something wrong, get out with error
                    com       0,X                 Get one's complement of final CRC value
                    cmpa      0,X                 Is it the same as the one calculated
                    bne       ?LoadS19.Error      No, exit with error

          ;See if we're done
                    lda       4,X                 Check record type
                    cmpa      #'9'
                    beq       ?LoadS19.Exit       Done, get out without errors
                    bsr       ?LoadS19.ToEOL
                    pulx:4
                    bra       ?LoadS19            Go back to read another line

?LoadS19.Exit       ldx       1,X                 Load X with address
                    stx       X_,y                for use by OS11
                    bsr       ?LoadS19.ToEOL
                    pulx:4
                    clc
                    rts
?LoadS19.Error      bsr       ?LoadS19.ToEOL      1.01 addition (exit at end-of-line)
                    pulx:4
                    ldb       #errFailure         for use by OS11
                    sec
                    rts
?LoadS19.EOL        bsr       ?LoadS19.ToEOL
                    bra       ?LoadS19.Error      Always out from the error exit

?LoadS19.ToEOL      cmpa      #CR
                    beq       ?LoadS19.ToEOL.OK
                    cmpa      #LF
                    beq       ?LoadS19.ToEOL.OK
                    bsr       ?LoadS19.ReadByte
                    bra       ?LoadS19.ToEOL
?LoadS19.ToEOL.OK   rts

?LoadS19.UpdateCRC  psha                          Adjust the CRC for the record
                    adda      0,X
                    sta       0,X
                    pula
                    rts

?LoadS19.StoreByte  pshx                          Save RegA to current [PC] address
                    ldx       1,X                 Get address in X
                    sta       ,X
                    pulx
                    pshd                          Adjust the PC value by 1
                    ldd       1,X
                    incd
                    std       1,X
                    puld
                    rts

          ;Sets Carry of end-of-line character found
?LoadS19.ReadByte   pshx
                    ldx       #REGS
                    brclr     [SCSR,X,#$20,*      Wait for a character
                    lda       SCDR                Read character
                    cmpa      #CR
                    bne       ?LoadS19.ReadByte0
                    cmpa      #LF
                    bne       ?LoadS19.ReadByte0
                    sec
                    bra       ?LoadS19.ReadByte1
?LoadS19.ReadByte0  clc
?LoadS19.ReadByte1  pulx
                    rts

?LoadS19.ReadHex    bsr       ?LoadS19.ReadByte   Get next character
                    bcs       ?LoadS19.ReadHex1   if EOL, get out with error
                    bsr       ?HexToBin.Local     Convert from Hex to Binary
                    bcs       ?LoadS19.ReadHex1   if EOL, get out with error
                    lsla:4                        Move to high nibble
                    sta       5,X                 and save in temporary location
                    bsr       ?LoadS19.ReadByte   Get next 2 characters
                    bcs       ?LoadS19.ReadHex1   if EOL, get out with error
                    bsr       ?HexToBin.Local     Convert from Hex to Binary
                    bcs       ?LoadS19.ReadHex1   Invalid character, ignore rest of line
                    ora       5,X                 combine LSN with MSN
                    sta       5,X                 and save it back (optional)
                    clc                           Error-free exit
                    rts
?LoadS19.ReadHex1   ldb       #errFailure
                    sec                           Error exit
                    rts
$endif

$ifdef fHexToBin
; Routine: HexToBin
; Purpose: Convert a hex digit (0-F) in RegA to its binary equivalent
; Input  : A holds binary number
; Output : A holds ASCII equivalent
?HexToBin           lda       A_,y
                    bsr       ?HexToBin.Local
                    sta       A_,y
                    clc
                    rts
$endif

$ifdef fGarble
; Routine: Garble
; Purpose: Garble buffer at X with seed in RegA
; Input  : X->buffer
;        : A=8-bit seed to use for garbling
;        : B=Length of buffer
; Output : Buffer at X is garbled with seed in RegA
?Garble             ldx       X_,y
                    ldb       B_,y
?Garble.Loop        beq       ?Garble.Exit        no more bytes to process
                    lda       ,x                  get character
                    eora      A_,y                XOR with user seed
                    sta       ,x                  put character back in buffer
                    inx                           point to next character
                    decb
                    bra       ?Garble.Loop
?Garble.Exit        clc
                    rts
$endif

;
; ---------------- add more OS routines here -------------------------
;

$PAGE
*********************************************************************
*                     GENERAL-PURPOSE ROUTINES                      *
*********************************************************************

; Routine: Upcase
; Purpose: Convert character in A to uppercase
; Input  : A=s
; Output : A=S
Upcase              cmpa      #'a'                 less than 'A'?
                    blo       ?Upcase.Exit         yes, skip
                    cmpa      #'z'                 greater than 'Z'?
                    bhi       ?Upcase.Exit         yes, skip
                    suba      #'a'-'A'             do the conversion
?Upcase.Exit        rts

; Purpose: Convert a binary number to ASCII equivalent
; Input  : B holds binary number
; Output : B holds ASCII equivalent
ToDigit             addb      #'0'                convert to ASCII
                    cmpb      #'9'
                    bls       ?ToDigit.Exit
                    addb      #'A'-'0'-10         adjust for appropriate letter
?ToDigit.Exit       clc
                    rts

; Purpose: Convert a hex digit (0-F) in RegA to its binary equivalent
; Input  : A holds binary number
; Output : A holds ASCII equivalent
?HexToBin.Local     cmpa      #'0'
                    blo       ?HexToBin.Error
                    cmpa      #'9'
                    bhi       ?HexToBin.Letter
                    suba      #'0'
                    bra       ?HexToBin.OK
?HexToBin.Letter    cmpa      #'A'
                    blo       ?HexToBin.Error
                    cmpa      #'F'
                    bhi       ?HexToBin.Error
                    suba      #'A'-10
?HexToBin.OK        clc                           Error-free exit
                    rts
?HexToBin.Error     sec                           Error exit
                    rts

?Delay50            pshx
                    ldx       #50
                    os        fDelayMS
                    pulx
                    rts

$PAGE
$ifdef LCD
                    $RAM
?LCD.Addr           rmb       1
?LCD.Dir            rmb       1

                    $ROM
*********************************************************************
*                         LCD EQUATES                               *
*********************************************************************
?LCD_E              equ       Bit1.               for modified EVBU
?LCD_RW             equ       Bit2.               for modified EVBU
?LCD_RS             equ       Bit0.               for modified EVBU
?LCD_BUSY           equ       Bit7.               for modified EVBU

?LCD_CTRL           equ       PORTB               for modified EVBU
?LCD_Data           equ       PORTC               for modified EVBU
?LCD_DDR            equ       DDRC                for modified EVBU

*********************************************************************
*                        LCD COMMANDS                               *
*********************************************************************
?cCLS               equ       %00000001           Clear LCD display
?cHOME              equ       %00000010           Home the cursor
?cDECNSHF           equ       %00000100           Decrement cursor, no display shift
?cDECSHF            equ       %00000101           Decrement cursor, display shift
?cINCNSHF           equ       %00000110           Increment cursor, no display shift
?cINCSHF            equ       %00000111           Increment cursor, display shift
?cND                equ       %00001000           Display OFF
?cDNCNB             equ       %00001100           Display ON, no cursor, no blink
?cDNCB              equ       %00001101           Display ON, no cursor, blink
?cDCNB              equ       %00001110           Display ON, cursor on, no blink
?cDCB               equ       %00001111           Display ON, cursor on, blink
?cSHIFTL            equ       %00011000           Display Shift Left
?cSHIFTR            equ       %00011100           Display Shift Right
?cMOVEL             equ       %00010000           Cursor Move Left
?cMOVER             equ       %00010100           Cursor Move Right

*********************************************************************
*                     LCD-RELATED ROUTINES                          *
*********************************************************************

$ifdef fLCDCls
*****
* CLS - Clear the LCD display
*****
?LCD.Cls            lda       #?cCLS
                    bra       ?WCommLCD
$endif

$ifdef fLCDHome
*****
* Home - Home the cursor
*****
?LCD.Home           lda       #?cHOME
                    bra       ?WCommLCD
$endif

$ifdef fLCDPutChar
*****
* Input: RegA holds ASCII character to display
* NOTES: Character is displayed and cursor advanced
*****
?LCD.PutChar        lda       A_,y
                    jsr       ?ASCIILCD           convert standard ASCII to LCD codes
                    bra       ?WDataLCD           write it out
$endif

$ifdef fLCDInit
*****
* INPUT : Nothing
* OUTPUT: Nothing
* NOTES : LCD is initialized
*****
?LCD.Init           lda       #$FF                set LCD port for writing
                    sta       ?LCD_DDR
                    bsr       ?Delay50            wait for 50ms (45ms only required)
                    lda       #%00111000          set 8-bit data, 1/16 duty, 5x7 dots
                    bsr       ?CommandModeLCD
                    bsr       ?WriteLCD
                    bsr       ?Delay50            wait for 50ms (4.1ms only required)
;do it once again in case it was originally set at 4-bit bus
                    bsr       ?WriteLCD
                    bsr       ?Delay50            wait for 50ms (not needed, just do)
                    bsr       ?WriteLCD
                    jsr       ?WaitBusy
                    bsr       ?WCommLCD
                    bsr       ?LCD.Cls
                    bsr       ?DispOn
                    jsr       ?DirRght
                    bra       ?LoadGR             Load the Greek Character set
$endif

$ifdef fLCDon
*****
* Turn the LCD display on, no cursor
*****
?DispOn             lda       #?cDNCNB
                    bra       ?WCommLCD
$endif

*****
* Write the LCD command (NO REGISTERS ARE DESTROYED) A holds instruction
*****
?WCommLCD           jsr       ?WaitBusy
                    bsr       ?CommandModeLCD
                    bsr       ?WriteLCD
                    jmp       ?WaitBusy           get cursor address

*****
* Write LCD Data and toggle Write enable on/off
*****
?WriteLCD           pshx
                    ldx       #REGS
                    bset      [?LCD_CTRL,x,?LCD_E enable Write operation
                    sta       ?LCD_Data           put command data on bus
                    bclr      [?LCD_CTRL,x,?LCD_E disable Write operation
                    pulx
                    rts

*****
* Write the LCD data (NO REGISTERS ARE DESTROYED) A holds data
*****
?WDataLCD           jsr       ?WaitBusy
                    bsr       ?DataModeLCD
                    bsr       ?WriteLCD
                    jmp       ?WaitBusy           get cursor address

?DataModeLCD        pshx
                    ldx       #REGS
                    bset      [?LCD_CTRL,x,?LCD_RS --- select data
                    bclr      [?LCD_CTRL,x,?LCD_RW --- write mode
                    pulx
                    rts

?CommandModeLCD     pshx
                    ldx       #REGS
                    bclr      [?LCD_CTRL,x,?LCD_RS+?LCD_RW select command write mode
                    pulx
                    rts

*****
* SetCGRAM - Set CG RAM address (A holds address 0-255)
*****
?SetCGRAM           anda      #%00111111          mask off non-address bits
                    ora       #%01000000          mix with SET CGRAM command
                    bra       ?WCommLCD

*****
* FixAddr - Fix LCD DDRAM address format to SetDDRAM address format
*           A holds LCD format address on input, SetDDRAM format on output
*****
?FixAddr            cmpa      #$40                is it line 2?
                    blo       ?FixAddr.Line1      address is in correct format
                    suba      #$40                convert to column without line info
                    ora       #$80                add line 2 info
?FixAddr.Line1      rts

*****
* LoadGR - Load the Greek character set into unused LCD characters
*****
?LoadGR             clra                          beginning address to write
                    bsr       ?SetCGRAM
                    ldx       #?GreekTab          start at beginning of data table
?LoadGR1            lda       ,x
                    jsr       ?WaitBusy
                    bsr       ?DataModeLCD
                    anda      #%00011111          mask off unused bits just in case
                    bsr       ?WriteLCD
                    inx                           point to next sequence or code
                    cpx       #?GreekTabEnd       have we loaded all characters?
                    blo       ?LoadGR1
                    clra                          get away from CG RAM and into DD RAM
                    bra       ?SetDDRAM.Local
?GreekTab           equ       *                   Table of Greek 5x7(8) dot characters
* PI
                    fcb       %00011111           *****
                    fcb       %00001010            * *
                    fcb       %00001010            * *
                    fcb       %00001010            * *
                    fcb       %00001010            * *
                    fcb       %00001010            * *
                    fcb       %00001010            * *
                    fcb       %00000000           
* OMEGA
                    fcb       %00001110            ***
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00001010            * *
                    fcb       %00001010            * *
                    fcb       %00011011           ** **
                    fcb       %00000000           
* KSI
                    fcb       %00011111           *****
                    fcb       %00000000
                    fcb       %00000000
                    fcb       %00001110            ***
                    fcb       %00000000
                    fcb       %00000000
                    fcb       %00011111           *****
                    fcb       %00000000           
* GAMMA
                    fcb       %00011111           *****
                    fcb       %00010000           *
                    fcb       %00010000           *
                    fcb       %00010000           *
                    fcb       %00010000           *
                    fcb       %00010000           *
                    fcb       %00010000           *
                    fcb       %00000000           
* DELTA
                    fcb       %00000100             *
                    fcb       %00001010            * *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00011111           *****
                    fcb       %00000000           
* LAMDA
                    fcb       %00000100             *
                    fcb       %00001010            * *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00010001           *   *
                    fcb       %00000000           
* PHI
                    fcb       %00000100             *
                    fcb       %00001110            ***
                    fcb       %00010101           * * *
                    fcb       %00010101           * * *
                    fcb       %00001110            ***
                    fcb       %00000100             *
                    fcb       %00000100             *
                    fcb       %00000000           
* PSI
                    fcb       %00010101           * * *
                    fcb       %00010101           * * *
                    fcb       %00010101           * * *
                    fcb       %00001110            ***
                    fcb       %00000100             *
                    fcb       %00000100             *
                    fcb       %00000100             *
                    fcb       %00000000           
?GreekTabEnd        equ       *                   End of Table of Greek

$ifdef fSetAddr
*****
* INPUT : Param (or A) holds address as follows:
*         Row (either 1 or 2) is in bit 7 as 0 or 1 respectively
*         Column (0 through 39) is in bits 6-0
* OUTPUT: SetDDRAM instruction is executed for the given address
*****
?SetDDRAM           lda       A_,y
?SetDDRAM.Local     bita      #$80                is address given for row 2?
                    beq       ?SetDRAM2           no, skip row 2 adjustment
?SetDRAM1           adda      #$40                yes, adjust column address for row 2
?SetDRAM2           ora       #$80                mix address with SET DDRAM command
                    jmp       ?WCommLCD
$endif

*****
* Convert standard ASCII to LCD, actually only DOS-Greek codes are affected
*****
?ASCIILCD           cmpa      #' '                is it below space
                    bhi       ?ASCII.1
                    lda       #' '                yes, convert to space
                    bra       ?ASCII.Exit         and get out
?ASCII.1            cmpa      #''                is it Greek Alpha?
                    blo       ?ASCII.Exit         no, no more checks
                    cmpa      #''                is it higher than Greek Omega?
                    bhi       ?ASCII.Exit         yes, no more checks
                    suba      #''                Less Greek Alpha for zero-based
                    tab
                    ldx       #?ASCII.Table
                    abx
                    lda       ,x                  get corresponding character
?ASCII.Exit         rts
?ASCII.Table        fcb       'AB',3,4,'EZH',$F2,'IK',5,'MN',2,'O',0,'P',$F6,'TY',6,'X',7,1

$ifdef fClrEOL
*****
* fClrEOL - Clear to end of line
*****
?ClrEOL             bsr       ?WaitBusy           get the current DDRAM address in LCD.Addr
                    lda       ?LCD.Addr           save current DDRAM address in A
                    psha
                    cmpa      #$40                is it line 1 or 2?
                    blo       ?ClrEOL.Line1       skip subtraction
                    suba      #$40                get column number regardless of line
?ClrEOL.Line1       nega                          column = -column
                    adda      #40                 A = line length - column
                    tab                           Move counter to B
                    lda       #' '                get a space character in A
?ClrEOL.Loop        jsr       ?WDataLCD
                    decb
                    bne       ?ClrEOL.Loop
                    pula
                    jsr       ?FixAddr            convert to proper format for SetDDRAM
                    bra       ?SetDDRAM.Local
$endif

*****
* DirLeft - Set write direction towards left (eg. numbers)
* DirRght - Set write direction towards right (normal)
*****
$ifdef fDirLeft
?DirLeft            lda       #?cDECNSHF
                    bra       ?SetDirection
$endif

$ifdef fDirRght
?DirRght            lda       #?cINCNSHF
$endif

?SetDirection       sta       ?LCD.Dir
                    jmp       ?WCommLCD

*****
* Wait while LCD is busy. On exit LCD.Addr holds address read from LCD
*****
?WaitBusy           psha
                    pshx
                    ldx       #REGS
                    clr       [?LCD_DDR,x         change Data register to inputs
                    bclr      [?LCD_CTRL,x,?LCD_RS configure check busy flag/address mode
                    bset      [?LCD_CTRL,x,?LCD_RW+?LCD_E enable Read operation
                    brset     [?LCD_Data,x,?LCD_BUSY,* wait for busy flag to reset
                    lda       ?LCD_Data           get the Address
                    bclr      [?LCD_CTRL,x,?LCD_E+?LCD_RW disable Read operation and change to instruction mode
                    anda      #$7F                mask off busy flag
                    sta       ?LCD.Addr           save address read
                    com       [?LCD_DDR,x         set Data register to outputs
                    pulx
                    pula
                    rts

****
* Clear the top LCD line
****
?ClearLine.1        psha
                    clra                          point to line 1
                    jsr       ?SetDDRAM.Local
                    os        fClrEOL
                    pula
                    rts

****
* Clear the bottom LCD line
****
?ClearLine.2        psha
                    lda       #$40                point to line 2
                    jsr       ?SetDDRAM.Local
                    os        fClrEOL
                    pula
                    rts

          * Print message pointed by X on LCD line 2
LCDPrintLine2       psha
                    lda       #$40                point to line 2
                    jsr       ?SetDDRAM.Local
                    bra       ?PrintIt.Loop
          * Print message pointed by X on LCD line 1
LCDPrintLine1       clra                          point to line 1
                    jsr       ?SetDDRAM.Local
          * Print message pointed by X on current line and cursor position
LCDPrint            psha
?PrintIt.Loop       lda       ,x
                    beq       ?PrintIt.Exit
                    os        fLCDPutChar
                    inx
                    bra       ?PrintIt.Loop
?PrintIt.Exit       pula
                    os        fClrEOL             and clear to the end of line
                    rts
$endif

$PAGE
*********************************************************************
*                   M I S C E L L A N E O U S                       *
*********************************************************************

Halt_MCU            equ       *                   Globally accessed
                    ldx       #10
$ifdef ?OS_16BIT
                    osw       fDelayMS            Let pending hardware events complete
$else
                    os        fDelayMS            Let pending hardware events complete
$endif
                    sei                           Disable interrupts
; --- Here you could add code to shut down all peripherals (on and off chip)
; --- such as the A/D, SCI, SPI, external chips, etc.
                    clr       OPTION              Power-down A/D
                    clr       SCCR2               Disable SCI
                    clr       SPCR                Disable SPI
                    clr       TMSK1               Disable IC/OC interrupts
                    clr       TMSK2               Disable Timer interrupts

                    os        fDelayMS            Let subsystems turn off

                    cls                           Enable STOP instruction
$ifdef DEBUG
                    nop                           Instead of STOP (for simulator)
$else
                    stop                          and stay here in deep sleep
$endif
                    bra       *-1                 in case of XIRQ, go back to sleep