/*
 * MultiMail offline mail reader
 * Blue Wave class

 Copyright (c) 1996 Toth Istvan <stoty@vma.bme.hu>
 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "bw.h"
#include "compress.h"
#include "../interfac/mysystem.h"

#include <sys/utsname.h>

// ---------------------------------------------------------------------------
// The Blue Wave methods
// ---------------------------------------------------------------------------

bluewave::bluewave(mmail *mmA)
{
	mm = mmA;
	init();
}

void bluewave::init()
{
	ID = 0;
	bodyString = NULL;

	findInfBaseName();
	infFile = openFile("INF");

	initInf();
	readMixRec();
	initMixID();

	fclose(infFile);

	body = new struct bodytype *[maxConf];

	ftiFile = openFile("FTI");
	datFile = openFile("DAT");
}

bluewave::~bluewave()
{
	cleanup();
}

void bluewave::cleanup()
{
	while (maxConf)
		delete body[--maxConf];
	delete body;
	delete mixID;
	delete areas;
	delete mixRecord;
	delete bodyString;

	fclose(ftiFile);
	fclose(datFile);
}

int bluewave::getNoOfAreas()
{
	return maxConf;
}

void bluewave::resetAll()
{
	cleanup();
	init();
}

int bluewave::numcmp(const char *first, const char *second)
{
	for (int c = 0; c < 6; c++)
		if (toupper(first[c]) != toupper(second[c]))
			return 1;
		else
			if (!first[c] && !second[c])
				break;
	return 0;
}

area_header *bluewave::getNextArea()
{
	area_header *tmp;
	int x, totmsgs, numpers, flags_raw, flags_cooked;

	x = mixID[ID];

	if (x != -1) {
		totmsgs = getshort(mixRecord[x].totmsgs);
		numpers = getshort(mixRecord[x].numpers);
	} else
		totmsgs = numpers = 0;

	body[ID] = new struct bodytype[totmsgs];

	flags_raw = getshort(areas[ID].area_flags);
	flags_cooked = ((x != -1) ? ACTIVE : 0) |
		((flags_raw & INF_ALIAS_NAME) ? ALIAS : 0) |
		((flags_raw & INF_NETMAIL) ? NETMAIL : 0) |
		((areas[ID].network_type == INF_NET_INTERNET) ? INTERNET : 0);

	tmp = new area_header(mm, ID + mm->driverList->getOffset(this),
		(char *) areas[ID].areanum, (char *) areas[ID].echotag,
		(char *) areas[ID].title, "Blue Wave", flags_cooked,
		totmsgs, numpers);
	ID++;
	return tmp;
}

void bluewave::selectArea(int area)
{
	currentArea = area;
	resetLetters();
}

void bluewave::resetLetters()
{
	currentLetter = 0;
}

int bluewave::getNoOfLetters()
{
	int x = mixID[currentArea];

	return ((x != -1) ? getshort(mixRecord[x].totmsgs) : 0);
}

letter_header *bluewave::getNextLetter()
{
	FTI_REC ftiRec;
	net_address na;

	if (currentLetter >= (int) getshort(
		mixRecord[mixID[currentArea]].totmsgs))
			return NULL;

	fseek(ftiFile, getlong(mixRecord[mixID[currentArea]].msghptr) +
		currentLetter * ftiStructLen, SEEK_SET);

	if (!(fread(&ftiRec, sizeof(FTI_REC), 1, ftiFile)))
		fatalError("Error reading .FTI file");

	long msglength = getlong(ftiRec.msglength);

	body[currentArea][currentLetter].pointer = getlong(ftiRec.msgptr);
	body[currentArea][currentLetter].msgLength = msglength;

	cropesp((char *) ftiRec.from);
	cropesp((char *) ftiRec.to);
	cropesp((char *) ftiRec.subject);

	int flags = getshort(ftiRec.flags);

	na.zone = getshort(ftiRec.orig_zone);
	na.net = getshort(ftiRec.orig_net);
	na.node = getshort(ftiRec.orig_node);
	na.point = 0;	// :-(

	return new letter_header(mm, stripre((char *) ftiRec.subject),
		(char *) ftiRec.to, (char *) ftiRec.from, (char *) ftiRec.date,
		getshort(ftiRec.replyto), currentLetter++,
		getshort(ftiRec.msgnum), currentArea,
		(flags & MSG_NET_PRIVATE), msglength, this, &na);
}

// returns the body of the requested letter in the active area
char *bluewave::getBody(int AreaID, int LetterID)
{
	unsigned char *p;
	int c, kar, lfig = 0;		// ^A flag

	delete bodyString;
	bodyString = new char[body[AreaID][LetterID].msgLength + 1];
	fseek(datFile, body[AreaID][LetterID].pointer, SEEK_SET);

	for (c = 0, p = (unsigned char *) bodyString;
	     c < body[AreaID][LetterID].msgLength; c++) {
		kar = fgetc(datFile);

		// Substitute for the code that Renewave crashes:
		if (!c && (kar == ' '))
			kar = fgetc(datFile);

		if (!kar)
			kar = ' ';

		// Ctrl-A line filtering

		// More should be done with the hidden lines (they shouldn't
		// just be discarded like this).

		if (kar == '\r') {
			*p++ = '\n';
			if (lfig)
				p--;
			lfig = 0;
		} else
			if (kar == 1)
				lfig = 1;
			else
				if (!lfig && (kar != '\n'))
					*p++ = kar;
	}
	*p = '\0';

	return bodyString;
}

void bluewave::findInfBaseName()
{
	mm->workList->gotoFile(0);
	const char *q = mm->workList->getNext(".inf");
	int len = strlen(q) - 4;
	strncpy(packetBaseName, q, len);
	packetBaseName[len] = '\0';
}

// Opens the .INF file, fills the infoHeader struct and sets the 
// record lengths. Reads the misc data from the header.
void bluewave::initInf()
{
	INF_HEADER infoHeader;

	if (!fread(&infoHeader, sizeof(INF_HEADER), 1, infFile))
		fatalError("Error reading .INF file");

	infoHeaderLen = getshort(infoHeader.inf_header_len);
	if (infoHeaderLen < ORIGINAL_INF_HEADER_LEN)
		infoHeaderLen = ORIGINAL_INF_HEADER_LEN;

	infoAreainfoLen = getshort(infoHeader.inf_areainfo_len);
	if (infoAreainfoLen < ORIGINAL_INF_AREA_LEN)
		infoAreainfoLen = ORIGINAL_INF_AREA_LEN;

	maxConf = (mm->workList->getSize() - infoHeaderLen) / infoAreainfoLen;

	mixStructLen = getshort(infoHeader.mix_structlen);
	if (mixStructLen < ORIGINAL_MIX_STRUCT_LEN)
		mixStructLen = ORIGINAL_MIX_STRUCT_LEN;

	ftiStructLen = getshort(infoHeader.fti_structlen);
	if (ftiStructLen < ORIGINAL_FTI_STRUCT_LEN)
		ftiStructLen = ORIGINAL_FTI_STRUCT_LEN;

	mm->resourceObject->set(LoginName, (char *) infoHeader.loginname);
	mm->resourceObject->set(AliasName, (char *) infoHeader.aliasname);
	mm->resourceObject->set(SysOpName, (char *) infoHeader.sysop);
	mm->resourceObject->set(BBSName, (char *) infoHeader.systemname); 
}

// Match .INF records to .MIX records
void bluewave::initMixID()
{
	int mixIndex, c, d;

	areas = new INF_AREA_INFO[maxConf];
	mixID = new int[maxConf];

	for (d = 0; d < maxConf; d++) {
		mixIndex = -1;
		fseek(infFile, (infoHeaderLen + d * infoAreainfoLen),
			SEEK_SET);
		if (!fread(&areas[d], sizeof(INF_AREA_INFO), 1, infFile))
			fatalError("Premature EOF in bluewave::initInf");
		for (c = 0; c < noOfMixRecs; c++)
			if (!numcmp((char *) areas[d].areanum,
				(char *) mixRecord[c].areanum)) {
				mixIndex = c;
				break;
			}
		mixID[d] = mixIndex;
	}
}

void bluewave::readMixRec()
{
	FILE *mixFile = openFile("MIX");

	noOfMixRecs = (int) (mm->workList->getSize() / mixStructLen);
	mixRecord = new MIX_REC[noOfMixRecs];

	fread(mixRecord, mixStructLen, noOfMixRecs, mixFile);
	fclose(mixFile);
}

FILE *bluewave::openFile(const char *extent)
{
	FILE *tmp;
	char fname[13];

	sprintf(fname, "%s.%s", packetBaseName, extent);

	if (!(tmp = mm->workList->ftryopen(fname, "rb"))) {
		sprintf(fname, "Could not open .%s file", extent);
		fatalError(fname);
	}
	return tmp;
}

// ---------------------------------------------------------------------------
// The Blue Wave reply methods
// ---------------------------------------------------------------------------

bwreply::bwreply(mmail *mmA)
{
	mm = mmA;
	init();
}

bwreply::~bwreply()
{
	if (replyExists)
		cleanup();
}

void bwreply::init()
{
	uplHeader = new UPL_HEADER;
	replyText = NULL;
	newFileName(mm->resourceObject->get(PacketName));
	mychdir(mm->resourceObject->get(ReplyDir));
	replyExists = !access(replyPacketName, R_OK | W_OK);

	if (replyExists) {
		uncompress();
		readUpl();
		currentLetter = 1;
	} else {
		uplListHead = NULL;
		noOfLetters = currentLetter = 0;
	}
}

void bwreply::cleanup()
{
	upl_list *next, *curr = uplListHead;

	for (int c = 0; c < noOfLetters; c++) {
		remove(curr->fname);
		next = curr->nextRecord;
		delete curr;
		curr = next;
	}
	delete replyText;
	delete uplHeader;
}

void bwreply::newFileName(const char *bwFile)
{
	int c;

	for (c = 0; (bwFile[c] != '.') && bwFile[c]; c++)
		replyPacketName[c] = tolower(bwFile[c]);
	strcpy(replyPacketName + c, ".new");
}

void bwreply::uncompress()
{
	char fname[256];

	sprintf(fname, "%s/%s", mm->resourceObject->get(ReplyDir),
		replyPacketName);
	uncompressFile(mm->resourceObject, fname,
		mm->resourceObject->get(UpWorkDir));
}

int bwreply::getNew1(FILE *uplFile, upl_list *l, int recnum)
{
	const char *orgname;

	fseek(uplFile, getshort(uplHeader->upl_header_len) + recnum *
		getshort(uplHeader->upl_rec_len), SEEK_SET);
	if (fread(&(l->uplRec), sizeof(UPL_REC), 1, uplFile) != 1)
		return -1;

	orgname = upWorkList->exists((char *) l->uplRec.filename);
	mytmpnam(l->fname);
	l->msglen = fromdos(orgname, l->fname);
	remove(orgname);

	return 0;
}

void bwreply::readUpl()
{
	FILE *uplFile;
	upl_list baseUplList, *currUplList;
	char uplName[13];

	upWorkList = new file_list(mm->resourceObject->get(UpWorkDir));
	sprintf(uplName, "%s.upl", findBaseName(replyPacketName));

	if (!(uplFile = upWorkList->ftryopen(uplName, "rb")))
		fatalError("Error opening UPL");

	if (!fread(uplHeader, sizeof(UPL_HEADER), 1, uplFile))
		fatalError("Error reading UPL file");

	noOfLetters = (upWorkList->getSize() -
			getshort(uplHeader->upl_header_len)) /
			getshort(uplHeader->upl_rec_len);

	currUplList = &baseUplList;
	for (int c = 0; c < noOfLetters; c++) {
		currUplList->nextRecord = new upl_list;
		currUplList = currUplList->nextRecord;
                if (getNew1(uplFile, currUplList, c)) {    // ha eof/error
                        delete currUplList;
                        break;
                }
	}
	uplListHead = baseUplList.nextRecord;

	fclose(uplFile);
	remove(upWorkList->exists(uplName));
	delete upWorkList;
}

int bwreply::getNoOfAreas()
{
	return 1;
}

void bwreply::resetAll()
{
	cleanup();
	init();
}

area_header *bwreply::getNextArea()
{
	return new area_header(mm, 0, "REPLY", "REPLIES",
		"Letters written by you", "Blue Wave replies",
		(COLLECTION | REPLYAREA | ACTIVE), noOfLetters, 0);
}

void bwreply::selectArea(int ID)
{
	if (ID == 0)
		resetLetters();
}

void bwreply::resetLetters()
{
	currentLetter = 1;
	uplListCurrent = uplListHead;
}

int bwreply::getNoOfLetters()
{
	return noOfLetters;
}

letter_header *bwreply::getNextLetter()
{
	letter_header *newLetter;
	net_address na;
	time_t unixTime;
	char *date;
	int areaNo;

	unixTime = (time_t) getlong(uplListCurrent->uplRec.unix_date);

	date = ctime(&unixTime);
	date[strlen(date) - 1] = '\0';

	areaNo = 0;
	for (int c = 0; c < mm->areaList->noOfAreas(); c++)
		if (!strcmp(mm->areaList->getName(c),
			    (char *) uplListCurrent->uplRec.echotag))
			areaNo = c;

	na.zone = getshort(uplListCurrent->uplRec.destzone);
	if (na.zone) {
		na.net = getshort(uplListCurrent->uplRec.destnet);
		na.node = getshort(uplListCurrent->uplRec.destnode);
		na.point = getshort(uplListCurrent->uplRec.destpoint);
	}

	newLetter = new letter_header(mm, (char *) uplListCurrent->uplRec.subj,
					(char *) uplListCurrent->uplRec.to,
				(char *) uplListCurrent->uplRec.from, date,
				getlong(uplListCurrent->uplRec.replyto),
				currentLetter, currentLetter, areaNo,
		(getshort(uplListCurrent->uplRec.msg_attr) & UPL_PRIVATE),
					uplListCurrent->msglen, this, &na);

	currentLetter++;
	uplListCurrent = uplListCurrent->nextRecord;
	return newLetter;
}

char *bwreply::getBody(int area, int ID)
{
	FILE *replyFile;
	upl_list *actUplList;
	int msglen;

	area = area;	// warning suppression
	delete replyText;

	actUplList = uplListHead;
	for (int c = 1; c < ID; c++)
		actUplList = actUplList->nextRecord;

	if ((replyFile = fopen(actUplList->fname, "rt"))) {
		msglen = actUplList->msglen;
		replyText = new char[msglen + 1];
		msglen = fread(replyText, 1, msglen, replyFile);
		fclose(replyFile);
		replyText[msglen] = '\0';
	} else
		replyText = NULL;

	return replyText;
}

void bwreply::enterLetter(letter_header *newLetter, char *newLetterFileName,
				int length)
{
	net_address *na = newLetter->getNetAddr();
	upl_list *newList = new upl_list;
	memset(newList, 0, sizeof(upl_list));

	int msg_attr = 0;

	// fill the fields of UPL_REC
	strncpy((char *) newList->uplRec.from, newLetter->getFrom(), 35);
	strncpy((char *) newList->uplRec.to, newLetter->getTo(), 35);
	strncpy((char *) newList->uplRec.subj, newLetter->getSubject(), 71);
	strcpy((char *) newList->uplRec.echotag, mm->areaList->getName());

	strcpy(newList->fname, newLetterFileName);
	putlong(newList->uplRec.unix_date, (long) time(NULL));
	if (na->zone) {
		msg_attr |= UPL_NETMAIL;
		putshort(newList->uplRec.destzone, na->zone);
		putshort(newList->uplRec.destnet, na->net);
		putshort(newList->uplRec.destnode, na->node);
		putshort(newList->uplRec.destpoint, na->point);
	};
	putlong(newList->uplRec.replyto,
		(unsigned long) newLetter->getReplyTo());

	if (newLetter->getPrivate())
		msg_attr |= UPL_PRIVATE;
	if (newLetter->getReplyTo())
		msg_attr |= UPL_IS_REPLY;
	putshort(newList->uplRec.msg_attr, msg_attr);

	newList->msglen = length;

	if (!noOfLetters)
		uplListHead = newList;
	else {
		upl_list *workList = uplListHead;
		for (int c = 1; c < noOfLetters; c++)	//go to last elem
			workList = workList->nextRecord;
		workList->nextRecord = newList;
	}

	noOfLetters++;
	replyExists = 1;
}

void bwreply::killLetter(int letterNo)
{
	upl_list *actUplList, *tmpUplList;

	if ((noOfLetters == 0) || (letterNo < 1) || (letterNo > noOfLetters))
		fatalError("Internal error in bwreply::killLetter");

	if (letterNo == 1) {
		tmpUplList = uplListHead;
		uplListHead = uplListHead->nextRecord;
	} else {
		actUplList = uplListHead;
		for (int c = 1; c < letterNo - 1; c++)
			actUplList = actUplList->nextRecord;
		tmpUplList = actUplList->nextRecord;
		actUplList->nextRecord = (letterNo == noOfLetters) ? 0 :
			actUplList->nextRecord->nextRecord;
	}
	noOfLetters--;
	remove(tmpUplList->fname);
	delete tmpUplList;
	resetLetters();
}

area_header *bwreply::refreshArea()
{
	return getNextArea();
}

void bwreply::addNew1(FILE *uplFile, upl_list *l)
{
	strcpy((char *) l->uplRec.filename, freeFileName());
	todos(l->fname, (char *) l->uplRec.filename);
	fwrite(&(l->uplRec), sizeof(UPL_REC), 1, uplFile);
}

void bwreply::makeReply()
{
	FILE *uplFile;
	UPL_HEADER newUplHeader;
	upl_list *actUplList;
	char uplFileName[13], tmp[256];

	if (mychdir(mm->resourceObject->get(UpWorkDir)))
		fatalError("Could not cd to upworkdir in bwreply::makeReply");

	// Delete old file -- hmm, could trash old replies if the pack fails
	sprintf(tmp, "%s/%s", mm->resourceObject->get(ReplyDir),
		replyPacketName);
	remove(tmp);

	if (!noOfLetters)
		return;

	// fill the UPL_HEADER struct
	memset(&newUplHeader, 0, sizeof(UPL_HEADER));

	putshort(newUplHeader.upl_header_len, sizeof(UPL_HEADER));
	putshort(newUplHeader.upl_rec_len, sizeof(UPL_REC));

	newUplHeader.reader_major = MM_MAJOR;
	newUplHeader.reader_minor = MM_MINOR;
	sprintf((char *) newUplHeader.vernum, "%d.%d", MM_MAJOR, MM_MINOR);
	for (int c = 0; newUplHeader.vernum[c]; newUplHeader.vernum[c++] -= 10);

	struct utsname buf;
	uname(&buf);
	int tearlen = sprintf((char *) newUplHeader.reader_name,
		MM_NAME "/%s", buf.sysname);
	strncpy((char *) newUplHeader.reader_tear, ((tearlen < 16) ?
		(char *) newUplHeader.reader_name : MM_NAME), 15);

	strcpy((char *) newUplHeader.loginname,
		mm->resourceObject->get(LoginName));
	strcpy((char *) newUplHeader.aliasname,
		mm->resourceObject->get(AliasName));

	sprintf(uplFileName, "%s.upl",
		findBaseName(mm->resourceObject->get(PacketName)));
	uplFile = fopen(uplFileName, "wb");	//!! no check yet

	fwrite(&newUplHeader, sizeof(UPL_HEADER), 1, uplFile);

	actUplList = uplListHead;
	for (int c = 0; c < noOfLetters; c++) {
		addNew1(uplFile, actUplList);
		actUplList = actUplList->nextRecord;
	}

	fclose(uplFile);

	//pack the files
	sprintf(tmp, "%s *.msg", uplFileName);
	compressAddFile(mm->resourceObject,
		mm->resourceObject->get(ReplyDir), replyPacketName, tmp);

	// clean up the work area
	clearDirectory(".");
}

char *bwreply::freeFileName()
{
	static char testFileName[13];

	for (long c = 0; c <= 99999; c++) {
		sprintf(testFileName, "%05ld.msg", c);
		if (access(testFileName, R_OK) == -1)
			break;
	}
	return testFileName;
}

int bwreply::todos(const char *org, const char *dest)
{
	FILE *orgfile, *destfile;
	int c, count = 0;

	if ((orgfile = fopen(org, "rt"))) {
		if ((destfile = fopen(dest, "wb"))) {
			while ((c = fgetc(orgfile)) != EOF) {
				if (c == '\n') {
					fputc('\r', destfile);
					count++;
				}
				fputc(c, destfile);
				count++;
			}
			fclose(destfile);
		}
		fclose(orgfile);
	}
	return count;
}

int bwreply::fromdos(const char *org, const char *dest)
{
	FILE *orgfile, *destfile;
	int c, count = 0;

	if ((orgfile = fopen(org, "rb"))) {
		if ((destfile = fopen(dest, "wt"))) {
			while ((c = fgetc(orgfile)) != EOF) {
				if (c != '\r') {
					fputc(c, destfile);
					count++;
				}
			}
			fclose(destfile);
		}
		fclose(orgfile);
	}
	return count;
}
