(************************************************)
(*                                              *)
(*  RS485.PAS (01/15/99)                        *)
(*                                              *)
(*  Test program for 2-wire RS485.              *)
(*                                              *)
(*  This program assumes that RTS controls the  *)
(*  UART transmitter, as is typical in 2-wire   *)
(*  RS-485 circuits. Note that in a 2-wire      *)
(*  setup, all data is echoed back locally.     *)
(*                                              *) 
(*  Execute this program from the DOS command   *)
(*  line (boot to DOS in Windows).              *)
(*                                              *)
(*  This program is donated to the Public       *)
(*  Domain by MarshallSoft Computing, Inc.      *)
(*  It is provided as an example of the use     *)
(*  of the Personal Communications Library.     *)
(*                                              *)
(************************************************)

program rs485;
uses crt, pcl4p;

var
   BaudCode : Integer;
   RetCode  : Integer;
   Byte : Char;
   i    : Integer;
   Port : Integer;
   ResetFlag : Boolean;
   BufPtr    : Pointer;
   BufSeg    : Word;

procedure SayError( Code : Integer );
var
   RetCode : Integer;
begin
   if Code < 0 then RetCode := SioError(Code)
   else if (Code and (FramingError or ParityError or OverrunError)) <> 0 then
      begin (* Port Error *)
         if (Code and FramingError) <> 0 then writeln('Framing Error');
         if (Code and ParityError)  <> 0 then writeln('Parity Error');
         if (Code and OverrunError) <> 0 then writeln('Overrun Error')
      end
end;

procedure MyHalt( Code : Integer );
var
   RetCode : Integer;
begin
   if Code < 0 then SayError(Code);
   if ResetFlag then RetCode := SioDone(Port);
   writeln('*** HALTING ***');
   Halt;
end;

function MatchBaud(BaudString : String) : Integer;
const
   BaudRateArray : array[1..10] of LongInt =
       (0,300,1200,2400,4800,9600,19200,38400,57600,115200);
var
   i : Integer;
   BaudRate: LongInt;
   RetCode : Integer;
begin
  Val(BaudString,BaudRate,RetCode);
  if RetCode <> 0 then
  begin
    MatchBaud := -1;
    exit;
  end;
  for i := 1 to 10 do if BaudRateArray[i] = BaudRate then
  begin
    MatchBaud := i - 1;
    exit;
  end;
  (* no match *)
  MatchBaud := -1;
end;

begin   (* main program *)
   ResetFlag := FALSE;
   (* fetch the port # from command line *)
   if ParamCount <> 2 then
      begin
         writeln('USAGE: "rs485 <port> <baud rate>" where port = 1 to 20');
         halt;
      end;
   Val( ParamStr(1),Port, RetCode );
   if RetCode <> 0 then
      begin
         writeln('Port must be 1 to 16');
         Halt;
      end;
   (* COM1 = 0, COM2 = 1, etc. *)
   Port := Port - 1;
   if (Port<COM1) or (Port>COM16) then
      begin
         writeln('Port must be 1 to 16');
         Halt
      end;
   (* get baud rate *)
   BaudCode := MatchBaud(ParamStr(2));
   (* setup 1K receive buffer *)
   GetMem(BufPtr,1024+16);
   BufSeg := Seg(BufPtr^) + ((Ofs(BufPtr^)+15) SHR 4);
   RetCode := SioRxBuf(Port, BufSeg, Size1024);
   if RetCode < 0 then MyHalt(RetCode);
   if SioInfo('I') > 0 then
     begin
       (* setup 128 transmit buffer *)
       GetMem(BufPtr,128+16);
       BufSeg := Seg(BufPtr^) + ((Ofs(BufPtr^)+15) SHR 4);
       RetCode := SioTxBuf(Port, BufSeg, Size128);
       if RetCode < 0 then MyHalt(RetCode);
     end;
   (* reset port *)
   RetCode := SioReset(Port,BaudCode);
   (* if error then try one more time *)
   if RetCode <> 0 then RetCode := SioReset(Port,BaudCode);
   (* was port reset ? *)
   if RetCode <> 0 then
     begin
        writeln('Cannot reset COM',Port+1);
        MyHalt(RetCode);
     end;
   (* Port successfully reset *)
   writeln;
   writeln('COM',1+Port,' @ ',ParamStr(2));
   ResetFlag := TRUE;
   (* specify parity, # stop bits, and word length for port *)
   RetCode := SioParms(Port, NoParity, OneStopBit, WordLength8);
   if RetCode < 0 then MyHalt(RetCode);
   (* set FIFO level if have 16550/16650/16750 *)
   RetCode := SioFIFO(Port, LEVEL_4TH);
   if RetCode < 0 then MyHalt(RetCode); 
   case RetCode of
       0: WriteLn('8250/16450 detected.'); 
       1: WriteLn('16550 detected.');
       2: WriteLn('16650 detected.');
       3: WriteLn('16750 detected.');
   end;   
   (* begin terminal loop *)
   writeln('Enter terminal loop ( Type ^Z to exit )');
   while TRUE do
      begin
         (* did user press Ctrl-BREAK ? *)
         if SioBrkKey then
            begin
               writeln('User typed Ctl-BREAK');
               RetCode := SioDone(Port);
               Halt;
            end;
         (* anything incoming over serial port ? *)
         RetCode := SioGetc(Port,0);
         if RetCode < -1 then MyHalt( RetCode );
         if RetCode > -1 then Write( chr(RetCode) );
         (* has user pressed keyboard ? *)
         if KeyPressed then
            begin
               (* read keyboard *)
               Byte := ReadKey;
               (* quit if user types ^Z *)
               if Byte = chr($1a) then
                  begin
                     writeln('User typed ^Z');
                     RetCode := SioDone(Port);
                     Halt;
                  end;
               (* set RTS to enable UART driver *)
               RetCode := SioRTS(Port, 'S');
               (* send out over serial line *)
               RetCode := SioPutc(Port, Byte );
               (* wait for last bit to be sent *)
               RetCode := SioEmpty(Port);
               (* drop RTS *)
               RetCode := SioRTS(Port, 'C')
            end
      end
end.
