/* name2num
 * copyright 1996 David J. Binette
 * ALL RIGHT RESERVED
 * Possesion of this source code does not constitute ownership
 */

/* written with Tab stops set to 4 */

#include <stdio.h>
#include <stdlib.h>
#include <sys\stat.h>
#include <string.h>
#include <io.h>

typedef unsigned long dword;
typedef unsigned char byte;

/* this logfile is created in the CURRENT directory */
#define LOGFILE "name2num.log"
#define	EOS	'\0'

/* maximum length of a Fully Qualified pathname */
#define LENMAX	2048

/* ---------------------------------------------------------------- */

char * Progname;
int	Crcerror;
char * Dirstr;				/* directory component */
char * Namstr;				/* filename component */
char * Extstr;				/* file extension */

/* the path/filename seperator character
 * this is for DOS
 * For *nix, change it to: "/"
 */
char * Sepstr= "\\";		/* a 1 character string */

/* ---------------------------------------------------------------- */

/* this is the CRC32 lookup table
 * thanks Gary S. Brown 
 * 64 lines of 4 values for a 256 dword table (1024 bytes)
 */
dword crc_tab[256] =
{					 /* CRC polynomial 0xedb88320 */
	(dword)0x00000000, (dword)0x77073096, (dword)0xee0e612c, (dword)0x990951ba,
	(dword)0x076dc419, (dword)0x706af48f, (dword)0xe963a535, (dword)0x9e6495a3,
	(dword)0x0edb8832, (dword)0x79dcb8a4, (dword)0xe0d5e91e, (dword)0x97d2d988,
	(dword)0x09b64c2b, (dword)0x7eb17cbd, (dword)0xe7b82d07, (dword)0x90bf1d91,
	(dword)0x1db71064, (dword)0x6ab020f2, (dword)0xf3b97148, (dword)0x84be41de,
	(dword)0x1adad47d, (dword)0x6ddde4eb, (dword)0xf4d4b551, (dword)0x83d385c7,
	(dword)0x136c9856, (dword)0x646ba8c0, (dword)0xfd62f97a, (dword)0x8a65c9ec,
	(dword)0x14015c4f, (dword)0x63066cd9, (dword)0xfa0f3d63, (dword)0x8d080df5,
	(dword)0x3b6e20c8, (dword)0x4c69105e, (dword)0xd56041e4, (dword)0xa2677172,
	(dword)0x3c03e4d1, (dword)0x4b04d447, (dword)0xd20d85fd, (dword)0xa50ab56b,
	(dword)0x35b5a8fa, (dword)0x42b2986c, (dword)0xdbbbc9d6, (dword)0xacbcf940,
	(dword)0x32d86ce3, (dword)0x45df5c75, (dword)0xdcd60dcf, (dword)0xabd13d59,
	(dword)0x26d930ac, (dword)0x51de003a, (dword)0xc8d75180, (dword)0xbfd06116,
	(dword)0x21b4f4b5, (dword)0x56b3c423, (dword)0xcfba9599, (dword)0xb8bda50f,
	(dword)0x2802b89e, (dword)0x5f058808, (dword)0xc60cd9b2, (dword)0xb10be924,
	(dword)0x2f6f7c87, (dword)0x58684c11, (dword)0xc1611dab, (dword)0xb6662d3d,
	(dword)0x76dc4190, (dword)0x01db7106, (dword)0x98d220bc, (dword)0xefd5102a,
	(dword)0x71b18589, (dword)0x06b6b51f, (dword)0x9fbfe4a5, (dword)0xe8b8d433,
	(dword)0x7807c9a2, (dword)0x0f00f934, (dword)0x9609a88e, (dword)0xe10e9818,
	(dword)0x7f6a0dbb, (dword)0x086d3d2d, (dword)0x91646c97, (dword)0xe6635c01,
	(dword)0x6b6b51f4, (dword)0x1c6c6162, (dword)0x856530d8, (dword)0xf262004e,
	(dword)0x6c0695ed, (dword)0x1b01a57b, (dword)0x8208f4c1, (dword)0xf50fc457,
	(dword)0x65b0d9c6, (dword)0x12b7e950, (dword)0x8bbeb8ea, (dword)0xfcb9887c,
	(dword)0x62dd1ddf, (dword)0x15da2d49, (dword)0x8cd37cf3, (dword)0xfbd44c65,
	(dword)0x4db26158, (dword)0x3ab551ce, (dword)0xa3bc0074, (dword)0xd4bb30e2,
	(dword)0x4adfa541, (dword)0x3dd895d7, (dword)0xa4d1c46d, (dword)0xd3d6f4fb,
	(dword)0x4369e96a, (dword)0x346ed9fc, (dword)0xad678846, (dword)0xda60b8d0,
	(dword)0x44042d73, (dword)0x33031de5, (dword)0xaa0a4c5f, (dword)0xdd0d7cc9,
	(dword)0x5005713c, (dword)0x270241aa, (dword)0xbe0b1010, (dword)0xc90c2086,
	(dword)0x5768b525, (dword)0x206f85b3, (dword)0xb966d409, (dword)0xce61e49f,
	(dword)0x5edef90e, (dword)0x29d9c998, (dword)0xb0d09822, (dword)0xc7d7a8b4,
	(dword)0x59b33d17, (dword)0x2eb40d81, (dword)0xb7bd5c3b, (dword)0xc0ba6cad,
	(dword)0xedb88320, (dword)0x9abfb3b6, (dword)0x03b6e20c, (dword)0x74b1d29a,
	(dword)0xead54739, (dword)0x9dd277af, (dword)0x04db2615, (dword)0x73dc1683,
	(dword)0xe3630b12, (dword)0x94643b84, (dword)0x0d6d6a3e, (dword)0x7a6a5aa8,
	(dword)0xe40ecf0b, (dword)0x9309ff9d, (dword)0x0a00ae27, (dword)0x7d079eb1,
	(dword)0xf00f9344, (dword)0x8708a3d2, (dword)0x1e01f268, (dword)0x6906c2fe,
	(dword)0xf762575d, (dword)0x806567cb, (dword)0x196c3671, (dword)0x6e6b06e7,
	(dword)0xfed41b76, (dword)0x89d32be0, (dword)0x10da7a5a, (dword)0x67dd4acc,
	(dword)0xf9b9df6f, (dword)0x8ebeeff9, (dword)0x17b7be43, (dword)0x60b08ed5,
	(dword)0xd6d6a3e8, (dword)0xa1d1937e, (dword)0x38d8c2c4, (dword)0x4fdff252,
	(dword)0xd1bb67f1, (dword)0xa6bc5767, (dword)0x3fb506dd, (dword)0x48b2364b,
	(dword)0xd80d2bda, (dword)0xaf0a1b4c, (dword)0x36034af6, (dword)0x41047a60,
	(dword)0xdf60efc3, (dword)0xa867df55, (dword)0x316e8eef, (dword)0x4669be79,
	(dword)0xcb61b38c, (dword)0xbc66831a, (dword)0x256fd2a0, (dword)0x5268e236,
	(dword)0xcc0c7795, (dword)0xbb0b4703, (dword)0x220216b9, (dword)0x5505262f,
	(dword)0xc5ba3bbe, (dword)0xb2bd0b28, (dword)0x2bb45a92, (dword)0x5cb36a04,
	(dword)0xc2d7ffa7, (dword)0xb5d0cf31, (dword)0x2cd99e8b, (dword)0x5bdeae1d,
	(dword)0x9b64c2b0, (dword)0xec63f226, (dword)0x756aa39c, (dword)0x026d930a,
	(dword)0x9c0906a9, (dword)0xeb0e363f, (dword)0x72076785, (dword)0x05005713,
	(dword)0x95bf4a82, (dword)0xe2b87a14, (dword)0x7bb12bae, (dword)0x0cb61b38,
	(dword)0x92d28e9b, (dword)0xe5d5be0d, (dword)0x7cdcefb7, (dword)0x0bdbdf21,
	(dword)0x86d3d2d4, (dword)0xf1d4e242, (dword)0x68ddb3f8, (dword)0x1fda836e,
	(dword)0x81be16cd, (dword)0xf6b9265b, (dword)0x6fb077e1, (dword)0x18b74777,
	(dword)0x88085ae6, (dword)0xff0f6a70, (dword)0x66063bca, (dword)0x11010b5c,
	(dword)0x8f659eff, (dword)0xf862ae69, (dword)0x616bffd3, (dword)0x166ccf45,
	(dword)0xa00ae278, (dword)0xd70dd2ee, (dword)0x4e048354, (dword)0x3903b3c2,
	(dword)0xa7672661, (dword)0xd06016f7, (dword)0x4969474d, (dword)0x3e6e77db,
	(dword)0xaed16a4a, (dword)0xd9d65adc, (dword)0x40df0b66, (dword)0x37d83bf0,
	(dword)0xa9bcae53, (dword)0xdebb9ec5, (dword)0x47b2cf7f, (dword)0x30b5ffe9,
	(dword)0xbdbdf21c, (dword)0xcabac28a, (dword)0x53b39330, (dword)0x24b4a3a6,
	(dword)0xbad03605, (dword)0xcdd70693, (dword)0x54de5729, (dword)0x23d967bf,
	(dword)0xb3667a2e, (dword)0xc4614ab8, (dword)0x5d681b02, (dword)0x2a6f2b94,
	(dword)0xb40bbe37, (dword)0xc30c8ea1, (dword)0x5a05df1b, (dword)0x2d02ef8d
};

/* ---------------------------------------------------------------- */

void usage(int longhelp)
{
	fprintf(stderr,
			"%s, Version 1.0 Beta, copyright 1996 David J. Binette\n"
			"ALL RIGHTS RESERVED\n"
			"email to dbin@sce.de\n"
			"CRC32 code courtesy Gary S. Brown\n",
			Progname);

	fprintf(stderr,
			"usage: %s [-d] [-f] [-h] [-n] [-u] [-v] {[path\\]filename}....\n",
			Progname);

	if(!longhelp)
	{
	fprintf(stderr,
			"Use %s -h for license and usage information BEFORE using.\n",
			Progname);
	fprintf(stderr,
			"sample usage for automated non-interactive use:\n"
			"%s -d -f -v *.jpg *.gif\n",
			Progname);
	}
	else
	{
	fprintf(stderr,
			"-d   Delete, delete duplicate files\n"
			"-f   Force, do not display startup warning\n"
			"-n   NoLog, Do not append log info to %s (in current dir)\n"
			"-u   Update file comment (not yet implimented)\n"
			"-v   Verbose output\n"
			"-h   This Help\n",
			LOGFILE);
	fprintf(stderr,
			"Renames files to CRC32 of contents with the original filename extension.\n"
			"Multiple identical copies of the same file can optionally be deleted.\n"
			"NOTE: It is highly unlikely but NOT impossible that two files\n"
			"      will have the same CRC32. If -d is specified the NEW copy is discarded.\n");

	fprintf(stderr,
            "      !-!-!-!-!-! USE AT YOUR OWN RISK !-!-!-!-!-!\n"
			"Use of this program is restricted to personal private use\n"
			"and not for profit organizations.  Other use requires payment of $100(US)\n"
			"Make cheques payable to:\n"
			"David J. Binette\n"
			"Box 12 Kootenay Bay\n"
			"British Columbia\n"
			"Canada V0B 1X0\n");
	}
	exit(1);
}

/* ---------------------------------------------------------------- */

char *basename(char *s)
{
	char *p;
	p = strrchr(s,*Sepstr);
	if(p && *p)
	{
		char *q;
		p++;
		q = strrchr(p, '.');
		if(q && *q)
			*q = EOS;
	}
	return p;
}

/* ---------------------------------------------------------------- */

/* open a file
 * calculate the CRC32 of the entire contents
 * return the CRC
 * if there is an error rdet the global variable Crcerror
 */
dword readfilecrc(char *fname)
{
	register int c;
	register dword crc;
	FILE *fp;

	Crcerror=0;

	if((fp = fopen(fname, "rb")) == (FILE*)0)
	{
		perror(fname);
		Crcerror=1;
		return -1;
	}

	crc = (dword)0xFFFFFFFF;

	while((c = getc(fp)) != EOF)
		crc =  ~(crc_tab[(int)((crc ^ c) & 0xff)] ^ ((crc >> 8) & (dword)0x00FFFFFF));

	fclose(fp);
	return crc;
}

/* ---------------------------------------------------------------- */

/* apply the existing Directory
 * and filename extension
 * to the specified filename
 * and return the fully qualified name
 */
char * DNE2fqn(char *name)
{
	static char tmp[LENMAX+1];
	tmp[0] = EOS;

	if(Dirstr && *Dirstr)
	{
		strcat(tmp,Dirstr);
		strcat(tmp,Sepstr);
	}

	if(name && *name)
		strcat(tmp,name);
	
	if(Extstr && *Extstr)
	{
		strcat(tmp,".");
		strcat(tmp,Extstr);
	}
	return tmp;
}

/* ---------------------------------------------------------------- */

/* Given a fully qualified name
 * break it up into 3 parts, for Directory ,FileName and Extension
 * The global variables for 
 * Dirstr to the directory name
 * Namstr to the file name
 * Extstr to the extension
 * return 0 if OK, 1 if error 
 */


int fqn2DNE(char *fqn)
{
	static char tmp[LENMAX+1];
	char *p1;
	char *p2;

	Dirstr = NULL;
	Namstr = NULL;
	Extstr = NULL;

	if(!fqn || !*fqn)
		return 1;

	if(strlen(fqn) > LENMAX)
		return 1;

	strcpy(tmp, fqn);

	p1 = strrchr(tmp,*Sepstr);
	p2 = strrchr(tmp,'.');

	if((p2 && *p2) && (p2 > p1))
	{
		*p2 = EOS;
		Extstr = p2+1;
	}

	if(p1 && *p1)
	{
		*p1 = EOS;
		Namstr = p1+1;
		Dirstr = tmp;
	}
	else
		Namstr = tmp;
	return 0;
}

/* ---------------------------------------------------------------- */

void main(int argc, char *argv[])
{
	struct stat statbuf;
	dword crc;
	int arg = 0;
	char *origname;
	char newname[9];
	char *newfqn;
	char tmp[80];
	int	wantlog		= 1;
	int	wantforce	= 0;
	int	wantdelete	= 0;
	int	wantverbose	= 0;
	int	flagger		= 0;

	Progname = basename(argv[0]);

	if(argc == 1)
		usage(0);

	/* process the command line */
	while(++arg < argc)
	{
		/* stop processing option on first filename */
		if(argv[arg][0] != '-')
			break;

		/* process command line options */
		switch(argv[arg][1])
		{
		case 'd': wantdelete=1;	break;
		case 'f': wantforce=1;	break;
		case 'n': wantlog=0;	break;
		case 'u': 				break;
		case 'v': wantverbose=1;break;
		default:	usage(1);	break;
		}
	}
	arg--;

	/* ask user if its OK to do it */
	if(!wantforce)
	{
		fprintf(stderr,
			"%s:Enter 'y' to process %d FILES : ",
			Progname,
			argc-1);

		gets(tmp);
		if((*tmp != 'y') && (*tmp != 'Y'))
		{
			fprintf(stderr,"Quitting\n");
			exit(1);
		}
	}

	/* process each specifie file */
	while(++arg < argc)
	{
		/* when processing many files this is a progress indicator */
		if(wantverbose)
		{
			if((++flagger) & 1)
				write(1,"+\r",2);
			else
				write(1,"-\r",2);
		}

		origname = argv[arg];

		/* dont handle NAME2NUM.log */
		if(!stricmp(origname,LOGFILE))
			continue;

		if(stat(origname, &statbuf))
		{
			perror(origname);
			continue;
		}

		/* only process normal files */
		if((statbuf.st_mode & S_IFREG) == 0)
		{
			if(wantverbose)
				fprintf(stderr,"%s is not a plain file.\n", origname);
			continue;
		}

		/* split up oldfilename into Directory, Name and Extension */
		if(fqn2DNE(origname))
			continue;

		/* calculate the CRC of the entire file contents */
		crc = readfilecrc(origname);
		if(Crcerror)
			continue;

		sprintf(newname,"%08lX", crc);

		/* if the name IS the CRC, we can skip to the next file */
		if(!strcmp(newname, Namstr))
			continue;

		/* Generate the new filename with the original Dir and extension */
		newfqn = DNE2fqn(newname);

		/* if the new file already exists, handle that */
		if(access(newfqn,0) == 0)
		{
			if(!wantdelete)
			{
				fprintf(stderr,
					"%s EXISTS AS %s\n"
					"delete %s (y/N)? ",
					origname,
					newfqn,
					origname);
				*tmp = '\0';
				gets(tmp);	/* hope the user isn't keyboard happy */
			}
			else
				*tmp = 'y';

			if((*tmp == 'y') || (*tmp != 'Y'))
			{
				if(wantverbose)
					fprintf(stderr,"deleting      %s\n", origname);

				if(unlink(origname))
					perror(origname);
			}
			continue;
		}

		/* handle rename of file */


		if(rename(origname, newfqn))
			fprintf(stderr,"rename FAILED %s <- %s\n", newfqn, origname);
		else
		{
			if(wantverbose)
				fprintf(stderr,"rename OK     %s <- %s\n", newfqn, origname);
		}

		if(wantlog)
		{
			FILE *fp;
			if((fp = fopen(LOGFILE,"a")) != (FILE*)0)
			{
				fprintf(fp,"rename %s %s\n", newfqn, origname);
				fclose(fp);
			}
		}
	}
	exit(0);
}

/* ---------------------------------------------------------------- */
