/*							zetac.c
 *
 *	Riemann zeta function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, zetac();
 *
 * y = zetac( x );
 *
 *
 *
 * DESCRIPTION:
 *
 *
 *
 *                inf.
 *                 -    -x
 *   zetac(x)  =   >   k   ,   x > 1,
 *                 -
 *                k=2
 *
 * is related to the Riemann zeta function by
 *
 *	Riemann zeta(x) = zetac(x) + 1.
 *
 * Extension of the function definition for x < 1 is implemented.
 * Zero is returned for x > 127.
 * Overflow may occur for x < -30.8148.
 * The routine deals with the following cases.
 *	Tabulated values stored for integer x between 0 and 30
 *	Defining series if x greater than 30 or noninteger > 25
 *	Euler-Maclaurin summation formula for noninteger x < 25
 *	Transformation for negative x
 *
 *
 * ACCURACY:
 *
 * Tabulated values have full machine accuracy.
 * Tested near integer values; relative accuracy is 4e-16 at
 * x = 2, improving to 1e-16 at x > 2.5.
 * Other values spot checked against tables.
 *
 */

/*							zetac	*/

/*
 * By AMS55 #23.2.9,
 *
 * zeta(s) =
 *
 *                          inf.
 *  n            1-s          -
 *  -   -s      n            | |  x - [x]
 *  >  k    +  -----  -  s   |    ------- dx
 *  -           s-1        | |       s+1
 * k=1                      -       x
 *                           n
 *
 * for x = Re(s) > 0 and n > 0.  The term with the integral is
 * expanded by the Euler-Maclaurin summation formula into
 *
 *
 *           inf.  B   s(s+1)...(1+j-1)
 *    1       -     2j
 * - ---  +   >    --------------------
 *     s      -               s+j
 *   2n      j=1       (2j)! n
 *
 *  where the B2j are Bernoulli numbers.
 *
 *
 * References:
 *
 * Jahnke, E., and F. Emde, Tables of Functions, 
 * pp 269-274; Dover, 1945 (Note: one of the zeta function
 * expansion coefficients given by them is incorrect.)
 *
 * Froberg, C-E, Introduction to Numerical Analysis, 2nd ed.,
 * pp 231-233; Addison-Wesley, 1969.
 *
 */

/*							zetac	*/

/* Cephes Math Library Release 1.1:  March, 1985
 * Copyright 1985 by Stephen L. Moshier
 * Direct inquiries to 30 Frost Street, Cambridge, MA 02140 */

#include <stdio.h>
#include "qhead.h"

/* Expansion coefficients
 * for Euler-Maclaurin summation formula
 * (2k)! / B2k
 * where B2k are Bernoulli numbers
 */
#if WORDSIZE ==32
static QELT A[30][NQ] = {
/* 1/(6* 2!) */
{0,EXPONE-4,0,0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaaa,
0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaaa,0xaaaaaaab},
/* -1/(30* 4!) */
{0,EXPONE-10,0,0xb60b60b6,0x0b60b60b,0x60b60b60,0xb60b60b6,0x0b60b60b,
 0x60b60b60,0xb60b60b6,0x0b60b60b,0x60b60b60,0xb60b60b6,0x0b60b60b},
/* 1/(42* 6!) */
{0,EXPONE-15,0,0x8ab355e0,0x08ab355e,0x008ab355,0xe008ab35,0x5e008ab3,
 0x55e008ab,0x355e008a,0xb355e008,0xab355e00,0x8ab355e0,0x08ab355e},
/* -1/(30* 8!) */
{0,EXPONE-21,0,0xddebbc99,0xa7785563,0x34111eef,0xccdaab88,0x96674452,
 0x23000dde,0xbbc99a77,0x85563341,0x11eefccd,0xaab88966,0x74452230},
/* 5/(66* 10!) */
{0,EXPONE-26,0,0xb354795f,0xb875ed14,0xb089f4cc,0x1f12f738,0x121a8f20,
 0xc1c727a7,0x0eb26d4e,0x7b431225,0x1113de62,0xf68ff2e8,0xc563d339},
/* -691/(2730* 12!) */
{0,EXPONE-31,0,0x91402eb2,0x2133f7ba,0x56337ff2,0x7cb7da47,0xbf24da67,
 0x4691c785,0x3f543dfa,0x53423721,0xa93b90e7,0x64b84284,0x0c17b0d2},
/* 7/(6* 14!) */
{0,EXPONE-37,0,0xeb6d9627,0x048b0fdc,0x257768c2,0x74f0337c,0xcf8fd4d8,
 0x4bda21ad,0x5e201178,0x19499c6a,0x6d10e4f0,0x32a018d3,0x1044097b},
/* -3617/(510* 16!) */
{0,EXPONE-42,0,0xbed270fb,0xccaae25f,0xf129a06f,0x42d0dc42,0x1ca96a3a,
 0xeae69ce3,0x6e56be72,0xab3307c5,0xb2705d50,0xd7e350f5,0x6bde3a3a},
/* 43867/(798* 18!) */
{0,EXPONE-47,0,0x9aac38eb,0x2974ecee,0x56665d7d,0x7bc1dbfa,0x0670bed3,
 0xad62d19a,0xf8bcacbb,0x549a14b5,0xc92ee27e,0xb8f78962,0xe2c3b3be},
/* -174611/(330* 20!) */
{0,EXPONE-53,0,0xfabecb46,0x55678663,0xce0f50ad,0xaa0b8212,0x52baefe8,
 0xd958c04f,0x4d240803,0x6fb1d83f,0xd907f569,0xa191f100,0x6d029269},
/* 854513/(138* 22!) */
{0,EXPONE-58,0,0xcb3f0f84,0xe1bb77d8,0x2b4adb04,0x8eaa5e7b,0x530b0ee9,
 0x176625db,0xd1b55d3d,0xcb5d39ce,0x9184092e,0xedb35982,0xff7f810a},
/* -236364091/(2730* 24!) */
{0,EXPONE-63,0,0xa4bec819,0xd15ae370,0x87e6555a,0x6c876af6,0x54e1cc08,
 0x69e26dea,0xb864b682,0xa98aad8f,0xff209833,0x43dd00ea,0x7bd3e661},
/* 8553103/(6* 26!) */
{0,EXPONE-68,0,0x858996be,0x35683203,0xa8a4d91c,0xebb6a6c7,0x923e385d,
 0xa8b53868,0x8ef70706,0xe9677213,0x9cc286bf,0x87b55fca,0xef69972b},
/* -23749461029/(870* 28!) = -8.9535174270375468504e-23 */
{0,EXPONE-74,0,0xd87b96ac,0xf8e0b3e6,0x16e913f6,0xcf2056ca,0x177d84d2,
0x9cb6a9c2,0x2916f42b,0x30802bde,0xd5533fa3,0xf3cf5749,0xb6b0755d},
/* 8615841276005/(14322* 30!) =  2.2679524523376830603e-24 */
{0,EXPONE-79,0,0xaf796d26,0x65136ad7,0x3275bfba,0x8ce6d0ed,0x18c831dd,
0x563cab76,0xa4d169bf,0x0565a958,0x84b9327f,0xee243811,0x001b3cdc},
/* -7709321041217/(510* 32!) = -5.74479066887220244526e-26 */
{0,EXPONE-84,0,0x8e3befcb,0x6f1c57e2,0x53a622f2,0xcc53f6c7,0x227bd799,
0x96c5068c,0x8bc04ed3,0x542c128e,0xb3c075c9,0xe4d0e0a1,0xcee836c3},
/* 2577687858367/(6* 34!) =  1.45517247561486490187e-27 */
{0,EXPONE-90,0,0xe694cef2,0x90db0d7f,0x140f2fbc,0x2f9641dc,0x2a098ff8,
0x490d2aa4,0xf33e000f,0x1f812bc5,0xf527f6b4,0x8c4c23f0,0x02d9ce33},
/* -26315271553053477373/(1919190* 36!) = -3.68599494066531017818e-29 */
{0,EXPONE-95,0,0xbae6f32b,0x2ba53676,0x30634b2a,0xe8812ec9,0xc32eadd1,
0x8804da1e,0xdf2b73de,0xd1f4c251,0x4c4f34fd,0xf48d23e6,0xc477531c},
/* 2929993913841559/(6* 38!) =  9.3367342570950446720e-31 */
{0,EXPONE-100,0,0x977f46d9,0xda56f633,0xf1698e12,0x2c11f788,0xff40fd1a,
0x6fb90bbd,0xca18ec5e,0xa8c18f52,0xe8b23588,0x21792787,0x1ee0c150},
/* -261082718496449122051/(13530* 40!) = -2.36502241570062993456e-32 */
{0,EXPONE-106,0,0xf5991482,0x3b0ff767,0xbe9068b5,0x98935e3a,0x0735e5f2,
0xc8096598,0x54c4518b,0x163fffae,0xb8ed3ea7,0x961588db,0x2e89a3ef},
/* 1520097643918070802691/(1806* 42!) =  5.99067176248213430466e-34 */
{0,EXPONE-111,0,0xc712ffc9,0x94232185,0x7b060807,0xe1a612c5,0xba91fbc9,
0x91d93be4,0x8d43c5c7,0x95bbdb55,0xadb45290,0x0c4470ac,0xfde25c40},
/* -27833269579301024235023/(690* 44!) */
{0,EXPONE-116,0,0xa15d0d1a,0x4dae965b,0x1957c82a,0x844c0b09,0x61fe019f,
0x644bf076,0xffa96c7c,0x88218e00,0xb554f791,0x79c905dc,0x28258c5e},
/* 596451111593912163277961/(282* 46!) =  3.8437581254541882322e-37 */
{0,EXPONE-121,0,0x82cbdb0e,0x59869c78,0x6d77d91f,0xb4588bcd,0x1acd0687,
0xe8b5db8a,0xf5c85518,0xfd76cc62,0xdb014105,0xad61f0bc,0x0b7622e9},
/* -5609403368997817686249127547/(46410* 48!) = -9.73635307264669103527e-39 */
{0,EXPONE-127,0,0xd409fb75,0x5383969d,0xa5f874c5,0x38b7836e,0x56f90a54,
0x7bd8de26,0xd5c1783e,0x08e4e59f,0xb1f92897,0xcfe15f8d,0xbdd3c601},
/* 495057205241079648212477525/(66* 50!) = 2.4662470442006809571e-40 */
{0,EXPONE-132,0,0xabdf514a,0x87891d0b,0xea270db1,0x6c47aa9b,0xcc19b238,
0xbd4d90bd,0x50bccc8b,0xffbdbfe7,0x6ecc0a4c,0xf8428cba,0x7b64c8e8},
/* -801165718135489957347924991853/(1590* 52!) = -6.2470767418207436931e-42 */
{0,EXPONE-137,0,0x8b5080e2,0xfef4b9b3,0x96f8faef,0x3cdda828,0xdabbb3ef,
0xe381d59c,0xf9c57a1e,0x432b19b4,0xff944ac2,0x14ff247d,0xf868a79a},
/* 29149963634884862421418123812691/(798* 54!) = 1.58240302446449142975e-43 */
{0,EXPONE-143,0,0xe1d91d82,0xf1cfc905,0xc83a5d11,0xa4abd879,0xa0853391,
0x7658dd4a,0x06049d04,0xe0b00c28,0x5be5e7b4,0xb5c49e6a,0x8633d023},
/* -2479392929313226753685415739663229/(870* 56!) = -4.00827368594893597e-45 */
{0,EXPONE-148,0,0xb710c9d7,0x24b6aad8,0x1c1d54f1,0x267cb789,0xcf344624,
0xb84de752,0xb8f46713,0xecdafedf,0x01799c95,0x10df04e7,0xf5d81d1b},
/* 84483613348880041862046775994036021/(354* 58!) = 1.01530758555695563e-46 */
{0,EXPONE-153,0,0x94632aab,0xf5152c21,0x9d63999a,0xd3fa40cc,0xd94ab82c,
0x538af3b0,0xf006b60b,0x01eac181,0xb68f8dba,0x643de039,0xa68f7cf5},
/* -1215233140483755572040304994079820246041491/(56786730* 60!)
 = -2.57180415824187e-48 */
{0,EXPONE-159,0,0xf08e799e,0x319541b8,0xa3195fa6,0x9e351760,0xbec65995,
0x6962b00c,0xcf73fdd9,0xec1df101,0xe801d8a8,0x47b70f6f,0xb5cf5494},
};

/* Table of logarithms for power function */
static QELT logs[33][NQ] = {
/* 6.9314718055994530941723212145817656807550013436025525412068000949339362E-1*/
{0,EXPONE-1,0,0xb17217f7,0xd1cf79ab,0xc9e3b398,0x03f2f6af,0x40f34326,
0x7298b62d,0x8a0d175b,0x8baafa2b,0xe7b87620,0x6debac98,0x559552fb},
/*1.0986122886681096913952452369225257046474905578227494517346943336374943E0*/
{0,EXPONE,0,0x8c9f53d5,0x681854bb,0x520cc6aa,0x829dbe5a,0xdf0a216c,
0xdbf046f8,0x1ecbf775,0x28a49ac6,0x554bc169,0x06634c28,0x2838fb9a},
/*1.3862943611198906188344642429163531361510002687205105082413600189867872E0*/
{0,EXPONE,0,0xb17217f7,0xd1cf79ab,0xc9e3b398,0x03f2f6af,0x40f34326,
0x7298b62d,0x8a0d175b,0x8baafa2b,0xe7b87620,0x6debac98,0x559552fb},
/*1.6094379124341003746007593332261876395256013542685177219126478914741790E0*/
{0,EXPONE,0,0xce020fbf,0x6c699b57,0xefbbd28b,0x03ac98fa,0x24a63860,
0x7b6f9706,0x4285b156,0x3e0addcf,0x3881ca9f,0x2c5a4a6d,0x6e16ee03},
/*1.7917594692280550008124773583807022727229906921830047058553743431308879E0*/
{0,EXPONE,0,0xe5585fd1,0x51001191,0x36fea076,0x849739b2,0x7f83c300,
0x153ca20e,0xe3d28322,0xee7a17dc,0x4927fc79,0x3d592274,0x5303a517},
/*1.9459101490553133051053527434431797296370847295818611884593901499375799E0*/
{0,EXPONE,0,0xf9139571,0x92d2baa3,0x7b4a4b67,0x930df023,0x261ae732,
0x274e1ffd,0x5732a55e,0xb97b611d,0x1ef0beb1,0x439817a0,0x48d1b5ab},
/*2.0794415416798359282516963643745297042265004030807657623620400284801809E0*/
{0,EXPONE+1,0,0x851591f9,0xdd5b9b40,0xd76ac6b2,0x02f63903,0x70b6725c,
0xd5f288a2,0x2789d184,0xa8c03ba0,0xedca5898,0x5270c172,0x402ffe3c},
/*2.1972245773362193827904904738450514092949811156454989034693886672749886E0*/
{0,EXPONE+1,0,0x8c9f53d5,0x681854bb,0x520cc6aa,0x829dbe5a,0xdf0a216c,
0xdbf046f8,0x1ecbf775,0x28a49ac6,0x554bc169,0x06634c28,0x2838fb9a},
/*2.3025850929940456840179914546843642076011014886287729760333279009675726E0*/
{0,EXPONE+1,0,0x935d8ddd,0xaaa8ac16,0xea56d62b,0x82d30a28,0xe28fecf9,
0xda5df90e,0x83c61e82,0x01f02d72,0x962f02d7,0xb1a8105c,0xcc70cbc0},
/*2.3978952727983705440619435779651292998217068539374171752185677091305736E0*/
{0,EXPONE+1,0,0x99771dbb,0xf9ba5dbd,0xe2e5fe65,0x3f697d19,0x4d69bb8c,
0xd8c396f1,0x55057edb,0xd31d3ac7,0x198edbf4,0x015ad951,0xac5493dc},
/*2.4849066497880003102297094798388788407984908265432599599760543526242815E0*/
{0,EXPONE+1,0,0x9f08b5e6,0x9cf3e733,0x8df83d21,0x43485a85,0x0ffeb249,
0xa7447e92,0xd46c8768,0x5a27ca79,0x1e821bc4,0xba277c60,0x3ee7274a},
/*2.5649493574615367360534874415653186048052679447602071164190455106634647E0*/
{0,EXPONE+1,0,0xa4282159,0x8c6287f4,0x33ee77c5,0x1fec1040,0xc9dab82d,
0x7da45a8e,0xa22109df,0x903f943b,0x1dea3905,0x2c17d0f0,0xa764f12b},
/*2.6390573296152586145225848649013562977125848639421164425800701594309735E0*/
{0,EXPONE+1,0,0xa8e650b6,0xbddd3bbc,0xb01e1299,0xca83b5bd,0x634a4462,
0xb04d3d8a,0x0e1c9886,0x3fa86f19,0x89667ce0,0xbd46f6f6,0x39ce2f94},
/*2.7080502011022100659960045701487133441730919120912671736473422251116733E0*/
{0,EXPONE+1,0,0xad50b1ca,0x6a40f809,0xa0e44c9a,0xc3252baa,0x81d82ce6,
0xabafeeff,0x30a8d465,0xb357bc4a,0xc6e6c604,0x195ecb4a,0xcb27f4ce},
/*2.7725887222397812376689284858327062723020005374410210164827200379735745E0*/
{0,EXPONE+1,0,0xb17217f7,0xd1cf79ab,0xc9e3b398,0x03f2f6af,0x40f34326,
0x7298b62d,0x8a0d175b,0x8baafa2b,0xe7b87620,0x6debac98,0x559552fb},
/*2.8332133440562160802495346178731265355882030125857447872972377378822926E0*/
{0,EXPONE+1,0,0xb5535e0f,0xd3fbce77,0xf97ce1c5,0xe15954d5,0x75c51673,
0x5005ff93,0x266e5f1e,0x3168a03e,0x40152cbb,0x658d3099,0xf9b71bb2},
/*2.8903717578961646922077225953032279773704812500057541575900686767683822E0*/
{0,EXPONE+1,0,0xb8fbd9d3,0x5c8c3326,0x4485b390,0x839a7c06,0xaf46f236,
0x78967483,0x814f3d4c,0x0b8f5951,0x4f39def1,0x21de374e,0x3d9e5058},
/*2.9444389791664404600090274318878535372373792612991288185379602364092927E0*/
{0,EXPONE+1,0,0xbc71b030,0x25996390,0x03d4b89b,0x9484d42e,0x764a5ffb,
0xf215afb9,0xfa385711,0x41f50473,0x6327359f,0xae787116,0xbb8a662d},
/*2.9957322735539909934352235761425407756766016229890282301540079104609662E0*/
{0,EXPONE+1,0,0xbfba13db,0x9f1c8a81,0xdccfc311,0x83cfc7d4,0xb2ccbdc3,
0x77042699,0xe6496458,0xe4daebfd,0x901d205f,0xcd22fb82,0xe1d6207f},
/*3.0445224377234229965005979803657054342845752874046106401940844835750742E0*/
{0,EXPONE+1,0,0xc2d974a3,0x7d7587af,0x66ab8909,0x0ad5d73f,0x0292844f,
0x819f337a,0xbaff4e69,0xf10ffdf1,0xba1e400d,0x24fdb1e4,0x388558a2},
/*3.0910424533583158534791756994233058678972069882976724293392477186239672E0*/
{0,EXPONE+1,0,0xc5d3a3b9,0xee2e3c28,0xd55eeb4b,0x40663ac5,0x1da68c56,
0x7569c47c,0xb788c4b2,0xb607f952,0x137cf97c,0x1cd5c477,0xc1b9e89a},
/*3.1354942159291496908067528318101961184423803148404357419986353774829932E0*/
{0,EXPONE+1,0,0xc8abefee,0x8d9f8048,0xac300792,0xd7e27019,0xf86b869a,
0x1bfe8c1c,0x7931b56e,0xd47c15ec,0x7bcfc190,0xc09d43e3,0x06bc321a},
/*3.1780538303479456196469416012970554088739909609035152140967343621176752E0*/
{0,EXPONE+1,0,0xcb653be4,0x9167c59e,0x80712a07,0x44451830,0xe03b8313,
0x43eaac1e,0x36efcd3f,0x3d128904,0x1870394c,0xd5a26786,0x544c7c09},
/*3.2188758248682007492015186664523752790512027085370354438252957829483580E0*/
{0,EXPONE+1,0,0xce020fbf,0x6c699b57,0xefbbd28b,0x03ac98fa,0x24a63860,
0x7b6f9706,0x4285b156,0x3e0addcf,0x3881ca9f,0x2c5a4a6d,0x6e16ee02},
/*3.2580965380214820454707195630234951728807680791204623705397255201568583E0*/
{0,EXPONE+1,0,0xd084a757,0x80d6665f,0x266764ab,0x20e8cdec,0x9a1788f7,
0x1a4a881a,0x04a44fb6,0x732a52c6,0x17d8568d,0x4792bc16,0xbcca45ea},
/*3.2958368660043290741857357107675771139424716734682483552040830009124829E0*/
{0,EXPONE+1,0,0xd2eefdc0,0x1c247f18,0xfb1329ff,0xc3ec9d88,0x4e8f3223,
0x49e86a74,0x2e31f32f,0xbcf6e829,0x7ff1a21d,0x8994f23c,0x3c557966},
/*3.3322045101752039239398169863595328657880849983023716967007501689243671E0*/
{0,EXPONE+1,0,0xd542d6b4,0xb2511a27,0xa296ff7f,0xcb807369,0x3387152c,
0x4cf36b15,0x709fde5d,0x22932da4,0x83549a68,0xd8c1e21c,0x4f338453},
/*3.3672958299864740271832720323619116054945129139227440789216703516427808E0*/
{0,EXPONE+1,0,0xd781c65e,0x6ff0f5eb,0x3504f782,0xd1a67235,0x43cf47f3,
0x52df24d1,0xda1d9982,0xa486773d,0x5ca79b0e,0x67bdec7e,0xf77d3e90},
/*3.4011973816621553754132366916068899122485920464515224277680222346050669E0*/
{0,EXPONE+1,0,0xd9ad37c8,0x5eb4d674,0x935d3980,0xc421e956,0x5214fdb0,
0x48561c8a,0x932c1a3c,0x96427ad5,0xc0d4e38c,0x34d9b670,0xe08d498d},
/*3.4339872044851462459291643245423572104499389304805919717567180724749814E0*/
{0,EXPONE+1,0,0xdbc67244,0x895acf88,0xb97511c3,0x69dea09e,0xf517c701,
0x05a722f5,0x32ef44e8,0x56203500,0xda3d59d8,0x7d79b3c1,0x165cdf14},
/*3.4657359027997265470861606072908828403775006718012762706034000474669681E0*/
{0,EXPONE+1,0,0xddce9df5,0xc6435816,0xbc5ca07e,0x04efb45b,0x113013f0,
0x0f3ee3b8,0xec905d32,0x6e95b8b6,0xe1a693a8,0x896697be,0x6afaa7ba},
/*3.4965075614664802354571888148876550044691974117601666269532620427680679E0*/
{0,EXPONE+1,0,0xdfc6c7a6,0xadc6881b,0x8bec61ba,0x80b85c46,0xbceecc43,
0x46bbba6d,0x646b7a96,0x676f882a,0x4434bca8,0x848c7f65,0xc07111a9},
/*3.5263605246161613896667667393313031036637031469460000414179177473756862E0*/
{0,EXPONE+1,0,0xe1afe40d,0xc86face2,0xebf5ceab,0xe2561281,0x4601e73c,
0xecac2d1e,0x88f1a4f5,0x14535ec9,0x3a034a43,0x81081bc0,0x0f1c7070},
};
#else
/* WORDSIZE = 16 */
#if NQ < 24

static short A[30][NQ] = {
/* 1/(6* 2!) */
{0x0000,EXPONE-4,0x0000,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,
0xaaaa,0xaaaa,0xaaaa,0xaaab,},
/* -1/(30* 4!) */
{0xffff,EXPONE-10,0x0000,0xb60b,0x60b6,0x0b60,0xb60b,0x60b6,
0x0b60,0xb60b,0x60b6,0x0b61,},
/* 1/(42* 6!) */
{0x0000,EXPONE-15,0x0000,0x8ab3,0x55e0,0x08ab,0x355e,0x008a,
0xb355,0xe008,0xab35,0x5e01,},
/* -1/(30* 8!) */
{0xffff,EXPONE-21,0x0000,0xddeb,0xbc99,0xa778,0x5563,0x3411,
0x1eef,0xccda,0xab88,0x9667,},
/* 5/(66* 10!) */
{0x0000,EXPONE-26,0x0000,0xb354,0x795f,0xb875,0xed14,0xb089,
0xf4cc,0x1f12,0xf738,0x121b,},
/* -691/(2730* 12!) */
{0xffff,EXPONE-31,0x0000,0x9140,0x2eb2,0x2133,0xf7ba,0x5633,
0x7ff2,0x7cb7,0xda47,0xbf25,},
/* 7/(6* 14!) */
{0x0000,EXPONE-37,0x0000,0xeb6d,0x9627,0x048b,0x0fdc,0x2577,
0x68c2,0x74f0,0x337c,0xcf90,},
/* -3617/(510* 16!) */
{0xffff,EXPONE-42,0x0000,0xbed2,0x70fb,0xccaa,0xe25f,0xf129,
0xa06f,0x42d0,0xdc42,0x1ca9,},
/* 43867/(798* 18!) */
{0x0000,EXPONE-47,0x0000,0x9aac,0x38eb,0x2974,0xecee,0x5666,
0x5d7d,0x7bc1,0xdbfa,0x0671,},
/* -174611/(330* 20!) */
{0xffff,EXPONE-53,0x0000,0xfabe,0xcb46,0x5567,0x8663,0xce0f,
0x50ad,0xaa0b,0x8212,0x52bb,},
/* 854513/(138* 22!) */
{0x0000,EXPONE-58,0x0000,0xcb3f,0x0f84,0xe1bb,0x77d8,0x2b4a,
0xdb04,0x8eaa,0x5e7b,0x530c,},
/* -236364091/(2730* 24!) */
{0xffff,EXPONE-63,0x0000,0xa4be,0xc819,0xd15a,0xe370,0x87e6,
0x555a,0x6c87,0x6af6,0x54e2,},
/* 8553103/(6* 26!) */
{0x0000,EXPONE-68,0x0000,0x8589,0x96be,0x3568,0x3203,0xa8a4,
0xd91c,0xebb6,0xa6c7,0x923e,},
/* -23749461029/(870* 28!) = -8.9535174270375468504e-23 */
{0xffff,EXPONE-74,0x0000,0xd87b,0x96ac,0xf8e0,0xb3e6,0x16e9,
0x13f6,0xcf20,0x56ca,0x177e,},
/* 8615841276005/(14322* 30!) =  2.2679524523376830603e-24 */
{0x0000,EXPONE-79,0x0000,0xaf79,0x6d26,0x6513,0x6ad7,0x3275,
0xbfba,0x8ce6,0xd0ed,0x18c8,},
/* -7709321041217/(510* 32!) = -5.74479066887220244526e-26 */
{0xffff,EXPONE-84,0x0000,0x8e3b,0xefcb,0x6f1c,0x57e2,0x53a6,
0x22f2,0xcc53,0xf6c7,0x227c,},
/* 2577687858367/(6* 34!) =  1.45517247561486490187e-27 */
{0x0000,EXPONE-90,0x0000,0xe694,0xcef2,0x90db,0x0d7f,0x140f,
0x2fbc,0x2f96,0x41dc,0x2a0a,},
/* -26315271553053477373/(1919190* 36!) = -3.68599494066531017818e-29 */
{0xffff,EXPONE-95,0x0000,0xbae6,0xf32b,0x2ba5,0x3676,0x3063,
0x4b2a,0xe881,0x2ec9,0xc32f,},
/* 2929993913841559/(6* 38!) =  9.3367342570950446720e-31 */
{0x0000,EXPONE-100,0x0000,0x977f,0x46d9,0xda56,0xf633,0xf169,
0x8e12,0x2c11,0xf788,0xff41,},
/* -261082718496449122051/(13530* 40!) = -2.36502241570062993456e-32 */
{0xffff,EXPONE-106,0x0000,0xf599,0x1482,0x3b0f,0xf767,0xbe90,
0x68b5,0x9893,0x5e3a,0x0736,},
/* 1520097643918070802691/(1806* 42!) =  5.99067176248213430466e-34 */
{0x0000,EXPONE-111,0x0000,0xc712,0xffc9,0x9423,0x2185,0x7b06,
0x0807,0xe1a6,0x12c5,0xba92,},
/* -27833269579301024235023/(690* 44!) */
{0xffff,EXPONE-116,0x0000,0xa15d,0x0d1a,0x4dae,0x965b,0x1957,
0xc82a,0x844c,0x0b09,0x61fe,},
/* 596451111593912163277961/(282* 46!) =  3.8437581254541882322e-37 */
{0x0000,EXPONE-121,0x0000,0x82cb,0xdb0e,0x5986,0x9c78,0x6d77,
0xd91f,0xb458,0x8bcd,0x1acd,},
/* -5609403368997817686249127547/(46410* 48!) = -9.73635307264669103527e-39 */
{0xffff,EXPONE-127,0x0000,0xd409,0xfb75,0x5383,0x969d,0xa5f8,
0x74c5,0x38b7,0x836e,0x56f9,},
/* 495057205241079648212477525/(66* 50!) = 2.4662470442006809571e-40 */
{0x0000,EXPONE-132,0x0000,0xabdf,0x514a,0x8789,0x1d0b,0xea27,
0x0db1,0x6c47,0xaa9b,0xcc1a,},
/* -801165718135489957347924991853/(1590* 52!) = -6.2470767418207436931e-42 */
{0xffff,EXPONE-137,0x0000,0x8b50,0x80e2,0xfef4,0xb9b3,0x96f8,
0xfaef,0x3cdd,0xa828,0xdabc,},
/* 29149963634884862421418123812691/(798* 54!) = 1.58240302446449142975e-43 */
{0x0000,EXPONE-143,0x0000,0xe1d9,0x1d82,0xf1cf,0xc905,0xc83a,
0x5d11,0xa4ab,0xd879,0xa085,},
/* -2479392929313226753685415739663229/(870* 56!) = -4.00827368594893597e-45 */
{0xffff,EXPONE-148,0x0000,0xb710,0xc9d7,0x24b6,0xaad8,0x1c1d,
0x54f1,0x267c,0xb789,0xcf34,},
/* 84483613348880041862046775994036021/(354* 58!) = 1.01530758555695563e-46 */
{0x0000,EXPONE-153,0x0000,0x9463,0x2aab,0xf515,0x2c21,0x9d63,
0x999a,0xd3fa,0x40cc,0xd94b,},
/* -1215233140483755572040304994079820246041491/(56786730* 60!)
 = -2.57180415824187e-48 */
{0xffff,EXPONE-159,0x0000,0xf08e,0x799e,0x3195,0x41b8,0xa319,
0x5fa6,0x9e35,0x1760,0xbec6,},
};

/* Table of logarithms for power function */
static short logs[33][NQ] = {
/* 6.9314718055994530941723212145817656807550013436025525412068000949339362E-1*/
{0x0000,EXPONE-1,0x0000,0xb172,0x17f7,0xd1cf,0x79ab,0xc9e3,
0xb398,0x03f2,0xf6af,0x40f3,},
/*1.0986122886681096913952452369225257046474905578227494517346943336374943E0*/
{0x0000,EXPONE,0x0000,0x8c9f,0x53d5,0x6818,0x54bb,0x520c,
0xc6aa,0x829d,0xbe5a,0xdf0a,},
/*1.3862943611198906188344642429163531361510002687205105082413600189867872E0*/
{0x0000,EXPONE,0x0000,0xb172,0x17f7,0xd1cf,0x79ab,0xc9e3,
0xb398,0x03f2,0xf6af,0x40f3,},
/*1.6094379124341003746007593332261876395256013542685177219126478914741790E0*/
{0x0000,EXPONE,0x0000,0xce02,0x0fbf,0x6c69,0x9b57,0xefbb,
0xd28b,0x03ac,0x98fa,0x24a6,},
/*1.7917594692280550008124773583807022727229906921830047058553743431308879E0*/
{0x0000,EXPONE,0x0000,0xe558,0x5fd1,0x5100,0x1191,0x36fe,
0xa076,0x8497,0x39b2,0x7f84,},
/*1.9459101490553133051053527434431797296370847295818611884593901499375799E0*/
{0x0000,EXPONE,0x0000,0xf913,0x9571,0x92d2,0xbaa3,0x7b4a,
0x4b67,0x930d,0xf023,0x261b,},
/*2.0794415416798359282516963643745297042265004030807657623620400284801809E0*/
{0x0000,EXPONE+1,0x0000,0x8515,0x91f9,0xdd5b,0x9b40,0xd76a,
0xc6b2,0x02f6,0x3903,0x70b6,},
/*2.1972245773362193827904904738450514092949811156454989034693886672749886E0*/
{0x0000,EXPONE+1,0x0000,0x8c9f,0x53d5,0x6818,0x54bb,0x520c,
0xc6aa,0x829d,0xbe5a,0xdf0a,},
/*2.3025850929940456840179914546843642076011014886287729760333279009675726E0*/
{0x0000,EXPONE+1,0x0000,0x935d,0x8ddd,0xaaa8,0xac16,0xea56,
0xd62b,0x82d3,0x0a28,0xe290,},
/*2.3978952727983705440619435779651292998217068539374171752185677091305736E0*/
{0x0000,EXPONE+1,0x0000,0x9977,0x1dbb,0xf9ba,0x5dbd,0xe2e5,
0xfe65,0x3f69,0x7d19,0x4d6a,},
/*2.4849066497880003102297094798388788407984908265432599599760543526242815E0*/
{0x0000,EXPONE+1,0x0000,0x9f08,0xb5e6,0x9cf3,0xe733,0x8df8,
0x3d21,0x4348,0x5a85,0x0fff,},
/*2.5649493574615367360534874415653186048052679447602071164190455106634647E0*/
{0x0000,EXPONE+1,0x0000,0xa428,0x2159,0x8c62,0x87f4,0x33ee,
0x77c5,0x1fec,0x1040,0xc9db,},
/*2.6390573296152586145225848649013562977125848639421164425800701594309735E0*/
{0x0000,EXPONE+1,0x0000,0xa8e6,0x50b6,0xbddd,0x3bbc,0xb01e,
0x1299,0xca83,0xb5bd,0x634a,},
/*2.7080502011022100659960045701487133441730919120912671736473422251116733E0*/
{0x0000,EXPONE+1,0x0000,0xad50,0xb1ca,0x6a40,0xf809,0xa0e4,
0x4c9a,0xc325,0x2baa,0x81d8,},
/*2.7725887222397812376689284858327062723020005374410210164827200379735745E0*/
{0x0000,EXPONE+1,0x0000,0xb172,0x17f7,0xd1cf,0x79ab,0xc9e3,
0xb398,0x03f2,0xf6af,0x40f3,},
/*2.8332133440562160802495346178731265355882030125857447872972377378822926E0*/
{0x0000,EXPONE+1,0x0000,0xb553,0x5e0f,0xd3fb,0xce77,0xf97c,
0xe1c5,0xe159,0x54d5,0x75c5,},
/*2.8903717578961646922077225953032279773704812500057541575900686767683822E0*/
{0x0000,EXPONE+1,0x0000,0xb8fb,0xd9d3,0x5c8c,0x3326,0x4485,
0xb390,0x839a,0x7c06,0xaf47,},
/*2.9444389791664404600090274318878535372373792612991288185379602364092927E0*/
{0x0000,EXPONE+1,0x0000,0xbc71,0xb030,0x2599,0x6390,0x03d4,
0xb89b,0x9484,0xd42e,0x764a,},
/*2.9957322735539909934352235761425407756766016229890282301540079104609662E0*/
{0x0000,EXPONE+1,0x0000,0xbfba,0x13db,0x9f1c,0x8a81,0xdccf,
0xc311,0x83cf,0xc7d4,0xb2cd,},
/*3.0445224377234229965005979803657054342845752874046106401940844835750742E0*/
{0x0000,EXPONE+1,0x0000,0xc2d9,0x74a3,0x7d75,0x87af,0x66ab,
0x8909,0x0ad5,0xd73f,0x0293,},
/*3.0910424533583158534791756994233058678972069882976724293392477186239672E0*/
{0x0000,EXPONE+1,0x0000,0xc5d3,0xa3b9,0xee2e,0x3c28,0xd55e,
0xeb4b,0x4066,0x3ac5,0x1da7,},
/*3.1354942159291496908067528318101961184423803148404357419986353774829932E0*/
{0x0000,EXPONE+1,0x0000,0xc8ab,0xefee,0x8d9f,0x8048,0xac30,
0x0792,0xd7e2,0x7019,0xf86c,},
/*3.1780538303479456196469416012970554088739909609035152140967343621176752E0*/
{0x0000,EXPONE+1,0x0000,0xcb65,0x3be4,0x9167,0xc59e,0x8071,
0x2a07,0x4445,0x1830,0xe03c,},
/*3.2188758248682007492015186664523752790512027085370354438252957829483580E0*/
{0x0000,EXPONE+1,0x0000,0xce02,0x0fbf,0x6c69,0x9b57,0xefbb,
0xd28b,0x03ac,0x98fa,0x24a6,},
/*3.2580965380214820454707195630234951728807680791204623705397255201568583E0*/
{0x0000,EXPONE+1,0x0000,0xd084,0xa757,0x80d6,0x665f,0x2667,
0x64ab,0x20e8,0xcdec,0x9a18,},
/*3.2958368660043290741857357107675771139424716734682483552040830009124829E0*/
{0x0000,EXPONE+1,0x0000,0xd2ee,0xfdc0,0x1c24,0x7f18,0xfb13,
0x29ff,0xc3ec,0x9d88,0x4e8f,},
/*3.3322045101752039239398169863595328657880849983023716967007501689243671E0*/
{0x0000,EXPONE+1,0x0000,0xd542,0xd6b4,0xb251,0x1a27,0xa296,
0xff7f,0xcb80,0x7369,0x3387,},
/*3.3672958299864740271832720323619116054945129139227440789216703516427808E0*/
{0x0000,EXPONE+1,0x0000,0xd781,0xc65e,0x6ff0,0xf5eb,0x3504,
0xf782,0xd1a6,0x7235,0x43cf,},
/*3.4011973816621553754132366916068899122485920464515224277680222346050669E0*/
{0x0000,EXPONE+1,0x0000,0xd9ad,0x37c8,0x5eb4,0xd674,0x935d,
0x3980,0xc421,0xe956,0x5215,},
/*3.4339872044851462459291643245423572104499389304805919717567180724749814E0*/
{0x0000,EXPONE+1,0x0000,0xdbc6,0x7244,0x895a,0xcf88,0xb975,
0x11c3,0x69de,0xa09e,0xf518,},
/*3.4657359027997265470861606072908828403775006718012762706034000474669681E0*/
{0x0000,EXPONE+1,0x0000,0xddce,0x9df5,0xc643,0x5816,0xbc5c,
0xa07e,0x04ef,0xb45b,0x1130,},
/*3.4965075614664802354571888148876550044691974117601666269532620427680679E0*/
{0x0000,EXPONE+1,0x0000,0xdfc6,0xc7a6,0xadc6,0x881b,0x8bec,
0x61ba,0x80b8,0x5c46,0xbcef,},
/*3.5263605246161613896667667393313031036637031469460000414179177473756862E0*/
{0x0000,EXPONE+1,0x0000,0xe1af,0xe40d,0xc86f,0xace2,0xebf5,
0xceab,0xe256,0x1281,0x4602,},
};

#else

static short A[30][NQ] = {
/* 1/(6* 2!) */
{0x0000,EXPONE-4,0x0000,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,
0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,
0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaaa,0xaaab,},
/* -1/(30* 4!) */
{0xffff,EXPONE-10,0x0000,0xb60b,0x60b6,0x0b60,0xb60b,0x60b6,
0x0b60,0xb60b,0x60b6,0x0b60,0xb60b,0x60b6,0x0b60,0xb60b,
0x60b6,0x0b60,0xb60b,0x60b6,0x0b60,0xb60b,0x60b6,0x0b61,},
/* 1/(42* 6!) */
{0x0000,EXPONE-15,0x0000,0x8ab3,0x55e0,0x08ab,0x355e,0x008a,
0xb355,0xe008,0xab35,0x5e00,0x8ab3,0x55e0,0x08ab,0x355e,
0x008a,0xb355,0xe008,0xab35,0x5e00,0x8ab3,0x55e0,0x08ab,},
/* -1/(30* 8!) */
{0xffff,EXPONE-21,0x0000,0xddeb,0xbc99,0xa778,0x5563,0x3411,
0x1eef,0xccda,0xab88,0x9667,0x4452,0x2300,0x0dde,0xbbc9,
0x9a77,0x8556,0x3341,0x11ee,0xfccd,0xaab8,0x8966,0x7445,},
/* 5/(66* 10!) */
{0x0000,EXPONE-26,0x0000,0xb354,0x795f,0xb875,0xed14,0xb089,
0xf4cc,0x1f12,0xf738,0x121a,0x8f20,0xc1c7,0x27a7,0x0eb2,
0x6d4e,0x7b43,0x1225,0x1113,0xde62,0xf68f,0xf2e8,0xc564,},
/* -691/(2730* 12!) */
{0xffff,EXPONE-31,0x0000,0x9140,0x2eb2,0x2133,0xf7ba,0x5633,
0x7ff2,0x7cb7,0xda47,0xbf24,0xda67,0x4691,0xc785,0x3f54,
0x3dfa,0x5342,0x3721,0xa93b,0x90e7,0x64b8,0x4284,0x0c18,},
/* 7/(6* 14!) */
{0x0000,EXPONE-37,0x0000,0xeb6d,0x9627,0x048b,0x0fdc,0x2577,
0x68c2,0x74f0,0x337c,0xcf8f,0xd4d8,0x4bda,0x21ad,0x5e20,
0x1178,0x1949,0x9c6a,0x6d10,0xe4f0,0x32a0,0x18d3,0x1044,},
/* -3617/(510* 16!) */
{0xffff,EXPONE-42,0x0000,0xbed2,0x70fb,0xccaa,0xe25f,0xf129,
0xa06f,0x42d0,0xdc42,0x1ca9,0x6a3a,0xeae6,0x9ce3,0x6e56,
0xbe72,0xab33,0x07c5,0xb270,0x5d50,0xd7e3,0x50f5,0x6bde,},
/* 43867/(798* 18!) */
{0x0000,EXPONE-47,0x0000,0x9aac,0x38eb,0x2974,0xecee,0x5666,
0x5d7d,0x7bc1,0xdbfa,0x0670,0xbed3,0xad62,0xd19a,0xf8bc,
0xacbb,0x549a,0x14b5,0xc92e,0xe27e,0xb8f7,0x8962,0xe2c4,},
/* -174611/(330* 20!) */
{0xffff,EXPONE-53,0x0000,0xfabe,0xcb46,0x5567,0x8663,0xce0f,
0x50ad,0xaa0b,0x8212,0x52ba,0xefe8,0xd958,0xc04f,0x4d24,
0x0803,0x6fb1,0xd83f,0xd907,0xf569,0xa191,0xf100,0x6d03,},
/* 854513/(138* 22!) */
{0x0000,EXPONE-58,0x0000,0xcb3f,0x0f84,0xe1bb,0x77d8,0x2b4a,
0xdb04,0x8eaa,0x5e7b,0x530b,0x0ee9,0x1766,0x25db,0xd1b5,
0x5d3d,0xcb5d,0x39ce,0x9184,0x092e,0xedb3,0x5982,0xff80,},
/* -236364091/(2730* 24!) */
{0xffff,EXPONE-63,0x0000,0xa4be,0xc819,0xd15a,0xe370,0x87e6,
0x555a,0x6c87,0x6af6,0x54e1,0xcc08,0x69e2,0x6dea,0xb864,
0xb682,0xa98a,0xad8f,0xff20,0x9833,0x43dd,0x00ea,0x7bd4,},
/* 8553103/(6* 26!) */
{0x0000,EXPONE-68,0x0000,0x8589,0x96be,0x3568,0x3203,0xa8a4,
0xd91c,0xebb6,0xa6c7,0x923e,0x385d,0xa8b5,0x3868,0x8ef7,
0x0706,0xe967,0x7213,0x9cc2,0x86bf,0x87b5,0x5fca,0xef6a,},
/* -23749461029/(870* 28!) = -8.9535174270375468504e-23 */
{0xffff,EXPONE-74,0x0000,0xd87b,0x96ac,0xf8e0,0xb3e6,0x16e9,
0x13f6,0xcf20,0x56ca,0x177d,0x84d2,0x9cb6,0xa9c2,0x2916,
0xf42b,0x3080,0x2bde,0xd553,0x3fa3,0xf3cf,0x5749,0xb6b0,},
/* 8615841276005/(14322* 30!) =  2.2679524523376830603e-24 */
{0x0000,EXPONE-79,0x0000,0xaf79,0x6d26,0x6513,0x6ad7,0x3275,
0xbfba,0x8ce6,0xd0ed,0x18c8,0x31dd,0x563c,0xab76,0xa4d1,
0x69bf,0x0565,0xa958,0x84b9,0x327f,0xee24,0x3811,0x001b,},
/* -7709321041217/(510* 32!) = -5.74479066887220244526e-26 */
{0xffff,EXPONE-84,0x0000,0x8e3b,0xefcb,0x6f1c,0x57e2,0x53a6,
0x22f2,0xcc53,0xf6c7,0x227b,0xd799,0x96c5,0x068c,0x8bc0,
0x4ed3,0x542c,0x128e,0xb3c0,0x75c9,0xe4d0,0xe0a1,0xcee8,},
/* 2577687858367/(6* 34!) =  1.45517247561486490187e-27 */
{0x0000,EXPONE-90,0x0000,0xe694,0xcef2,0x90db,0x0d7f,0x140f,
0x2fbc,0x2f96,0x41dc,0x2a09,0x8ff8,0x490d,0x2aa4,0xf33e,
0x000f,0x1f81,0x2bc5,0xf527,0xf6b4,0x8c4c,0x23f0,0x02da,},
/* -26315271553053477373/(1919190* 36!) = -3.68599494066531017818e-29 */
{0xffff,EXPONE-95,0x0000,0xbae6,0xf32b,0x2ba5,0x3676,0x3063,
0x4b2a,0xe881,0x2ec9,0xc32e,0xadd1,0x8804,0xda1e,0xdf2b,
0x73de,0xd1f4,0xc251,0x4c4f,0x34fd,0xf48d,0x23e6,0xc477,},
/* 2929993913841559/(6* 38!) =  9.3367342570950446720e-31 */
{0x0000,EXPONE-100,0x0000,0x977f,0x46d9,0xda56,0xf633,0xf169,
0x8e12,0x2c11,0xf788,0xff40,0xfd1a,0x6fb9,0x0bbd,0xca18,
0xec5e,0xa8c1,0x8f52,0xe8b2,0x3588,0x2179,0x2787,0x1ee1,},
/* -261082718496449122051/(13530* 40!) = -2.36502241570062993456e-32 */
{0xffff,EXPONE-106,0x0000,0xf599,0x1482,0x3b0f,0xf767,0xbe90,
0x68b5,0x9893,0x5e3a,0x0735,0xe5f2,0xc809,0x6598,0x54c4,
0x518b,0x163f,0xffae,0xb8ed,0x3ea7,0x9615,0x88db,0x2e8a,},
/* 1520097643918070802691/(1806* 42!) =  5.99067176248213430466e-34 */
{0x0000,EXPONE-111,0x0000,0xc712,0xffc9,0x9423,0x2185,0x7b06,
0x0807,0xe1a6,0x12c5,0xba91,0xfbc9,0x91d9,0x3be4,0x8d43,
0xc5c7,0x95bb,0xdb55,0xadb4,0x5290,0x0c44,0x70ac,0xfde2,},
/* -27833269579301024235023/(690* 44!) */
{0xffff,EXPONE-116,0x0000,0xa15d,0x0d1a,0x4dae,0x965b,0x1957,
0xc82a,0x844c,0x0b09,0x61fe,0x019f,0x644b,0xf076,0xffa9,
0x6c7c,0x8821,0x8e00,0xb554,0xf791,0x79c9,0x05dc,0x2826,},
/* 596451111593912163277961/(282* 46!) =  3.8437581254541882322e-37 */
{0x0000,EXPONE-121,0x0000,0x82cb,0xdb0e,0x5986,0x9c78,0x6d77,
0xd91f,0xb458,0x8bcd,0x1acd,0x0687,0xe8b5,0xdb8a,0xf5c8,
0x5518,0xfd76,0xcc62,0xdb01,0x4105,0xad61,0xf0bc,0x0b76,},
/* -5609403368997817686249127547/(46410* 48!) = -9.73635307264669103527e-39 */
{0xffff,EXPONE-127,0x0000,0xd409,0xfb75,0x5383,0x969d,0xa5f8,
0x74c5,0x38b7,0x836e,0x56f9,0x0a54,0x7bd8,0xde26,0xd5c1,
0x783e,0x08e4,0xe59f,0xb1f9,0x2897,0xcfe1,0x5f8d,0xbdd4,},
/* 495057205241079648212477525/(66* 50!) = 2.4662470442006809571e-40 */
{0x0000,EXPONE-132,0x0000,0xabdf,0x514a,0x8789,0x1d0b,0xea27,
0x0db1,0x6c47,0xaa9b,0xcc19,0xb238,0xbd4d,0x90bd,0x50bc,
0xcc8b,0xffbd,0xbfe7,0x6ecc,0x0a4c,0xf842,0x8cba,0x7b65,},
/* -801165718135489957347924991853/(1590* 52!) = -6.2470767418207436931e-42 */
{0xffff,EXPONE-137,0x0000,0x8b50,0x80e2,0xfef4,0xb9b3,0x96f8,
0xfaef,0x3cdd,0xa828,0xdabb,0xb3ef,0xe381,0xd59c,0xf9c5,
0x7a1e,0x432b,0x19b4,0xff94,0x4ac2,0x14ff,0x247d,0xf869,},
/* 29149963634884862421418123812691/(798* 54!) = 1.58240302446449142975e-43 */
{0x0000,EXPONE-143,0x0000,0xe1d9,0x1d82,0xf1cf,0xc905,0xc83a,
0x5d11,0xa4ab,0xd879,0xa085,0x3391,0x7658,0xdd4a,0x0604,
0x9d04,0xe0b0,0x0c28,0x5be5,0xe7b4,0xb5c4,0x9e6a,0x8634,},
/* -2479392929313226753685415739663229/(870* 56!) = -4.00827368594893597e-45 */
{0xffff,EXPONE-148,0x0000,0xb710,0xc9d7,0x24b6,0xaad8,0x1c1d,
0x54f1,0x267c,0xb789,0xcf34,0x4624,0xb84d,0xe752,0xb8f4,
0x6713,0xecda,0xfedf,0x0179,0x9c95,0x10df,0x04e7,0xf5d8,},
/* 84483613348880041862046775994036021/(354* 58!) = 1.01530758555695563e-46 */
{0x0000,EXPONE-153,0x0000,0x9463,0x2aab,0xf515,0x2c21,0x9d63,
0x999a,0xd3fa,0x40cc,0xd94a,0xb82c,0x538a,0xf3b0,0xf006,
0xb60b,0x01ea,0xc181,0xb68f,0x8dba,0x643d,0xe039,0xa68f,},
/* -1215233140483755572040304994079820246041491/(56786730* 60!)
 = -2.57180415824187e-48 */
{0xffff,EXPONE-159,0x0000,0xf08e,0x799e,0x3195,0x41b8,0xa319,
0x5fa6,0x9e35,0x1760,0xbec6,0x5995,0x6962,0xb00c,0xcf73,
0xfdd9,0xec1d,0xf101,0xe801,0xd8a8,0x47b7,0x0f6f,0xb5cf,},
};

/* Table of logarithms for power function */
static short logs[33][NQ] = {
/* 6.9314718055994530941723212145817656807550013436025525412068000949339362E-1*/
{0x0000,EXPONE-1,0x0000,0xb172,0x17f7,0xd1cf,0x79ab,0xc9e3,
0xb398,0x03f2,0xf6af,0x40f3,0x4326,0x7298,0xb62d,0x8a0d,
0x175b,0x8baa,0xfa2b,0xe7b8,0x7620,0x6deb,0xac98,0x5595,},
/*1.0986122886681096913952452369225257046474905578227494517346943336374943E0*/
{0x0000,EXPONE,0x0000,0x8c9f,0x53d5,0x6818,0x54bb,0x520c,
0xc6aa,0x829d,0xbe5a,0xdf0a,0x216c,0xdbf0,0x46f8,0x1ecb,
0xf775,0x28a4,0x9ac6,0x554b,0xc169,0x0663,0x4c28,0x2839,},
/*1.3862943611198906188344642429163531361510002687205105082413600189867872E0*/
{0x0000,EXPONE,0x0000,0xb172,0x17f7,0xd1cf,0x79ab,0xc9e3,
0xb398,0x03f2,0xf6af,0x40f3,0x4326,0x7298,0xb62d,0x8a0d,
0x175b,0x8baa,0xfa2b,0xe7b8,0x7620,0x6deb,0xac98,0x5595,},
/*1.6094379124341003746007593332261876395256013542685177219126478914741790E0*/
{0x0000,EXPONE,0x0000,0xce02,0x0fbf,0x6c69,0x9b57,0xefbb,
0xd28b,0x03ac,0x98fa,0x24a6,0x3860,0x7b6f,0x9706,0x4285,
0xb156,0x3e0a,0xddcf,0x3881,0xca9f,0x2c5a,0x4a6d,0x6e17,},
/*1.7917594692280550008124773583807022727229906921830047058553743431308879E0*/
{0x0000,EXPONE,0x0000,0xe558,0x5fd1,0x5100,0x1191,0x36fe,
0xa076,0x8497,0x39b2,0x7f83,0xc300,0x153c,0xa20e,0xe3d2,
0x8322,0xee7a,0x17dc,0x4927,0xfc79,0x3d59,0x2274,0x5304,},
/*1.9459101490553133051053527434431797296370847295818611884593901499375799E0*/
{0x0000,EXPONE,0x0000,0xf913,0x9571,0x92d2,0xbaa3,0x7b4a,
0x4b67,0x930d,0xf023,0x261a,0xe732,0x274e,0x1ffd,0x5732,
0xa55e,0xb97b,0x611d,0x1ef0,0xbeb1,0x4398,0x17a0,0x48d2,},
/*2.0794415416798359282516963643745297042265004030807657623620400284801809E0*/
{0x0000,EXPONE+1,0x0000,0x8515,0x91f9,0xdd5b,0x9b40,0xd76a,
0xc6b2,0x02f6,0x3903,0x70b6,0x725c,0xd5f2,0x88a2,0x2789,
0xd184,0xa8c0,0x3ba0,0xedca,0x5898,0x5270,0xc172,0x4030,},
/*2.1972245773362193827904904738450514092949811156454989034693886672749886E0*/
{0x0000,EXPONE+1,0x0000,0x8c9f,0x53d5,0x6818,0x54bb,0x520c,
0xc6aa,0x829d,0xbe5a,0xdf0a,0x216c,0xdbf0,0x46f8,0x1ecb,
0xf775,0x28a4,0x9ac6,0x554b,0xc169,0x0663,0x4c28,0x2839,},
/*2.3025850929940456840179914546843642076011014886287729760333279009675726E0*/
{0x0000,EXPONE+1,0x0000,0x935d,0x8ddd,0xaaa8,0xac16,0xea56,
0xd62b,0x82d3,0x0a28,0xe28f,0xecf9,0xda5d,0xf90e,0x83c6,
0x1e82,0x01f0,0x2d72,0x962f,0x02d7,0xb1a8,0x105c,0xcc71,},
/*2.3978952727983705440619435779651292998217068539374171752185677091305736E0*/
{0x0000,EXPONE+1,0x0000,0x9977,0x1dbb,0xf9ba,0x5dbd,0xe2e5,
0xfe65,0x3f69,0x7d19,0x4d69,0xbb8c,0xd8c3,0x96f1,0x5505,
0x7edb,0xd31d,0x3ac7,0x198e,0xdbf4,0x015a,0xd951,0xac55,},
/*2.4849066497880003102297094798388788407984908265432599599760543526242815E0*/
{0x0000,EXPONE+1,0x0000,0x9f08,0xb5e6,0x9cf3,0xe733,0x8df8,
0x3d21,0x4348,0x5a85,0x0ffe,0xb249,0xa744,0x7e92,0xd46c,
0x8768,0x5a27,0xca79,0x1e82,0x1bc4,0xba27,0x7c60,0x3ee7,},
/*2.5649493574615367360534874415653186048052679447602071164190455106634647E0*/
{0x0000,EXPONE+1,0x0000,0xa428,0x2159,0x8c62,0x87f4,0x33ee,
0x77c5,0x1fec,0x1040,0xc9da,0xb82d,0x7da4,0x5a8e,0xa221,
0x09df,0x903f,0x943b,0x1dea,0x3905,0x2c17,0xd0f0,0xa765,},
/*2.6390573296152586145225848649013562977125848639421164425800701594309735E0*/
{0x0000,EXPONE+1,0x0000,0xa8e6,0x50b6,0xbddd,0x3bbc,0xb01e,
0x1299,0xca83,0xb5bd,0x634a,0x4462,0xb04d,0x3d8a,0x0e1c,
0x9886,0x3fa8,0x6f19,0x8966,0x7ce0,0xbd46,0xf6f6,0x39ce,},
/*2.7080502011022100659960045701487133441730919120912671736473422251116733E0*/
{0x0000,EXPONE+1,0x0000,0xad50,0xb1ca,0x6a40,0xf809,0xa0e4,
0x4c9a,0xc325,0x2baa,0x81d8,0x2ce6,0xabaf,0xeeff,0x30a8,
0xd465,0xb357,0xbc4a,0xc6e6,0xc604,0x195e,0xcb4a,0xcb28,},
/*2.7725887222397812376689284858327062723020005374410210164827200379735745E0*/
{0x0000,EXPONE+1,0x0000,0xb172,0x17f7,0xd1cf,0x79ab,0xc9e3,
0xb398,0x03f2,0xf6af,0x40f3,0x4326,0x7298,0xb62d,0x8a0d,
0x175b,0x8baa,0xfa2b,0xe7b8,0x7620,0x6deb,0xac98,0x5595,},
/*2.8332133440562160802495346178731265355882030125857447872972377378822926E0*/
{0x0000,EXPONE+1,0x0000,0xb553,0x5e0f,0xd3fb,0xce77,0xf97c,
0xe1c5,0xe159,0x54d5,0x75c5,0x1673,0x5005,0xff93,0x266e,
0x5f1e,0x3168,0xa03e,0x4015,0x2cbb,0x658d,0x3099,0xf9b7,},
/*2.8903717578961646922077225953032279773704812500057541575900686767683822E0*/
{0x0000,EXPONE+1,0x0000,0xb8fb,0xd9d3,0x5c8c,0x3326,0x4485,
0xb390,0x839a,0x7c06,0xaf46,0xf236,0x7896,0x7483,0x814f,
0x3d4c,0x0b8f,0x5951,0x4f39,0xdef1,0x21de,0x374e,0x3d9e,},
/*2.9444389791664404600090274318878535372373792612991288185379602364092927E0*/
{0x0000,EXPONE+1,0x0000,0xbc71,0xb030,0x2599,0x6390,0x03d4,
0xb89b,0x9484,0xd42e,0x764a,0x5ffb,0xf215,0xafb9,0xfa38,
0x5711,0x41f5,0x0473,0x6327,0x359f,0xae78,0x7116,0xbb8a,},
/*2.9957322735539909934352235761425407756766016229890282301540079104609662E0*/
{0x0000,EXPONE+1,0x0000,0xbfba,0x13db,0x9f1c,0x8a81,0xdccf,
0xc311,0x83cf,0xc7d4,0xb2cc,0xbdc3,0x7704,0x2699,0xe649,
0x6458,0xe4da,0xebfd,0x901d,0x205f,0xcd22,0xfb82,0xe1d6,},
/*3.0445224377234229965005979803657054342845752874046106401940844835750742E0*/
{0x0000,EXPONE+1,0x0000,0xc2d9,0x74a3,0x7d75,0x87af,0x66ab,
0x8909,0x0ad5,0xd73f,0x0292,0x844f,0x819f,0x337a,0xbaff,
0x4e69,0xf10f,0xfdf1,0xba1e,0x400d,0x24fd,0xb1e4,0x3885,},
/*3.0910424533583158534791756994233058678972069882976724293392477186239672E0*/
{0x0000,EXPONE+1,0x0000,0xc5d3,0xa3b9,0xee2e,0x3c28,0xd55e,
0xeb4b,0x4066,0x3ac5,0x1da6,0x8c56,0x7569,0xc47c,0xb788,
0xc4b2,0xb607,0xf952,0x137c,0xf97c,0x1cd5,0xc477,0xc1ba,},
/*3.1354942159291496908067528318101961184423803148404357419986353774829932E0*/
{0x0000,EXPONE+1,0x0000,0xc8ab,0xefee,0x8d9f,0x8048,0xac30,
0x0792,0xd7e2,0x7019,0xf86b,0x869a,0x1bfe,0x8c1c,0x7931,
0xb56e,0xd47c,0x15ec,0x7bcf,0xc190,0xc09d,0x43e3,0x06bc,},
/*3.1780538303479456196469416012970554088739909609035152140967343621176752E0*/
{0x0000,EXPONE+1,0x0000,0xcb65,0x3be4,0x9167,0xc59e,0x8071,
0x2a07,0x4445,0x1830,0xe03b,0x8313,0x43ea,0xac1e,0x36ef,
0xcd3f,0x3d12,0x8904,0x1870,0x394c,0xd5a2,0x6786,0x544c,},
/*3.2188758248682007492015186664523752790512027085370354438252957829483580E0*/
{0x0000,EXPONE+1,0x0000,0xce02,0x0fbf,0x6c69,0x9b57,0xefbb,
0xd28b,0x03ac,0x98fa,0x24a6,0x3860,0x7b6f,0x9706,0x4285,
0xb156,0x3e0a,0xddcf,0x3881,0xca9f,0x2c5a,0x4a6d,0x6e17,},
/*3.2580965380214820454707195630234951728807680791204623705397255201568583E0*/
{0x0000,EXPONE+1,0x0000,0xd084,0xa757,0x80d6,0x665f,0x2667,
0x64ab,0x20e8,0xcdec,0x9a17,0x88f7,0x1a4a,0x881a,0x04a4,
0x4fb6,0x732a,0x52c6,0x17d8,0x568d,0x4792,0xbc16,0xbcca,},
/*3.2958368660043290741857357107675771139424716734682483552040830009124829E0*/
{0x0000,EXPONE+1,0x0000,0xd2ee,0xfdc0,0x1c24,0x7f18,0xfb13,
0x29ff,0xc3ec,0x9d88,0x4e8f,0x3223,0x49e8,0x6a74,0x2e31,
0xf32f,0xbcf6,0xe829,0x7ff1,0xa21d,0x8994,0xf23c,0x3c55,},
/*3.3322045101752039239398169863595328657880849983023716967007501689243671E0*/
{0x0000,EXPONE+1,0x0000,0xd542,0xd6b4,0xb251,0x1a27,0xa296,
0xff7f,0xcb80,0x7369,0x3387,0x152c,0x4cf3,0x6b15,0x709f,
0xde5d,0x2293,0x2da4,0x8354,0x9a68,0xd8c1,0xe21c,0x4f34,},
/*3.3672958299864740271832720323619116054945129139227440789216703516427808E0*/
{0x0000,EXPONE+1,0x0000,0xd781,0xc65e,0x6ff0,0xf5eb,0x3504,
0xf782,0xd1a6,0x7235,0x43cf,0x47f3,0x52df,0x24d1,0xda1d,
0x9982,0xa486,0x773d,0x5ca7,0x9b0e,0x67bd,0xec7e,0xf77d,},
/*3.4011973816621553754132366916068899122485920464515224277680222346050669E0*/
{0x0000,EXPONE+1,0x0000,0xd9ad,0x37c8,0x5eb4,0xd674,0x935d,
0x3980,0xc421,0xe956,0x5214,0xfdb0,0x4856,0x1c8a,0x932c,
0x1a3c,0x9642,0x7ad5,0xc0d4,0xe38c,0x34d9,0xb670,0xe08d,},
/*3.4339872044851462459291643245423572104499389304805919717567180724749814E0*/
{0x0000,EXPONE+1,0x0000,0xdbc6,0x7244,0x895a,0xcf88,0xb975,
0x11c3,0x69de,0xa09e,0xf517,0xc701,0x05a7,0x22f5,0x32ef,
0x44e8,0x5620,0x3500,0xda3d,0x59d8,0x7d79,0xb3c1,0x165d,},
/*3.4657359027997265470861606072908828403775006718012762706034000474669681E0*/
{0x0000,EXPONE+1,0x0000,0xddce,0x9df5,0xc643,0x5816,0xbc5c,
0xa07e,0x04ef,0xb45b,0x1130,0x13f0,0x0f3e,0xe3b8,0xec90,
0x5d32,0x6e95,0xb8b6,0xe1a6,0x93a8,0x8966,0x97be,0x6afb,},
/*3.4965075614664802354571888148876550044691974117601666269532620427680679E0*/
{0x0000,EXPONE+1,0x0000,0xdfc6,0xc7a6,0xadc6,0x881b,0x8bec,
0x61ba,0x80b8,0x5c46,0xbcee,0xcc43,0x46bb,0xba6d,0x646b,
0x7a96,0x676f,0x882a,0x4434,0xbca8,0x848c,0x7f65,0xc071,},
/*3.5263605246161613896667667393313031036637031469460000414179177473756862E0*/
{0x0000,EXPONE+1,0x0000,0xe1af,0xe40d,0xc86f,0xace2,0xebf5,
0xceab,0xe256,0x1281,0x4601,0xe73c,0xecac,0x2d1e,0x88f1,
0xa4f5,0x1453,0x5ec9,0x3a03,0x4a43,0x8108,0x1bc0,0x0f1c,},
};
#endif
#endif

/*							zetac	*/
/*
 * Riemann zeta function, minus one
 */

extern QELT qone[];
static QELT notwo = 0;
int qzetac();

/* special entry to start summation at 3 instead of 2 */
int qzeta3( x, y )
QELT x[], y[];
{
notwo = 1;
qzetac( x, y );
return 0;
}


int qzetac( x, y )
QELT x[], y[];
{
int i, ia, conv;
long iw;
QELT min;
static QELT a[NQ], b[NQ], k[NQ], s[NQ], w[NQ], t[NQ];
double fabs();
/*double sin(), abs(), gamma(), power();*/

/*
if( x < 0.0 )
	{
	if( x < -30.8148 )
		{
		puts("zetac arg out of range" );
		return(0.0);
		}
	s = 1.0 - x;
	w = zetac( s );
	b = sin(0.5*PI*x) * power(2.0*PI, x) * gamma(s) * (1.0 + w) / PI;
	return(b - 1.0);
	}

if( x >= 127.0 )
	return(0);

i = iround(x);
w = fabs(x - i);
if( w == 0.0 )
	{
	if( i < 31 )
		return( azetac[i] );
	goto pseres;
	}
*/

/*							zetac	*/

/* Euler-Maclaurin summation formula */
/* Accuracy table
 *	iw	bits
 *	10	 75
 *	15	116
 *	17	129
 *	19	139
 *	20	144
 */

iw = 22;		/* w = 32.0 */
ltoq( &iw, w );
qclear( s );		/* s = 0.0  */

if( notwo == 1 )
	ia = 2;
else
	ia = 1;			/* a = 1.0 */


do
	{
	ia += 1;		/* a += 1.0 */
	/* b = power( a, -x ) */
	qmov( &logs[ia-2][0], t );	/* qlog( a, t ) */
	qmul( x, t, t );
	qexp( t, b );
	qdiv( b, qone, b );
	qadd( s, b, s );	/* s += b */
	if( ((int) s[1] - (int) b[1]) > NBITS/2 )
		goto done;
	}
while( ia < iw );	/* while a < w */

qsub( qone, x, t );	/* s += b*w/(x-1.0) */
qdiv( t, w, t );
qmul( t, b, t );
qadd( s, t, s );
qmov( b, t );		/* s -= 0.5 * b */
t[1] -= 1;
qsub( t, s, s );
qmov( qone, a );	/* a = 1.0 */
qclear( k );		/* k = 0.0 */
min = MAXEXP;
conv = 0;
for( i=0; i<30; i++ )
	{
	qadd( x, k, t );	/* a *= x + k */
	qmul( a, t, a );
	qdiv( w, b, b );	/* b /= w */
	qmul( &A[i][0], b, t );	/* t = a*b*A[i] */
	qmul( a, t, t );
	qadd( s, t, s );	/* s = s + t */
	if( t[1] >= min )
		break;
	min = t[1];
	conv = (int) s[1] - (int) t[1];
	if( conv > 144 )
		goto done;
	qadd( qone, k, k );	/* k += 1.0 */
	qadd( x, k, t );	/* a *= x + k */
	qmul( a, t, a );
	qdiv( w, b, b );	/* b /= w */
	qadd( qone, k, k );	/* k += 1.0 */
	}
printf( "zetac %d bits, %d terms\n", conv, i );
done:
qmov( s, y );


/* Basic sum of inverse powers */
/*
pseres:

s = 0.0;
a = 1.0;
do
	{
	a += 2.0;
	b = power( a, -x );
	s += b;
	}
while( b/s > 1.0e-17 );

b = power( 2.0, -x );
s = (s + b)/(1.0-b);
return(s);
*/
return 0;
}
