/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details.  */
#include <jlib.h>


/*+----------------------------------------------------------------------+*/
/*|stamp a sprite frame shadow on buff without clipping                  |*/
/*+----------------------------------------------------------------------+*/
void buff_stamp_sprite_colorNC(sprite_system *ssys, int frame, buffer_rec *obuff,int x, int y, UBYTE col)
{
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *out     = B_OFFSET(obuff,y) + x;
   int    bwidth  = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
          height  = ssys->sprite_data[frame]->height;

   JLIB_ENTER("buff_stamp_sprite_colorNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STAMP_COLOR(out,col,datum);
         out += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|Stamp a sprite frame shadow on buff with clipping.                    |*/
/*+----------------------------------------------------------------------+*/
void buff_stamp_sprite_color(sprite_system *ssys, int frame, buffer_rec *obuff,int x, int y, UBYTE col)
{
   int width, height, bwidth, bheight, clip=0;
   UBYTE *data = (UBYTE *)0;

   JLIB_ENTER("buff_stamp_sprite_color");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   bwidth = B_MAX_X(obuff);
   bheight= B_MAX_Y(obuff);
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       int sprwidth = ssys->sprite_data[frame]->width;
       UBYTE *out;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             if(data[w]){
                out[w] = col;
             }
             else{
                out[w] = 0;
             }
          }
          out  += bwidth;
          data += sprwidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;
    
           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STAMP_COLOR(out,col,datum);
                 out += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_stamp_sprite_colorNC(ssys,frame,obuff,x,y,col);
       }
    }
    JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|Stencil a sprite frame shadow on buff without clipping                |*/
/*+----------------------------------------------------------------------+*/
void buff_stencil_sprite_colorNC(sprite_system *ssys, int frame, buffer_rec *obuff,int x, int y, UBYTE col)
{
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *out     = B_OFFSET(obuff,y) + x;
   int    bwidth  = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
          height  = ssys->sprite_data[frame]->height;

   JLIB_ENTER("buff_stencil_sprite_colorNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STENCIL_COLOR(out,col,datum);
         out += (datum & 15);
      }
      out +=  bwidth;
   }

   JLIB_LEAVE;
}

/*+----------------------------------------------------------------------+*/
/*|Stencil a sprite frame shadow on buff without clipping.               |*/
/*+----------------------------------------------------------------------+*/
void buff_stencil_sprite_color(sprite_system *ssys, int frame, buffer_rec *obuff,int x, int y, UBYTE col)
{
   int width, height, bwidth, bheight, clip=0;
   UBYTE *data = (UBYTE *)0;

   JLIB_ENTER("buff_stencil_sprite_color");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_sprite_system(ssys);
#endif

   bwidth = B_MAX_X(obuff);
   bheight= B_MAX_Y(obuff);
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       int sprwidth = ssys->sprite_data[frame]->width;
       UBYTE *out;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             if(data[w]){
                out[w] = col;
             }
          }
          out  += bwidth;
          data += sprwidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;
    
           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STENCIL_COLOR(out,col,datum);
                 out += (datum & 15);
              }
              out +=  bwidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_stencil_sprite_colorNC(ssys,frame,obuff,x,y,col);
       }
    }
    JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|Stamp a textured sprite frame out of sbuff to obuff without clipping  |*/
/*+----------------------------------------------------------------------+*/
void buff_stamp_sprite_buffNC(sprite_system *ssys,int frame,buffer_rec *obuff,int x, int y,buffer_rec * sbuff)
{
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *out     = B_OFFSET(obuff,y) + x,
         *texture = B_BUFF_PTR(sbuff);
   int bwidth  = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       twidth  = B_X_SIZE(sbuff) - ssys->sprite_data[frame]->width,
       height  = ssys->sprite_data[frame]->height;

   JLIB_ENTER("buff_stamp_sprite_buffNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STAMP_COPY(out,texture,datum);
         out += (datum & 15);
         texture += (datum & 15);
      }
      out +=  bwidth;
      texture += twidth;
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|Stamp a textured sprite frame out of sbuff to obuff with clipping     |*/
/*+----------------------------------------------------------------------+*/
void buff_stamp_sprite_buff(sprite_system *ssys,int frame,buffer_rec *obuff,int x, int y,buffer_rec * sbuff)
{
   int twidth, width, height, bwidth, bheight, clip=0;
   UBYTE *data = (UBYTE *)0, *texture;

   JLIB_ENTER("buff_stamp_sprite_buff");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_buffer(sbuff);
   jlib_check_sprite_system(ssys);
#endif

   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   twidth  = B_X_SIZE(sbuff);
   texture = B_BUFF_PTR(sbuff);

#ifndef JLIB_PRODUCTION
   /* the texture must be big enough to cover the sprite */
   if((B_MAX_X(sbuff) < width) || (B_MAX_Y(sbuff) < height)){
      jlib_exit("Texture too small");
   }
#endif

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      texture -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       int sprwidth = ssys->sprite_data[frame]->width;
       UBYTE *out;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
             texture += twidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             if(data[w]){
                out[w] = texture[w];
             }
             else{
                out[w] = 0;
             }
          }
          out  += bwidth;
          data += sprwidth;
          texture += twidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
             texture += twidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;
           twidth -= ssys->sprite_data[frame]->width;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STAMP_COPY(out,texture,datum);
                 out += (datum & 15);
                 texture += (datum & 15);
              }
              out +=  bwidth;
              texture += twidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_stamp_sprite_buffNC(ssys,frame,obuff,x,y,sbuff);
       }
    }
    JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Stencil a textured sprite frame out of sbuff to obuff without clipping| */
/*+----------------------------------------------------------------------+ */
void buff_stencil_sprite_buffNC(sprite_system *ssys,int frame,buffer_rec *obuff,int x, int y,buffer_rec * sbuff)
{
   UBYTE *pattern = ssys->sprite_data[frame]->pattern,
         *out     = B_OFFSET(obuff,y) + x,
         *texture = B_BUFF_PTR(sbuff);
   int bwidth  = B_X_SIZE(obuff) - ssys->sprite_data[frame]->width,
       twidth  = B_X_SIZE(sbuff) - ssys->sprite_data[frame]->width,
       height  = ssys->sprite_data[frame]->height;

   JLIB_ENTER("buff_stencil_sprite_buffNC");

   while(height--){
      int width = *pattern++;
      while(width--){
         UBYTE datum = *pattern++;
         SPR_STENCIL_COPY(out,texture,datum);
         out += (datum & 15);
         texture += (datum & 15);
      }
      out +=  bwidth;
      texture += twidth;
   }

   JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+*/
/*|Stencil a textured sprite frame out of sbuff to obuff with clipping   |*/
/*+----------------------------------------------------------------------+*/
void buff_stencil_sprite_buff(sprite_system *ssys,int frame,buffer_rec *obuff,int x, int y,buffer_rec * sbuff)
{
   int twidth, width, height, bwidth, bheight, clip=0;
   UBYTE *data = (UBYTE *)0, *texture;

   JLIB_ENTER("buff_stencil_sprite_buff");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(obuff);
   jlib_check_buffer(sbuff);
   jlib_check_sprite_system(ssys);
#endif

   bwidth  = B_MAX_X(obuff);
   bheight = B_MAX_Y(obuff);
   width   = ssys->sprite_data[frame]->width;
   height  = ssys->sprite_data[frame]->height;
   twidth  = B_X_SIZE(sbuff);
   texture = B_BUFF_PTR(sbuff);

#ifndef JLIB_PRODUCTION
   /* the texture must be big enough to cover the sprite */
   if((B_MAX_X(sbuff) < width) || (B_MAX_Y(sbuff) < height)){
      jlib_exit("Texture too small");
   }
#endif

   /* x clipping */
   if(x + width > bwidth){
      width = bwidth - x + 1;
      clip |= C_RIGHT;
   }
   else{
      if(x + width < 0){
        JLIB_LEAVE;
        return;
      }
   }

   if(x < 0){
      width += x;
      data  -= x;
      texture -= x;
      x = 0;
      clip |= C_LEFT;
   }
   else{
      if((x > bwidth)){
        JLIB_LEAVE;
        return;
      }
   }


   if (clip) {
       /* worst case: x is clipped */
       int sprwidth = ssys->sprite_data[frame]->width;
       UBYTE *out;
       data += (unsigned int)ssys->sprite_data[frame]->data;

       bwidth++;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             data += sprwidth;
             texture += twidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }

       out = B_OFFSET(obuff,y) + x;
   
       while(height--){
          int w = width;
          while(w--){
             if(data[w]){
                out[w] = texture[w];
             }
          }
          out  += bwidth;
          data += sprwidth;
          texture += twidth;
       }
       JLIB_LEAVE;
       return;
   }
   else{
       UBYTE *pattern = ssys->sprite_data[frame]->pattern;

       /* y clipping */
       if(y + height > bheight){
          height = bheight - y + 1;
          clip |= C_DOWN;
       }
       else{
          if(y + height < 0){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(y < 0){
          height += y;
          do{
             pattern += *pattern + 1; /* skip through RLE pattern */
             texture += twidth;
          }while(++y != 0);
          clip |= C_UP;
       }
       else{
          if(y > bheight){
            JLIB_LEAVE;
            return;
          }
       }
    
       if(clip){
           /* average case: y is clipped */
           UBYTE *out = B_OFFSET(obuff,y) + x;
           bwidth = bwidth - width + 1;
           twidth -= ssys->sprite_data[frame]->width;

           while(height--){
              int w = *pattern++;
              while(w--){
                 UBYTE datum = *pattern++;
                 SPR_STENCIL_COPY(out,texture,datum);
                 out += (datum & 15);
                 texture += (datum & 15);
              }
              out +=  bwidth;
              texture += twidth;
           }
       }
       else{
           /* best case: no clipping */
           buff_stencil_sprite_buffNC(ssys,frame,obuff,x,y,sbuff);
       }
    }
    JLIB_LEAVE;
}
