/*
 * This file is part of the M-Edit program (Copyright 1996 by Vincenzo Morello)
 * Permission to use is hereby granted for any purpose.
 *
 * Contains customization callbacks.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "mgui.h"
#include "medit.h"
#include "me.h"

#ifdef UNIX
#define RMODE	"r"
#define WMODE	"w"
#else
#define RMODE	"rb"
#define WMODE	"wb"
#endif

extern MOBJECT windows[MAX_N_WINDOW];

PREFERENCES default_pref =
{
    {HELV_MEDIUM, 0xa0, 0xc0, 0x90, 0, 0, 0},
    {FIXED_MEDIUM, 0xc5, 0xc5, 0x90, 0, 0, 0},
    {FIXED_MEDIUM, 0x35, 0x70, 0x70, 0xe0, 0xe0, 0xb0},
    False, 80, 20, 8
};

PREFERENCES prefs;

void 
LoadPrefs(void)
{
    int val;

    if (MGetIntOption("MEdit", "autoIndent", &val))
	default_pref.auto_indent = val;
    if (MGetIntOption("MEdit", "tabSize", &val))
	default_pref.tab_size = val;
    if (MGetIntOption("MEdit", "initialWidth", &val))
	default_pref.initial_width = val;
    if (MGetIntOption("MEdit", "initialHeight", &val))
	default_pref.initial_height = val;
    prefs = default_pref;
}

static void 
SetPrefs(PREFERENCES * pp)
{
    char str[32];

    MTButtonSetActivationStatus(auto_indent_tb, pp->auto_indent);

    sprintf(str, "%d", pp->initial_width);
    MObjectSetText(init_w_e, str);
    sprintf(str, "%d", pp->initial_height);
    MObjectSetText(init_h_e, str);
    sprintf(str, "%d", pp->tab_size);
    MObjectSetText(tab_size_e, str);
}

void 
SetWindowOptions(MOBJECT shell)
{
    FILE_DATA *pfd;

    pfd = (FILE_DATA *) MObjectGetUserData(shell);
    MMenuItemSetCheckStatus(pfd->auto_indent,
			prefs.auto_indent);
    MEditSetAutoIndent(pfd->edit, prefs.auto_indent);
    MEditSetTabSize(pfd->edit, prefs.tab_size);
    MEditSetClipWindow(pfd->edit, prefs.initial_width, prefs.initial_height);
}

void 
PickOptions(void)
{
    char str[32];

    prefs.auto_indent = MTButtonGetActivationStatus(auto_indent_tb);
    MObjectGetText(init_w_e, str);
    prefs.initial_width = atoi(str);
    MObjectGetText(init_h_e, str);
    prefs.initial_height = atoi(str);
    MObjectGetText(tab_size_e, str);
    prefs.tab_size = atoi(str);
}

void 
CustomizeCB(MOBJECT p, void *od, void *ad)
{
    MOBJECT shell;

    shell = MDCreatecustom_shell();

    SetPrefs(&prefs);

    MShellRealize(shell);
}

void
OptionsOkCB(MOBJECT p, void *od, void *ad)
{
    int ii;

    PickOptions();
    for (ii = 0; ii < MAX_N_WINDOW; ii++)
	if (windows[ii] != NULL)
	{
	    MObjectUnmap(windows[ii]);
	    SetWindowOptions(windows[ii]);
	    MObjectMap(windows[ii]);
	}
    MShellDestroy(p);
}

void
OptionsSaveCB(MOBJECT p, void *od, void *ad)
{
    PickOptions();
    MSetIntOption("MEdit", "autoIndent", prefs.auto_indent);
    MSetIntOption("MEdit", "tabSize", prefs.tab_size);
    MSetIntOption("MEdit", "initialWidth", prefs.initial_width);
    MSetIntOption("MEdit", "initialHeight", prefs.initial_height);
    MSaveOptions();
}

void 
OptionsDefaultCB(MOBJECT p, void *od, void *ad)
{
    SetPrefs(&default_pref);
}

void 
OptionsCancelCB(MOBJECT p, void *od, void *ad)
{
    MShellDestroy(p);
}

void 
ResetStatusLineCB(TIMEOUT_ID id, void *ud)
{
    MObjectResize(ud, -1, -1);
    MObjectMap(ud);
}

/*
 * This callback is called when the filename label is resized.
 * If it occurs due to a font change, then the new font is
 * also set to all other status labels.
 */

void 
StatusLineChangeCB(MOBJECT obj, MEvent * pe, void *ad)
{
    FILE_DATA *pfd = (FILE_DATA *) ad;
    MOBJECT parent = MObjectParent(obj);

    if (pe->type == E_RESIZE && pe->resize.height != MObjectHeight(pfd->modified_label))
    {
	MTFont font = MObjectGetFont(obj);

	MObjectUnmap(parent);
	MObjectSetFont(pfd->modified_label, font, False);
	MObjectSetFont(pfd->curs_x_label, font, False);
	MObjectSetFont(pfd->curs_y_label, font, False);
/*
 * We must force a resize in the status line form (parent)
 * when this catched event has been processed by the object
 */
	MAddTimeout(1L, ResetStatusLineCB, parent);
    }
    else if (pe->type == E_SET_COLOR)
    {
	MTColor bgc = MObjectBackgroundColor(obj);
	MTColor fgc = MObjectForegroundColor(obj);

	MObjectSetColor(pfd->modified_label, bgc, fgc);
	MObjectSetColor(pfd->curs_x_label, bgc, fgc);
	MObjectSetColor(pfd->curs_y_label, bgc, fgc);
    }
}
