/*
 * This file is part of the M-Edit program (Copyright 1996 by Vincenzo Morello)
 * Permission to use is hereby granted for any purpose.
 *
 * Contains customization callbacks.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "mguipp.h"
#include "medit.hpp"

#ifdef UNIX
#define RMODE	"r"
#define WMODE	"w"
#else
#define RMODE	"rb"
#define WMODE	"wb"
#endif

PREFERENCES default_pref =
{
    {HELV_MEDIUM, 0xa0, 0xc0, 0x90, 0, 0, 0},
    {FIXED_MEDIUM, 0xc5, 0xc5, 0x90, 0, 0, 0},
    {FIXED_MEDIUM, 0x35, 0x70, 0x70, 0xe0, 0xe0, 0xb0},
    False, 80, 20, 8
};

PREFERENCES prefs;

void
LoadPrefs(void)
{
    int val;

    if (MGetIntOption("MEdit", "autoIndent", &val))
	default_pref.auto_indent = (unsigned char)val;
    if (MGetIntOption("MEdit", "tabSize", &val))
	default_pref.tab_size = (unsigned char)val;
    if (MGetIntOption("MEdit", "initialWidth", &val))
	default_pref.initial_width = (unsigned char)val;
    if (MGetIntOption("MEdit", "initialHeight", &val))
	default_pref.initial_height = (unsigned char)val;
    prefs = default_pref;
}

void Ccustom_shell::setPrefs(PREFERENCES * pp)
{
    char str[32];

    if (pp->auto_indent)
	auto_indent_tb->arm();
    else
	auto_indent_tb->disarm();

    sprintf(str, "%d", pp->initial_width);
    init_w_e->setText(str);
    sprintf(str, "%d", pp->initial_height);
    init_h_e->setText(str);
    sprintf(str, "%d", pp->tab_size);
    tab_size_e->setText(str);
}

void Cmain_shell::setWindowOptions(void)
{
    auto_indent_mi->setCheckStatus(prefs.auto_indent);
    text_mle->setAutoIndent(prefs.auto_indent);
    text_mle->setTabSize(prefs.tab_size);
    text_mle->setClipWindow(prefs.initial_width, prefs.initial_height);
}

void Ccustom_shell::pickOptions(void)
{
    char str[32];

    prefs.auto_indent = (unsigned char)auto_indent_tb->isArmed();
    init_w_e->getText(str);
    prefs.initial_width = (unsigned char)atoi(str);
    init_h_e->getText(str);
    prefs.initial_height = (unsigned char)atoi(str);
    tab_size_e->getText(str);
    prefs.tab_size = (unsigned char)atoi(str);
}

void Ccustom_shell::OptionsOkCB(void)
{
    pickOptions();
    for (Cmain_shell *ms=owner->getFirst(); ms; ms=ms->getNext())
    {
	ms->unmap();
	ms->setWindowOptions();
	ms->map();
    }
    delete this;
}

void Ccustom_shell::OptionsSaveCB(void)
{
    pickOptions();
    MSetIntOption("MEdit", "autoIndent", prefs.auto_indent);
    MSetIntOption("MEdit", "tabSize", prefs.tab_size);
    MSetIntOption("MEdit", "initialWidth", prefs.initial_width);
    MSetIntOption("MEdit", "initialHeight", prefs.initial_height);
    MSaveOptions();
    delete this;
}

void Ccustom_shell::OptionsDefaultCB(void)
{
    setPrefs(&default_pref);
    delete this;
}

void Ccustom_shell::OptionsCancelCB(void)
{
    delete this;
}

void Cmain_shell::CustomizeCB(int)
{
    Ccustom_shell *shell = new Ccustom_shell(this);
    shell->setPrefs(&prefs);
    shell->realize();
}

void Cmain_shell::resetStatusLineCB(void)
{
    CmObject *parent = filename_l->getParent();
    parent->resize(-1, -1);
    parent->map();
    delete reset_tmt;
}

/*
 * This callback is called when the filename label is resized.
 * If it occurs due to a font change, then the new font is
 * also set to all other status labels.
 */

void Cmain_shell::statusLineChangeCB(MEvent *pe)
{
    CmObject *parent = filename_l->getParent();

    if (pe->type == E_RESIZE && pe->resize.height != modified_l->getHeight())
    {
	MTFont font = filename_l->getFont();

	parent->unmap();
	modified_l->setFont(font, False);
	curs_x_l->setFont(font, False);
	curs_y_l->setFont(font, False);
/*
 * We must force a resize in the status line form (parent)
 * when this catched event has been processed by the object
 */
	reset_tmt = new CmTimeout(this, (VOID_CB)&Cmain_shell::resetStatusLineCB, 1L);
    }
    else if (pe->type == E_SET_COLOR)
    {
	CmColor *bgc = filename_l->getBackgroundColor();
	CmColor *fgc = filename_l->getForegroundColor();

	modified_l->setColor(bgc, fgc);
	curs_x_l->setColor(bgc, fgc);
	curs_y_l->setColor(bgc, fgc);
    }
}
