#ifndef __SLANG_VARIABLE_H_
#define __SLANG_VARIABLE_H_

#include "simple.h"
#include "kh_error.h"
#include <alloc.h>
#include <string.h>

/*  Variables are ordered in "memory" by addresses. C(HAR), I(NT), L(ONG),
    D(OUBLE) and P(OINTER) are default types, USER_TYPE is user-defined
    type, for example, struct. KNOW-HOW work only in LARGE model, so sizeof(P)
    is 4 bytes. If variable is marked-to-remove,
    adresses[n] -> -adresses[n] - 1 (< 0 in any case). 
    In this case types[n] keep size of this marked area (positive).
    To use pointers arithmetic, Slang should have information
    about type of elements of array. To distinguish between TYPE and pointer
    to this type we use sign of types[n]:

    int type, size;
    if(types[n] > 0)
	type = size = types[n];
    else
	{ type = P; size = -types[n]; }
    There are two types of arrays: allocated by "new" and pointed in from
    "memory" and arrary in "memory". C++ analog:
    char* s = new char[2];    // In heap
    char s[2];                // In stack
    In Slang in first case types[n] == -C, and memory[arrays[n]] points to
    the array. Size of array element is C, so when we mark it to remove,
    we know how many bytes in "memory" we left. We could call delete to
    delete space occupated by this array.
    In second case types[n] == -C - USER_TYPE, memory[arrays[n]] is not
    pointer but first element of the array. We could not use types[n] to
    determine size of "memory" occupied by array, but we could use
    arrays[n + 1] - arrays[n].
    Array of pointers is the same as array of long (types[n] == -L-USER_TYPE)
    or types[n] == -P).
    If we add array (not pointer) to the memory, we must know the number of
    the elements to copy. If we remove it, we must know the size of free
    memory we get. It is always possible because we have "memory_used"
    for last variable and adresses[n+1] - adresses[n] for others.

    Current version does not support user-defined types.
*/

#define MEMORY_STEP 32//1024
#define ADRESS_STEP 4//64

enum { FREE = 0, C = 1, I = 2, L = 4, D = 8, P = 4, USER_TYPE = 16,
       PC = -C-USER_TYPE, PI = -I-USER_TYPE, PL = -L-USER_TYPE,
       PD = -D-USER_TYPE };

class Var_Table
    {
    protected:
	long memory_used;
	long memory_size;
	char far* memory;         // Area in memory where Slang keep data

	int size;
	int used;
	long* adresses;          // Adress of n-th variable is adresses[n]
	int* types;               // Types of variables FILE

	int marked;               // Num. of marked_to_remove
	long mem_marked;         // Size of fragmented memory

	int type;                 // Last call to get_type() function set
	int size_of;              // this two variables

	int get_type(int n);      // Set type and size variables. Return type.
	int get_size(int n);      // Return size in "memory"
	int check_memory_expand();    // Expand memory if necessary
	int check_memory_compress();  // Remove marked variables if necessary

    public:
	Var_Table();
	~Var_Table();

    // Add to position n or to the end variable of type t. Return n (OK) or -1
	int add(void* v, int t, int n = -1, uint elems = 1);
    // Remove variable from the list (and dump if no memory)
    // Return pointer if type is P or NULL. If area is marked-to-remove,
    // types[n] keep size of this area. !!! Pointer to NULL is legal too,
    // but in this case types[n] < 0 !!!
	void* remove(int n, bool check = ON);
	long find(int n);              // Set type and size, return address
	int memory_compress();         // Remove marked variables
    };

#endif __SLANG_VARIABLE_H_
