***********************************************
* Program   :
* Programmer: Tony Papadimitriou
* Purpose   :
* Language  : MC68HC11 (ASM11 v1.48+)
* History   : 99.xx.xx Version 1.00    Original
***********************************************
DEBUG     ;Comment this line for production

$EXTRAON       -- Allow extra mnemonics
$OPTRELON      -- Show optimization warnings for JMP/JSR
$OPTRTSOFF     -- Do not show optimization warnings for JSR/BSR, RTS
$SPACESOFF     -- Do not allow spaces within expressions (v1.48 or later)

$IFDEF DEBUG
  $MESSAGE Using DEBUG/SIMULATOR Mode (NOT for burning device)
$ENDIF

$LISTOFF -- Turn off listing of equates
$INCLUDE 711E9.INC  -- MC68HC711E9 equates
$INCLUDE COMMON.INC -- MCU-independent equates
$LISTON  -- Turn listing back on for main program


*********************************************************************
*                        LOCAL EQUATES                              *
*********************************************************************
SECOND_DELAY        equ       122                 One second delay at 8.2ms RTI
$ifdef DEBUG
BPS_RATE            equ       125                 for SIM11x debugging, use maximum speed
$else
BPS_RATE            equ       96                  9600 bps for the SCI
$endif



*********************************************************************
*                 USER RAM AND STACK ORGANIZATION                   *
*********************************************************************
                    org       RAM
timer               rmb       1                   sample used by RTI
;
; ------------- add your global variables here ----------------------
;



*********************************************************************
*                           PROGRAM CODE                            *
*********************************************************************
                    org       ROM

; The following is a standard block of information to be found in every program
Prog.Name           db        'xxxxxxxx'          Program Name
Prog.Author         db        'xxxxxxxxxxxxxxxxx' Programmer's Name
Prog.Company        db        'xxxxxxxxxxxxx'     Company name
Prog.Version        db        1,0                 Major, minor version numbers
Prog.Date           db        $99,$01,$01         Date in YY,MM,DD in BCD format

Start               lds       #STACKTOP           Initialize the Stack Pointer
                    bsr       InitMCU             Initialize the MCU
                    bsr       InitUserVars        Initialize User variables
;
; ---------------- add more instructions here ------------------------
;
; Sample code (to be removed) follows
                    os        fPrint
                    fcs       'Program begins...',CR,LF
; Sample main loop
                    lda       #SECOND_DELAY       set up a ~1 second delay
                    sta       timer
MainLoop            wait                          Enable ints and wait for one
                    os        fPrint              Display a dot after each interrupt
                    fcs       '.'
                    bsr       ServiceRequests     Service each request in detail
                    bra       MainLoop

ServiceRequests     equ       *                   Sample Routine
                    tst       timer               is timer expired?
                    bne       ServiceRequests.Out no, get out of here
                    nop                           yes, do some work
                    nop                           ..like a few NOPs, if you can't
                    nop                           ..think of something better
                    os        fNewLine
                    os        fPrint              Display a message when timer expires
                    fcs       '[Timer Reset]'
                    os        fNewLine
                    lda       #SECOND_DELAY       reset timer
                    sta       timer
                    os        255                 FORCE ILLEGAL OPCODE TRAP TEST
ServiceRequests.Out rts
;
; ---------------- add more instructions here ------------------------
;
Halt_MCU            sei                           Disable interrupts
; --- Here you could add code to shut down all peripherals (on and off chip)
; --- such as the A/D, SCI, SPI, external chips, etc.
                    clr       OPTION              Power-down A/D
                    clr       SCCR2               Disable SCI
                    clr       SPCR                Disable SPI
                    clr       TMSK1               Disable IC/OC interrupts
                    clr       TMSK2               Disable Timer interrupts
;
                    cls                           Enable STOP instruction
$ifdef DEBUG
                    bra       *                   Some emulators can't STOP
$else
                    stop                          and stay here until reset
                    bra       *-1                 in case of interrupt stay here
$endif

$mapoff -- Turn mapping off
InitUserVars        equ       *
; first, clear all RAM except for stacked data
                    pshx
                    tsx
                    dex
                    clr       ,x
                    dex
                    bne       *-3
                    clr       ,x
                    pulx
;
; next, set specific non-zero values
;
; ---------------- add your code here --------------------------------
;
                    rts

*****
* InitMCU - Initialize the MCU
*****
InitMCU             equ       *
; --- NOTE: The following STAs must occur within next 53 cycles (64 cycle-limit)

     ;Setup INIT (even the default reset value is required if using a PRU)
                    lda       #REGS>12            Changing REGS will affect this line!
     ;$103D must be hardcoded or it won't work, do NOT use INIT because it may not be $103D
                    sta       $103D               RAM and I/O Mapping

                    lda       #IRQE.              IRQ edge triggered (use your own value here)
                    sta       OPTION              System Configuration Options

                    lda       #$03                (use your own value here)
                    sta       TMSK2               Timer Interrupt Mask 2
; --- NOTE: The following STAs will affect write-once register bits (even outside 64-cycle limit)
                    ldx       #REGS
                    bset      [HPRIO,x,Bit4.      Turn off E-clock output (for single chip mode)
; ---
$IFDEF DEBUG
                    clrd                          SIM11x doesn't like uninitialized regs
                    clrx
                    clry
$ENDIF
                    bsr       InitPortDirs
                    bsr       InitRTI
                    bsr       InitSCI
                    bsr       InitSPI
                    bsr       InitAD
                    rts

*********************************************************************

; --- Initialize Port Directions and status (where applicable)
InitPortDirs        equ       *
                    ldx       #REGS
; first, zero all port data
                    clr       PORTA
                    clr       PORTB
                    clr       PORTC
                    clr       PORTD
                    clr       PORTE
; next, set port pin directions
                    clr       PACTL
                    bset      [PACTL,x,#%00000000 use appropriate value
       * only these two bits may be set ^___^___ for PA7 and PA3
                    lda       #%00000000
                    sta       DDRC
                    lda       #%00000000          (use your own value here)
                    sta       DDRD
                    rts

; --- Initialize the Real-Time Interrupt (8.2ms RTI @ 2MHz MCU E-Clock)
InitRTI             equ       *
                    bclr      [PACTL,x,3          (use your own value here)
                    bset      [TMSK2,x,$40        Enable RTI subsystem
                    rts

; --- Initialize the Serial Communications Interface for 9600bps, 8-N-1
;     (Using polled mode, no interrupts; change fSetBAUD call for other needs)
InitSCI             equ       *
                    lda       #BPS_RATE
                    clc
                    os        fSetBAUD
                    rts

; --- Initialize the Synchronous Peripheral Interface
InitSPI             equ       *
                    lda       #$14                (use your own value here)
                    sta       SPCR
                    rts

; --- Initialize the Analog-to-Digital Converter
InitAD              equ       *
                    lda       #$00                (use your own value here)
                    sta       ADCTL
                    rts
;$mapon -- Turn mapping back on (remove comment to enable this directive)

*********************************************************************
*                     INTERRUPT SERVICE ROUTINES                    *
*********************************************************************



**********************************************************************
* SWI routines table (OpCodes $00 to $FF may be used for 8-bit mode) *
*       (Opcodes $0000 to $7FFF may be used for 16-bit mode)         *
* They must be listed in the table in sequence from lowest to highest*
* If a number must be left unused in between, point it to Op_Unused  *
* When calling with OS or OSW, use the fXXX name, eg.  OS fPrint     *
* You may change the order of the calls as long as you move both EQU *
* and related DW lines together.  They are separated with spaces for *
* easy identification.                                               *
**********************************************************************
OSCommands          equ       *

fDelayMS            equ       *-OSCommands/2
                    dw        DelayMS             Delay variable number of ms

fDelay10MS          equ       *-OSCommands/2
                    dw        Delay10MS           Delay 10ms

fCopyMem            equ       *-OSCommands/2
                    dw        CopyMem             Copy a block of memory

fSetBAUD            equ       *-OSCommands/2
                    dw        SetBAUD             Set SCI bps ("baud") rate

fPrint              equ       *-OSCommands/2
                    dw        Print               ..the following FCS string

fBin2Dec            equ       *-OSCommands/2
                    dw        Bin2Dec             Binary to Decimal conversion

fBin2BCD            equ       *-OSCommands/2
                    dw        Bin2BCD             Binary to BCD conversion (actually any base)

fWrite              equ       *-OSCommands/2
                    dw        Write               Write a "Pascal" string to SCI

fWriteln            equ       *-OSCommands/2
                    dw        Writeln             fWrite followed by CR,LF pair

fNewLine            equ       *-OSCommands/2
                    dw        NewLine             Send a CR,LF pair to the SCI

fUpString           equ       *-OSCommands/2
                    dw        UpString            Convert ASCIIZ string to uppercase

fUpChar             equ       *-OSCommands/2
                    dw        UpChar              Convert char in RegA to uppercase

fWriteEE            equ       *-OSCommands/2
                    dw        WriteEE             Write one EEPROM byte

fEraseEE            equ       *-OSCommands/2
                    dw        EraseEE             Erase one EEPROM byte

fBulkEraseEE        equ       *-OSCommands/2
                    dw        BulkEraseEE         Bulk Erase EEPROM

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;?fCallName?        equ       *-OSCommands/2      Keep adding more functions
;                   dw        ?Routine_Name?      ..like with these two lines
;
;                   dw        Op_Unused           This is one line only (no EQU needed)
;
;?fCallName?        equ       *-OSCommands/2      Another two line entry
;                   dw        ?Routine_Name?      ..for another call
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

OSCommands.Entries  equ       *-OSCommands/2      Number of opcodes defined

$if OSCommands.Entries > $FF
OS_16BIT            ;Use (16 bit) OSW instruction, rather than (8-bit) OS
$MESSAGE OS calls are 16-bit (make sure to use OSW instruction)
$endif



**********************************************************************
*                        OS Error Codes                              *
*            (Any number from 0 to 255 may be defined)               *
**********************************************************************
errNone             equ       0                   No Error
errUnused           equ       1                   Opcode unused
errBadParm          equ       2                   Bad parameter
errFailure          equ       3                   Operation failed
errOutOfRange       equ       4                   Value out of range
errOther            equ       5                   Other error




**********************************************************************
*                       OS CALLS COME HERE                           *
**********************************************************************
* Call format: OS CallName (OSW CallName for 16-bit opcodes)         *
*            : Parameters are passed in registers depending on call  *
*            : Results either in memory or regs depending on call    *
*            : No registers (other than results) are destroyed       *
* Returns    : No Carry if no error, B is unaffected                 *
*            : Carry if error, B holds error code                    *
* Note(s)    : Each subcall should get parameters from Reg??,y and   *
*            : return results in Reg??_,y and error code in B_,y     *
*            : EXCEPT for A,B, (or D) which could be used directly,  *
*            : User need not save the Y index as it is saved by the  *
*            : dispatcher.                                           *
* Gen Reg Use: Generally, registers used have a similar role in each *
*            : routine.                                              *
*            : X is used as a destination memory pointer             *
*            :   (if only one pointer is used, it is X, not Y)       *
*            : Y is used as a source memory pointer                  *
*            : D (or just B if dealing with 8-bit quantities) is     *
*            :   used as a counter.                                  *
*            : This arrangement may be violated when necessary, for  *
*            : example, if there is a need for one pointer and two   *
*            : counters, X will be the pointer, D and Y the primary  *
*            : and secondary counters, respectively.                 *
* Total Stack: 13 (9 for caller + 4 for local storage without subs)  *
**********************************************************************
* OS Command Dispatcher.  Based on function, it calls appropriate subroutine.
SWI_Handler         tsy                           point Y to SWI stack frame
                    ldx       PC_,y               adjust PC for opcode
$ifdef OS_16BIT
                    ldd       ,x                  OSW type calls
                    inx                           skip 2nd opcode byte
$else
                    clra                          we're gonna use D (for OS calls)
                    ldb       ,x                  get OS opcode byte
$endif
                    inx                           skip 1st opcode byte
                    stx       PC_,y               and save new PC to return to
                    cpd       #OSCommands.Entries is it a valid OpCode?
                    bhs       SWI_Ret
                    lsld                          multiply by 2 (the size of table entries)
                    addd      #OSCommands
                    xgdx                          put result in X
                    ldx       ,x                  get jmp vector
                    cpx       #Op_Unused          is it an invalid OpCode?
                    beq       SWI_Ret             ..if so, go handle the exception
                    lda       A_,y                get D for user
                    ldb       B_,y
                    psha
                    lda       CCR_,y              and CCR
                    tap
                    pula
                    clc
                    pshy
                    cli                           Enable interrupts (optional) -.
                    jsr       ,x                  Perform function              |
                    sei                           Disable interrupts (optional)-'
                    puly
;now, we should prepare the stack frame to be returned with appropriate values
                    bcs       SWI_Error
                    bclr      CCR_,y,C.           clear Carry flag
                    bra       SWI_NoError
SWI_Error           bset      CCR_,y,C.           set Carry flag
                    stb       B_,y                save error code for user
SWI_NoError         rti
* The following section handles invalid System Opcodes *
SWI_Ret             ldx       PC_,y               adjust PC to start of opcode
                    dex                           ..back one byte for operand
                    dex                           ..and one for the OS (SWI) opcode
                    stx       PC_,y               and save new PC to return to
                    ldx       ILLOP_VECTOR
                    cpx       #ERASED_STATE       Check if not initialized (for EPROM, ie., $FFFF)
                    beq       SWI_Ret.2
                    jmp       ,x                  Invalid System Call
SWI_Ret.2           jmp       Halt_MCU            In case ILLOP handler not installed

**********************************************************************
*                 Operating System routines expanded                 *
**********************************************************************

Op_Unused           sec                           Dummy call for missing opcodes
                    ldb       #errUnused          ..always returns error
AnRTS               rts                           ..use your own non-zero error codes

$ifdef fDelayMS
; Purpose: Delay ~ RegX milliseconds (not counting caller's overhead)
DelayMS             ldx       X_,y                Get milliseconds to delay
DelayMS.01          pshx
                    ldx       #333                constant for 1 millisecond
                    dex
                    bne       *-1
                    pulx
                    dex
                    bne       DelayMS.01
                    clc                           No errors in any case
                    rts
$endif

$ifdef fDelay10MS
; Purpose: Delay 10 msec (20000 cycles @500nsec/cycle assuming 2MHz E clock)
; Note(s): All involved registers are preserved so it can be called internally
;          This routine executes in 20003 cycles from entry (including BSR or
;          JSR instruction) to completion of RTS
;          Because of the BNE loop, on exit Carry is always Clear = No Error
Delay10MS           equ       *
                    pshx                          4 cycles
                    ldx       #3330               3 cycles
Delay10MS.01        dex                           3 cycles
                    bne       Delay10MS.01        3 cycles
                    pulx                          5 cycles
                    rts                           5 cycles
$endif

$ifdef fCopyMem
; Purpose: Copy (move) bytes from source to destination memory
; Input  : D holds number of bytes to copy
;        : Y points to start of source memory
;        : X points to start of destination memory
; Note(s): Correctly handles copies (even partially) to EEPROM
CopyMem             pshy
                    ldx       X_,y                load parameters (D is OK)
                    ldy       Y_,y
                    cpd       #0                  any bytes to copy?
                    beq       CopyMem.NoError     no, get out of here
CopyMem.Loop        pshd                          save counter between loops
                    cpx       #EEPROM             check destination for EEPROM
                    blo       CopyMem.RAM
                    cpx       #EEPROM_END
                    bhi       CopyMem.RAM
CopyMem.EEPROM      lda       ,y                  get a byte from source
                    jsr       WriteEE.Local       write EEPROM byte A to ,X
                    tstb                          was there an error?
                    beq       CopyMem.LoopEnd     no, continue
                    bra       CopyMem.Error
CopyMem.RAM         lda       ,y                  get byte from source
                    sta       ,x                  put byte to destination
CopyMem.LoopEnd     inx                           bump up both pointers
                    iny
                    puld                          get loop counter
                    decd                          and count down
                    bne       CopyMem.Loop
                    bra       CopyMem.NoError
CopyMem.Error       puld
                    puly
                    ldb       #errFailure
                    sec
                    bra       CopyMem.Exit
CopyMem.NoError     puly
                    clc
CopyMem.Exit        rts
$endif

$ifdef fSetBAUD
; Purpose: Set SCI BAUD rate to one of the following values (in SWI_D,y)
;          3(00), 12(00), 24(00), 48(00), 96(00), 125(000)
;          (use number outside parentheses for corresponding baud rate)
; Input  : RegA = Baud Rate
;        : RegB = Length of buffer (required only for interrupt mode)
;        : RegX = Pointer to buffer to use (required only for interrupt mode)
;        : Carry = Set when we want interrupt driven mode
;        : Carry = Clear when we want polling mode
SetBAUD             lda       A_,y
                    cmpa      #3                  is it 300bps?
                    bne       SetBAUD.1200
                    lda       #$35
                    bra       SetBAUD.SetBaud
SetBAUD.1200        cmpa      #12                 is it 1200bps?
                    bne       SetBAUD.2400
                    lda       #$33
                    bra       SetBAUD.SetBaud
SetBAUD.2400        cmpa      #24                 is it 2400bps?
                    bne       SetBAUD.4800
                    lda       #$32
                    bra       SetBAUD.SetBaud
SetBAUD.4800        cmpa      #48                 is it 4800bps?
                    bne       SetBAUD.9600
                    lda       #$31
                    bra       SetBAUD.SetBaud
SetBAUD.9600        cmpa      #96                 is it 9600bps?
                    bne       SetBAUD.125000
                    lda       #$30
                    bra       SetBAUD.SetBaud
SetBAUD.125000      cmpa      #125                is it 125000bps?
                    bne       SetBAUD.Error
                    clra
SetBAUD.SetBaud     clr       SCCR2               disable SCI while changing BAUD
                    sta       BAUD
                    clr       SCCR1
                    lda       CCR_,y
                    bita      #1                  is carry set?
                    bne       SetBAUD.WithInt
                    lda       #$0C                No interrupt mode
                    sta       SCCR2
                    bra       SetBAUD.Exit        exit without errors
SetBAUD.WithInt     lda       #$2C                Interrupt mode
                    sta       SCCR2
* Clear the buffer to be used for transmission/receipt of data
                    ldx       X_,y                get pointer to buffer
                    ldb       B_,y                get length of buffer
                    clr       ,x                  zero length in buffer[1]
SetBAUD.Loop        inx
                    clr       ,x                  clear buffer and advance pointer
                    decb
                    bne       SetBAUD.Loop
SetBAUD.Exit        clc                           no error exit
                    rts
SetBAUD.Error       ldb       #errBadParm         return error code in B
                    sec
                    rts
$endif

$ifdef fPrint
; Purpose: Print ASCIIZ string following OS instruction (with FCS)
; Parms  : None
Print               ldx       PC_,y               get address following OS
                    pshy
                    ldy       #REGS
Print.Loop          lda       ,x
                    beq       Print.Exit
                    brclr     [SCSR,y,$80,*       wait for ready
                    sta       SCDR
                    inx
                    bra       Print.Loop
Print.Exit          inx                           point after NUL
                    puly
                    stx       PC_,y               ..to use as return address
                    clc                           no errors
                    rts
$endif

$ifdef fBin2Dec
; Purpose: Convert a binary number to decimal
; Input  : D=Unsigned binary number from 0 to 65535 ($FFFF)
; Output : D=First two ASCII digits of result (first char is blank)
;        : X=Second two ASCII digits of result
;        : Y=Third two ASCII digits of result
Bin2Dec             ldx       #'00'
                    pshx                          put zeros on stack
                    pshx                          where the result will go
                    pshx
                    pshy                          original Y
                    tsy                           and put it's pointer in Y
                    iny
                    iny
Bin2Dec.01.0        subd      #10000              figure out tens of thousands
                    bcs       Bin2Dec.01
                    inc       1,y                 increment tens of thousands
                    bra       Bin2Dec.01.0
Bin2Dec.01          addd      #10000
Bin2Dec.01.1        subd      #1000               figure out thousands
                    bcs       Bin2Dec.02
                    inc       2,y                 increment thousands
                    bra       Bin2Dec.01.1
Bin2Dec.02          addd      #1000
Bin2Dec.02.1        subd      #100
                    bcs       Bin2Dec.03
                    inc       3,y                 increment hundreds
                    bra       Bin2Dec.02.1
Bin2Dec.03          addd      #100
Bin2Dec.03.1        subd      #10
                    bcs       Bin2Dec.04
                    inc       4,y                 increment tens
                    bra       Bin2Dec.03.1
Bin2Dec.04          addd      #10+'0'
                    stb       5,y                 finally, save units
                    puly                          get original Y
                    puld
                    sta       A_,y
                    stb       B_,y
                    puld
                    std       X_,y
                    puld
                    std       Y_,y
Bin2Dec.Exit        clc
                    rts
$endif

$ifdef fWrite
; Purpose: Write (send) a string to the SCI
; Input  : X->buffer of Pascal-like string, ie., LengthByte,Char1,Char2,...
; Note(s): buffer should not be used for incoming chars, for it will be
;          messed up.
Write               ldx       X_,y
                    ldy       #REGS
                    ldb       ,x                  get length of string
                    beq       Write.NoError       if 0, nothing to send
                    inx                           point to data
Write.Loop          brclr     [SCSR,y,$80,*
                    lda       ,x
                    sta       SCDR
                    inx
                    decb
                    bne       Write.Loop
                    bra       Write.NoError
Write.Error         ldb       #errFailure
                    sec
                    bra       Write.Exit
Write.NoError       clc
Write.Exit          rts
$endif

$ifdef fWriteln
; Purpose: Writeln (send) a string to the SCI followed by a CR,LF pair
; Input  : X->buffer of Pascal-like string, ie., LengthByte,Char1,Char2,...
Writeln             bsr       Write               do regular fWrite
                    bcc       Writeln.Exit        on failure, exit
; Purpose: Advance a line sending a CR,LF pair to the SCI
NewLine             ldy       #REGS
                    brclr     [SCSR,y,$80,*       wait for ready
                    lda       #CR                 send a CR
                    sta       SCDR
                    brclr     [SCSR,y,$80,*       wait for ready
                    lda       #LF                 send a LF
                    sta       SCDR
                    clc
Writeln.Exit        rts
$endif

$ifdef fUpString
; Purpose: Convert ASCIIZ string pointed to by X to uppercase
; Input  : X_,y->string
; Output : X_,y->STRING
UpString            ldx       X_,y                point X to string
UpString.Loop       lda       ,x
                    beq       UpString.Exit
                    jsr       Upcase              convert to uppercase
                    sta       ,x
                    cpx       #RAM_END            do not run wild outside of RAM
                    bhs       UpString.Exit
                    inx
                    bne       UpString.Loop
UpString.Exit       clc
                    rts
$endif

$ifdef fUpChar
; Purpose: Convert ASCII character in A to uppercase
; Input  : RegA->char
; Output : RegA->CHAR
UpChar              lda       A_,y
                    jsr       Upcase
                    sta       A_,y
                    clc
                    rts
$endif

$ifdef fWriteEE
; Purpose: Program an internal EEPROM location
; Input  : A_,y->value to be programmed
;        : X_,y->address to program
WriteEE             lda       A_,y
                    ldx       X_,y
WriteEE.Local       equ       *                   local call entry point
                    ldb       #$02                EELAT = 1
                    stb       PPROG               Set EELAT bit
                    sta       ,x                  save data to address
                    ldb       #$03                EELAT = 1 and EPGM = 1
                    stb       PPROG               Turn on programming voltage
                    jsr       Delay10MS           Delay 10 msec
                    clr       PPROG               Turn off programming voltage
                    cmpa      ,x                  Compare with value written
                    beq       WriteEE.01          if equal, get out, no errors
                    ldb       #errFailure
                    sec
                    bra       WriteEE.Exit
WriteEE.01          clc
WriteEE.Exit        rts
$endif

$ifdef fEraseEE
; Purpose: Byte Erase an internal EEPROM location
; Input  : X_,y->address to erase
EraseEE             ldx       X_,y
                    ldb       #$16                BYTE = 1, ERASE = 1, EELAT = 1
                    stb       PPROG
                    stb       ,x                  any data to this address will do
                    ldb       #$17                BYTE = 1, ERASE = 1, EELAT = 1, EPGM = 1
                    stb       PPROG               Turn on programming voltage
                    jsr       Delay10MS           Delay 10 msec
                    clr       PPROG               Turn off programming voltage
                    lda       ,x                  Read back the byte for check
                    cmpa      #[ERASED_STATE      is it erased ($FF)?
                    beq       EraseEE.01          if equal, get out, no errors
                    ldb       #errFailure
                    sec
                    bra       EraseEE.Exit
EraseEE.01          clc
EraseEE.Exit        rts
$endif

$ifdef fBulkEraseEE
; Purpose: Bulk Erase EEPROM
BulkEraseEE         ldb       #$06                EELAT = 1
                    stb       PPROG
                    sta       EEPROM              any EEPROM address will do
                    ldb       #$07                EELAT = 1 and EPGM = 1
                    stb       PPROG               Turn on programming voltage
                    jsr       Delay10MS           Delay 10 msec
                    clr       PPROG               Turn off programming voltage
;here each and every byte in EEPROM should have an $FF value
                    clc
                    rts
$endif

$ifdef fBin2BCD
; Purpose: Convert a binary word value to BCD or other base
; Input  : D holds word value
;        : X points to output buffer
;        : Y holds length of buffer
;        : First byte of buffer holds conversion base
; Note(s): Although user's stack frame is used for temporary storage,
;        : it is left intact.
Bin2BCD             equ       *
          ; save original stack frame
                    ldx       Y_,y                Save stack frame's Y
                    pshx
                    ldx       X_,y                Save stack frame's X (pointer to buffer)
                    pshx
          ; check conversion base for validity
                    clra                          Get conversion base in D
                    ldb       ,x
                    cmpb      #2                  Check base to be 2 to 36
                    blo       Bin2BCD.Error
                    cmpb      #36
                    bhi       Bin2BCD.Error
          ; make buffer pointer point to end of buffer (last character)
                    xgdx                          Now X holds base, D holds buffer pointer
                    addd      Y_,y                add length of buffer
                    decd                          minus 1 to point to end of buffer
                    std       X_,y                and save in temp space

                    lda       A_,y                Load D register with word
                    ldb       B_,y                ..to convert
                    pshx                          push needed for first pass of next loop

Bin2BCD.01          pulx
                    pshx
                    idiv                          X should be holding base here
                    bsr       ToDigit

          ; save current digit to buffer and adjust buffer pointer backwards
                    pshx
                    ldx       X_,y                Point to current buffer character
                    stb       ,x                  Save remainder in output buffer
                    dex                           decrement pointer toward start of buffer
                    stx       X_,y                and save it
                    pulx

                    xgdx                          and use integer result as input for
                    dec       Y_+1,y              decrement [low byte] of Y counter
                    bne       Bin2BCD.01          2nd.last digit? if no, loop back
Bin2BCD.Exit        pulx                          dummy PULL to adjust stack
                    pulx                          restore stack frame's X
                    stx       X_,y
                    pulx                          restore stack frame's Y
                    stx       Y_,y
                    clc
                    rts
Bin2BCD.Error       pulx                          dummy PULLs to adjust stack
                    pulx
                    sec
                    ldb       #errBadParm
                    rts
$endif

;
; ---------------- add more OS routines here -------------------------
;

*********************************************************************
*                     GENERAL-PURPOSE ROUTINES                      *
*********************************************************************

; Routine: Upcase
; Purpose: Convert character in A to uppercase
; Input  : A=s
; Output : A=S
Upcase              cmpa      #'a'                 less than 'A'?
                    blo       Upcase.Exit          yes, skip
                    cmpa      #'z'                 greater than 'Z'?
                    bhi       Upcase.Exit          yes, skip
                    suba      #'a'-'A'             do the conversion
Upcase.Exit         rts

; Purpose: Convert a binary number to ASCII equivalent
; Input  : B holds binary number
; Output : B holds ASCII equivalent
ToDigit             addb      #'0'                convert to ASCII
                    cmpb      #'9'
                    bls       ToDigit.Exit
                    addb      #'A'-'0'-10         adjust for appropriate letter
ToDigit.Exit        clc
                    rts

; Purpose: Kick the COP timer (Call this to prevent COP timeout resets)
KickCOP             equ       *
                    psha
                    lda       #$55
                    sta       COPRST
                    lda       #$AA
                    sta       COPRST
                    pula
                    rts

**********************************************************************
* RTI - Real Time Interrupt requests come here                       *
**********************************************************************
RTI_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
; Sample code below allows loading timer with number of ticks to wait for
                    tst       timer               is timer expired?
                    beq       RTI_Exit            yes, get out
                    dec       timer               decrement timer
;
; ---------------- add your code here --------------------------------
;
RTI_Exit            ldx       #REGS
                    bset      [TFLG2,x,#$40       Reset the RTI int
                    rti

**********************************************************************
* IRQ - External Interrupt requests come here                        *
**********************************************************************
IRQ_Handler         equ       *
                    lda       PIOC                Determine if IRQ was from STRA
                    bita      #STAF.              Is the STAF set?
                    beq       IRQ_Pin             No, go to IRQ pin interrupt
                    lda       PORTCL              Read latched data to clear STAF
;
; ---------------- add your code here --------------------------------
;
                    rti

IRQ_Pin             equ       *
;
; ---------------- add your code here --------------------------------
;
IRQ_Exit            rti

**********************************************************************
* SCI - Serial Communications Interface requests come here           *
**********************************************************************
SCI_Handler         equ       *
                    ldb       SCSR
                    andb      #$20
                    beq       SCI_Exit
                    lda       SCDR                get received data
; the following three lines of code are only needed if ECHO is desired
; WARNING: This is NOT the best place to put this code as it could
;          slow down this routine enough to lose characters or other
;          interrupts.
                    ldx       #REGS
                    brclr     [SCSR,x,Bit7.,*     wait for ready
                    sta       SCDR                and echo it back
;
; ---------------- add your code here --------------------------------
;
SCI_Exit            rti
SCI_Error           lda       SCDR                get data but ignore it
AnRTI               rti

**********************************************************************
* SPI - Serial Peripheral Interface requests come here               *
**********************************************************************
SPI_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
SPI_Exit            rti

**********************************************************************
* PAI - Pulse Accumulator Input interrupt requests come here         *
**********************************************************************
PAI_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
                    lda       #Bit5.              Reset the interrupt
                    sta       TFLG2
PAI_Exit            rti

**********************************************************************
* PAO - Pulse Accumulator Overflow interrupt requests come here      *
**********************************************************************
PAO_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
                    lda       #Bit4.              Reset the interrupt
                    sta       TFLG2
PAO_Exit            rti

**********************************************************************
* RTO - Real Time Overflow interrupt requests come here              *
**********************************************************************
RTO_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
                    lda       #Bit7.              Reset the interrupt
                    sta       TFLG2
RTO_Exit            rti

**********************************************************************
* TIC4/TOC5 - TIC4 and/or TOC5 interrupt requests come here          *
**********************************************************************
TIC4_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TIC4_Exit           rti

**********************************************************************
* TOC4 - TOC4 interrupt requests come here                           *
**********************************************************************
TOC4_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TOC4_Exit           rti

**********************************************************************
* TOC3 - TOC3 interrupt requests come here                           *
**********************************************************************
TOC3_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TOC3_Exit           rti

**********************************************************************
* TOC2 - TOC2 interrupt requests come here                           *
**********************************************************************
TOC2_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TOC2_Exit           rti

**********************************************************************
* TOC1 - TOC1 interrupt requests come here                           *
**********************************************************************
TOC1_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TOC1_Exit           rti

**********************************************************************
* TIC3 - TIC3 interrupt requests come here                           *
**********************************************************************
TIC3_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TIC3_Exit           rti

**********************************************************************
* TIC2 - TIC2 interrupt requests come here                           *
**********************************************************************
TIC2_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TIC2_Exit           rti

**********************************************************************
* TIC1 - TIC1 interrupt requests come here                           *
**********************************************************************
TIC1_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
TIC1_Exit           rti

**********************************************************************
* XIRQ - External Interrupt requests come here                       *
**********************************************************************
XIRQ_Handler        equ       *
;
; ---------------- add your code here --------------------------------
;
; make sure we don't re-enter this routine (optional)
                    tsx
                    lda       CCR_,x              Get original CCR
                    ora       #%01000000          Disable further XIRQs
                    sta       CCR_,x              Put CCR in return stack frame
XIRQ_Exit           rti

**********************************************************************
* ILLOP - Illegal Opcode Interrupt requests come here                *
* Display a message with the offending PC address in hex and halt    *
* If stack isn't usable (due to possible corruption) simply halt     *
* Stack Needed: 22 (9 for caller + 13 for calls without subs)        *
**********************************************************************
ILLOP_Handler       equ       *
                    tsx                           First, check stack availability
                    cmpx      #RAM+30             Is enough stack space available?
                    blo       ILLOP_Halt          No, show simple message only
                    cmpx      #STACKTOP-9         Is it above possible stack top (after call)?
                    bhs       ILLOP_Halt          Yes, show simple message only
                    bsr       ILLOP_SetSCI        Reset the SCI to 9600bps 8-N-1
                    bcs       ILLOP_Exit          SCI setup failed, don't show message
                    tsy                           get pointer to stack frame in Y
                    os        fPrint              print first part of message
                    fcs       'Illegal opcode at $'
                    ldx       #'00'               make buffer on stack (4 bytes)
                    pshx                          ..initialized to string 0000
                    pshx
                    tsx                           and get its pointer in X
                    lda       #16                 conversion base
                    sta       ,x                  ..set in buffer[1]
                    ldd       PC_,y               get offending PC address in RegD
          ; Display the offending address
                    ldy       #4                  the length of the buffer in Y
                    os        fBin2BCD            do the conversion of PC address to hex
                    lda       #4                  and a fifth byte with the length
                    psha
                    dex                           point to beginning of string
                    os        fWrite              and print hex PC address
                    pula                          restore stack
                    pulx
                    pulx
                    os        fPrint
                    fcs       '. '
ILLOP_Halt          lds       #STACKTOP           we can safely reset the stack now
                    bsr       ILLOP_SetSCI        Reset the SCI to 9600bps 8-N-1
                    bcs       ILLOP_Exit          SCI setup failed, don't show message
                    os        fPrint
                    fcs       'System halted!'
ILLOP_Exit          jmp       Halt_MCU            Go wait for processor RESET
ILLOP_SetSCI        lda       #BPS_RATE           Next, re-set the SCI
                    clc                           ..in case it got screwed up
                    os        fSetBAUD            ..by random code
                    rts

**********************************************************************
* COP - Computer Operating Properly RESET requests come here         *
**********************************************************************
COP_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
COP_Exit            jmp       Start               Cannot return with RTI

**********************************************************************
* CMF - Clock Monitor Failure RESET requests come here               *
**********************************************************************
CMF_Handler         equ       *
;
; ---------------- add your code here --------------------------------
;
CMF_Exit            jmp       Start               Cannot return with RTI

; ------------------ VECTOR SETUP --------------------
                    org       VECTORS
                    dw        SCI_Handler         SCI
                    dw        SPI_Handler         SPI
                    dw        PAI_Handler         Pulse Accumulator Input
                    dw        PAO_Handler         Pulse Accumulator Overflow
                    dw        RTO_Handler         Timer Overflow
                    dw        TIC4_Handler        TIC4 / TOC5
                    dw        TOC4_Handler        Timer Output Compare 4
                    dw        TOC3_Handler        Timer Output Compare 3
                    dw        TOC2_Handler        Timer Output Compare 2
                    dw        TOC1_Handler        Timer Output Compare 1
                    dw        TIC3_Handler        Timer Input Capture 3
                    dw        TIC2_Handler        Timer Input Capture 2
                    dw        TIC1_Handler        Timer Input Capture 1
                    dw        RTI_Handler         Timer interrupt
                    dw        IRQ_Handler         IRQ for module select
                    dw        XIRQ_Handler        XIRQ
                    dw        SWI_Handler         SoftWare Interrupt
                    dw        ILLOP_Handler       Illegal Opcode Trap
                    dw        COP_Handler         COP Failure (RESET)
                    dw        CMF_Handler         Clock Monitor Failure (RESET)
                    dw        Start               RESET vector
$IF *-VECTORS/2 < 21
  $ERROR You have not defined enough vectors
$ENDIF

                          *** EOF ***
