                                                  
/*
**  Video Output Functions.
**
**  Copyright 1993-97 by Paul D. Burgin. All rights reserved.
*/
                                
#include "conio.h"
#include "graphics.h"
#include "dos.h"
#include "bios.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "key.h"
#include "6809cpu.h"
#include "6809regs.h"

	signed char 		vmode;					/* Current graphic mode.  */
	unsigned int  		screen_base;			/* Current scr base addr. */
	unsigned char		screen_size;			/* Current scr size.      */
	unsigned int  		screen_end;				/* Current scr end addrs. */
	unsigned char		height;					/* Curr byte horiz rpt.   */
	unsigned char		increment;				/* Vert scan byte inc.    */
	boolean				double_width;           /* Curr byte vert rpt.    */
	unsigned char 		palette;				/* Current palette num.   */
	boolean				four_colours;			/* 2 or 4 color mode flag.*/
	signed char			new_vmode;				/* New graphic mode.      */
	unsigned int  		new_screen_base;		/* New scr base addr.     */
	unsigned char		new_screen_size;		/* New scr size.          */
	unsigned char 		new_palette;			/* New palette num.       */
	boolean				new_four_colours;		/* New 2 or 4 color flag. */
	unsigned int		last_text_base;			/* Ad of last text screen.*/
	boolean				video_enabled = TRUE;	/* Used by debugger.      */
	boolean				lower_case = FALSE;		/* Invrse/lower case flag.*/
	int					cursortype = _NOCURSOR;	/* Type of text cursor.   */

	unsigned char		xx, yy;					/* Temp co-ordinates.     */
	unsigned int		xxx, yyy;				/* More temp co-ordinates.*/

	boolean				graphinit = FALSE;		/* Flag indicates whether */
												/* initgraph() has been   */
												/* called.                */

	enum text_modes		text_mode = C40;		/* PC text mode to use.   */

	/**************************************************************/
	/* 'vmode' is based on the value in the SAM video register,   */
	/* but with 8 added for narrow modes. It's therefore used as: */
	/*		-2	Mode not set yet (emulator is in text mode).      */
	/*		-1	Text Mode                                         */
	/*		0+	Graphics Mode                                     */
	/**************************************************************/


/* EGA driver initialisation/error code variables. */
int 					gdriver = EGA;
int						gmode = EGALO;
int						gerror;

/* Normal foreground and background colours for text. */
unsigned char			back_ary[2] = {GREEN, BROWN};
unsigned char			fore_ary[2] = {BLACK, BLACK};

/* Colours used to emulate orange. */
unsigned char			orange_ary[2] = {BROWN, WHITE};

/* Mapping of dragon text mode colours to PC text mode colours. */
/* (Orange changes according to the palette mode since orange   */
/* on orange would be invisible!)                               */
unsigned char color_ary[]
	= { GREEN, YELLOW, BLUE, RED, LIGHTGRAY, CYAN, MAGENTA, 0 /* orange */ };

/* Mapping of dragon graphic blocks to PC extended ascii characters. */
const unsigned char block_ary[]
	= {32,254,254,220,254,222,254,178,254,254,221,178,223,178,178,219};

/* Number of bytes of RAM used as video memory for all modes. */
const unsigned int vdg_sizes[]
	= {512, /* text mode */
		512,1024,2048,1536,3072,3072,6144,3072,    /* graphics */
		512,3072,2048,4608,3072,9216,6144,9216};   /*  modes   */

/* VDG scanner memory increment for each horizontal line. */
const unsigned char vdg_increments[]
	= {32,16,32,16,32,16,32,16,
		32,16,32,48,32,48,32,48};

/* Vertical byte repeat counts. */
const unsigned char vdg_heights[]
	= {12,3,3,2,2,1,1,1,
		12,1,3,1,2,1,1,1};

/* Mapping of dragon graphics colours to EGA colours. */
unsigned char vmode_ary[2][2][6]
	= {	{{EGA_BLACK,		EGA_BLACK,			EGA_GREEN,
		  EGA_GREEN,		EGA_BLACK,			EGA_GREEN},
		 {EGA_BLACK,		EGA_BLACK,			EGA_WHITE,
		  EGA_WHITE,		EGA_BLACK,			EGA_WHITE}},

		{{EGA_GREEN,		EGA_YELLOW,			EGA_YELLOW,
		  EGA_BLUE,			EGA_BLUE,			EGA_RED},
		 {EGA_WHITE,		EGA_CYAN,			EGA_CYAN,
		  EGA_LIGHTMAGENTA,	EGA_LIGHTMAGENTA,	6 /* BROWN */}}	  };

/* Border colours. */
unsigned char border_ary[2][2]
	= {{EGA_GREEN, EGA_WHITE}, {EGA_GREEN, EGA_WHITE}};

/* Toggle between 2 and 4 colour modes. */
void toggle_2or4_colours(void)
{
	switch(memory[0xff22] >> 4)
	{
		case 0x8:	memory[0xff22] |= 0xd0;
					break;
		case 0x9:
		case 0xb:
		case 0xd:	memory[0xff22] &= 0x8f;
					break;
		case 0xa:
		case 0xc:
		case 0xe:	memory[0xff22] |= 0xf0;
					break;
		case 0xf:	memory[0xff22] &= 0xef;
					break;
	}
	set_vmode(memory[0xff22]);
}

/* Toggle width of pixels in current graphics mode. */
void toggle_pixel_width(void)
{
	switch(memory[0xff22] >> 4)
	{
		case 0x8:	memory[0xff22] |= 0xe0;
					break;
		case 0x9:
		case 0xb:
		case 0xd:	memory[0xff22] |= 0xf0;
					break;
		case 0xa:
		case 0xc:
		case 0xe:	memory[0xff22] &= 0x8f;
					break;
		case 0xf:	memory[0xff22] &= 0xdf;
					break;
	}
	set_vmode(memory[0xff22]);
}

/* Assess current artifacted colours mode. */
unsigned char get_artifact(void)
{
	if ((vmode_ary[0][1][0] == EGA_BLACK)
		&& (vmode_ary[0][1][5] == EGA_WHITE))
	{
		if ((vmode_ary[0][1][1] == EGA_RED)
				&& (vmode_ary[0][1][2] == EGA_RED)
				&& (vmode_ary[0][1][3] == EGA_BLUE)
				&& (vmode_ary[0][1][4] == EGA_BLUE))
			return 1;
		else if ((vmode_ary[0][1][1] == EGA_BLUE)
			&& (vmode_ary[0][1][2] == EGA_BLUE)
			&& (vmode_ary[0][1][3] == EGA_RED)
			&& (vmode_ary[0][1][4] == EGA_RED))
			return 2;
	}
	return 0;
}

/* Change artifacted colours mode. */
void set_artifact(unsigned char art_action)
{
	vmode_ary[0][1][0] = EGA_BLACK;
	vmode_ary[0][1][5] = EGA_WHITE;

	switch (art_action)
	{
		/* Artifacting off. */
		case 0:	vmode_ary[0][1][1]	= EGA_BLACK;
				vmode_ary[0][1][2]	= EGA_WHITE;
				vmode_ary[0][1][3]	= EGA_WHITE;
				vmode_ary[0][1][4]	= EGA_BLACK;
				border_ary[0][1]	= EGA_WHITE;
				break;

		/* Blue edged artifacting. */
		case 1:	vmode_ary[0][1][1]	= EGA_RED;
				vmode_ary[0][1][2]	= EGA_RED;
				vmode_ary[0][1][3]	= EGA_BLUE;
				vmode_ary[0][1][4]	= EGA_BLUE;
				border_ary[0][1]	= EGA_BLACK;
				break;

		/* Red edged artifacting. */
		case 2:	vmode_ary[0][1][1]	= EGA_BLUE;
				vmode_ary[0][1][2]	= EGA_BLUE;
				vmode_ary[0][1][3]	= EGA_RED;
				vmode_ary[0][1][4]	= EGA_RED;
				border_ary[0][1]	= EGA_BLACK;
				break;
	}
}

/* Setup PC screen for Dragon text mode emulation. */
void setup_screen(void)
{
	/* Change to 40 character colour mode. */
	textmode(text_mode);
	_setcursortype(cursortype);
	textattr((BLACK << 4) + LIGHTGRAY);
	clrscr();
	new_window(WIN_DRAGON);
}

/* Align PC text cursor with dragon text cursor address. */
void update_cursor(void)
{
	unsigned int cursadd = (dp_reg.u_value << 8) + 0x88;

	if ((vmode < 0)
		&& (memory[cursadd] >= (screen_base >> 8))
		&& (memory[cursadd] <= (screen_end  >> 8)))
			gotoxy((memory[cursadd+1] & 0x1f) + 1, ((((memory[cursadd] << 8)
				- screen_base) + memory[cursadd + 1]) >> 5) + 1);
}

/* Check result of graphics library operations. */
void check_gerror(void)
{
	if ((gerror = graphresult()) != grOk)
		quit(grapherrormsg(gerror),1);
}

/* Re-assign graphics mode palette. */
void refresh_palette(void)
{
	setpalette(0, border_ary[four_colours][palette]);
	setpalette(1, vmode_ary[four_colours][palette][0]);
	setpalette(2, vmode_ary[four_colours][palette][1]);
	setpalette(3, vmode_ary[four_colours][palette][2]);
	setpalette(4, vmode_ary[four_colours][palette][3]);
	setpalette(5, vmode_ary[four_colours][palette][4]);
	setpalette(6, vmode_ary[four_colours][palette][5]);
}

/* Redraw any dragon video display. */
void refresh_video(void)
{
	unsigned int address;

	/* Note: '!=' is used instead of '<' so that screen wraps $FFFF to 0. */
	for (address = screen_base; address != (screen_end + 1); address++)
			video_out(memory[address], address);
}

/* Setup graphics mode pixel width. Return: 0=wide, 8=normal. */
unsigned char vmode_type(unsigned char new_ff22)
{
	switch(new_ff22 >> 4)
	{
		case 0x8:
		case 0x9:
		case 0xb:
		case 0xd:
			return(0);
	}
	return(8);
}

/* Calculate end address of next display mode. */
unsigned int new_screen_end(void)
{
	return(new_screen_base + (vdg_sizes[new_vmode + 1] - 1));
}

/* Setup new dragon video mode when address 0xff22 accessed. */
void set_vmode(unsigned char ff22)
{

	/*******************************************************************/
	/* This function assesses what the next Dragon mode should be, but */
	/* does not switch the emulator into that mode (unless it can be   */
	/* done quickly). This avoids thrashing when several writes are    */
	/* required to set up a given Dragon mode.                         */
	/*                                                                 */
	/* When the Dragon mode has stabalised it will be changed in the   */
	/* emulator some time later, after a couple of IRQ's. This is      */
	/* performed by the function execute_vmode(). The details of the   */
	/* current Dragon mode are held in variables vmode, palette,       */
	/* screen_size, screen_base and four_colours. The details of the   */
	/* next mode are prefixed by "new_".                               */
	/*******************************************************************/

	new_palette = ((ff22 >> 3) & 0x01);

	/* Are we trying to do semigraphics? */
	if ((ff22 < 0x80) && (new_screen_size > 0))
		ff22 |= 0xe0;

	if (ff22 < 0x80)
	{
		/* New mode is text mode. */
		new_vmode = -1;
	}
	else
	{
		/* New mode is a graphics mode. */
		new_vmode			= vmode_type(ff22) + new_screen_size;
		new_four_colours	= ((ff22 & 0x10) == 0);
	}

	/* Check for scroll lock. */
	if (((bioskey(2) & 0x10) == 0) && !(new_int9_set && keys[kSCROLLLOCK]))
	{
		/* Need to refresh or change colours? */
		if ((new_palette != palette)
			|| (new_four_colours != four_colours)
			|| (new_vmode != vmode))
		{
			if (new_vmode < 0)
			{
				/* May as well change palette and 'orange' now. */
				palette = new_palette;
				color_ary[7] = orange_ary[palette];

				/* Okay to refresh if we are currently in text mode.   */
				/* (But not if we just exit a dialogue box - screen    */
				/* end could be set to more than 512 bytes after       */
				/* screen_base if we were previously in a semigraphics */
				/* mode.)                                              */
				if (vmode == -1)
					refresh_video();
			}
			else
			{
				if (vmode == new_vmode)	/* Could change this to FALSE */
										/* to defeat annoying games   */
										/* which thrash the palette.  */
				{
					/* Okay to change palette & colours if */
					/* we are currently in graphics mode.  */
					palette = new_palette;
					four_colours = new_four_colours;
					refresh_palette();
				}
			}
		}
	}
	refresh_count = 0;
}

/* Change to new dragon video mode. */
void execute_vmode(boolean forced_mode)
{
	int gdet;

	/**************************************************************/
	/* This function changes the video mode which the emulator is */
	/* displaying. It is called when necessary, in sync with the  */
	/* Dragon IRQ. Also called in response to some function keys. */
	/* If forced_mode is FALSE then the mode will not be changed  */
	/* if SCROLL LOCK is engaged. Function keys need to override  */
	/* SCROLL LOCK and set forced_mode to TRUE.                   */
	/**************************************************************/

	/* Make sure vmode is setup right (it might have changed). */
	set_vmode(memory[0xff22]);

	/* Check for scroll lock. */
	if ((((bioskey(2) & 0x10) == 0) || forced_mode)
		&& !(new_int9_set && keys[kSCROLLLOCK]))
	{
		/* Do we need to change between text and graphics mode? */
		if ((new_vmode < 0) && (vmode >= 0))
		{
			/* Change to text mode. */
#ifdef BUGFIX
			closegraph();
#else
			restorecrtmode();
#endif
			_setcursortype(cursortype);
		}
		else if (new_vmode >= 0)
		{
			if (vmode < 0)
			{
				/* Initialise graphics system if necessary. */
				if (!graphinit)
				{
#ifndef BUGFIX
					graphinit = TRUE;
#endif

					/* Look for link-loaded graphics driver. */
					gerror = registerfarbgidriver(EGAVGA_driver_far);
					if (gerror < 0)
						check_gerror();

					/* Look for EGA card; init graphics system. */
					detectgraph(&gdet, &gerror);
					check_gerror();
					if (gdet < gdriver)
						quit("EGA Video Card Not Detected",1);

					initgraph(&gdriver, &gmode, "");
				}
				else
				{
					/* Go back into graphics mode. */
					setgraphmode(gmode);
				}

				/* Check result of mode change. */
				check_gerror();

				/* Define reference points. */
				setviewport(64,4,575,195, 1);

				/* Set border to colour 0. */
				setbkcolor(0);
			}
		}

		/* Update mode information. */
		vmode			= new_vmode;
		screen_base		= new_screen_base;
		screen_size		= new_screen_size;
		screen_end		= new_screen_end();
		height			= vdg_heights[vmode];
		increment		= vdg_increments[vmode];
		double_width	= (vmode < 8);
		palette			= new_palette;
		four_colours    = new_four_colours;

		/* Need to re-install palette colours for graphics modes. */
		if (vmode >= 0)
			refresh_palette();
		else
		{
			/* Remember the address that we last displayed a text screen. */
			last_text_base = screen_base;
		}

		/* Draw screen contents. */
		refresh_video();
	}
}

/* Switch back to 80 column text mode for emulator F-key functions. */
void temp_textmode(void)
{
	if (vmode >= 0)
#ifdef BUGFIX
		closegraph();
#else
		restorecrtmode();
#endif

	textmode(C80);
	if (debug_disp || (text_mode == C80))
		clrscr();
	_setcursortype(_NOCURSOR);
	debug_disp = FALSE;
}

/* Return immediately to appropriate dragon mode. */
void restore_vmode(void)
{
	vmode = -2;
	setup_screen();
	execute_vmode(TRUE);
}

/* Update a series of pixels on screen. */
void plot(unsigned char stage, unsigned int colour)
{
	char h_count;
	unsigned int x_from = (xxx << 4) + (stage << 2);

	switch (colour)
	{
		case 0:	setcolor(1);
				for (h_count = 0; h_count < height; h_count++)
					line(x_from, yyy+h_count, x_from+3, yyy+h_count);
				break;

		case 1:	setcolor(2);
				for (h_count = 0; h_count < height; h_count++)
					line(x_from, yyy+h_count, x_from+1, yyy+h_count);
				setcolor(3);
				for (h_count = 0; h_count < height; h_count++)
					line(x_from+2, yyy+h_count, x_from+3, yyy+h_count);
				break;

		case 2:	setcolor(4);
				for (h_count = 0; h_count < height; h_count++)
					line(x_from, yyy+h_count, x_from+1, yyy+h_count);
				setcolor(5);
				for (h_count = 0; h_count < height; h_count++)
					line(x_from+2, yyy+h_count, x_from+3, yyy+h_count);
				break;

		case 3:	setcolor(6);
				for (h_count = 0; h_count < height; h_count++)
					line(x_from, yyy+h_count, x_from+3, yyy+h_count);
				break;
	}
}

/* Update a series of double width pixels on screen. */
void double_plot(unsigned char stage, unsigned int colour)
{
	char h_count;
	unsigned int x_from = (xxx << 5) + (stage << 3);

	if (x_from < 574)
	{
		switch (colour)
		{
			case 0:	setcolor(1);
					for (h_count = 0; h_count < height; h_count++)
						line(x_from, yyy+h_count, x_from+7, yyy+h_count);
					break;

			case 1:	setcolor(2);
					for (h_count = 0; h_count < height; h_count++)
						line(x_from, yyy+h_count, x_from+3, yyy+h_count);
					setcolor(3);
					for (h_count = 0; h_count < height; h_count++)
						line(x_from+4, yyy+h_count, x_from+7, yyy+h_count);
					break;

			case 2:	setcolor(4);
					for (h_count = 0; h_count < height; h_count++)
						line(x_from, yyy+h_count, x_from+3, yyy+h_count);
					setcolor(5);
					for (h_count = 0; h_count < height; h_count++)
						line(x_from+4, yyy+h_count, x_from+7, yyy+h_count);
					break;

			case 3:	setcolor(6);
					for (h_count = 0; h_count < height; h_count++)
						line(x_from, yyy+h_count, x_from+7, yyy+h_count);
					break;
		}
	}
}

/* Output text character or 1 byte of graphics to screen. */
void video_out(unsigned char vid_char, unsigned int vid_address)
{
	unsigned char m_vid_char = vid_char;

	/* Textmode. */
	if (vmode < 0)
	{
		if (video_enabled)
		{
			/* Move cursor to appropriate screen address. */
			gotoxy((vid_address & 0x001f) + 1, ((vid_address - screen_base) >> 5) + 1);

			/* Transform dragon text output to pc ascii chars and colours. */
			if (vid_char > 0x7f)
			{
				/* Graphic block. */
				m_vid_char = block_ary[vid_char & 0x0f];
				textattr((BLACK << 4) | (color_ary[(vid_char & 0x70) >> 4]));
			}
			else if ((vid_char > 0x3f) || lower_case)
			{
				/* Normal video. */
				textattr((back_ary[palette] << 4) | fore_ary[palette]);
			}
			else
			{
				/* Reverse video. */
				textattr((fore_ary[palette] << 4) | back_ary[palette]);
			}
			if (m_vid_char <= 0x1f)
			{
				if (lower_case && (m_vid_char >= 1) && (m_vid_char <= 26))
					m_vid_char += 0x60;
				else
					m_vid_char += 0x40;
			}
			else
			{
				if ((m_vid_char >= 0x60) && (m_vid_char <= 0x7f))
					m_vid_char -= 0x40;
			}
			/* Arrow symbols. */
			if (m_vid_char == 94)
				m_vid_char = 24;
			else if (m_vid_char == 95)
				m_vid_char = 27;

			/* Output modified ascii character to window. */
			putch(m_vid_char);

			/* Put cursor back where it was. */
			update_cursor();
		}
	}

	/* Graphics modes. */
	else
	{
		/* Calculate most common x and y start co-ords for new lines. */
		if (increment <= 32)
			xxx = vid_address & (increment -1);
		else
			xxx = vid_address % increment;
		yyy = ((vid_address - screen_base) / increment) * height;

		/* Double width modes are simple. */
		if (double_width)
		{
			double_plot(0, vid_char >> 6);
			double_plot(1, (vid_char >> 4) & 0x03);
			double_plot(2, (vid_char >> 2) & 0x03);
			double_plot(3, vid_char & 0x03);
		}
		else
		{
			if (vmode == 11)
			{
				/* Mode that repeats 16 bytes every 48 bytes. */
				yyy <<= 1;
				if (xxx < 16)
				{
					plot(0, vid_char >> 6);
					plot(1, (vid_char >> 4) & 0x03);
					plot(2, (vid_char >> 2) & 0x03);
					plot(3, vid_char & 0x03);
					xxx += 16;
				}
				else
				{
					xxx -= 16;
					yyy += height;
				}
			}
			plot(0, vid_char >> 6);
			plot(1, (vid_char >> 4) & 0x03);
			plot(2, (vid_char >> 2) & 0x03);
			plot(3, vid_char & 0x03);
			if (vmode == 9)
			{
				/* Mode that repeats 16 bytes every line. */
				xxx += 16;
				plot(0, vid_char >> 6);
				plot(1, (vid_char >> 4) & 0x03);
				plot(2, (vid_char >> 2) & 0x03);
				plot(3, vid_char & 0x03);
			}
		}
	}
}
