/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPETHER.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/7/31
**
** DESCRIPTION: Contains the ethernet interface driver and
**              ARP implementation.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   IPETHER.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IFACE/VCS/IPETHER.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:38:08   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 15:34:46   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IFACE/VCS/IPETHER.C_V   1.1   21 Nov 1990 14:38:08   etstjan  $";
#endif

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <dos.h>
#include    <pkt.h>                  /* Packet Driver Specification   */
#include    <power3.h>
#include    <beholder.h>

#include    "ip.h"                   /* global IP type definitions    */
#include    "ipcodes.h"              /* contains all the return codes */
#include    "iplib.h"
#include    "ipif.h"
#include    "ipevents.h"             /* event and timer processing    */
#include    "ipether.h"              /* header file for this file     */
#include    "iplayer.h"              /* IP report facility            */

/* prototypes */
void        ARPReceive(EVENTUNION *ARPEvent);
void        IPEtherRec(DPBUF *RecPacket);
void        UpdateARP(ARP_ENTRY *ThisEntry, HARDW_ADD NewAdd);
ARP_ENTRY   *SearchARP(ADDRESS_T InterAdd);
ARP_ENTRY   *AddARP(ADDRESS_T InterAdd, HARDW_ADD EthAdd,
                    IFDESCRIPTOR *AsoIf);
void        DeleteARP(ARP_ENTRY *ThisEntry);
void        ARPTimeOut(EVENTUNION *ARPUnion);
int         SendByPDS(BYTE *ThisPacket, USHORT PackLen, HARDW_ADD Destin,
                      HARDW_ADD Source, USHORT Inter);
int         SendARP(BYTE *ThisARP, USHORT Inter);
void        RetryARP(EVENTUNION *ThisUnion);


/* global variables */
IFDESCRIPTOR  *EtherDescr;           /* pointer to if descriptor      */
HARDW_ADD     BrdCastAdd = {         /* ethernet broadcast address    */
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff
                };
ARP_ENTRY     ARPList = {            /* start of ARP table            */
                NULL,
                NULL
                };
BYTE          ARPPacket[28] = {      /* initialized packet for ARP's  */
                0, 1, 8, 0,          /* Hardware and Protocol         */
                6, 4, 0, 1           /* HLen, Len and Operation       */
                };
BYTE          SendBuffer[ETHER_SIZE];/* send buffer                   */

/************************************************************************
** NAME:        IPEtherInit
** SYNOPSIS:    int IPEtherInit(IFDESCRIPTOR *IfDesc);
** DESCRIPTION: Initializes the ethernet interface.
**              Parameters are stored in strings, should later
**              be read from initialization file.
**              IP Buffer Management MUST have been initialized
**              before calling IPLoopInit.
**              The necessary field in the interface descriptor
**              are initialized as well.
** RETURNS:     NO_ERR -->   no error
**              else         error code
*************************************************************************/
int IPEtherInit(IFDESCRIPTOR *IfDesc)
{
  int      UsRet, AdrLen;

  /* set global interface descriptor fields */
  EtherDescr = IfDesc;
  EtherDescr->Flags |= IF_UP | IF_RUNNING;
  EtherDescr->SendFun = IPEtherSndFun;
  
  if (DpDevActive[0].PktInt == 0) {
    return NOT_FOUND;
    }

  EtherDescr->IntNo = DpDevActive[0].PktInt;
  AdrLen = sizeof(EtherDescr->HardwAddr);
  UsRet = PktGetAddress(EtherDescr->IntNo, EtherDescr->HardwAddr, &AdrLen);

  return (UsRet) ? ETHINITERR : NO_ERR;
}

/**************************************************************
** NAME:        IPEtherSndFun
** SYNOPSIS:    int   IPEtherSndFun(LLREQ Request,
**                                  DATALINK* Packet,
**                                  ADDRESS_T Dest)
** DESCRIPTION: Implements the Ethernet interface send
**              function.
**              In case of a 'Broadcast Request', the packet
**              is broadcasted through the ethernet.
**              In other cases, the ARP table and ARP
**              mechanism are used to find the destination's
**              ethernet address.
**              buffer and a receive event is scheduled.
** RETURNS:     NO_ERR   -->   no error
**              IF_DOWN  -->   interface is not available
**************************************************************/
int   IPEtherSndFun(LLREQ Request, DATALINK* Packet, ADDRESS_T Dest)
{
  USHORT      PackLen;
  BYTE        *DestinAdd;              /* pointer to the destination addr. */
  ARP_ENTRY   *CurEntry;
  BYTE        *TmpPnt;
  EVENTUNION  ARPRetrUn;
  int         RetCode;

  EtherDescr->NrSendReq++;
  if ((EtherDescr->Flags & IF_UP) != IF_UP) {
     return IF_DOWN;
     }
  switch (Request) {
    case BROADREQ:
      DestinAdd = BrdCastAdd;
      break;
    case SENDREQ :
      if ((CurEntry = SearchARP(Dest)) != NULL) {
        /* already a packet waiting for ARP response? */
        if (CurEntry->WaitLen != 0) {
          return ARPBUSY;
          }
        CurEntry->RefCount++;
        DestinAdd = CurEntry->EthAddress;
        }
      else {
        /* make new ARP entry with this packet waiting */
        PackLen = DLinkLen(Packet);
        if (PackLen < 46)
          PackLen = 46;
        if ((TmpPnt = IPBufGet(PackLen + 14)) == NULL) {
          IPReport(NOSPACE, NULL, NULL);
          return NOSPACE;
          }
        if ((CurEntry = AddARP(Dest, BrdCastAdd,
                               EtherDescr)) == NULL) {
          IPReport(NOSPACE, NULL, NULL);
          EtherDescr->NrSysErrors++;
          IPBufFree(TmpPnt);
          return NOSPACE;
          }
        CurEntry->WaitLen = DLinkToByte(TmpPnt + 14, Packet);
        /* save first hop destination address */
        *(ADDRESS_T *)TmpPnt = Dest;
        CurEntry->Waiting = TmpPnt;
        CurEntry->RetryCount = ARPRETRIES;
        ARPRetrUn.ARPData.ARPEntry = CurEntry;
        RetryARP(&ARPRetrUn);
        return NO_ERR;
        }
      break;
    default :
      return ILL_REQUEST;
      }

  PackLen = DLinkToByte(SendBuffer + 14, Packet);
  if ((RetCode = SendByPDS(SendBuffer, PackLen, DestinAdd,
                   EtherDescr->HardwAddr, EtherDescr->IntNo)) == NO_ERR) {
    EtherDescr->NrSend++;
    }
  return RetCode;
}


/**************************************************************
** NAME:        IPEtherRec
** SYNOPSIS:    void  IPEtherRec(DPBUF *RecPacket);
** DESCRIPTION: Receive function for Ethernet IP packets.
**              In case of an IP or an ARP message addressed
**              to this host, buffer space is allocated and
**              the packet is copied.
**              Next, an event is scheduled to process the
**              IP or ARP message during 'FreeTime'.
**              for the IP package.
*** RETURNS:    void
**************************************************************/
void  IPEtherRec(DPBUF *RecPacket)
{
  if (RecPacket->Dev == 0)
  {
    DATALINK   *PassPacket;
    USHORT      PackSize;
    EVENTUNION  EtherUnion;
    int         BCasted;
    int         Type;

    Type = *(USHORT *)(RecPacket->pBuf + 12);
    if ((Type==ETHERIP)||(Type == ETHERARP))        /* Type OK? */
    {
      BCasted = (memcmp(RecPacket->pBuf, BrdCastAdd, 6)==0);
      if (BCasted || (memcmp(RecPacket->pBuf, EtherDescr->HardwAddr, 6)==0))
      {
        EtherDescr->NrRecReq++;
        PackSize = RecPacket->Size - 14;
        if ((PassPacket = IPDataLinkGet(PackSize)) == NULL) {
          IPReport(NOSPACE, NULL, NULL);
          EtherDescr->NrSysErrors++;
          return;
          }
        memcpy(PassPacket->DataStart, RecPacket->pBuf + 14, PackSize);

        PassPacket->ThisType = (Type==ETHERIP) ? (BYTE)DATAHD : (BYTE)ARPDATAHD;
        EtherUnion.RecData.If = EtherDescr;
        EtherUnion.RecData.RecPacket = PassPacket;
        EtherUnion.RecData.LinkBrdCst = BCasted;
        if (Schedule( HIGH,
                      (Type==ETHERIP) ? IPReceive : ARPReceive,
                      &EtherUnion) != NO_ERR) {
           IPReport(NOSPACE, NULL, PassPacket);
           EtherDescr->NrSysErrors++;
           return;
           }
        EtherDescr->NrReceived++;
      }
    }
  }
}


/**************************************************************
** NAME:        ARPReceive
** SYNOPSIS:    void ARPReceive(EVENTUNION *RecMessage);
** 
** DESCRIPTION: Handles the reception of ARP packets.
**              The current ARP table is updated with the
**              source of the ARP message and pending
**              messages are sent.
**              If the message is an ARP request, a reply is
**              sent.
**              If the message is an ARP reply, all
**              processing needed is handled by the updating
**              of the ARP table.
** RETURNS:     
**************************************************************/
void ARPReceive(EVENTUNION *RecMessage)
{
  BYTE          *Packet = RecMessage->RecData.RecPacket->DataStart;
  int           Merge = 0;
  ARP_ENTRY     *CurEntry;
  IFDESCRIPTOR  *IfList;

  if (RecMessage->RecData.RecPacket->Length < 28) {
    IPStat.NrARPShort++;
    IPReport(ARPSHORT, NULL, RecMessage->RecData.RecPacket);
    return;
    }

  /* first, check for hardware type and protocol type */
  if (*(ULONG *)(Packet) == htonl(0x00010800l)) {
    if ((*(Packet + 7) == ARPREQOP) || (*(Packet + 7) == ARPREPOP)) {
      IPStat.ARPRec[*(Packet + 7)]++;

      /* search for source address in ARP table */
      CurEntry = SearchARP(*(ADDRESS_T *)(Packet + 14));
      if ((Merge = (CurEntry != NULL)) != 0) {
        UpdateARP(CurEntry, Packet + 8);
        if (CurEntry->WaitLen != 0) {
          ResetTimer(&(CurEntry->RetryTimer));
          if (SendByPDS(CurEntry->Waiting, CurEntry->WaitLen,
                        CurEntry->EthAddress, CurEntry->ThisIf->HardwAddr,
                        CurEntry->ThisIf->IntNo) == NO_ERR) {
            CurEntry->ThisIf->NrSend++;
            }
          IPBufFree(CurEntry->Waiting);
          CurEntry->WaitLen = 0;
          }
        }
      for (IfList = FirstIf;
          IfList != NULL; IfList = IfList->NextIf) {
        if (((IfList->Flags & (IF_UP | IF_NOARP)) == IF_UP) &&
            (AddressSearch(IfList->FirstAdd, *(ULONG *)(Packet + 24),
                           0xffffffffl) != NULL)) {
          break;
          }
        }
      if (IfList != NULL) {
        if (!Merge) {
          AddARP(*(ULONG *)(Packet + 14), Packet + 8, IfList);
          }
        if (*(Packet + 7) == ARPREQOP) {
          *(USHORT *)(Packet + 6) = htons(ARPREPOP);
          AddressSwap((ULONG *)(Packet + 14), (ULONG *)(Packet + 24));
          memcpy(Packet + 18, Packet + 8, 6);
          memcpy(Packet + 8, IfList->HardwAddr, 6);
          IPStat.NrARPRepS++;
          SendARP(Packet, IfList->IntNo);
          }
        }
      }
    else {
      IPStat.ARPRec[0]++;
      }
    }
  DiscardAll(RecMessage->RecData.RecPacket);
}



/**************************************************************
** NAME:        UpdateARP
** SYNOPSIS:    void UpdateARP(ARP_ENTRY *ThisEntry,
**                             HARDW_ADD *NewAdd);
** 
** DESCRIPTION: Updates ARP entry ThisEntry with Ethernet
**              address NewAdd. The TimeOut value is reset
**              as well.     
** RETURNS:
**************************************************************/
void UpdateARP(ARP_ENTRY *ThisEntry, HARDW_ADD NewAdd)
{
  memcpy(ThisEntry->EthAddress, NewAdd, 6);
  _disable();
  ThisEntry->TimeValue = ARPTIME;     /* reset timer of cache entry */
  _enable();
}

/**************************************************************
** NAME:        SearchARP
** SYNOPSIS:    ARP_ENTRY *SearchARP(ADDRESS_T InterAdd);
** 
** DESCRIPTION: Searches for the IP address InterAdd in
**              the ARP entry table.
** RETURNS:     NULL -->   entry not found
**              else -->   pointer to ARP entry
**************************************************************/
ARP_ENTRY *SearchARP(ADDRESS_T InterAdd)
{
  ARP_ENTRY   *AddPoint = ARPList.NextEntry;

  while (AddPoint != NULL) {
    if (ntohl(AddPoint->IPAddress) >= ntohl(InterAdd)) {
      break;
      }
    AddPoint = AddPoint->NextEntry;
    }

  return (AddPoint == NULL) ? NULL :
         (AddPoint->IPAddress == InterAdd) ? AddPoint : NULL;
}

/**************************************************************
** NAME:        AddARP
** SYNOPSIS:    ARP_ENTRY *AddARP(ADDRESS_T InterAdd,
**                                HARDW_ADD EthAdd,
**                                IFDESCRIPTOR *AsoIf)
**              
** DESCRIPTION: Adds an entry to the ARP table.
**              If an entry for InterAdd already exists, it
**              is updated with EthAdd.
**              In the new entry, ThisIf is set to Asof,
**              RefCount is set to 0 and Waiting to NULL.
**              If a new entry is created, a timer is
**              started for a time out.
** RETURNS:     NULL  --> no buffer space left
**              else  --> a pointer to the new entry
**************************************************************/
ARP_ENTRY *AddARP(ADDRESS_T InterAdd, HARDW_ADD EthAdd,
                  IFDESCRIPTOR *AsoIf)
{
  ARP_ENTRY    *PrevARP = &ARPList;
  ARP_ENTRY    *ARPPoint = ARPList.NextEntry;
  ARP_ENTRY    *NewARP;
  EVENTUNION   ARPUnion;

  while (ARPPoint != NULL) {
    if (ntohl(ARPPoint->IPAddress) >= ntohl(InterAdd)) {
      break;
      }
    PrevARP = ARPPoint;
    ARPPoint = ARPPoint->NextEntry;
    }

  /* if entry already exists, update it */
  if ((ARPPoint != NULL) && (ARPPoint->IPAddress == InterAdd)) {
    UpdateARP(ARPPoint, EthAdd);
    return  ARPPoint;
    }

  /* else, make a new entry */
  if ((NewARP = IPBufGet(sizeof(ARP_ENTRY))) == NULL) {
    IPReport(NOSPACE, NULL, NULL);
    return NULL;
    }

  ARPUnion.ARPData.ARPEntry = NewARP;
  if (SetTimer(LOW, ARPTimeOut, &ARPUnion,
               &(NewARP->TimeValue), ARPTIME) != NO_ERR) {
    IPReport(NOSPACE, (CIPHEAD *)NewARP, NULL);
    return NULL;
    }

  NewARP->NextEntry = ARPPoint;
  NewARP->PrevEntry = PrevARP;
  PrevARP->NextEntry = NewARP;
  if (ARPPoint != NULL) {
    ARPPoint->PrevEntry = NewARP;
    }
  NewARP->IPAddress = InterAdd;
  memcpy(NewARP->EthAddress, EthAdd, 6);
  NewARP->RefCount = 0;
  NewARP->WaitLen = 0;
  NewARP->ThisIf = AsoIf;

  return NewARP;
}

/**************************************************************
** NAME:        DeleteARP
** SYNOPSIS:    void DeleteARP(ARP_ENTRY ThisEntry);
**              
** DESCRIPTION: Deletes an entry from the ARP table.
**              The timer for the time out is reset,
**              the allocated buffer space is freed.
** RETURNS:  
**************************************************************/
void DeleteARP(ARP_ENTRY *ThisEntry)
{
  ResetTimer(&(ThisEntry->TimeValue));
  ResetTimer(&(ThisEntry->RetryTimer));
  if (ThisEntry->WaitLen != 0) {
    IPBufFree(ThisEntry->Waiting);
    }
  ThisEntry->PrevEntry->NextEntry = ThisEntry->NextEntry;
  if (ThisEntry->NextEntry != NULL) {
    ThisEntry->NextEntry->PrevEntry = ThisEntry->PrevEntry;
    }
  IPBufFree(ThisEntry);
}

/**************************************************************
** NAME:        ARPTimeOut
** SYNOPSIS:    void ARPTimeOut(EVENTUNION *ARPUnion);
**
** DESCRIPTION: Flushes an ARP entry after a time out.
** RETURNS:   
**************************************************************/
void ARPTimeOut(EVENTUNION *ARPUnion)
{
  DeleteARP(ARPUnion->ARPData.ARPEntry);
}

/**************************************************************
** NAME:        SendByPDS
** SYNOPSIS:    int SendByPDS(BYTE *ThisPacket,
**                            USHORT PackLen,
**                            HARDW_ADD Destin,
**                            HARDW_ADD Source,
**                            USHORT Inter);
**           
** DESCRIPTION: Sends ThisPacket through the Ethernet
**              interface. Destin and Source are used as the
**              Ethernet Destination and Source address.
**              Inter is the used interrupt number.
**              The first 14 bytes of ThisPacket are used
**              for adding the addresses and the IP type.
**              ThisPacket should be large enough to contain
**              60 bytes.
** RETURNS:     NO_ERR     -->   no error
**              ERRORETHER -->   error occured
**************************************************************/
int SendByPDS(BYTE *ThisPacket, USHORT PackLen, HARDW_ADD Destin,
              HARDW_ADD Source, USHORT Inter)
{
  memcpy(ThisPacket, Destin, 6);
  memcpy(ThisPacket + 6, Source, 6);
  *(USHORT *)(ThisPacket + 12) = ETHERIP;
  while (PackLen < 46) {
    ThisPacket[14 + PackLen++] = 0;
    }
  return (PktSendPkt(Inter, ThisPacket, PackLen + 14) == PKT_OK)
        ? NO_ERR : ERRORETHER;
}

/**************************************************************
** NAME:        SendARP
** SYNOPSIS:    int SendARP(BYTE *ThisARP, USHORT Inter);
**           
** DESCRIPTION: Sends the ARP message in ThisARP through the
**              Ethernet. Destination and Source are derived
**              from the ARP message.
**              Inter is the used interrupt number.
** RETURNS:     NO_ERR     -->   no error
**              ERRORETHER -->   error occured
**************************************************************/
int SendARP(BYTE *ThisARP, USHORT Inter)
{
  BYTE   ARPMess[60];

  memcpy(ARPMess, ThisARP + 18, 6);
  memcpy(ARPMess + 6, ThisARP + 8, 6);
  *(USHORT *)(ARPMess + 12) = ETHERARP;
  memcpy(ARPMess + 14, ThisARP, 28);
  memset(ARPMess + 42, 0, 18);
  return (PktSendPkt(Inter, ARPMess, 60) == PKT_OK) ? NO_ERR : ERRORETHER;
}

/**************************************************************
** NAME:        RetryARP
** SYNOPSIS:    void RetryARP(EVENTUNION *ThisUnion);
**
** DESCRIPTION: Processes the time out of an ARP request.
**              If the retry counter is not 0, another
**              ARP request will be sent.
**              If an error occures in setting a timer or if
**              the retry counter has reached 0, the ARP
**              entry is deleted and the associated packet
**              thrown away.
** RETURNS:  
**************************************************************/
void RetryARP(EVENTUNION *ThisUnion)
{
  ARP_ENTRY   *ThisEntry = ThisUnion->ARPData.ARPEntry;
  EVENTUNION  ARPRetrUn;

  if ((ThisEntry->RetryCount)-- != 0) {
    memcpy(ARPPacket + 8, ThisEntry->ThisIf->HardwAddr, 6);
    *(ADDRESS_T *)(ARPPacket + 14) = ThisEntry->ThisIf->FirstAdd->ThisAddress;
    memcpy(ARPPacket + 18, BrdCastAdd, 6);
    *(ADDRESS_T *)(ARPPacket + 24) = *(ADDRESS_T *)(ThisEntry->Waiting);
    IPStat.NrARPReqS++;
    SendARP(ARPPacket, ThisEntry->ThisIf->IntNo);
    ARPRetrUn.ARPData.ARPEntry = ThisEntry;
    if (SetTimer(LOW, RetryARP, &ARPRetrUn,
                 &(ThisEntry->RetryTimer), ARPRETRTIME) != NO_ERR) {
      IPReport(NOSPACE, NULL, NULL);
      DeleteARP(ThisEntry);
      }
    }
  else {
    DeleteARP(ThisEntry);
    }
}


