/*
 * make.c
 *
 *      An imitation of the Unix MAKE facility
 *
 *      Copyright (C) 1984 by Larry Campbell, 73 Concord St., Maynard, Mass.
 *      Rewritten w/modifications by Mike Hickey, University of DC
 *
 *      This software may be freely copied and disseminated for noncommercial 
 *      purposes, if and only if this entire copyright statement and notice 
 *      is included intact.  This software, and the information contained 
 *      herein, may not be used for commercial purposes without my prior 
 *      written permission.
 *
 *      This program runs a new shell (COMMAND.COM) for each command specified 
 *      in the makefile.  This, I recommend that you put a copy of the shell in
 *      ramdisk (if you have one).  Assuming your ramdisk is called, say, drive
 *      F:, you would:
 *      
 *
 *              COPY A:\COMMAND.COM F:
 *              SET COMSPEC=F:\COMMAND.COM
 *
 */

/* Certain portions of this software are Copyright (C) 1985 by Dan Grayson,
   2409 S. Vine St, Urbana, IL 61801, namely all those lines which differ
   between versions 2.10 and 2.11.  Qualitative descriptions of these 
   changes are described in the edit history below.
        Provided this copyright notice is not changed, and VERS211 below
   is not changed, and VERS211 is still prints when 'usage()' runs, these
   portions may be freely copied and used for any purpose, with one exception,
   namely:  those persons or corporations who claim a copyright on this
   program or a part of it may not use these portions for any commercial
   purpose whatsoever.  In particular, they may not collect royalties on
   any version of this program which includes these portions. */

/*
 * Edit history:
 *
 *  2.14      Allowed a default file MAKE.INI that is always read.
              Now it makes the prerequisites in left right order.
              Fixed various bugs involving default rules.
              Improved the behavior with names that don't correspond to
              existing files.  -dan grayson
                         
 *  2.13      Allowed the use of multiple default rules for a given
 *            default target, enforced the requirement of a single
 *            instance of a given explicit target:
 *                    *.obj : *.asm
 *                            asm $*;
 *                    *.obj : *.c
 *                            cc $* -ia
 *            is legal, but:
 *                    foo.obj: bar.asm
 *                            asm bar,foo;
 *                    foo.obj: bar.c
 *                            cc bar -ia -ofoo
 *            is not (and not terribly useful or interesting).
                Peter da Silva (the mad Australian)
                        UUCP: ...!shell!neuro1!{hyd-ptd,baylor,datafac}!peter
 *  2.12      Made it compilable by Lattice C version 2.14. Use '-dLATTICE'
 *            on the command line for this. Default is Microsoft C.
 *  2.11        Fixed breakout_symbols, which tried to return a pointer to
                        a local variable!
                Made symbol substitution occur in all lines, not just shell
                        command lines.
                Fixed breakout_symbols, which blew up when a symbol was 
                        undefined.
                Allowed blank lines, which are ignored now.
                Change command line length to 1000.
                Fixed it so MAKE, when no targets are specified on the command
                        line, will simply make the first target in the makefile.
                Remove the command line symbol definition option.
                Changed the line continuation character to \ (it was - )
                Now symbol definition lines do NOT begin with $.
                Fixed numerous bugs dealing with character arrays and their
                        lengths.
                Now a shell command line which begins with @ is not echoed.
                A shell command line beginning with + is executed through
                        command.com ; this makes io redirection and pipes
                        available, but the exit code of the program cannot
                        be checked due to a misfeature of command.com.
                A shell command line beginning with - may return a nonzero
                        exit code without halting 'make'.
                Fixed it so a target:prerequisite line followed by no how-to
                        lines is interpreted not as an error, and not as 
                        sharing the how-to lines following the next
                        target:prerequisite line, but is considered fulfilled
                        by no action other than making all the 
                        prerequisites.
                Fixed the bug which meant the return code from the commmand
                        was never dicovered.  This resulted from using 
                        "system", which uses "command.com", which hides the
                        return code of the program it runs.  Resident 
                        commands can still be used, nevertheless.
                Error messages now include the line number of the makefile,
                        if relevant.
                Made the return code of the command print out if nonzero.
                Now the copyright notice only prints when the usage appears.
                Convert to Microsoft vers 3.00, large memory model.
                                - dan grayson
 *  2.10        Fix bug in abort routine, update copyright notice
 *  2.09        Set up for command line parsing of macros
 *  2.08        Remove edit 2.05; keep debug a compile-time option
 *  2.07        Finish macro parsing
 *  2.06        Add initial code for macro handling
 *  2.05        Add -d (debug) switch
 *  2.04        Add error message handling (doserror).
 *  2.03        Add -i (ignore errors) switch.
 *  2.02        Add -s (silent run) switch.
 *  2.01        Convert to Lattice 2.14.  Clean up code for faster execution.
 *  1.11        Make default makefilename be MAKEFILE.
 *  1.10        Add -n (trace) switch.
 *  1.09        Allow multiple targets before colon.
 *  1.08        Cleanup and speedup of makefile parsing.
 *  1.07        Add continuation lines (hyphen as last char of line)
 *  1.06        Don't need to make dummy FCBs, zero seems to work OK.
 *  1.05        Fix bug finding COMSPEC when it's not first env variable
 *  1.04        Wordier usage text, including copyright notice.
 *              Remove printf's (except when DEBUG on) to shrink EXE
 *  1.03        Don't uppercase shell command and fix datetime bug
 *  1.02        Random cleanup
 *  1.01        The beginning
 */

#define VERSION "MAKE ver. 2.10 Copyright (C) 1984 by Larry Campbell, Maynard Mass."
#define VERS211 "MAKE ver. 2.14 Portions copyright (C) 1985 by Dan Grayson, Urbana IL."

#ifdef LATTICE
#define assert(a)
#define perror error
#include <dos.h>
#include <stdio.h>
#include <ctype.h>
extern char *strchr();

#else

#include <assert.h>
#include <process.h>
#include <dos.h>
#include <stdio.h>
#include <malloc.h>
#include <ctype.h>
#include <string.h>

#endif

#include "make.h"
#include "mstring.h"

#define MAKEINI "make.ini"

char *dos_commands[] = {
      "dir", "type", "rem", "pause", "date", "time",
      "ren", "rename", "ver", "vol", "break", "verify",
      "mkdir", "md", "exit", "ctty", "echo", "if", "cls",
      "chdir", "cd", "rmdir", "rd", "copy", "del", "erase",
                         NULL };

#define PREREQ_MAGIC 123
#define FILE_MAGIC 543
#define SHELL_MAGIC 678
#define TARG_MAGIC 987
#define SYMBOL_MAGIC 653
#define MAXLIN 1000
#define SYMLEN 1000
#define MAXTARGETS 100


#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#define EQUAL 0

#define EOS '\0'                /* End Of String */

#define LINESIZE 1000            /* max length of input line */
#define MAXCOMMON 8             /* max no. of targets with common prereqs */

#define NEGINF 0x8000           /* smallest int */
#define LONGNEGINF 0x80000000L  /* smallest long int */
#define LONGPOSINF 0x7fffffffL  /* largest long int */

#define tfree(x) if (x) free(x),x=NULL

#ifdef LATTICE
extern char _dos;
#else
extern unsigned char _osmajor;           /* in MS ver 3, gives version of OS to left of point */
extern int _doserrno;
#endif

char  *talloc(), *strperm();

extern int linenumber;   /* defined in mstring.c */
long getdatetime ();

    static int dontworry=0;

union REGS inregs, outregs;

struct {
    char reserved[21];
    char attr;
    unsigned time, date, size_l, size_h;
    char pname[13];
} 
find_buf;

/*
 * MAKE parses the make file and constructs a somewhat tangled
 * directed graph describing dependencies.  There is a list of
 * TargNode structures, each of which points to a list of
 * prereq_node structures.  These both point to FileNode structures,
 * which contain information about files such as date-time of write
 * and the filename.  To keep things simple, MAKE insures that only
 * one FileNode exists for any given file.
 */

typedef struct FileNode {
    #ifndef NDEBUG
    int magic;
    #endif
    char *fname;
    struct FileNode *chain;
    }    *fileptr;
fileptr  FileNodeList, NewFileNode ();

typedef struct TargNode {
    #ifndef NDEBUG
    int magic;
    #endif
    struct TargNode *next;
    fileptr file;
    struct prereq_node *PreqList;
    struct shell_node *shell_list;
    }    *targptr;
targptr first_targ, target_list, new_target (), lookup_target ();

typedef struct prereq_node {
    #ifndef NDEBUG
    int magic;
    #endif
    struct prereq_node *next;
    fileptr file;
    } *preqptr;
preqptr NewPreqNode ();

typedef struct shell_node {
    #ifndef NDEBUG
    int magic;
    #endif
    struct shell_node *next;
    char *command;
    unsigned quiet : 1, ignore : 1, useshell : 1;
    } *shellptr;

typedef struct symbol_node {
    #ifndef NDEBUG
    int magic;
    #endif
    char *token, *value;
    struct symbol_node *next;
    }    *symbptr;
symbptr SymbolList;

static char *makefilename;
static int status, tracing, quietly, ignore_errors;

usage ()
{
    puts (VERS211);
    puts (VERSION);
    puts ("This program may be copied freely for noncommercial purposes.  It may");
    puts ("not be copied for commercial use without the author's written permission.\n");
    puts ("This  program is an imitation of the MAKE program supplied with Unix.");
    puts ("Usage: make [target ...] [options ...]");
    puts ("Options:");
    puts ("        -f filename specify makefile, default is MAKEFILE");
    puts ("        -i          ignore errors while processing");
    puts ("        -n          trace and print, but don't execute commands");
    puts ("        -s          suppress MAKE echoing commands");
    exit (21);
}

main (argc, argv)
int argc;
char **argv;
{
    int
        argi,
        targi,
        linlen;

    char  *targname[MAXTARGETS];

    if (
        #ifdef LATTICE
          _dos
        #else
          _osmajor
        #endif
                 < 2) error ("Must have DOS 2.XX or higher");
    makefilename = "MAKEFILE";
    tracing = quietly = ignore_errors = FALSE;
    target_list = 0;
    SymbolList = 0;
    FileNodeList = 0;
    targi = 0;
    for (argi = 1; argi < argc; argi++) {
        if (argv[argi][0] == '-')               /* switch */
            switch (argv[argi][1]) {            /* switch character */
            case 'f':
                if (argi < (argc - 1))      makefilename = argv[++argi];
                else                        usage ();
                break;
            case 'i':                ignore_errors = TRUE;             break;
            case 'n':                tracing = TRUE;                   break;
            case 's':                quietly = TRUE;                   break;
            default:                 usage ();
            }
        else {                          /* target name */
            if (targi == MAXTARGETS)    error ("Too many target files");
            targname[targi] = strperm (argv[argi]);
            targi++;
        }
    }
    if (tracing && quietly) quietly = 0;

    { FILE *fd, *fopenp(char *,char *);
       if (fd = fopenp(MAKEINI,      "r")) parse(fd);
       first_targ = NULL;
       if (fd = fopen (makefilename, "r")) parse(fd); else /* no makefile, okay */ ;
    }

    if (targi)
            /* make the targets specified on the command line */
            for (argi = 0; argi < targi; argi++) make (targname[argi]);
    else if (first_targ) make(first_targ->file->fname);
    else error ("No targets specified");
    return 0;
}

parse (fd)
FILE *fd;
{
    int targi=0,   i;
    char  c,  *sp,  *dp;  int wlen;
    mstring input=NULL;
    targptr targ[MAXTARGETS];

    while (1) {
        tfree(input);
        input = mfgets(fd);
        if (input==NULL) break;
        #if DEBUG
            printf ("Makefile Input: \"%s\"\n", input);
            #endif
        sp = input;
        passpace(&sp);
        if (*sp==0 || *sp=='#') continue;
           /* ignore comment lines and blank lines */

        if (isspace(*input)) {   /* if leading space, then this is a shell line */
                if (targi == 0) error ("Target line must come before shell lines");
                sp = input; passpace(&sp);
                for (i = 0; i < targi; i++)  NewShellLine (targ[i], sp);
                continue;
        }

        {               /* substitute for symbols - this will be done later
                           for shell lines, to take special symbols like $*
                           into account, which can only be known at run time
                           */
              breakout_symbols(&input);
        }

        {       /*** check for the form 'name = value'   ***/
                char *endword;
                sp=input;
                password(&sp);    endword = sp;
                passpace(&sp);
                if (*sp == '=') {
                    targi=0;
                    sp++;
                    *endword = EOS;
                    SetSymbol (input, sp) ;
                    continue;
                }
        }                               /* end of macro parsing */


        /**** now we know this is a 'targets : prerequisite' line ***/


        for ( targi=0, wlen=1, dp = sp = input; 1 ; wlen++, sp++)        /*** collect the targets ***/
            if (    *sp == ':'
                #ifdef MSDOS
                 && (wlen!=2 || isspace(sp[1]))   /* allow filenames like C:FOO.BAR, so the drive can be specified */
                #endif
                 ||  isspace (*sp)) { /* space or colon ends target name */
                if (targi == MAXTARGETS) error ("Too many targets in one line");
                c = *sp;  *sp = EOS; targ[targi] = new_target (dp); *sp = c;
                if (first_targ == NULL  &&  *dp != '*') first_targ = targ[targi];
                targi++;
                passpace(&sp);
                if (*sp == ':') break;
                dp = sp;
                wlen = 0;
            }
            else if (*sp == EOS) error ("no separating ':' ");

        sp++;   /* pass over the ':' */
        if (targi == 0) error ("No target file before ':' ");
        while(1) {              /*** collect the prerequisites ***/
            passpace(&sp);
            if (*sp == EOS) break;            /* end of line */
            dp = sp;                    /* beginning of prereq word */
            passnonsp(&sp);
            c = *sp;
            *sp = EOS ;                /* end of prereq word */
            for (i = 0; i < targi; i++) LinkPreq (targ[i], NewFileNode(dp) );
            *sp = c;
        }
    }   /* end while */
    tfree(input);
    fclose (fd);
    linenumber = 0;
}

/*
 * new_target
 *
 *      We think we have a new target;  this routine scans the
 *      target list and if we've already seen this name returns
 *      the node we already built.  Otherwise, a new node is
 *      allocated and linked.
 */

        targptr
new_target (name)
    char *name; {
    targptr targ ;

    #if DEBUG
      printf ("new_target (\"%s\")\n", name);
      #endif
    if (name[0] != '*') { /* look for name already on target list, unless it's a default rule */
        targptr p = target_list;
        while (p) {
                assert (p->magic == TARG_MAGIC);
                if (EQUAL == strcmp(name,p->file->fname)) return p;
                p = p->next;
                }
        }
    
    /* add new target to top of list */
    targ = (targptr) talloc (sizeof (struct TargNode));
    #ifndef NDEBUG
       targ->magic = TARG_MAGIC;
       #endif
    targ->file = NewFileNode (name);
    targ->next = target_list;
    targ->shell_list = NULL;
    targ->PreqList = NULL;
    target_list = targ;
    return targ;
}

SetSymbol (name, value)
char *name, *value;
{
    symbptr sym;
    for (sym = SymbolList; sym; sym = sym->next)
         if (0==strcmp(sym->token,name)) {
                free(sym->value);
                sym->value = strperm(value) ;
                return;
                }
    sym = (symbptr) talloc (sizeof (struct symbol_node));
    #ifndef NDEBUG
        sym->magic = SYMBOL_MAGIC;
        #endif
    sym->token = strperm (name);
    sym->value = strperm (value);
    sym->next = SymbolList;
    SymbolList = sym;
}

/*
 * NewShellLine
 *
 *      Add a shell command to the list of commands needed to update
 *      a target file
 */

NewShellLine (targ, line)
targptr targ;
char *line;
{
    shellptr snode, new;

    #if DEBUG
         printf ("NewShellLine (%x, \"%s\")\n", targ, line);
         #endif
    new = (shellptr) talloc (sizeof (struct shell_node));
    new->next = 0;
    #ifndef NDEBUG
        new->magic = SHELL_MAGIC;
        #endif
    new->useshell = new->ignore = 0;
    new -> quiet = quietly ;
    for ( ; 1 ; line++, passpace(&line) )
       if      (line[0] == '@')  new->quiet = 1;
       else if (line[0] == '+')  new->useshell = 1;
       else if (line[0] == '-')  new->ignore = 1;
       else break;
    new->command = strperm (line);
    snode = targ->shell_list;
    if (snode) {
        assert (snode->magic == SHELL_MAGIC);
        while (snode->next) {
                snode = snode->next;
                assert (snode->magic == SHELL_MAGIC);
                }
        snode->next = new;
    }
    else
        targ->shell_list = new;
}

/*
 * LinkPreq
 *
 *      Link a new prerequisite file onto prereq list for a target.
 */

LinkPreq (targ, fnode)
   targptr targ;  fileptr fnode; {
   preqptr p,np=NewPreqNode(fnode);

    #if DEBUG
        printf ("LinkPreq (\"%s\")\n", fnode->fname );
        #endif

   /* we attach the prerequisite to the end of the list so prerequisites
      are made in left to right order */

   p = targ->PreqList;
   if (p) {
        assert (p->magic == PREREQ_MAGIC);
        while (p->next) {p=p->next; assert (p->magic == PREREQ_MAGIC);}
        p->next=np;
        }
   else {
        targ->PreqList = np;
        }
    }

/*
 * NewPreqNode
 *
 *      Allocate and return a new prerequisite node
 */

        preqptr 
NewPreqNode (fnode)
        fileptr fnode;
{
    preqptr new;

    #if DEBUG
        printf ("NewPreqNode (struct FileNode *%x, \"%s\")\n",fnode,fnode->fname);
        #endif
    new = (preqptr) talloc (sizeof (struct prereq_node));
    new->next = NULL;
    #ifndef NDEBUG
        new->magic = PREREQ_MAGIC;
        #endif
    new->file = fnode;
    return new;
}

/*
 * NewFileNode
 *
 *      Return FileNode pointer for a file;  returns pointer to
 *      existing FileNode if this file already seen, else allocates
 *      and inits a new node
 */

        fileptr
NewFileNode (name)
        char *name;
{
    fileptr fnode;

    #if DEBUG
        printf ("NewFileNode (\"%s\")\n", name);
        #endif
    for ( fnode = FileNodeList; fnode; fnode = fnode->chain) {
        assert (fnode->magic == FILE_MAGIC);
        if (strcmp (name, fnode->fname) == 0) {
    #if DEBUG
        printf ("NewFileNode returning existing node\n");
        #endif
            return fnode;
        }
        
    }
    fnode = (fileptr) talloc (sizeof (struct FileNode));
    fnode->fname = strperm (name);
    fnode->chain = 0;
    #ifndef NDEBUG
        fnode->magic = FILE_MAGIC;
        #endif
    fnode -> chain = FileNodeList;
    FileNodeList = fnode;
    return fnode;
}

/*
 * 
getdatetime
 *
 *      Return date-time of a file squished into a long so compares
 *      are easy
 */

        long 
getdatetime (name,deflt)
        char *name;      long deflt;
{
    long datetime;
    int dma_off;

    #ifdef DEBUG
        printf("getdatetime(%s)\n",name);
        #endif

    inregs.x.ax = 0x2F00;            /* get current DMA address */
    intdos (&inregs, &outregs);   /*  .. */
    dma_off = outregs.x.bx;           /* and save for later restoration */
    inregs.x.dx = (int) & find_buf;  /* set DMA to GNJFN block */
    inregs.x.ax = 0x1A00;
    intdos (&inregs, &outregs);

    inregs.x.dx = (int) name;       /* pathname */
    inregs.x.cx = 0;                 /* attributes */
    inregs.x.ax = 0x4E00;            /* GTJFN */
    outregs.x.cflag = 0;
    status = intdos (&inregs, &outregs);
    if (
         #ifndef LATTICE
           outregs.x.cflag ||
         #endif
           (status & 1)
                         ) {
    #if DEBUG
        printf ("File \"%s\" does not exist\n", name);
        #endif
    return deflt;
    }

    inregs.x.dx = dma_off;
    inregs.x.ax = 0x1A00;
    intdos (&inregs, &outregs);

    /* the ^LONGNEGINF in the next line is an artifice so we can get by without
       unsigned longs - future comparisons of datetimes will work AS THOUGH
       we were using unsigned longs, as we switch here all the negative
       numbers with the positive ones.  */
    datetime = ((((long) find_buf.date) << 16) | find_buf.time) ^ LONGNEGINF;

    #if DEBUG
      printf ("file=%s date=%04x time=%04x datetime=%08lx\n",
        find_buf.pname, find_buf.date,find_buf.time,datetime);
        #endif

    return datetime;
}

/*
 * make(name)
 *
 *      This routine actually does the work.  It scans the list of
 *      targets parsed from the makefile, and checks the target's
 *      prerequisites date/time values against the target's.  If
 *      the prerequisite is itself a target (present in target_list),
 *      we call make recursively to check it.  Then, if any of our
 *      prerequisites are newer than we are, we execute all our shell
 *      commands.  If there are no prerequisites specified at all, then
 *      also execute all our shell commands.
 */

        int
make (targname)         /* use fnode instead of fname here */
    char *targname;
{          
    targptr targ;
    preqptr prereq;
    long NewestPreq=LONGNEGINF+1, targtime=LONGNEGINF; /* the +1 will arrange it so the commands will execute if there are no prerequisites and the target file does not exist */

    #if DEBUG
        printf ("Making %s\n", targname);
        #endif

    if ((targ = lookup_target (targname)) == NULL)
       return TryDefault( targname , 0);
    prereq = targ->PreqList; 
    if (prereq)
        {
        for ( ; prereq; prereq = prereq->next) {
            long date;
            assert (prereq->magic == PREREQ_MAGIC);
            make (prereq->file->fname);             /* recursively make */
            date =  getdatetime(prereq->file->fname,LONGPOSINF);
            if (date > NewestPreq) NewestPreq = date;
            }
        }
    targtime = getdatetime(targ->file->fname, LONGNEGINF);
    #if DEBUG
        printf ("Target \"%s\" datetime is %08lx, newest prereq is %08lx\n", targ->file->fname, targtime, NewestPreq);
        #endif
    if (targ->shell_list)
        if (targtime < NewestPreq) build(targ);
        else ;
    else {   /* no shell lines specified, so there might be some implicit prerequisites */
         int i;
         dontworry ++;
         i = TryDefault( targname , targtime < NewestPreq );
         dontworry --;
         return i;
         }
    return 1;
}


TryDefault(targname, outofdate)
        char *targname;
        int outofdate; /* non zero if the target is out of date with respect to some other prerequisites */
        {
        targptr targ;
        char * ext = strchr (targname, '.');
        dontworry ++;
        if (ext != NULL)
            for (targ = target_list ; targ ; targ = targ -> next )
                if (targ->file->fname[0] == '*' &&
                    0 == strcmp ( ext , targ->file->fname+1 ) ) {
                        char *root = msubstr( targname , 0 , ext-targname );
                        char *cname;
                        int worked;
                        long cnamedate;
                        cname = mstrcat( root ,targ->PreqList->file->fname+1 );
                        worked = make ( cname ) ;
                        SetSymbol ( "*" , root ) ;  /* should really have a way of freeing this space later... */
                        if (worked && !outofdate) cnamedate = getdatetime(cname,LONGNEGINF);
                        free(cname);
                        free(root);
                        if (!worked) continue; /*** was: goto ret0; ***/
                        if (outofdate || cnamedate > getdatetime(targname,LONGNEGINF)) build (targ) ;
                        goto ret1;
                        }

        if (getdatetime(targname,LONGNEGINF) > LONGNEGINF) goto ret1;

        ret0:                   /* unsuccessful return */
            if (--dontworry) return 0;
            else {
                error(mstrcat("Don't know how to make ",targname));
                }

        ret1:                   /* successful return */
            dontworry--;
            return 1;
        }




/*
 * build
 *
 *      Invoke shell commands to build a target file
 */

build (targ)
    targptr targ;
    {
    shellptr snode;
    char *cmd;
    int  runsts = 0;

    #if DEBUG
        printf ("Building \"%s : %s...\"\n", targ->file->fname, targ->PreqList->file->fname);
        #endif
    for ( snode = targ->shell_list; snode; snode = snode->next, free(cmd) ) {
        char *p, **q, *cmdstart, *cmdname, *cmdtail, sys=FALSE;

        assert (snode->magic == SHELL_MAGIC);
        cmd = strperm(snode->command);
        breakout_symbols(&cmd);           /* notice that this may introduce a space at the beginning of the command line */
        cmdstart = cmd; passpace(&cmdstart);

        if (!snode->quiet)  fputs (cmdstart, stdout);
        if (tracing) {
               puts ("");   /* EXEC does newline, so must be faked */
               continue;
               }

        /* separate the name of the command */
        p = cmdstart ; passnonsp(&p);
        if (*p)  *p = EOS, cmdname = strperm(cmdstart), *p = ' ';
        else               cmdname = strperm(cmdstart);

        cmdtail = p;  passpace(&cmdtail);

        /* find whether it is a dos command */
        strlwr(cmdname);          /* lower  case for comparison */
        for (q=dos_commands ; *q ; q++) if (0==strcmp(*q,cmdname)) break;

        if (*q || snode->useshell)        /* must we use command.com ? */

        #ifndef LATTICE
                if (0==strcmp(cmdname,"chdir") || 0==strcmp(cmdname,"cd"))
                        if (*cmdtail) {   /* chdir with arg */
                            char *q=cmdtail;
                            passnonsp(&q);  *q = EOS;
                            runsts = chdir(cmdtail), sys=FALSE;
                            }
                        else {                      /* chdir without arg */
                            char name[200];
                            if (getcwd(name,200)) {
                                if (!snode->quiet) putchar('\n');
                                fputs(name,stdout);
                                runsts = 0;
                                }
                            else error("path name too long");
                             }
                else
        #endif
                        {                           /* resident command */
                        if (strlen(cmdstart) > 128) error("shell command line too long");
                        #if DEBUG
                           printf("\nsystem -- %s --",cmdstart);
                           #endif

                        #ifdef LATTICE
                        runsts = forklp(cmdname,cmdname,"",cmdtail,NULL) ?
                                         -1         /* error */    :
                                         wait()   /* get status */ ;
                        #else
                        runsts = system(cmdstart), sys = TRUE;
                        #endif

                        }
        else    {                                   /* transient command */
                if (!snode->quiet) putchar ('\n');
                #if DEBUG
                  printf ("spawnvp -- %s --\n",cmdstart);
                  #endif
                runsts = mspawn(cmdstart), sys = FALSE;
                /* can't use 'system()' here, because command.com does not
                   return the exit code of the program */
                }
        putchar('\n');      /* some programs do not end with return */
        if (runsts == -1)
           perror(
                sys ? "command.com not found" : "can't run program"
                ),
           exit(23);
        if (runsts > 0 && !snode->ignore && !ignore_errors) {
              fputs( " --- return code ", stderr);
              fputint(runsts,stderr);
              fputs(" ---\7",stderr);
              exit(runsts);
              }

    free(cmdname);
    free(cmd);
                     
    }
}

        targptr 
lookup_target (name)
char *name;
{
    targptr targ;
    for ( targ = target_list; targ ; targ = targ->next)
        if (strcmp (name, targ->file->fname) == 0) break;
    return targ;
}

breakout_symbols (cmdlinptr)
char **cmdlinptr; {
    char *cmdlin = *cmdlinptr, *cmd = talloc(LINESIZE+100);
    symbptr sym;
    char   symcmp[SYMLEN];
    int i, paren, cmdptr;

    #if DEBUG
        printf("breakout_symbols (\"%s\")\n", cmdlin);
        #endif
    /* this routine doesn't check for overflow of the line ! */

    strcpy ( cmd, "");
    cmdptr = 0;
    while (1) {
        while (*cmdlin != '$' && *cmdlin != EOS) {
                if (cmdptr >= LINESIZE) error ("Line too long after symbol substitution");
                cmd[cmdptr++] = *cmdlin++;
                }
        if (cmdptr >= LINESIZE) error ("Line too long after symbol substitution");
        cmd[cmdptr] = EOS;
        if (0==*cmdlin) break;            /* end of line */
        cmdlin++;               /* pass the $ */
        /* now we know we are looking at a symbol */
        if (*cmdlin == '(') paren = 1, cmdlin++; else paren=0;
        for (i = 0; i < SYMLEN-1 && (*cmdlin == '*' || isalnum (*cmdlin)); )
            symcmp[i++] = *cmdlin++;
        symcmp[i] = EOS;
        if (paren)
           if (*cmdlin == ')') cmdlin++;
           else puts ("No closing paren on shell line macro");
        for ( sym = SymbolList; 1 ; sym = sym->next) {
            if (sym==NULL)  error (mstrcat("Undefined symbol ", symcmp) );
            assert (sym->magic == SYMBOL_MAGIC);
            if (strcmp (sym->token, symcmp) == 0) break;
        }
        strcpy ( cmd + cmdptr , sym->value );
        cmdptr = strlen ( cmd ) ;
    }

    free(*cmdlinptr);
    *cmdlinptr = strperm(cmd);

    #if DEBUG                    
        printf ("breakout_symbols returning (\"%s\")\n", cmd);
        #endif

    free(cmd);

}

#ifdef LATTICE

strlwr(p)
        char *p;{
        for ( ; *p ; p++) *p = tolower(*p);
        }

#endif
