/*****************************************************************************
*   "Irit" - the 3d polygonal solid modeller.				     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
* Main definition Header file  for Irit - the 3d polygonal solid modeller.   *
*****************************************************************************/

#ifndef	IRIT_H
#define	IRIT_H

#include <setjmp.h>	/* Used for the long jumps - to main iteration loop. */
#include "irit_sm.h"
#include "cagd_lib.h"	   /* We define curves and surfaces handles as well. */

#ifdef __MSDOS__
#define MAX_OBJ_LIST	50	   /* Maximum size of geometric object list. */
#else
#define MAX_OBJ_LIST	250
#endif /* __MSDOS__ */

#define DEFAULT_LOAD_COLOR 1  /* Default colors for object loaded using LOAD */
#define DEFAULT_BOOL_COLOR 2  /* command, for boolean result objects, for    */
#define DEFAULT_ICRV_COLOR 14 /* boolean intersection curves and for basic   */
#define DEFAULT_PRIM_COLOR 4  /* primitives colors, respectively.	     */

/* Objects (ObjectStruct - OBJECT_TYPE) types.				     */
/* Dont change the order of these objects as the over loading table (see     */
/* OverLoad.c) is hardwired to it. If you add objects update that module     */
/* properly.								     */
typedef enum {
    UNDEF_OBJ = 0,

    POLY_OBJ,
    NUMERIC_OBJ,
    VECTOR_OBJ,
    MATRIX_OBJ,
    CURVE_OBJ,
    SURFACE_OBJ,
    STRING_OBJ,
    OBJ_LIST_OBJ,
    CTLPT_OBJ,

    ANY_OBJ = 100		 /* Match any object type, in type checking. */
} IritObjectType;

#define NUM_OF_OBJECT_TYPES	8

#define DEFAULT_RESOLUTION	20	/* Used in Primitiv/Boolean modules. */
#define DEFAULT_DRAW_CTLPT	FALSE	/* If control mesh/poly to be drawn. */
#define DEFAULT_INTERNAL	FALSE		  /* View of Internal edges? */
#define DEFAULT_INTERCRV	FALSE	      /* Return intersection curves? */
#define DEFAULT_ECHOSRC		TRUE	            /* Echo sourced program? */
#define DEFAULT_DUMPLVL		0   /* Min. information. 1,2 are also valid. */

#define MACHINE_MSDOS		1
#define MACHINE_SGI		2
#define MACHINE_HP		3
#define MACHINE_APOLLO		4
#define MACHINE_SUN		5
#define MACHINE_UNIX		6

#define KV_MIN_LEGAL		-9999
#define KV_UNIFORM_OPEN		-10000
#define KV_UNIFORM_FLOAT	-10001

#define MAX_NUM_ATTRS		10

/*****************************************************************************
* Attributes - geometry types (Surfaces/Curves/Polygons/Polylines) have this *
* structure for keeping general attributes like colors etc.		     *
*****************************************************************************/
typedef struct AttributeStruct {
    ByteType Color;				       /* Color of geometry. */
    int NumStrAttribs;
    char *StrAttrName[MAX_NUM_ATTRS + 1];	    /* Generic string attrs. */
    char *StrAttrData[MAX_NUM_ATTRS + 1];
} AttributeStruct;

/*****************************************************************************
* Global data structures:						     *
* Objects in the system might be (real) scalars, (R3) vectors, matrices      *
* (4 by 4 - transformation matrix), strings of chars, lists of objects, or   *
* geometric objects. All but the last are simple and all their data is saved *
* in the object space itself. The last (geometric) object points on a	     *
* polygonal list of the form:						     *
*									     *
* Polygon -> Polygon -> Polygon -> Polygon -> .... -> NULL		     *
*    |		|	   |	      |					     *
*    V          V          V          V					     *
*  VList      VList      VList      VList	(VList = Vertex List)	     *
*									     *
* Each VList is a CIRCULAR vertex list. Each VList element (VertexStruct)    *
* implicitly defines an edge from this vertex, to the next. As each edge     *
* is used by exactly two polygons, a pointer to the other polygon using this *
* edge exists in the VertexStruct. Each polygon has also its Plane	     *
* definition for fast processing, with its normal pointing INTO the object.  *
*   Few other tags & flags are included in the data structures for different *
* modules.								     *
*   Note, vertices are not shared by few VLists/Polygons although it may     *
* decrease memory usage (suprisingly, not much). The main reason to that is  *
* the basic assumption of this solid modeller, which is simplicity...	     *
*****************************************************************************/

/*****************************************************************************
* Vertex Type - holds single 3D point, including some attributes on it as    *
* Tags & Count. The 3D coordinates are saved in Pt. Pointer to next in chain *
* is Pnext, and the pointer to the adjacent polygon (to the edge defined by  *
* this Vertex and Vertex->Pnext) is PAdj.				     *
*****************************************************************************/
typedef struct VertexStruct {
    ByteType Count, Tags;				 /* Same attributes. */
    PointType Pt;			       /* Holds X, Y, Z coordinates. */
    NormalType Normal;		       /* Hold Vertex normal into the solid. */
    struct PolygonStruct *PAdj;			     /* To adjacent polygon. */
    struct VertexStruct *Pnext;			        /* To next in chain. */
} VertexStruct;

/* Internal edge, or edge generated by the polygon decomposition stage when  */
/* only convex polygons are allowed. This edge was not in the input	     */
/* non-convex polygon, and therefore one may not want to see/display it.     */
/* Note bits 4-7 (high nibble of Tags) are reserved for the different	     */
/* modules to perform their local tasks and so should not be used here.	     */
#define INTERNAL_TAG	0x01	       /* Edge Tag - This edge is internal.  */

#define	IS_INTERNAL_EDGE(Vrtx)	((Vrtx)->Tags & INTERNAL_TAG)
#define	SET_INTERNAL_EDGE(Vrtx)	((Vrtx)->Tags |= INTERNAL_TAG)
#define	RST_INTERNAL_EDGE(Vrtx)	((Vrtx)->Tags &= ~INTERNAL_TAG)

/*****************************************************************************
* Polygon Type - holds single polygon - Its Plane definition, and a pointer  *
* to its vertices contour list V. As for VertexStruct, different attributes  *
* can be saved in Count & Tags. PAux can be used locally by different	     *
* modules, for local usage only, and nothing sould be assumed on entry.	     *
*****************************************************************************/
typedef struct PolygonStruct {
    ByteType Count, Tags;			         /* Same attributes. */
    PlaneType Plane;			 /* Holds Plane as Ax + By + Cz + D. */
    VoidPtr PAux;	  /* May be used locally (temporary!) by any module. */
    VertexStruct *V;			       /* To vertices circular list. */
    struct PolygonStruct *Pnext;		        /* To next in chain. */
} PolygonStruct;

/* Note bits 4-7 (high nibble of Tags) are reserved for the different	     */
/* modules to perform their local tasks and so should not be used here.	     */
#define CONVEX_TAG	0x01	  /* Convex Tag - Set if polygon is convex.  */

#define	IS_CONVEX_POLY(Poly)	((Poly)->Tags & CONVEX_TAG)
#define	SET_CONVEX_POLY(Poly)	((Poly)->Tags |= CONVEX_TAG)
#define	RST_CONVEX_POLY(Poly)	((Poly)->Tags &= ~CONVEX_TAG)

typedef struct PolyListStruct {
    AttributeStruct Attr;
    struct PolygonStruct *P;
    int IsPolyline;	             /* TRUE if polylines, FALSE if polygons */
} PolyListStruct;

#define	IS_POLYLINE_OBJ(Obj)		((int) (Obj->U.Pl.IsPolyline))
#define	SET_POLYLINE_OBJ(Obj)		(Obj->U.Pl.IsPolyline = TRUE)
#define	RST_POLYLINE_OBJ(Obj)		(Obj->U.Pl.IsPolyline = FALSE)

/*****************************************************************************
* Free form curves and surfaces. Follow dthe cagd lib data structure with    *
* some IRIT internal information.					     *
*****************************************************************************/
typedef struct CurveStruct {
    AttributeStruct Attr;
    CagdCrvStruct *Crv;
    CagdPolylineStruct *PLPolys; /* Piecewise linear polyline approx. curve. */
    CagdPolylineStruct *CtlPoly;			 /* Control Polygon. */
} CurveStruct;

typedef struct SurfaceStruct {
    AttributeStruct Attr;
    CagdSrfStruct *Srf;
    CagdPolylineStruct *PLPolys;  /* Piecewise linear polylines approx. srf. */
    CagdPolylineStruct *CtlMesh;			    /* Control Mesh. */
    struct ObjectStruct *Polygons; /* If srf has been converted to polygons. */
} SurfaceStruct;

/*****************************************************************************
* Object Type - main system structure, which holds all the objects defined   *
* in the system like Numeric, Geometric etc.				     *
*   Note that as the number of objects will be usually extermely low (100 is *
* high estimate!) we can waste some memory here...			     *
*****************************************************************************/
typedef struct ObjectStruct {
    char Name[OBJ_NAME_LEN];				  /* Name of object. */
    IritObjectType ObjType;	    /* Object Type: Numeric, Geometric, etc. */
    ByteType Count;	       /* Count Number of references to this object. */
    union {
	AttributeStruct Attr;	         /* Pl/Crv/Srf has it as first slot. */
	PolyListStruct Pl;		       	       /* Polygon/line list. */
	CurveStruct Crv;			      /* Free form curve(s). */
	SurfaceStruct Srf;			    /* Free form surface(s). */
	RealType R;				       /* Numeric real data. */
	VectorType Vec;			        /* Numeric real vector data. */
	CagdCtlPtStruct CtlPt;			      /* Control point data. */
	MatrixType Mat;		    /* Numeric 4 by 4 transformation matrix. */
	struct ObjectStruct *PObjList[MAX_OBJ_LIST];	 /* List of objects. */
	char Str[LINE_LEN];		  /* General string for text object. */
    } U;
    struct ObjectStruct *Pnext;			        /* To next in chain. */
} ObjectStruct;

#define IS_UNDEF_OBJ(Obj)	((Obj)->ObjType == UNDEF_OBJ)
#define IS_POLY_OBJ(Obj)	((Obj)->ObjType == POLY_OBJ)
#define IS_NUM_OBJ(Obj)		((Obj)->ObjType == NUMERIC_OBJ)
#define IS_VEC_OBJ(Obj)		((Obj)->ObjType == VECTOR_OBJ)
#define IS_CTLPT_OBJ(Obj)	((Obj)->ObjType == CTLPT_OBJ)
#define IS_MAT_OBJ(Obj)		((Obj)->ObjType == MATRIX_OBJ)
#define IS_STR_OBJ(Obj)		((Obj)->ObjType == STRING_OBJ)
#define IS_OLST_OBJ(Obj)	((Obj)->ObjType == OBJ_LIST_OBJ)
#define IS_CRV_OBJ(Obj)		((Obj)->ObjType == CURVE_OBJ)
#define IS_SRF_OBJ(Obj)		((Obj)->ObjType == SURFACE_OBJ)

#define IS_GEOM_OBJ(Obj)	(IS_UNDEF_OBJ(Obj) || \
				 (Obj)->ObjType == POLY_OBJ || \
				 (Obj)->ObjType == CURVE_OBJ || \
				 (Obj)->ObjType == SURFACE_OBJ)

extern ObjectStruct
    *GlblObjList;				   /* All objects on system. */

extern jmp_buf
    GlblLongJumpBuffer;			          /* Used in error recovery. */

extern FILE
    *GlblLogFile;		   /* If do log everything, it goes to here. */

extern int
#ifdef __MSDOS__		     /* Defaults for MSDOS intr_lib windows. */
    GlblWindowFrameWidth,
    GlblViewFrameColor,
    GlblViewBackColor,
    GlblTransFrameColor,
    GlblTransBackColor,
    GlblStatusFrameColor,
    GlblStatusBackColor,
    GlblInputFrameColor,
    GlblInputBackColor,
    GlblDrawHeader,
    GlblSmoothTextScroll,
    GlblIntrSaveMethod,
    GlblMouseSensitivity,	     /* Sensitivity control of mouse device. */
    GlblJoystickExists,
#endif /*__MSDOS__ */
    GlblMouseExists,
    GlblFatalError,		  /* True if disaster in system - must quit! */
    GlblPrintLogFile,		     /* If TRUE everything goes to log file. */
    GlblDoGraphics,		/* Control if running in graphics/text mode. */
    GlblLoadColor,	      /* Default colors for object loaded using LOAD */
    GlblBoolColor,	      /* command, for boolean result objects or      */
    GlblICrvColor,	      /* boolean intersection curves and for basic   */
    GlblPrimColor,	       /* primitives colors, respectively.	     */
    GlblTransformMode,		      /* Screen, Object coords. trans. mode. */
    GlblViewMode,		           /* Perspective, Orthographic etc. */
    GlblDrawSolid,			 /* Use hardware Z buffer rendering. */
    GlblDepthCue;				   /* Activate depth cueing. */

extern char 
#ifdef __MSDOS__		     /* Defaults for MSDOS intr_lib windows. */
    *GlblViewWndwPos,
    *GlblTransWndwPos,
    *GlblStatusWndwPos,
    *GlblInputWndwPos,
    *GlblIntrSaveDisk,
#endif /*__MSDOS__ */
#ifdef __GL__
    *GlblTransPrefPos,
    *GlblViewPrefPos,
#endif /* __GL__ */
    *GlblHelpFileName,
    *GlblStartFileName,			/* Name of startup file to executed. */
    *GlblLogFileName,					/* Name of log file. */
    *GlblEditPrgm;		/* Name of editor to execute (edit command). */

void MyExit(int ExitCode);
void FatalError(char * ErrorMsg);
#ifdef __MSDOS__
void cdecl DefaultFPEHandler(int Sig, int Type, int *RegList);
#else
void DefaultFPEHandler(int Type);
#endif /* __MSDOS__ */

#endif	/* IRIT_H */
