#include "copyleft.h"

/*
    GEPASI - a simulator of metabolic pathways and other dynamical systems
    Copyright (C) 1989, 1992  Pedro Mendes
*/

/*************************************/
/*                                   */
/*      reaction rate equations      */
/*                                   */
/*        Zortech C/C++ 3.0 r4       */
/*          MICROSOFT C 6.00         */
/*          Visual C/C++ 1.0         */
/*           QuickC/WIN 1.0          */
/*             ULTRIX cc             */
/*              GNU gcc              */
/*                                   */
/*   (include here compilers that    */
/*   compiled GEPASI successfully)   */
/*                                   */
/*************************************/

#include <math.h>
#include "globals.h"
#include "globvar.h"
#include "datab.h"

/*****************************************/
/*                                       */
/*              CONVENTIONS              */
/*                                       */
/* all functions take as arguments:      */
/* double s[] - vector of concentrations */
/* int r - the index of the reaction     */
/*                                       */
/*     all fuctions return a double.     */
/*                                       */
/*                                       */
/*****************************************/


/*                                   */
/* user defined rate-equations		 */
/*                                   */

double value_of( double s[], int r, struct treet *t, int idx )
{
 switch( t->node[idx].item )
 {
  case 'N': return (double) t->constant[(int)t->node[idx].val];
  case 'I': switch( (int) t->id[(int)t->node[idx].val][9] )
            {
             case 0: return params[r][ (int) t->id[(int)t->node[idx].val][0] ];
             case 1:
             case 2:
             case 3: return s[ eff[r][ (int) t->id[(int)t->node[idx].val][0] ] ];
            }
  case 'F': switch( t->node[idx].val )
            {
             case '+': return value_of( s, r, t, (int) t->node[idx].left );
             case '-': return - value_of( s, r, t, (int) t->node[idx].left );
             case 'L': return log10( value_of( s, r, t, (int) t->node[idx].left ) );
             case 'l': return log( value_of( s, r, t, (int) t->node[idx].left ) );
             case 'e': return exp( value_of( s, r, t, (int) t->node[idx].left ) );
             case 'S': return sin( value_of( s, r, t, (int) t->node[idx].left ) );
             case 'C': return cos( value_of( s, r, t, (int) t->node[idx].left ) );
            }
  case 'O': switch( t->node[idx].val )
            {
             case '+': return value_of( s, r, t, (int) t->node[idx].left ) +
                              value_of( s, r, t, (int) t->node[idx].right );
             case '-': return value_of( s, r, t, (int) t->node[idx].left ) -
                              value_of( s, r, t, (int) t->node[idx].right );
             case '*': return value_of( s, r, t, (int) t->node[idx].left ) *
                              value_of( s, r, t, (int) t->node[idx].right );
             case '/': return value_of( s, r, t, (int) t->node[idx].left ) /
                              value_of( s, r, t, (int) t->node[idx].right );
             case '^': return pow( value_of( s, r, t, (int) t->node[idx].left ),
                                   value_of( s, r, t, (int) t->node[idx].right ) );
            }
 }
}

double translate(double s[], int r)
{
 return value_of( s, r, &tree[kinetype[r]-MAX_TYP], (int) tree[kinetype[r]-MAX_TYP].node[0].left );
}


/*                                   */
/* the zero returning function       */
/*                                   */

double ret_zero( double s[], int r, int e )
{
 return (double) 0;
}


/*                                   */
/* derivatives for effectors         */
/* by finite differences             */
/*                                   */

double deff( double s[], int r, int e )
{
 double k, x, x1, f1, f2;

 x = k = s[e];
 /*
    if x is zero, we will calculate the derivative at a small positive
    value (no point in considering negative values!). let's stick with
    hrcz (the flux resolution)
 */
 if( x==0 ) x = options.hrcz;
 x1 = x * 0.01;
 s[e] = x - x1;
 f1 = rateq[r]( s, r );
 s[e] = x + x1;
 f2 = (rateq[r]( s, r ) - f1)/(x*0.02);
 s[e] = k;
 return f2;
}

/*                                   */
/* A -->  or  --> B                  */
/*                                   */

double i0(double s[], int r)
{
 return params[r][0];
}


/*                                   */
/* A --> B                           */
/*                                   */

double i11(double s[], int r)
{
 return s[eff[r][0]]*params[r][0];
}

/*                                 */
/* A <--> B                        */
/*                                 */

double r11(double s[], int r)
{
 return s[eff[r][0]]*params[r][0]-s[eff[r][1]]*params[r][1];
}

/*                                 */
/* A + B --> C                     */
/*                                 */

double i21(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*params[r][0];
}

/*                                 */
/* A + B <--> C                    */
/*                                 */

double r21(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*params[r][0]-s[eff[r][2]]*params[r][1];
}

/*                                   */
/* A --> B + C                       */
/*                                   */

double i12(double s[], int r)
{
 return s[eff[r][0]]*params[r][0];
}

/*                                 */
/* A <--> B + C                    */
/*                                 */

double r12(double s[], int r)
{
 return s[eff[r][0]]*params[r][0]-s[eff[r][1]]*s[eff[r][2]]*params[r][1];
}

/*                                 */
/* A + B + C --> D                 */
/*                                 */

double i31(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*s[eff[r][2]]*params[r][0];
}

/*                                 */
/* A + B + C <--> D                */
/*                                 */

double r31(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*s[eff[r][2]]*params[r][0]-s[eff[r][3]]*params[r][1];
}

/*                                   */
/* A --> B + C + D                   */
/*                                   */

double i13(double s[], int r)
{
 return s[eff[r][0]]*params[r][0];
}

/*                                 */
/* A <--> B + C + D                */
/*                                 */

double r13(double s[], int r)
{
 return s[eff[r][0]]*params[r][0]-s[eff[r][1]]*s[eff[r][2]]*s[eff[r][3]]*params[r][1];
}

/*                                 */
/* A + B --> C + D                 */
/*                                 */

double i22(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*params[r][0];
}

/*                                 */
/* A + B <--> C + D                */
/*                                 */

double r22(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*params[r][0]-s[eff[r][2]]*s[eff[r][3]]*params[r][1];
}

/*                                 */
/* A + B + C --> D + E             */
/*                                 */

double i32(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*s[eff[r][2]]*params[r][0];
}

/*                                 */
/* A + B + C <--> D + E            */
/*                                 */

double r32(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*s[eff[r][2]]*params[r][0]-s[eff[r][3]]*s[eff[r][4]]*params[r][1];
}

/*                                 */
/* A + B --> C + D + E             */
/*                                 */

double i23(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*params[r][0];
}

/*                                 */
/* A + B <--> C + D + E            */
/*                                 */

double r23(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*params[r][0]-s[eff[r][2]]*s[eff[r][3]]*s[eff[r][4]]*params[r][1];
}

/*                                 */
/* A + B + C --> D + E + F         */
/*                                 */

double i33(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*s[eff[r][2]]*params[r][0];
}

/*                                 */
/* A + B + C <--> D + E + F        */
/*                                 */

double r33(double s[], int r)
{
 return s[eff[r][0]]*s[eff[r][1]]*s[eff[r][2]]*params[r][0]-s[eff[r][3]]*s[eff[r][4]]*s[eff[r][5]]*params[r][1];
}


/*                                 */
/* derivatives for one substrate   */
/*                                 */

double d1Xs1( double s[], int r, int e )
{
 return params[r][0];
}


/*                                 */
/* derivatives for two substrates  */
/*                                 */

double d2Xs1( double s[], int r, int e )
{
 return params[r][0]*s[eff[r][1]];
}

double d2Xs2( double s[], int r, int e )
{
 return params[r][0]*s[eff[r][0]];
}


/*                                 */
/* derivatives for three substrates*/
/*                                 */

double d3Xs1( double s[], int r, int e )
{
 return params[r][0]*s[eff[r][1]]*s[eff[r][2]];
}

double d3Xs2( double s[], int r, int e )
{
 return params[r][0]*s[eff[r][0]]*s[eff[r][2]];
}

double d3Xs3( double s[], int r, int e )
{
 return params[r][0]*s[eff[r][0]]*s[eff[r][1]];
}


/*                                 */
/* derivatives for one product     */
/*                                 */

double dX1p1( double s[], int r, int e )
{
 return -params[r][1];
}


/*                                 */
/* derivatives for two products    */
/* (reactions with one substrate)  */
/*                                 */

double dX2p1s1( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][2]];
}

double dX2p2s1( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][1]];
}

/*                                 */
/* derivatives for two products    */
/* (reactions with two substrates) */
/*                                 */

double dX2p1s2( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][3]];
}

double dX2p2s2( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][2]];
}

/*                                 */
/* derivatives for two products    */
/* (reactions with 3 substrates)   */
/*                                 */

double dX2p1s3( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][4]];
}

double dX2p2s3( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][3]];
}


/*                                 */
/* derivatives for three products  */
/* (reactions with 1 substrate)   */
/*                                 */

double dX3p1s1( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][2]]*s[eff[r][3]];
}

double dX3p2s1( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][1]]*s[eff[r][3]];
}

double dX3p3s1( double s[], int r, int e )
{
 return  -params[r][1]*s[eff[r][1]]*s[eff[r][2]];
}

/*                                 */
/* derivatives for three products  */
/* (reactions with 2 substrates)   */
/*                                 */

double dX3p1s2( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][3]]*s[eff[r][4]];
}

double dX3p2s2( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][2]]*s[eff[r][4]];
}

double dX3p3s2( double s[], int r, int e )
{
 return  -params[r][1]*s[eff[r][2]]*s[eff[r][3]];
}

/*                                 */
/* derivatives for three products  */
/* (reactions with 3 substrates)   */
/*                                 */

double dX3p1s3( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][4]]*s[eff[r][5]];
}

double dX3p2s3( double s[], int r, int e )
{
 return -params[r][1]*s[eff[r][3]]*s[eff[r][5]];
}

double dX3p3s3( double s[], int r, int e )
{
 return  -params[r][1]*s[eff[r][3]]*s[eff[r][4]];
}



/*                                                   */
/* irreversible Henri-Michaelis-Menten rate equation */
/*                                                   */

double ihmm( double s[], int r)
          /* eff[r][0] - substrate           */
          /* params[r][0] - Km for substrate    */
          /* params[r][1] - max forward vel.    */
{
 double sb;

 sb = s[eff[r][0]];
 return (sb*params[r][1])  / ( sb + params[r][0] );
}

double dihmms( double s[], int r, int e )
{
 double c;

  c = 1 + s[eff[r][0]]/params[r][0];
 return params[r][1]/(c*c*params[r][0]);
}


/*                                                 */
/* reversible Henri-Michaelis-Menten rate equation */
/*                                                 */

double hmm( double s[], int r)
          /* eff[r][0] - substrate           */
          /* eff[r][1] - product             */
          /* params[r][0] - Km for substrate    */
          /* params[r][1] - Km for product      */
          /* params[r][2] - max forward vel.    */
          /* params[r][3] - max reverse vel.    */
{
 double nsb, npr;

 nsb = s[eff[r][0]]/params[r][0];
 npr = s[eff[r][1]]/params[r][1];
 return (nsb*params[r][2] - npr*params[r][3]) / (1 + nsb + npr);
}

double dhmms( double s[], int r, int e )
{
  double c;

  c = s[eff[r][1]]*params[r][0] + s[eff[r][0]]*params[r][1] + params[r][0]*params[r][1];
  return ( params[r][0]*params[r][1] * ( s[eff[r][1]]*(params[r][2]+params[r][3]) + params[r][2]*params[r][1] ) ) / ( c*c );
}

double dhmmp( double s[], int r, int e )
{
 double c;

 c = s[eff[r][1]]*params[r][0] + s[eff[r][0]]*params[r][1] + params[r][0]*params[r][1];
 return - ( params[r][0]*params[r][1] * ( s[eff[r][0]]*(params[r][2]+params[r][3]) + params[r][0]*params[r][3] ) ) / ( c*c );
}


/*                                                                     */
/* reversible Henri-Michaelis-Menten rate equation w/ competitive inh. */
/*                                                                     */

double hmmsi( double s[], int r )
          /* eff[r][0] - substrate           */
          /* eff[r][1] - product             */
          /* eff[r][2] - inhibitor           */
          /* params[r][0] - Km for substrate    */
          /* params[r][1] - Km for product      */
          /* params[r][2] - max forward vel.    */
          /* params[r][3] - max reverse vel.    */
          /* params[r][4] - inhibition const.   */
{
 double nsb, npr, in;

 nsb = s[eff[r][0]]/params[r][0];
 npr = s[eff[r][1]]/params[r][1];
 in = s[eff[r][3]];
 return (nsb*params[r][2] - npr*params[r][3]) / (1 + nsb + npr + in/params[r][4]);
}

double dhmmsis( double s[], int r, int e )
{
 double sb, pr, inn, c;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 inn = s[eff[r][2]]/params[r][4];
  c = 1 + sb/params[r][0] + pr/params[r][1] + inn;
 return (params[r][2]*(1+inn)+pr*(params[r][2]+params[r][3])/params[r][1]) / (c*c*params[r][0]);
}

double dhmmsip( double s[], int r, int e )
{
 double nsb, pr, in, c;

 nsb = s[eff[r][0]]/params[r][0];
 pr = s[eff[r][1]];
 in = s[eff[r][2]];
  c = 1 + nsb + pr/params[r][1] + in/params[r][4];
 return - (params[r][3]*(1+in/params[r][4])+nsb*(params[r][2]+params[r][3])) / (c*c*params[r][1]);
}

double dhmmsii( double s[], int r, int e )
{
 double nsb, npr, in, c;

 nsb = s[eff[r][0]]/params[r][0];
 npr = s[eff[r][1]]/params[r][1];
 in  = s[eff[r][2]];
  c = 1 + nsb + npr + in/params[r][4];
 return - (nsb*params[r][2] - npr*params[r][3]) / (c*c*params[r][4]);
}

/*                                                                         */
/* reversible Henri-Michaelis-Menten rate equation w/ non-competitive inh. */
/*                                                                         */

double hmmci( double s[], int r )
          /* eff[r][0] - substrate           */
          /* eff[r][1] - product             */
          /* eff[r][2] - inhibitor           */
          /* params[r][0] - Km for substrate    */
          /* params[r][1] - Km for product      */
          /* params[r][2] - max forward vel.    */
          /* params[r][3] - max reverse vel.    */
          /* params[r][4] - inhibition const.   */
{
 double nsb, npr, in;

 nsb = s[eff[r][0]]/params[r][0];
 npr = s[eff[r][1]]/params[r][1];
 in = s[eff[r][2]];
 return (nsb*params[r][2] - npr*params[r][3]) / ((1 + nsb + npr) * (1 + in/params[r][4]));
}

double dhmmcis( double s[], int r, int e )
{
 double npr, c;

 npr = s[eff[r][1]]/params[r][1];
   c = 1 + s[eff[r][0]]/params[r][0] + npr;
 return (params[r][2] + npr*(params[r][2]+params[r][3])) / (c*c*params[r][0]*(1+s[eff[r][2]]/params[r][4]));
}

double dhmmcip( double s[], int r, int e )
{
 double nsb, c;

 nsb = s[eff[r][0]]/params[r][0];
 c = 1 + nsb + s[eff[r][1]]/params[r][1];
 return -(params[r][3] + nsb*(params[r][2]+params[r][3])) / (c*c*params[r][1]*(1+s[eff[r][2]]/params[r][4]));
}

double dhmmcii( double s[], int r, int e )
{
 double nsb, npr, c, d;

 nsb = s[eff[r][0]]/params[r][0];
 npr = s[eff[r][1]]/params[r][1];
  c = 1 + nsb + npr;
  d = 1 + s[eff[r][2]]/params[r][4];
 return (npr*params[r][3] - nsb*params[r][2]) / (d*d*c*params[r][4]);
}

/*                                                                     */
/* reversible Henri-Michaelis-Menten rate equation w/ mixed inhibition */
/*                                                                     */

double hmmmi( double s[], int r )
          /* eff[r][0] - substrate            */
          /* eff[r][1] - product              */
          /* eff[r][2] - inhibitor            */
          /* params[r][0] - Km for substrate     */
          /* params[r][1] - Km for product       */
          /* params[r][2] - max forward vel.     */
          /* params[r][3] - max reverse vel.     */
          /* params[r][4] - spec. inh. const.    */
          /* params[r][5] - catal.inh. const.    */
{
 double sb, pr, in;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 in = s[eff[r][2]];
 return (sb*params[r][2]/params[r][0] - pr*params[r][3]/params[r][1]) /
        (1 + in/params[r][4] + (sb/params[r][0] + pr/params[r][1])*(1 + in/params[r][5]) );
}

double dhmmmis( double s[], int r, int e )
{
 double sb, pr, in, c, d;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 in = s[eff[r][2]];
  c = 1 + pr/params[r][1] + in/params[r][4] + pr*in/(params[r][5]*params[r][1]);
  d = c + sb/params[r][0]*(1 + in/params[r][5]);
 return (params[r][2]*c + params[r][3]*pr*(1+in/params[r][5])/params[r][1]) / (params[r][0]*d*d);
}

double dhmmmip( double s[], int r, int e )
{
 double sb, in, c, d;

 sb = s[eff[r][0]];
 in = s[eff[r][2]];
  c = 1 + sb/params[r][0] + in/params[r][4] + sb*in/(params[r][5]*params[r][0]);
  d = c + s[eff[r][1]]/params[r][1]*(1 + in/params[r][5]);
 return -(params[r][3]*c + params[r][2]*sb*(1+in/params[r][5])/params[r][0]) / (params[r][1]*d*d);
}


double dhmmmii( double s[], int r, int e )
{
 double sb, pr, in, c;

 sb = s[eff[r][0]]/params[r][0];
 pr = s[eff[r][1]]/params[r][1];
 in = s[eff[r][2]];
  c = 1  + in/params[r][4] + (sb + pr) * (1 + in/params[r][5]);
 return -(sb*params[r][2]-pr*params[r][3]) * (1/params[r][4]+(sb+pr)/params[r][5])
        / (c*c);
}


/*                                                                  */
/* reversible Henri-Michaelis-Menten rate equation w/ specific act. */
/*                                                                  */

double hmmsa( double s[], int r )
          /* eff[r][0] - substrate            */
          /* eff[r][1] - product              */
          /* eff[r][2] - activator            */
          /* params[r][0] - Km for substrate     */
          /* params[r][1] - Km for product       */
          /* params[r][2] - max forward vel.     */
          /* params[r][3] - max reverse vel.     */
          /* params[r][4] - activation const.    */
{
 double sb, pr, ac;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
 return  ac * (sb*params[r][2]*params[r][1] - pr*params[r][3]*params[r][0])
         /
         ( ac * ( params[r][0]*params[r][1] + params[r][0]*pr + params[r][1]* sb )
           + params[r][0]*params[r][1]*params[r][4] );
}

double dhmmsas( double s[], int r, int e )
{
 double pr, ac, c;

 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
  c =  ac * ( params[r][0]*params[r][1] + params[r][0]*pr + params[r][1]* s[eff[r][0]] )
       + params[r][0]*params[r][1]*params[r][4];
 return ac*params[r][0]*params[r][1] * (  params[r][1]*params[r][2]*(ac+params[r][4])
                                        + ac*pr*( params[r][2]+params[r][3])
                                       )
       / ( c * c );
}

double dhmmsap( double s[], int r, int e )
{
 double sb, ac, c;

 sb = s[eff[r][0]];
 ac = s[eff[r][2]];
  c =  ac * ( params[r][0]*params[r][1] + params[r][0]*s[eff[r][1]] + params[r][1]*sb )
       + params[r][0]*params[r][1]*params[r][4];
 return - ac*params[r][0]*params[r][1] * (  ac*sb*(params[r][2]+params[r][3])
                                         + params[r][0]*params[r][3]*(ac+params[r][4])
                                        )
        / ( c * c );
}

double dhmmsaa( double s[], int r, int e )
{
 double sb, pr, ac, c;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
  c =  ac * ( params[r][0]*params[r][1] + params[r][0]*pr + params[r][1]* s[eff[r][0]] )
       + params[r][0]*params[r][1]*params[r][4];
 return params[r][0]*params[r][1]*params[r][4] * (  sb*params[r][1]*params[r][2]
                                                  - pr*params[r][0]*params[r][3]
                                                 )
       / ( c * c );
}


/*                                                                         */
/* reversible Henri-Michaelis-Menten rate equation w/ catalytic activation */
/*                                                                         */

double hmmca( double s[], int r )
          /* eff[r][0] - substrate            */
          /* eff[r][1] - product              */
          /* eff[r][2] - activator            */
          /* params[r][0] - Km for substrate     */
          /* params[r][1] - Km for product       */
          /* params[r][2] - max forward vel.     */
          /* params[r][3] - max reverse vel.     */
          /* params[r][4] - activation const.    */
{
 double sb, pr, ac;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
 return   ac * ( sb*params[r][1]*params[r][2] - pr*params[r][0]*params[r][3] )
        / (   (ac+params[r][4])
            * (sb*params[r][1] + pr*params[r][0] + params[r][0]*params[r][1])
          );
}

double dhmmcas( double s[], int r, int e )
{
 double ac, pr, c;

 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
  c = s[eff[r][0]]*params[r][1] + pr*params[r][0] + params[r][0]*params[r][1];
 return   ac*params[r][0]*params[r][1] * (  pr*(params[r][2]+params[r][3])
                                          + params[r][1]*params[r][2]
                                         )
        / ( c * c * ( ac + params[r][4] ) );
}

double dhmmcap( double s[], int r, int e )
{
 double sb, ac, c;

 sb = s[eff[r][0]];
 ac = s[eff[r][2]];
  c = sb*params[r][1] + s[eff[r][1]]*params[r][0] + params[r][0]*params[r][1];
 return - ac*params[r][0]*params[r][1] * (  sb*(params[r][2]+params[r][3])
                                          + params[r][0]*params[r][3]
                                         )
        / ( c * c * ( ac + params[r][4] ) );
}

double dhmmcaa( double s[], int r, int e )
{
 double sb, pr, c;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
  c = s[eff[r][2]] + params[r][4];
 return   params[r][4] * ( sb*params[r][1]*params[r][2] + pr*params[r][0]*params[r][3] )
        / ( c * c * ( params[r][0]*params[r][1] + pr*params[r][0] + sb*params[r][1]) );

/* WHAT IS THIS DOING HERE???
 ( (params[r][4]*(sb*params[r][2]/params[r][0] - pr*params[r][3]/params[r][1])) / (c*c*(1+sb/params[r][0]+pr/params[r][1])) );
*/
}


/*                                                                     */
/* reversible Henri-Michaelis-Menten rate equation w/ mixed activation */
/*                                                                     */

double hmmma( double s[], int r )
          /* eff[r][0] - substrate           */
          /* eff[r][1] - product             */
          /* eff[r][2] - activator           */
          /* params[r][0] - Km for substrate    */
          /* params[r][1] - Km for product      */
          /* params[r][2] - max forward vel.    */
          /* params[r][3] - max reverse vel.    */
          /* params[r][4] - spec.act. const.    */
          /* params[r][5] - cat. act. const.    */
{
 double sb, pr, ac;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
 return   ac * (sb*params[r][2]*params[r][1] - pr*params[r][3]*params[r][0])
        / (  params[r][0]*params[r][1]*(ac+params[r][4])
           + (ac+params[r][5])*(sb*params[r][1]+pr*params[r][0])
          );
}

double dhmmmas( double s[], int r, int e )
{
 double pr, ac, c;

 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
  c =  params[r][0]*params[r][1]*(ac+params[r][4])
     + (ac+params[r][5])*(s[eff[r][0]]*params[r][1]+pr*params[r][0]);
 return   ac*params[r][0]*params[r][1] *
          (  params[r][1]*params[r][2]*(ac+params[r][4])
           + pr*(ac+params[r][5])*(params[r][2]+params[r][3])
          )
        / (c * c);
}

double dhmmmap( double s[], int r, int e )
{
 double sb, ac, c;

 sb = s[eff[r][0]];
 ac = s[eff[r][2]];
  c =  params[r][0]*params[r][1]*(ac+params[r][4])
     + (ac+params[r][5])*(sb*params[r][1]+s[eff[r][1]]*params[r][0]);
 return - ac*params[r][0]*params[r][1] *
          (  params[r][0]*params[r][3]*(ac+params[r][4])
           + sb*(ac+params[r][5])*(params[r][2]+params[r][3])
          )
        / (c * c);
}

double dhmmmaa( double s[], int r, int e )
{
 double sb, pr, ac, c;

 sb = s[eff[r][0]];
 pr = s[eff[r][1]];
 ac = s[eff[r][2]];
  c =  params[r][0]*params[r][1]*(ac+params[r][4])
     + (ac+params[r][5])*(sb*params[r][1]+pr*params[r][0]);
 return    (sb*params[r][1]*params[r][2] - pr*params[r][0]*params[r][3])
         * (  params[r][4]*params[r][0]*params[r][0]
            + params[r][5]*(sb*params[r][1]+pr*params[r][0])
           )
        / (c * c);
}


/*                    */
/* Hill rate equation */
/*                    */

double hill( double s[], int r )
          /* eff[r][0] - substrate           */
          /* params[r][0] - K for substrate    */
          /* params[r][1] - max forward vel.    */
          /* params[r][2] - Hill coef.      */
{
 double sb;

 sb = pow(s[eff[r][0]],params[r][2]);
 return sb*params[r][1]/(pow(params[r][0],params[r][2])+sb);
}

double dhills( double s[], int r, int e )
{
 double sb, kh, sb1, d;

 sb = pow(s[eff[r][0]], params[r][2]);
 kh = pow(params[r][0], params[r][2]);
 sb1 = pow(s[eff[r][0]], params[r][2] - 1);
 d = ( kh + sb );
 return (kh*sb1*params[r][2]*params[r][1]) / (d * d) ;
}
