/* texfont.c:  code to load TeX fonts for dvix
 *
 * based on code by: Eric Cooper, CMU, September 1985.
 *
 * Code derived from dvi-imagen.c.
 *
 * Modified for X.10 by Bob Scheifler, MIT LCS, January 1986.
 *
 * Modified by jonah@db.toronto.edu, UofT DCS, October 1988.
 */

#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <stdio.h>
#include <ctype.h>
#include "dvi.h"
#include "pxl.h"
#include "local.h"
#include "general.h"

FILE *fopen_path();
FILE *fopen_font_path();
char *font_path = DEFAULT_FONT_PATH;

#define PAPER_WIDTH	ROUNDUP(17*pixels_per_inch,shrink_factor*2)
#define PAPER_HEIGHT	ROUNDUP(11*pixels_per_inch,shrink_factor)
#define X_PAGE_OFFSET	ROUNDUP(pixels_per_inch,shrink_factor)
#define Y_PAGE_OFFSET	ROUNDUP(pixels_per_inch,shrink_factor)

#define pixel_round(x)      ((long) (conv * (double) (x) + 0.5))
#define dvi_round(x)        ((long) ((double) (x) / conv + 0.5))

struct frame {
	long pxl_h, dvi_h, pxl_v, dvi_v, w, x, y, z;
};


#define PXL_H   stack[stackp].pxl_h
#define PXL_V   stack[stackp].pxl_v
#define DVI_H   stack[stackp].dvi_h
#define DVI_V   stack[stackp].dvi_v
#define WW      stack[stackp].w
#define XX      stack[stackp].x
#define YY      stack[stackp].y
#define ZZ      stack[stackp].z

/*
 * Command line flags.
 */
extern int debug;
extern int list_fonts;

extern int pixels_per_inch;
extern int shrink_factor;

extern FILE *dvi_file;			/* user's file */

extern int font_not_found;
extern struct font *current_font;	/* ptr into circular list of fonts */

int n_open_fonts = 0;		/* for LRU management of fonts */

/*
 * DVI preamble and postamble information.
 */
extern double fraction, conv;
extern long numerator, denominator, magnification;

unsigned long num();
long snum();

extern char reverse_byte[];
char *malloc(), *calloc(), *index();

/********************************************/
/* This part of the file handles PK format  */
/********************************************/

#define PK_ID      89
#define PK_CMD_START 240
#define PK_X1     240
#define PK_X2     241
#define PK_X3     242
#define PK_X4     243
#define PK_Y      244
#define PK_POST   245
#define PK_NOOP   246
#define PK_PRE    247

#define Pxl1(fp) (one(fp->file))
#define Pxl2(fp) (two(fp->file))
#define Pxl4(fp) (four(fp->file))

#define FontRd1(fp) sone(fp->file)
#define FontRd2(fp) stwo(fp->file)
#define FontRd4(fp) sfour(fp->file)

int PK_flag_byte;
unsigned PK_input_byte;
int PK_bitpos;
int PK_dyn_f;
int PK_repeat_count;


int
PK_get_nyb( fp )
     register struct font *fp;
{
  unsigned temp;
  if( PK_bitpos < 0 )
    {
      PK_input_byte = Pxl1( fp );
      PK_bitpos = 4;
    }
  temp = PK_input_byte >> PK_bitpos;
  PK_bitpos -= 4;
  return( temp & 0xf );
}


int
PK_packed_num( fp )
     register struct font *fp;
{
  int i,j;
  if( ( i = PK_get_nyb( fp ) ) == 0 )
    {
      do
	{
	  j = PK_get_nyb( fp );
	  i++;
	}
      while( j == 0 );
      while( i > 0 )
	{
	  j = (j << 4) + PK_get_nyb( fp );
	  i--;
	}
      return( j - 15 + ( (13 - PK_dyn_f) << 4 ) + PK_dyn_f );
    }
  else
    {
      if( i <= PK_dyn_f ) return( i );
      if( i < 14 ) return( ( (i - PK_dyn_f - 1) << 4 ) +
			  PK_get_nyb( fp ) + PK_dyn_f + 1 );
      if( i == 14 ) PK_repeat_count = PK_packed_num( fp );
      else PK_repeat_count = 1;
      return( PK_packed_num( fp ) );
    }
}


void
PK_skip_specials( fp )
     register struct font *fp;
{
  int i,j;
  do
    {
      PK_flag_byte = Pxl1(fp);
      if( PK_flag_byte >= PK_CMD_START )
	{
	  switch( PK_flag_byte )
	    {
	    case PK_X1 :
	    case PK_X2 :
	    case PK_X3 :
	    case PK_X4 :
	      {
		i = 0;
		for( j = PK_CMD_START; j <= PK_flag_byte; j++ )
		  i = (i*256) + Pxl1(fp);
		for( j = 1; j <= i; j++ )
		  (void) Pxl1(fp);
		break;
	      }
	    case PK_Y :
	      (void) Pxl4(fp);
	    case PK_POST :
	    case PK_NOOP :
	      break;
	    default :
	      error("Unexpected %d in PK file %s\n",
		      PK_flag_byte, fp->fontname );
	      break;
	    }
	}
    }
  while( PK_flag_byte != PK_POST && PK_flag_byte >= PK_CMD_START );
}


RdPkFont(fp)
register struct font *fp;
{
  int i, hppp, vppp;
  /* 
   * Read the header of a packed pixel file.
   */
  (void) fseek(fp->file,0l,0); /* rewind the file, just in case */

  if (debug & DBG_PK)
    (void) fprintf(stderr, "Reading header for packed pixel file %s\n",
	    fp->fontname);

  if (Pxl1(fp) != PK_PRE) 
    {
      error("file %s lacks preamble command\n",fp->fontname);
    }

  if (Pxl1(fp) != PK_ID)
    {
      error("file %s has wrong PK id\n",fp->fontname);
    }

  for( i = Pxl1(fp); i > 0; i-- )
    (void) Pxl1(fp);		/* Skip comment */
  
  fp->design = FontRd4(fp);
  (void) Pxl4(fp);		/* Skip checksum */
  hppp = FontRd4(fp);
  vppp = FontRd4(fp);
  if( debug && hppp != vppp )
    error("Warning: aspect ratio not 1:1 for font %s\n",
	    fp->fontname);
  /*fp->f_scale = (int)( (( (float) hppp * 72.27 ) / (float) 65536 ) + 0.5 );*/
  
  /*if (fp->f_scale == 0) fp->f_scale = 1000;*/
  PK_skip_specials( fp );
  fp->glyphaddr = ftell(fp->file) - 1;
}


enum PK_pre_type { Short, Xtended, Long };
#define PK_row_size 100
int PK_row[ PK_row_size ];
int PK_rowix;
short PK_turn_on;
int PK_power[ 33 ] =
{
         0x1,        0x2,        0x4,        0x8,
        0x10,       0x20,       0x40,       0x80,
       0x100,      0x200,      0x400,      0x800,
      0x1000,     0x2000,     0x4000,     0x8000,
     0x10000,    0x20000,    0x40000,    0x80000,
    0x100000,   0x200000,   0x400000,   0x800000,
   0x1000000,  0x2000000,  0x4000000,  0x8000000,
  0x10000000, 0x20000000, 0x40000000, 0x80000000
};

int PK_gpower[ 34 ] =
{
         0x0,        0x1,        0x3,        0x7,
         0xf,       0x1f,       0x3f,       0x7f,
        0xff,      0x1ff,      0x3ff,      0x7ff,
       0xfff,     0x1fff,     0x3fff,     0x7fff,
      0xffff,    0x1ffff,    0x3ffff,    0x7ffff,
     0xfffff,   0x1fffff,   0x3fffff,   0x7fffff,
    0xffffff,  0x1ffffff,  0x3ffffff,  0x7ffffff,
   0xfffffff, 0x1fffffff, 0x3fffffff, 0x7fffffff,
  0xffffffff
};
  

int
PK_read_char( fp, cp )
     register struct font *fp;
     register char *cp;
{
  register cc;
  int bytes_left, thispos, i, j;
  int row_bit_pos, row_mask;
  char row_byte;
  enum PK_pre_type pre_type;
  register struct glyph *g;
  long fpwidth;
  int word, word_weight, rwidth;
  int rows_left, h_bit, count;

  thispos = ftell( fp->file ) - 1; /* has already read flag byte */
  PK_dyn_f = PK_flag_byte >> 4;
  PK_flag_byte &= 0xf;
  PK_turn_on = ( PK_flag_byte >= 8 );
  PK_flag_byte &= 0x7;
  if( PK_flag_byte == 7 )
    {
      /* read first part of long character preamble */
      pre_type = Long;
      bytes_left = FontRd4( fp ) - 28;
      cc = FontRd4( fp );
      if( cc >= MAXCHARS || cc < 0 )
	{
	  error("Character code %d outside range, file %s\n",
		  cc, fp->fontname );
	}
    }
  else
    if( PK_flag_byte > 3 )
      {
	/* read first part of extended short character preamble */
	pre_type = Xtended;
	bytes_left = ((PK_flag_byte - 4) << 16) + Pxl2( fp ) - 13;
	cc = Pxl1( fp );
	if( cc >= MAXCHARS )
	  {
	    error("Character code %d outside range, file %s\n",
		    cc, fp->fontname );
	  }
      }
    else
      {
	/* read first part of short character preamble */
	pre_type = Short;
	bytes_left = (PK_flag_byte << 8) + Pxl1( fp ) - 8;
	cc = Pxl1( fp );
	if( cc >= MAXCHARS)
	  {
	    error("Character code %d outside range, file %s\n",
		    cc, fp->fontname);
	  }
      }

  /* setup pointers for this char */
  g = fp->glyph + cc;
  g->addr = thispos;
  g->bitmap.bits = 0;		/* not in core -- may be altered below */

  if(debug & DBG_PK) {
    if( cp != NULL )
      (void) fprintf( stderr, "loading pk " );
    else
      (void) fprintf( stderr, "reading glyph data for pk " );
    (void) fprintf( stderr, "char %d", cc );
    (void) fprintf( stderr, ", char type " );
  }

  /* now read rest of character preamble */
  switch ( pre_type )
    {
    case Short :
      {
        if(debug & DBG_PK) (void) fprintf( stderr, "short" );
	fpwidth = Pxl1( fp ) << 16;
	fpwidth += Pxl2( fp );
	(void) Pxl1( fp );	/* vertical escapement */
	g->bitmap.w = Pxl1( fp );
	g->bitmap.h = Pxl1( fp );
	g->x = FontRd1( fp );
	g->y = FontRd1( fp );
	break;
      }
    case Xtended :
      {
        if(debug & DBG_PK) (void) fprintf( stderr, "extended" );
	fpwidth = Pxl1( fp ) << 16;
	fpwidth += Pxl2( fp );
	(void) Pxl2( fp );	/* vertical escapement */
	g->bitmap.w = Pxl2( fp );
	g->bitmap.h = Pxl2( fp );
	g->x = FontRd2( fp );
	g->y = FontRd2( fp );
	break;
      }
    case Long :
      {
	if(debug & DBG_PK) (void) fprintf( stderr, "long" );
	fpwidth = FontRd4( fp );
	(void) FontRd4( fp );	/* horizontal escapement */
	(void) FontRd4( fp );	/* vertical escapement */
	g->bitmap.w = FontRd4( fp );
	g->bitmap.h = FontRd4( fp );
	g->x = FontRd4( fp );
	g->y = FontRd4( fp );
	if( g->bitmap.w < 0 || g->bitmap.h < 0 )
/* 	   || g->bitmap.w > 0x7fff || g->bitmap.h > 0x7fff )
mod by GBP for MS-DOS, DV/X
Another bug.  Since g->bitmap.w is a short, it will never be > 0x7fff.
i.e. it will be interpreted as a negative number for 0x8000.
*/
	  {
	    error("Too large character (%d) in file %s\n",
		    cc, fp->fontname);
	  }
	break;
      }
    } /* end of switch */
  g->x /= shrink_factor;
  g->y /= shrink_factor;

  g->dvi_adv = ((double) fp->scale * fpwidth) / (1 << 20);
  g->pxl_adv = pixel_round(g->dvi_adv);

  if ( g->bitmap.w != 0 )
    {
      if(debug & DBG_PK)
      {
        (void) fprintf( stderr, ", size=%dx%d",g->bitmap.w,g->bitmap.h );
        (void) fprintf( stderr, ", dvi_adv=%d pxl_adv=%d",
			g->dvi_adv,g->pxl_adv );
      }
    }
  if(debug & DBG_PK) (void) fprintf( stderr, "\n" );

  if( cp != NULL && g->bitmap.w != 0 ) /* read character data into *cp */
    {
      g->bitmap.bits = cp;
      PK_bitpos = -1;
      if( PK_dyn_f == 14 )	/* get raster by bits */
	{
          rwidth = (g->bitmap.w + 15) / 16 * 2;
          bzero(g->bitmap.bits, g->bitmap.h * rwidth);
	  for (i = 0; i < g->bitmap.h; i++)  /* get all rows */
	    {
              cp = g->bitmap.bits + i * rwidth;
              row_bit_pos = 8;
	      for (j = 0; j < g->bitmap.w; j++)  /* get one row */
		{
                  if (--PK_bitpos < 0)
                    {
                      cc = Pxl1(fp);
                      PK_bitpos = 7;
                    }
                  if (--row_bit_pos < 0)
                    {
                      cp++;
                      row_bit_pos = 7;
                    }
                  if (cc & (1 << PK_bitpos))
                    *cp |= 1 << row_bit_pos;
                }
	    }
	}
      else
	{
	  /* get packed raster */
	  rows_left = g->bitmap.h;
	  h_bit = g->bitmap.w;
	  PK_repeat_count = 0;
	  word_weight = 32;
	  word = 0;
	  PK_rowix = 0;
	  while( rows_left > 0 )
	    {
	      count = PK_packed_num( fp );
	      while( count > 0 )
		{
		  if( count < word_weight && count < h_bit )
		    {
		      if( PK_turn_on )
			word += PK_gpower[ word_weight ]
			  - PK_gpower[ word_weight - count ];
		      h_bit -= count;
		      word_weight -= count;
		      count = 0;
		    }
		  else
		    if( count >= h_bit && h_bit <= word_weight )
		      {
			if( PK_turn_on )
			  word += PK_gpower[ word_weight ]
			    - PK_gpower[ word_weight - h_bit ];
			PK_row[ PK_rowix++ ] = word;
			/* "output" row(s) */
			rwidth = ((g->bitmap.w+15)/16)*2; /*## fix */
			for( i = 0; i <= PK_repeat_count; i++ )
			  {
			    for ( j = 0; j < rwidth; j++ )
			      {
				row_bit_pos = 24 - ( ( j & 0x3 ) << 3 );
				row_mask = 0xff << row_bit_pos;
				row_byte = (char) ( ( PK_row[ j >> 2 ] 
						     & row_mask )
						   >> row_bit_pos );
				*cp++ = row_byte;
			      }
			  }
			rows_left -= PK_repeat_count + 1;
			PK_repeat_count = 0;
			PK_rowix = 0;
			word = 0;
			word_weight = 32;
			count -= h_bit;
			h_bit = g->bitmap.w;
		      }
		    else
		      {
			if( PK_turn_on )
			  word |= PK_gpower[ word_weight ];
			PK_row[ PK_rowix++ ] = word;
			word = 0;
			count -= word_weight;
			h_bit -= word_weight;
			word_weight = 32;
		      }
		}
	      PK_turn_on = ( PK_turn_on ? 0 : 1 );
	    }
	  if( rows_left != 0 || h_bit != g->bitmap.w )
	    {
	      error("Bad pk file (%s), char %d, too many bits\n"
		      ,fp->fontname, cc );
	    }
	}
    }
  else
    {
      /* skip over data of character */
      for( i = 1; i <= bytes_left; i++ )
	(void) Pxl1( fp );
    }
}


RdPFont(fp)
register struct font *fp;
{
/* 
 * Read the header of a pixel file.
 */
	if (debug & DBG_PK)
		(void) fprintf(stderr, "Reading header for pixel file %s\n",
		fp->fontname);

	if (Pxl4(fp) != 1001) {
	    error("file %s is ill-formed PXL file (1)\n",fp->fontname);
	}
	(void) fseek(fp->file,(long)-(5*4),2);
	(void) Pxl4(fp);
	/*fp->f_scale = Pxl4(fp);*/
	/*if (fp->scale == 0) fp->scale = 1000;*/
	fp->design = Pxl4(fp);
	fp->glyphaddr = Pxl4(fp) * 4;  /* Pointer to glyph directory in file */
	if ((Pxl4(fp) != 1001)) {
    	    error("File %s is ill-formed PXL file (2)\n",fp->fontname);
	}
}


/********************************************/
/* end of PK routines                       */
/********************************************/


/*
**      define_font reads the rest of the fntdef command and then reads in
**      the specified PXL file, adding it to the global linked-list holding
**      all of the fonts used in the job.
*/
define_font(cmnd)
	ubyte cmnd;
{
        register struct font *fontp;
	int len;
	int unmodsize;
	float realsize;
	int size,pksize;
        long checksum;

	fontp = (struct font *) malloc((unsigned) sizeof(struct font));
	if (fontp == NULL)
		error("Can't allocate memory for font");
	fontp->TeXnumber = num(dvi_file, cmnd - FNTDEF1 + 1);
	checksum = four(dvi_file);
	fontp->scale = four(dvi_file);
	fontp->design = four(dvi_file);
	len = one(dvi_file) + one(dvi_file);
	fontp->fontname = malloc((unsigned) len + 10);	/* leave space for magnification */
	(void) fread(fontp->fontname, sizeof(char), len, dvi_file);
	fontp->fontname[len] = '\0';
	fontp->file = NULL;
	if(debug & DBG_PK)
	  (void) fprintf(stderr,"Define font \"%s\" scale=%d design=%d\n",
		  fontp->fontname,fontp->scale,fontp->design);
/*
**	In the actual implementation, scaled-size/design-size hasn't been
**	stored with sufficient precision, hence the messing around to find
**	its actual value.
*/
	realsize = (magnification/1000.)*((float) fontp->scale / fontp->design);
	unmodsize = (realsize * 1000) + 0.5;
	/* a real hack to correct for rounding in some cases */
	switch (unmodsize) {
	    case 1095:
		realsize = 1.095445;	/* stephalf */
		break;
	    case 1315:
		realsize = 1.314534;	/* stepihalf */
		break;
	    case 2074:
		realsize = 2.0736;	/* stepiv */
		break;
	    case 2488:
		realsize = 2.48832;	/* stepv */
		break;
	    case 2986:
		realsize = 2.985984;	/* stepiv */
		break;
	}
	/*
	 * the remaining magnification steps are represented
	 * with sufficient accuracy already
	 */
	size = (realsize * pixels_per_inch * 5) + 0.5;
	pksize = (realsize * pixels_per_inch) + 0.5;
        fontp->pksize = pksize;  /* mod for DV/X, MS-DOS GBP */
        fontp->size = size;      /* mod for DV/X, MS-DOS GBP */        
	
	(void) sprintf(&fontp->fontname[len], FONT_SUFFIX_PK, pksize);
	if (debug & DBG_PK)
	    (void) fprintf(stderr,"PKfilename=<%s>",fontp->fontname);
	if(open_pxl_file(fontp)) { /* THIS DOESN'T WORK QUITE RIGHT */
				/* since the open guesses on what it gets
				 * and ignores the extension... make this
				 * a site configureable option? Assume
				 * EITHER pk OR pxl??? [mwe/rlk] */
	  fontp->format = PkFormat;
	  read_glyphs(fontp);
	}
	else {
	  (void) sprintf(&fontp->fontname[len], FONT_SUFFIX_PXL, size);
	  if (debug & DBG_PK)
	    (void) fprintf(stderr,"PXLfilename=<%s>",fontp->fontname);
	  if(open_pxl_file(fontp)) {
	    fontp->format = PxlFormat;
	    read_glyphs(fontp);
	  }
	  else {
	    (void) fprintf(stderr,"%s [not found]\n",fontp->fontname);
	    font_not_found = 1;
	    return;
	  }
	}
	if (current_font == NULL) {
		fontp->next = fontp;
		fontp->prev = fontp;
	}
	else {
		fontp->next = current_font;
		fontp->prev = current_font->prev;
		current_font->prev->next = fontp;
		current_font->prev = fontp;
	}
	current_font = fontp;
}

open_pxl_file(font)
	struct font *font;
{
	char filename[300];
	extern int errno;
	if (font->file == NULL) {
		(void) sprintf(filename, "%s/%s", /* note: UNUSED */
				FONT_DIRECTORY, font->fontname);
		if (n_open_fonts == MAX_OPEN_FONTS)
			close_lru();
		font->file = fopen_font_path(font_path,
                                             font->fontname, "rb",
                                             font->pksize,
                                             font->size); /* mod for
                                                          DV/X, MS-DOS (GBP) */
		if (font->file == NULL) {
		    if (list_fonts)
                        (void) fprintf(stderr,"%ddpi %s [missing]\n",
                                       font->pksize,
                                       font->fontname);
                                       /* mod for DV/X, MS-DOS (GBP) */
                                       /* add extra info */
		    return (0);
		}
		n_open_fonts += 1;
	}
	if (list_fonts)
		(void) fprintf(stderr,"Loading %ddpi %s...\n",
                               font->pksize,
                               font->fontname);
                               /* mod for DV/X, MS-DOS (GBP) */
                               /* add extra info */
	return (1);
}

read_pxl_bitmap(ch, g)
	ubyte ch;
	register struct glyph *g;
{
	register struct bitmap *bitmap;
	register int file_bytes_wide;
	register unsigned char *ptr,*endbit;
	register int i, j;
	int bitmap_size;

	bitmap = &g->bitmap;

	/* in file, bitmap rows are multiples of 32 bits wide */
	file_bytes_wide = ROUNDUP(bitmap->w, BITS_PER_LONG)*BYTES_PER_LONG;
	/* width must be multiple of 16 bits for raster_op */
	bitmap->bytes_wide=ROUNDUP(bitmap->w, BITS_PER_SHORT)*BYTES_PER_SHORT;
	bitmap_size= (unsigned) bitmap->h * bitmap->bytes_wide;
	bitmap->bits = malloc((unsigned) bitmap_size);
	ptr = (unsigned char *) bitmap->bits;
	if (ptr == NULL)
		error("Can't allocate bitmap for character %d of font %s (%d by %d)",
			ch, current_font->fontname, bitmap->h, bitmap->w);
	if (!open_pxl_file(current_font))
		error("Can't find font file %s", current_font->fontname);
	(void) fseek(current_font->file, g->addr, 0);
	switch(current_font->format) {
	case PxlFormat:
		for (i = 0; i < bitmap->h; i += 1)
			for (j = 0; j < file_bytes_wide; j += 1)
				if (j < bitmap->bytes_wide)
					*ptr++ = one(current_font->file);
/*
	*ptr++ = reverse_byte[one(current_font->file)];
*/
				else
					one(current_font->file);
	    break;
	case PkFormat:
	    PK_skip_specials(current_font);
	    PK_read_char(current_font,bitmap->bits);
	    endbit=(unsigned char*)bitmap->bits+bitmap_size;
/*
	    for (ptr=bitmap->bits; ptr<endbit; ++ptr)
		*ptr = reverse_byte[(*ptr & 0377)];
*/
	    break;
	default:
		error("Internal error, bad format\n");
	}
	if (shrink_factor != 1)
		shrink_bitmap(bitmap, shrink_factor, shrink_factor); 
	if (debug & DBG_BITMAP)
		print_char(ch, g);
}

/*
 * Find font #n and move it to the head of the list.
 */
change_font(n)
	unsigned long n;
{
        register struct font *fontp;

	if (!current_font) return;
	fontp = current_font;
	for (;;) {
		if (fontp->TeXnumber == n)
                        break;
		fontp = fontp->next;
		if (fontp == current_font)
			error("Non-existent font #%d", n);
	}
	if (current_font == fontp)
		return;
	fontp->prev->next = fontp->next;
	fontp->next->prev = fontp->prev;
	fontp->next = current_font;
	fontp->prev = current_font->prev;
	current_font->prev->next = fontp;
	current_font->prev = fontp;
	current_font = fontp;
}

/*
 * Close the PXL file for the least recently used font.
 */
close_lru()
{
        register struct font *f;

	if (!current_font) return;
	f = current_font->prev;
	for (;;) {
		if (f->file != NULL)
                        break;
		f = f->prev;
		if (f == current_font->prev)
			error("Can't find an open PXL file to close");
	}
	(void) fclose(f->file);
	f->file = NULL;
	n_open_fonts -= 1;
}

reset_fonts()
{
        register struct font *f;
	register struct glyph *g;

	f = current_font;
	for (;f;) {
	    open_pxl_file(f);
	    for (g = &f->glyph[0]; g < &f->glyph[MAXCHARS]; g += 1) {
		if (g->bitmap.bits) free(g->bitmap.bits);
	    }
	    read_glyphs(f);
	    f = f->next;
	    if (f == current_font) break;
	}
}

read_P_glyphs (fontp)
        register struct font *fontp;
{
	register struct glyph *g;
        long checksum, magnify, design_size, font_dir_ptr, pxl_id_word;

	/* seek to trailer info */
	(void) fseek(fontp->file, (long) -(5 * BYTES_PER_LONG), 2);
        checksum = four(fontp->file);
        magnify = four(fontp->file);
        design_size = four(fontp->file);
        font_dir_ptr = sfour(fontp->file) * 4;
        pxl_id_word = four(fontp->file);
#ifdef lint
	magnify = design_size = pxl_id_word = checksum = magnify;
#endif
	/* seek to font directory */
	(void) fseek(fontp->file, font_dir_ptr, 0);
	for (g = &fontp->glyph[0]; g < &fontp->glyph[MAXPXLCHARS]; g += 1) {
		g->bitmap.bits = NULL;
		g->bitmap.w = two(fontp->file);	/* leave this for shrink_bitmap */
		g->bitmap.h = two(fontp->file);	/* leave this for shrink_bitmap */
		g->x = stwo(fontp->file) / shrink_factor;
		g->y = stwo(fontp->file) / shrink_factor;
		g->addr = four(fontp->file) * 4;
		/*
		**  The TFM-width word is kind of funny in the units
		**  it is expressed in.  It works this way:
		**
		**  If a glyph has width 'w' in a font with design-size
		**  'd' (both in same units), the TFM-width word is
		**
		**                    (w/d) * 2^20
		**
		**  Therefore, in order to find the glyph width in
		**  DVI units (1 / 2^16 points), we take the design-size
		**  'd' (in DVI's), the magnification 'm' of the PXL file
		**  and the TFM-width word 't' to the width (in DVI's)
		**  as follows:
		**
		**                     dmt
		**                w = -----
		**                    2^20
		**
		**  But the magnification of the PXL file is just the
		**  scaled size 's' over the design size, so the final
		**  expression for the width is
		**
		**                     st
		**                w = ----
		**                    2^20
		**
		*/      

		g->dvi_adv =
			((double) fontp->scale * four(fontp->file)) / (1 << 20);
		g->pxl_adv = pixel_round(g->dvi_adv);
#ifdef VERBOSE_DEBUG
		(void) fprintf(stderr,"char %d : dvi_adv=%d pxl_adv=%d\n",
		g - &fontp->glyph[0],g->dvi_adv,g->pxl_adv );
#endif
	}
}

read_PK_glyphs(fontp)
        register struct font *fontp;
{
	RdPkFont(fontp);
	(void) fseek(fontp->file,fontp->glyphaddr,0);
	PK_skip_specials( fontp );
	while( PK_flag_byte != PK_POST )
	  {
	    PK_read_char(fontp,(char *)0);	/* read "glyph directory" */
	    PK_skip_specials(fontp); /* (really a whole pass over file) */
	  }
}

read_glyphs(fontp)
        register struct font *fontp;
{
  switch(fontp->format)
    {
    case PxlFormat:
      read_P_glyphs(fontp);
      break;
    case PkFormat:
      read_PK_glyphs(fontp);
      break;
    }
}

