/**
 **
 ** SETPATH -- A utility for showing or setting the DOS environment path string
 **
 ** Version 1.2  09/08/91  (for Microsoft C Version 5.1)
 ** Copyright (c) D. E. Ekman 1990, 1991.  All rights reserved.
 **
 ** SETPATH either prints or sets the current PATH environment variable.  When
 ** setting the PATH variable, SETPATH expects a PATH string in the form 
 ** required by DOS, or the name of a file containing the wanted PATH string.
 ** In either case, SETPATH replaces any existing PATH environment variable with
 ** the new string.  The new string need not be the same length as the current
 ** string, but if it is longer, there must be enough available environment
 ** space to accommodate it.  In particular, the string may be longer than the
 ** 128-byte limit imposed by the DOS command line buffer, but to use this
 ** feature, the PATH string must come from a file.
 **
 ** Usage:
 **
 **  SETPATH
 **
 **  SETPATH path_string
 **
 **  SETPATH -f[d:\path]filename or SETPATH -F[d:\path]filename or
 **  SETPATH /f[d:\path]filename or SETPATH /F[d:\path]filename
 **
 ** where:
 **
 **   path_string is the new PATH environment variable string in the same
 **   form required by the DOS PATH command.  (Do not include "PATH=".)
 **
 **   filename is the name of a file containing the new PATH string.  The drive
 **   and path qualifiers are optional.  No space is permitted between the
 **   switch (-f, -F, /f, or /F) and its argument.  The file content must have
 **   the form:
 **
 **      PATH=any_wanted_text_string
 **
 **   The word PATH may be any combination of upper and lower case letters.
 **   This word and the following equals sign are required to help prevent
 **   using the wrong file, and must be the first five characters in the file
 **   except for spaces and control characters such as tabs and newlines.
 **   Before placing the file content in the environment, SETPATH changes any
 **   lower case characters in the word PATH to upper case and squeezes out all
 **   blanks and all control characters.  (SETPATH defines a control character
 **   as any character in the range 0x00-0x1f and the character 0x7f.)  If you
 **   do use newlines to separate the elements of your path string, do not
 **   forget the required semicolons; SETPATH will NOT add them automatically.
 **
 ** An error message is issued, and the current PATH string remains undisturbed,
 ** on any of the following conditions:
 **
 **   More than one argument
 **
 **   Specified file cannot be opened
 **
 **   Failure in fseek searching for end of file
 **
 **   Failure in ftell finding file length
 **
 **   Memory allocation failure creating buffer for new PATH contents
 **
 **   Specified file does not begin with "PATH="
 **
 **   Not enough environment space for new path
 **
 ** A successful completion message is issued when SETPATH inserts the new
 ** PATH string into the environment.
 **
 **/
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

main(argc,argv)
int argc;
char *argv[];
{
    int count, seekresult, pathlength, available, olength, ch;
    unsigned int envseg, envsize, envinuse;
    long int filelength;
    char *filebuff;
    char far *ptr, far *pathptr, far *endptr;
    FILE *ip;
    int numread = 0;
    
    printf("\nThe Ekman Utility SETPATH, V1.2\n");
    printf("Copyright (c) D. E. Ekman 1990, 1991.  All rights reserved.\n");
    
    if(argc > 2)
	{
	printf("\nSetpath:  Too many arguments\007\n");
	return (1);
	}
    if(argc > 1)
	{
	if(strnicmp(argv[1],"-f",2) &&
	   strnicmp(argv[1],"/f",2))
	    {
	    if((filebuff = (char *)malloc((strlen(argv[1])+8)*
	      sizeof(char))) == NULL)
		{
		printf("\nSetpath:  Failure allocating new PATH buffer\007\n");
		return (5);
		}
	    else
		{
		sprintf(filebuff,"PATH=%s",argv[1]);
		numread = strlen(filebuff);
		}
	    }
	else
	    {
	    if(!(ip = fopen(&argv[1][2],"r")))
		{
		printf("\nSetpath:  Can't open path file %s\007\n",&argv[1][2]);
		return (2);
		}
	    if(seekresult = fseek(ip, 0L, SEEK_END))
		{
		printf("\nSetpath:  Failure seeking end of %s\007\n",
		       &argv[1][2]);
		return (3);
		}
	    if((filelength = ftell(ip)) < 0)
		{
		printf("\nSetpath:  Failure in ftell with %s\007\n",
		       &argv[1][2]);
		return (4);
		}
	    rewind(ip);
	    if ((filebuff = (char *)malloc((filelength+2)*sizeof(char)))
		== NULL)
		{
		printf("\nSetpath:  Failure allocating new PATH buffer\007\n");
		return (5);
		}
	    ch = fgetc(ip);
	    for(numread = 0; (numread < filelength) && (!feof(ip)) ;)
		{
		if(isascii(ch) && (!iscntrl(ch)) && ch != ' ')
		    {
		    if(numread < 4)
			ch = toupper(ch);
		    *(filebuff + numread++) = ch;
		    }
		ch = fgetc(ip);
		}
	    *(filebuff + numread++) = '\0';
	    if(strncmp(filebuff, "PATH=", 5))
		{
		printf("\nSetpath:  Path file %s must begin with %s\007\n",
		       &argv[1][2],"PATH=");
		return (6);
		}
	    }
	}
    getsysenv(&envseg, &envsize, &envinuse);
    ptr = (char far *) ((long) envseg << 16);
    pathptr = NULL;
    while(*ptr != '\0')
	{
	if(*ptr == 'P' && *(ptr+1) == 'A' && *(ptr+2) == 'T' &&
	    *(ptr+3) == 'H' && *(ptr+4) == '=')
	    {
	    pathptr = ptr;
	    break;
	    }
	while(*ptr++ != '\0')
	    ;
	}
    if(!numread)
	{
	printf("\n");
	while(*pathptr)
	    putch(*pathptr++);
	puts("");
	return (0);
	}
    pathlength = 0;
    if(pathptr != NULL)
	{
	while(*(pathptr + pathlength++) != '\0')
	    ;
	}
    available = envsize - envinuse + pathlength;
    if(numread > available)
	{
	printf("\nSetpath:  Not enough environment space for new path\007\n");
	return (7);
	}
    ptr = (char far *) ((long) envseg << 16);
    olength = envinuse - (pathptr - ptr + pathlength);
    endptr = ptr + envinuse -1;
    if(pathptr != NULL)
	{
	if(numread == pathlength)
	    {
	    while(*pathptr++ = *filebuff++)
		;
	    }
	if(numread < pathlength)
	    {
	    ptr = pathptr + pathlength;
	    while(*pathptr++ = *filebuff++)
		;
	    for(count = 0; count < olength; count++)
		*pathptr++ = *ptr++;
	    }
	if(numread > pathlength)
	    {
	    ptr = endptr + numread - pathlength;
	    for(count = 0; count < olength; count++)
		*ptr-- = *endptr--;
	    while(*pathptr++ = *filebuff++)
		;
	    }
	}
    else
	{
	while(*ptr != '\0')
	    {
	    if((*ptr != 'C' || *(ptr+1) != 'O' || *(ptr+2) != 'M' ||
		*(ptr+3) != 'S' || *(ptr+4) != 'P' || *(ptr+5) != 'E' ||
		*(ptr+6) != 'C' || *(ptr+7) != '=') &&
		(*ptr != 'P' || *(ptr+1) != 'R' || *(ptr+2) != 'O' ||
		*(ptr+3) != 'M' || *(ptr+4) != 'P' || *(ptr+5) != 'T' ||
		*(ptr+6) != '='))
		{
		pathptr = ptr;
		break;
		}
	    while(*ptr++ != '\0')
		;
	    }
	ptr = endptr + numread;
	for(count = 0; count < olength; count++)
	    *ptr-- = *endptr--;
	while(*pathptr++ = *filebuff++)
	    ;
	}
    printf("\n New PATH string successfully placed in environment.\n");
    return(0);
}

getsysenv(envseg, envsize, envinuse)
unsigned *envseg, *envsize, *envinuse;
{
    unsigned getdospsp(), computeenvinuse(), dospsp, temp;
    unsigned far *ptr;
	
    dospsp = getdospsp();
    ptr = (unsigned far *) (((long) dospsp << 16) | 0x2c);
    temp = *ptr;
    if((temp != 0) && (_osmajor != 3 || _osminor <= 19 || _osminor >= 30))
	{
	*envseg = temp;
	}
    else
	{
	ptr = (unsigned far *) (((long) (dospsp-1) << 16) | 3);
	*envseg = dospsp + (*ptr) + 1;
	}
    ptr = (unsigned far *) (((long) ((*envseg)-1) << 16) | 3);
    *envsize = 16*(*ptr);
    *envinuse = computeenvinuse(*envsize, *envseg);
    return(0);
}

unsigned getdospsp()
{
    unsigned temp,temp1;
    unsigned far *ptr;

    ptr = (unsigned far *) (((long) _psp << 16) | 0x16);
    temp = *ptr;        
    while(1)
	{
	ptr = (unsigned far *) (((long) temp << 16) | 0x16);
	temp1 = *ptr;
	if((!temp1) || (temp1 == temp))
	    return(temp);
	else
	    temp = temp1;
	}
}

unsigned computeenvinuse(envsize, envseg)
unsigned envsize, envseg;
{
    unsigned j;
    unsigned far *ptr;
	
    for(j=0; j<=(envsize-3); j++)
	{
	ptr = (unsigned far *) (((long) envseg << 16) | j);
	if(*ptr == 0)
	    {
	    return(j+2);
	    }
	}
    return(envsize);
}
