//-------------------------------------------------------------------//
//  join

//  Syntax: join (s)
//          join (s, delimeter)

//  Description:

//  Join strings/numbers together to form a single string
//
//  s can be any class (list, matrix, vector) containing 
//  string/numeric data to be concatenated together.
//
//  delimiter is an optional string to be stuffed between 
//  strings/numbers.
// 
//  The output is a single string.
//

//  Example 1:
//  > s = ["January","February","March"]
//   s =
//  January   February  March
//  > t = join(s)
//   t =
//  JanuaryFebruaryMarch
//  > t = join(s," ")
//   t =
//  January February March
//  > t = join(s,", ")
//   t =
//  January, February, March
//  
 
//  Example 2:
//  > n = [1,2,3,4,5]
//   n =
//          1          2          3          4          5
//  > t = join(n)
//   t =
//  12345
//  > t = join(n,":")
//   t =
//  1:2:3:4:5
//

//  Example 3:
//  > r.[1] = "My zip code is";
//  > r.[2] = 94720;
//  > t = join(r," ")
//   t =
//  My zip code is 94720
//  >

//  Example 4:
//  > r = getline("file");
//  > line = join(r," ");
//

//  See Also: strsplt, strtod, num2str, getline, sprintf

//  Tzong-Shuoh Yang (tsyang@ce.berkeley.edu)  5/10/94
//
//-------------------------------------------------------------------//

join = function(s,delimiter)
{  
   local (i,del,n,ts,tmp,txt);

   if (exist(delimiter)) {
      if (class(delimiter) != "string") {
         error("Argument delimiter of function join() must be a string");
      }
      del = delimiter;
   else
      del = "";
   }
   
   txt = "";
   
   if (class(s) == "list") {
     n = length(s);
     for (i in 1:n) {
        if (class(s.[i])=="num") {
          if (type(s.[i]) == "real") {
             sprintf(tmp,"%g",s.[i]);
          else
             sprintf(tmp,"%g+%gi",real(s.[i]),imag(s.[i]));
          }
          txt = txt + tmp;
        else
          txt = txt + s.[i];       
        }
        if (i < n) {
           txt = txt + del;
        }
     }
  else
     ts = s[:];
     for (i in 1:ts.n) {
        if (class(ts[i])=="num") {
          if (type(ts[i]) == "real") {
             sprintf(tmp,"%g", ts[i]);
          else
             sprintf(tmp,"%g+%gi",real(ts[i]),imag(ts[i]));
          }        
          txt = txt + tmp;
        else
          txt = txt + ts[i];       
        }
        if (i < ts.n) {
           txt = txt + del;
        }
     }    
  }
  return txt;
};
