/*			      SUPCMPLX.CC
			A C++ implementation of
			   "super precision"
			    complex numbers.

                Copyright (c) 1997 William T. Wyatt, Jr.

	"Super precision" objects are implementations of a floating-point
data type having user-selectable precision and radix.  The intent is to
allow extremely long multiple-precision mantissas in the floating-point
representation.

	This file contains a supplemental class definition for a "complex
super precision" floating-point data type with similar properties.

*/

#ifndef already_super
#include "super.cc"
#endif


class super_complex					{ // CLASS DEFINITION
protected:
	static super_complex *zzc;	// returnable scratch object

public:
	super_precision real;		// real part
	super_precision imag;		// imaginary part

	super_complex ( void);					// constructor
	super_complex ( const int32, const int32);		// constructor
//	~super_complex ( void);					// destructor
	super_complex ( const int32);			//conversion constructor
	super_complex ( const double);			//conversion constructor
	void *operator new ( size_t);			// overload new
	void operator delete ( void *);			// overload delete
	static void setup ( const int32, const short i=10);	//initialization
	super_complex ( const super_complex &);		// ovld copy-constructor
	super_complex & operator= ( const super_complex &);	// overload a=b
	super_complex & operator= ( const int32);		// overload a=i
	super_complex & operator= ( const double);		// overload a=d
	super_complex & operator= ( const super_precision &);	// overload a=s
	super_complex operator- ( void);			// overload -b
	super_complex operator+ ( void);			// overload +b
	super_complex operator+ ( const super_complex &) const; // overload a+b
	super_complex & operator+=( const super_complex &);     // overload a+=b
	super_complex operator+ ( const int32) const;		// overload a+i
	super_complex & operator+=( const int32);		// overload a+=i
	super_complex operator+ ( const super_precision &) const;//overload a+s
	super_complex & operator+=( const super_precision &);	// overload a+=s
	super_complex operator- ( const super_complex &) const; // overload a-b
	super_complex & operator-=( const super_complex &);	// overload a-=b
	super_complex operator- ( const int32) const;		// overload a-i
	super_complex & operator-=( const int32);		// overload a-=i
	super_complex operator- ( const super_precision &) const;//overload a-s
	super_complex & operator-=( const super_precision &);	// overload a-=s
	super_complex operator* ( const super_complex &) const; // overload a*b
	super_complex & operator*=( const super_complex &);	// overload a*=b
	super_complex operator* ( const int32) const;		// overload a*i
	super_complex & operator*=( const int32);		// overload a*=i
	super_complex operator* ( const super_precision &) const;//overload a*s
	super_complex & operator*=( const super_precision &);	// overload a*=s
	super_complex operator/ ( const super_complex &) const; // overload a/b
	super_complex & operator/=( const super_complex &);	// overload a/=b
	super_complex operator/ ( const int32) const;		// overload a/i
	super_complex & operator/=( const int32);		// overload a/=i
	super_complex operator/ ( const super_precision &) const;//overload a/s
	super_complex & operator/=( const super_precision &);	// overload a/=s
	friend super_complex operator+ ( const int32,
				const super_complex &); 	// overload i+a
	friend super_complex operator- ( const int32,
				const super_complex &); 	// overload i-a
	friend super_complex operator* ( const int32,
				const super_complex &); 	// overload i*a
	friend super_complex operator/ ( const int32,
				const super_complex &); 	// overload i/a
	friend super_complex operator+ ( const super_precision &,
				const super_complex &); 	// overload s+a
	friend super_complex operator- ( const super_precision &,
				const super_complex &); 	// overload s-a
	friend super_complex operator* ( const super_precision &,
				const super_complex &); 	// overload s*a
	friend super_complex operator/ ( const super_precision &,
				const super_complex &); 	// overload s/a
	int operator==( const super_complex &) const;		// overload a==b
	int operator!=( const super_complex &) const;		// overload a!=b
	int operator==( const int32) const;			// overload a==i
	int operator!=( const int32) const;			// overload a!=i
	friend int operator==( const int32, const super_complex &);//overld i==a
	friend int operator!=( const int32, const super_complex &);//overld i!=a
	void super_dump ( void) const;			// dump super_complex
	friend super_complex pow ( const super_complex &,
				const int32);			// overload pow
	friend super_complex pow ( const super_complex &,
				const super_precision &);	// overload pow
	friend super_complex pow ( const super_complex &,
				const super_complex &);		// overload pow
	friend super_complex pow ( const super_precision &,
				const super_complex &);		// overload pow
	friend super_complex pow ( const int32,
				const super_complex &);		// overload pow
	friend super_complex cmplx ( const super_precision &,
				const super_precision &);	// store complex
	friend super_complex conjg ( const super_complex &);	// store conjug
	friend super_precision fabs ( const super_complex &);	// overload fabs
	friend super_complex sqrt ( const super_complex &);	// overload sqrt
	friend super_complex exp ( const super_complex &);	// overload exp
	friend super_complex log ( const super_complex &);	// overload log
	friend super_complex sin ( const super_complex &);	// overload sin
	friend super_complex cos ( const super_complex &);	// overload cos
	friend super_complex tan ( const super_complex &);	// overload tan
	friend super_precision norm ( const super_complex &);	// a = norm (b)
	friend super_precision argand ( const super_complex &);	// a = argand(b)
	friend super_complex polar ( const super_precision &,
				const super_precision &);	// a =polar(b,c)
	friend ostream & operator<<						// OVERLOAD <<
			( ostream &, const super_complex &);	// overload <<

};			// END OF CLASS DEFINITION of super_complex.

super_complex :: super_complex ( void)			// CLASS CONSTRUCTOR
{
}

super_complex :: super_complex ( const int32 i,		// CLASS CONSTRUCTOR
					const int32 j)
{
	this->real = i;
	this->imag = j;
}

void *super_complex :: operator new			// OVERLOAD new
				( size_t size)
{
	return (new super_precision [2]);
}

void super_complex :: operator delete			// OVERLOAD delete
				( void *ptr)
{
	delete [] ptr;
	ptr = 0;			// set ptr to NULL
}

super_complex :: super_complex		      // OVERLOAD copy-constructor
				( const super_complex &a)
{
	real = a.real;
	imag = a.imag;
}

super_complex :: super_complex			  //CONVERSION CONSTRUCTOR
				( const int32 i)
{
	this->real = i;			// Convert integer to super_complex.
	this->imag = 0;
}

super_complex :: super_complex			  //CONVERSION CONSTRUCTOR
				( const double d)
{
	this->real = d;			// Convert double to super_complex.
	this->imag = 0;
}


//
// ALLOCATE CLASS STATIC OBJECTS AT FILE SCOPE.
//
super_complex *super_complex :: zzc;


void super_complex :: setup					// PUBLIC FN
			( const int32 digits_, const short radix_)
{							// initializes stuff
	// Invoke super_precision setup function.
	super_precision :: setup ( digits_, radix_);

	// Allocate *zzc scratch super_complex object.
//	super_complex *super_complex :: zzc = new super_complex;
	super_complex :: zzc = new super_complex;

	return;
}

super_complex & super_complex :: operator=			// OVERLOAD =
				( const super_complex &b)
{								// ASSIGN a=b
	this->real = b.real;
	this->imag = b.imag;
	return ( *this);
}

super_complex & super_complex :: operator=			// OVERLOAD =
				( const int32 i)
{								// ASSIGN a=i
	this->real = i;
	this->imag = 0;
	return ( *this);
}

super_complex & super_complex :: operator=			// OVERLOAD =
				( const double d)
{								// ASSIGN a=d
	this->real = d;
	this->imag = 0;
	return ( *this);
}

super_complex & super_complex :: operator=			// OVERLOAD =
				( const super_precision &s)
{								// ASSIGN a=s
	this->real = s;
	this->imag = 0;
	return ( *this);
}

super_complex super_complex :: operator-			// OVERLOAD -
				( void)
{								// unary -b
	zzc->real = -this->real;
	zzc->imag = -this->imag;
	return ( *zzc);
}

super_complex super_complex :: operator+			// OVERLOAD +
				( void)
{								// unary +b
	zzc->real = this->real;
	zzc->imag = this->imag;
	return ( *zzc);
}

super_complex super_complex :: operator+			// OVERLOAD +
				( const super_complex &b) const
{								// a + b
	zzc->real = this->real + b.real;
	zzc->imag = this->imag + b.imag;
	return ( *zzc);
}

super_complex & super_complex :: operator+=			// OVERLOAD +=
				( const super_complex &b)
{								// a += b
	this->real += b.real;
	this->imag += b.imag;
	return ( *this);
}

super_complex super_complex :: operator-			// OVERLOAD -
				( const super_complex &b) const
{								// a - b
	zzc->real = this->real - b.real;
	zzc->imag = this->imag - b.imag;
	return ( *zzc);
}

super_complex & super_complex :: operator-=			// OVERLOAD -=
				( const super_complex &b)
{								// a -= b
	this->real -= b.real;
	this->imag -= b.imag;
	return ( *this);
}

super_complex super_complex :: operator+			// OVERLOAD +
				( const int32 i) const
{								// a + i
	zzc->real = this->real + i;
	zzc->imag = this->imag;
	return ( *zzc);
}

super_complex & super_complex :: operator+=			// OVERLOAD +=
				( const int32 i)
{								// a + i
	this->real += i;
	return ( *this);
}

super_complex super_complex :: operator+			// OVERLOAD +
				( const super_precision &s) const
{								// a + s
	zzc->real = this->real + s;
	zzc->imag = this->imag;
	return ( *zzc);
}

super_complex & super_complex :: operator+=			// OVERLOAD +=
				( const super_precision &s)
{								// a + s
	this->real += s;
	return ( *this);
}

super_complex super_complex :: operator-			// OVERLOAD -
				( const int32 i) const
{								// a - i
	zzc->real = this->real - i;
	zzc->imag = this->imag;
	return ( *zzc);
}

super_complex & super_complex :: operator-=			// OVERLOAD -=
				( const int32 i)
{								// a - i
	zzc->real = i;
	this->real -= zzc->real;
	return ( *this);
}

super_complex super_complex :: operator-			// OVERLOAD -
				( const super_precision &s) const
{								// a - s
	zzc->real = this->real - s;
	zzc->imag = this->imag;
	return ( *zzc);
}

super_complex & super_complex :: operator-=			// OVERLOAD -=
				( const super_precision &s)
{								// a - s
	this->real -= s;
	return ( *this);
}

super_complex super_complex :: operator*			// OVERLOAD *
				( const super_complex &b) const
{								// a * b
// This function obtains the complex product of *this and b using the method
// sometimes called Golub's method, where the product is done with
// 3 multiplications and 5 additions, instead of the usual method that uses
// 4 multiplications and 2 additions.
	super_precision_pool_ptr r_, s_, za_, zb_;
	super_precision &r = *r_;
	super_precision &s = *s_;
	super_precision &za = *za_;
	super_precision &zb = *zb_;

	r = this->real * b.imag;
	s = this->imag * b.real;
	za = r - s;
	r += s;
	s = this->real + this->imag;
	zb = b.real - b.imag;
	s *= zb;
	zzc->real = s + za;
	zzc->imag = r;

	return ( *zzc);
}

super_complex & super_complex :: operator*=			// OVERLOAD *=
				( const super_complex &b)
{								// a *= b
	super_precision_pool_ptr r_, s_, za_, zb_;
	super_precision &r = *r_;
	super_precision &s = *s_;
	super_precision &za = *za_;
	super_precision &zb = *zb_;

	r = this->real * b.imag;
	s = this->imag * b.real;
	za = r - s;
	r += s;
	s = this->real + this->imag;
	zb = b.real - b.imag;
	s *= zb;
	this->real = s + za;
	this->imag = r;

	return ( *this);
}

super_complex super_complex :: operator*			// OVERLOAD *
				( const int32 i) const
{								// a * i
	zzc->real = this->real * i;
	zzc->imag = this->imag * i;

	return ( *zzc);
}

super_complex & super_complex :: operator*=			// OVERLOAD *=
				( const int32 i)
{								// a *= i
	this->real *= i;
	this->imag *= i;

	return ( *this);
}

super_complex super_complex :: operator*			// OVERLOAD *
				( const super_precision &s) const
{								// a * s
	zzc->real = this->real * s;
	zzc->imag = this->imag * s;

	return ( *zzc);
}

super_complex & super_complex :: operator*=			// OVERLOAD *=
				( const super_precision &s)
{								// a *= s
	this->real *= s;
	this->imag *= s;

	return ( *this);
}

super_complex operator+					// OVERLOAD +
				( const int32 i, const super_complex &a)
{	// FRIEND FUNCTION 					// i + a
	super_complex :: zzc->real = a.real;
	super_complex :: zzc->real += i;
	super_complex :: zzc->imag = a.imag;

	return ( *super_complex :: zzc);
}

super_complex operator-					// OVERLOAD -
				( const int32 i, const super_complex &a)
{	// FRIEND FUNCTION 					// i - a

			// zzc cannot be the same object as a.
	super_complex :: zzc->real = i;
	super_complex :: zzc->real -= a.real;
	super_complex :: zzc->imag = a.imag;

	return ( *super_complex :: zzc);
}

super_complex operator*					// OVERLOAD *
				( const int32 i, const super_complex &a)
{	// FRIEND FUNCTION 					// i * a

	super_complex :: zzc->real = a.real;
	super_complex :: zzc->real *= i;
	super_complex :: zzc->imag = a.imag;
	super_complex :: zzc->imag *= i;

	return ( *super_complex :: zzc);
}

super_complex operator+					// OVERLOAD +
		( const super_precision &s, const super_complex &a)
{	// FRIEND FUNCTION 					// s + a
	super_complex :: zzc->real = s + a.real;
	super_complex :: zzc->imag = a.imag;

	return ( *super_complex :: zzc);
}

super_complex operator-					// OVERLOAD -
		( const super_precision &s, const super_complex &a)
{	// FRIEND FUNCTION 					// s - a
	super_complex :: zzc->real = s - a.real;
	super_complex :: zzc->imag = a.imag;

	return ( *super_complex :: zzc);
}

super_complex operator*					// OVERLOAD *
		( const super_precision &s, const super_complex &a)
{	// FRIEND FUNCTION 					// s * a

	super_complex :: zzc->real = s * a.real;
	super_complex :: zzc->imag = s * a.imag;

	return ( *super_complex :: zzc);
}

super_complex super_complex :: operator/			// OVERLOAD /
				( const super_complex &b) const
{								// a / b
	super_precision_pool_ptr r_, s_, za_, zb_, zc_;
	super_precision &r = *r_;
	super_precision &s = *s_;
	super_precision &za = *za_;
	super_precision &zb = *zb_;
	super_precision &zc = *zc_;

	zc = b.imag;
	zc = - zc;
	r = this->real * zc;
	s = this->imag * b.real;
	za = r - s;
	r += s;
	s = this->real + this->imag;
	zb = b.real - zc;
	s *= zb;
	s += za;
	za = b.real * b.real;
	zb = zc * zc;
	za += zb;
	zb = 1;
	za = zb / za;
	zzc->real = s * za;
	zzc->imag = r * za;

	return ( *zzc);
}

super_complex & super_complex :: operator/=			// OVERLOAD /=
				( const super_complex &b)
{								// a /= b
	super_precision_pool_ptr r_, s_, za_, zb_, zc_;
	super_precision &r = *r_;
	super_precision &s = *s_;
	super_precision &za = *za_;
	super_precision &zb = *zb_;
	super_precision &zc = *zc_;

	zc = b.imag;
	zc = - zc;
	r = this->real * zc;
	s = this->imag * b.real;
	za = r - s;
	r += s;
	s = this->real + this->imag;
	zb = b.real - zc;
	s *= zb;
	s += za;
	za = b.real * b.real;
	zb = zc * zc;
	za += zb;
	zb = 1;
	za = zb / za;
	this->real = s * za;
	this->imag = r * za;

	return ( *this);
}

super_complex super_complex :: operator/			// OVERLOAD /
				( const int32 i) const
{								// a / i
	zzc->real = this->real / i;
	zzc->imag = this->imag / i;

	return ( *zzc);
}

super_complex & super_complex :: operator/=			// OVERLOAD /=
				( const int32 i)
{								// a /= i
	this->real /= i;
	this->imag /= i;

	return ( *this);
}

super_complex super_complex :: operator/			// OVERLOAD /
				( const super_precision &s) const
{								// a / s
	super_precision_pool_ptr r_;
	super_precision &r = *r_;

	r = 1;
	r /= s;
	zzc->real = this->real * r;
	zzc->imag = this->imag * r;

	return ( *zzc);
}

super_complex & super_complex :: operator/=			// OVERLOAD /=
				( const super_precision &s)
{								// a /= s
	super_precision_pool_ptr r_;
	super_precision &r = *r_;

	r = 1;
	r /= s;
	this->real *= r;
	this->imag *= r;

	return ( *this);
}

int super_complex :: operator==					// OVERLOAD a==b
				( const super_complex &b) const
{								// a == b
// Returns a TRUE or FALSE for the numerical relationship
//                this->real == b.real && this->imag == b.imag.

	return ( ( this->real == b.real)
		&& ( this->imag == b.imag));
}

int super_complex :: operator!=					// OVERLOAD a!=b
				( const super_complex &b) const
{								// a != b
// Returns a TRUE or FALSE for the numerical relationship
//                this->real != b.real || this->imag != b.imag.

	return ( ( this->real != b.real)
		|| ( this->imag != b.imag));
}

int super_complex :: operator==					// OVERLOAD a==i
				( const int32 i) const
{								// a == b
// Returns a TRUE or FALSE for the numerical relationship
//                this->real == i && this->imag == 0.

	return ( ( this->real == i)
		&& ( this->imag == 0));
}

int super_complex :: operator!=					// OVERLOAD a!=i
				( const int32 i) const
{								// a != b
// Returns a TRUE or FALSE for the numerical relationship
//                this->real != i && this->imag == 0.

	return ( ( this->real != i)
		|| ( this->imag != 0));
}

int operator==							// OVERLOAD i==a
				( const int32 i, const super_complex &a)
{	// FRIEND FUNCTION					// i == a
// Returns a TRUE or FALSE for the numerical relationship
//                i == a.real && 0 == a.imag
	*super_complex :: zzc = i;
	return ( ( super_complex :: zzc->real == a.real)
		&& ( super_complex :: zzc->imag == a.imag));
}

int operator!=							// OVERLOAD i!=a
				( const int32 i, const super_complex &a)
{	// FRIEND FUNCTION					// i != a
// Returns a TRUE or FALSE for the numerical relationship
//                i != a.real || 0 != a.imag
	*super_complex :: zzc = i;
	return ( ( super_complex :: zzc->real != a.real)
		|| ( super_complex :: zzc->imag != a.imag));
}

void super_complex :: super_dump				// PUBLIC FN
				( void) const
{							// (const) dump a
	int32 i;

	cout << "\n/R/"; real.super_dump ();
	cout << "/I/"; imag.super_dump ();
	cout << endl;

	return;
}

super_complex pow						// FRIEND FN
			( const super_complex &b, const int32 ia)
{	// FRIEND FUNCTION					// a = pow(b,ia)
// This function calculates the complex quantity b raised to the ia power.
	int32 i, k, l;				// temporaries

	if ( b == 0)
	{
		if ( ia > 0)
		{			// Result is zero.
			*super_complex :: zzc = 0;
			return ( *super_complex :: zzc);
		}
		else
		{
			cout << "Fatal error -- pow taking nonpositive power";
			cout << " of zero.";
			exit (1);
		}
	}
	else
	{
		k = abs ( ia);
		if ( k == 0)
		{			// zeroth power yields unity
			*super_complex :: zzc = 1;
			return ( *super_complex :: zzc);
		}
		else
		{
			int first_store = 1;
			super_complex zv = b;
			while ( k)
			{
				i = k / 2;
				l = k - i - i;
				if ( l)
				{
					if ( first_store)
					{
						first_store = 0;
						*super_complex :: zzc = zv;
					}
					else
					{
						*super_complex :: zzc *= zv;
					}
				}
				if ( i) zv *= zv;
				k = i;
			}
						// positive exponent
			if ( ia > 0) return ( *super_complex :: zzc);
						// negative exponent
			else
			{
				zv = 1;
				return ( zv / *super_complex :: zzc);
			}
		}
	}
}			// End of function pow (b,ia).

super_complex pow						// FRIEND FN
			( const super_complex &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// a = pow(b,s)
// This function calculates the complex quantity b raised to the c power,
// where c is a super_precision object.

	if ( b.real == 0 && b.imag == 0)
	{
		if ( c > 0)
		{			// Result is zero.
			*super_complex :: zzc = 0;
			return ( *super_complex :: zzc);
		}
		else
		{
			cout << "Fatal error -- pow taking nonpositive power";
			cout << " of complex zero.";
			exit (1);
		}
	}
	else
	{
		if ( c == 0)
		{			// zeroth power yields unity
			*super_complex :: zzc = 1;
			return ( *super_complex :: zzc);
		}
		else
		{
			*super_complex :: zzc = log ( b);
			*super_complex :: zzc *= c;
			*super_complex :: zzc = exp ( *super_complex :: zzc);
			return ( *super_complex :: zzc);
		}
	}
}

super_complex pow						// FRIEND FN
			( const super_complex &b,
			const super_complex &c)
{	// FRIEND FUNCTION					// a = pow(b,c)
// This function calculates the complex quantity b raised to the c power,
// where c is a super_complex object.

	if ( b.real == 0 && b.imag == 0)
	{
		if ( c.real > 0)
		{			// Result is zero.
			*super_complex :: zzc = 0;
			return ( *super_complex :: zzc);
		}
		else
		{
			cout << "Fatal error -- pow taking power, with";
			cout << " nonpositive real part, of complex zero.";
			exit (1);
		}
	}
	else
	{
		*super_complex :: zzc = log ( b);
		*super_complex :: zzc *= c;
		*super_complex :: zzc = exp ( *super_complex :: zzc);
		return ( *super_complex :: zzc);
	}
}
	
super_complex pow						// FRIEND FN
			( const super_precision &b,
			const super_complex &c)
{	// FRIEND FUNCTION					// a = pow(s,c)
// This function calculates the super_precision object b raised to the c power,
// where c is a super_complex object.

	if ( b == 0)
	{
		if ( c.real > 0)
		{			// Result is zero.
			*super_complex :: zzc = 0;
			return ( *super_complex :: zzc);
		}
		else
		{
			cout << "Fatal error -- pow taking power, with";
			cout << " nonpositive real part, of zero.";
			exit (1);
		}
	}
	else
	{
		*super_complex :: zzc = log ( b);
		*super_complex :: zzc *= c;
		*super_complex :: zzc = exp ( *super_complex :: zzc);
		return ( *super_complex :: zzc);
	}
}
	
super_complex pow						// FRIEND FN
			( const int32 i,
			const super_complex &c)
{	// FRIEND FUNCTION					// a = pow(i,c)
// This function calculates the signed-integer i raised to the c power,
// where c is a super_complex object.

	if ( i == 0)
	{
		if ( c.real > 0)
		{			// Result is zero.
			*super_complex :: zzc = 0;
			return ( *super_complex :: zzc);
		}
		else
		{
			cout << "Fatal error -- pow taking power, with";
			cout << " nonpositive real part, of zero.";
			exit (1);
		}
	}
	else
	{
		super_complex :: zzc->real = i;
		*super_complex :: zzc = log ( super_complex :: zzc->real);
		*super_complex :: zzc *= c;
		*super_complex :: zzc = exp ( *super_complex :: zzc);
		return ( *super_complex :: zzc);
	}
}
	
super_complex cmplx ( const super_precision &a,			// FRIEND FN
				const super_precision &b)
{							// store complex
	super_complex :: zzc->real = a;
	super_complex :: zzc->imag = b;

	return *super_complex :: zzc;
}

super_complex conjg ( const super_complex &a)			// FRIEND FN
{							// store conjug
	super_complex :: zzc->real = a.real;
	super_complex :: zzc->imag = a.imag;
	super_complex :: zzc->imag = -super_complex :: zzc->imag;

	return *super_complex :: zzc;
}

super_precision fabs ( const super_complex &a)			// FRIEND FN
{							// overload fabs
	return hypot ( a.real, a.imag);
}
	
super_complex sqrt ( const super_complex &a)			// FRIEND FN
{							// overload sqrt
	if ( a.real == 0 && a.imag == 0)
	{
		return ( *super_complex :: zzc = 0);
	}

	if ( a.real < 0)
	{
		super_complex :: zzc->real = fabs ( a);
		super_complex :: zzc->real -= a.real;
		super_complex :: zzc->real /= 2;
		super_complex :: zzc->real = sqrt ( super_complex :: zzc->real);
		if ( a.imag < 0) super_complex :: zzc->real =
					- super_complex :: zzc->real;
		super_complex :: zzc->imag = super_complex :: zzc->real;
		super_complex :: zzc->real += super_complex :: zzc->real;
		super_complex :: zzc->real = a.imag / super_complex:: zzc->real;
	}
	else
	{
		super_complex :: zzc->real = fabs ( a);
		super_complex :: zzc->real += a.real;
		super_complex :: zzc->real /= 2;
		super_complex :: zzc->real = sqrt ( super_complex :: zzc->real);
		super_complex :: zzc->imag = super_complex :: zzc->real;
		super_complex :: zzc->imag += super_complex :: zzc->real;
		super_complex :: zzc->imag = a.imag / super_complex:: zzc->imag;
	}

	return *super_complex :: zzc;
}

super_complex exp ( const super_complex &a)			// FRIEND FN
{							// overload exp
	sincos ( a.imag, super_complex :: zzc->imag, super_complex:: zzc->real);

	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	b = exp ( a.real);
	super_complex :: zzc->real *= b;
	super_complex :: zzc->imag *= b;

	return *super_complex :: zzc;
}
	
super_complex log ( const super_complex &a)			// FRIEND FN
{							// overload log
	super_complex :: zzc->imag = atan2 ( a.imag, a.real);

	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	b = a.imag;
	b *= b;
	super_complex :: zzc->real = a.real;
	super_complex :: zzc->real *= super_complex :: zzc->real;
	b += super_complex :: zzc->real;
	super_complex :: zzc->real = log ( b);
	super_complex :: zzc->real /= 2;

	return *super_complex :: zzc;

}

super_complex sin ( const super_complex &a)			// FRIEND FN
{							// overload sin
	sincos ( a.real, super_complex :: zzc->real, super_complex:: zzc->imag);

	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	b = sinh ( a.imag);
	super_complex :: zzc->imag *= b;
	b *= b;
	b++;
	b = sqrt ( b);
	super_complex :: zzc->real *= b;

	return *super_complex :: zzc;
}

super_complex cos ( const super_complex &a)			// FRIEND FN
{							// overload cos
	sincos ( a.real, super_complex :: zzc->imag, super_complex:: zzc->real);
	super_complex :: zzc->imag = - super_complex :: zzc->imag;

	super_precision_pool_ptr b_;
	super_precision &b = *b_;
	b = sinh ( a.imag);
	super_complex :: zzc->imag *= b;
	b *= b;
	b++;
	b = sqrt ( b);
	super_complex :: zzc->real *= b;

	return *super_complex :: zzc;
}

super_complex tan ( const super_complex &a)			// FRIEND FN
{	// FRIEND FUNCTION				// overload tan
	super_complex b, c;
	b = sin ( a);
	c = cos ( a);

	return b / c;
}

super_precision norm ( const super_complex &b)			// FRIEND FN
{	// FRIEND FUNCTION				// a = norm (b)
	return ( b.real * b.real + b.imag * b.imag);
}

super_precision argand ( const super_complex &b)		// FRIEND FN
{	// FRIEND FUNCTION				// a = argand (b)
	return atan2 ( b.imag, b.real);
}

super_complex polar ( const super_precision &b,			// FRIEND FN
				const super_precision &c)
{	// FRIEND FUNCTION				// a = polar (b,c)
	sincos ( c, super_complex :: zzc->real, super_complex :: zzc->imag);
	*super_complex :: zzc *= b;

	return *super_complex :: zzc;
}

ostream & operator<<						// OVERLOAD <<
			( ostream &ost, const super_complex &a)
{	// FRIEND FUNCTION					// ost << a

	ost << a.real << "  " << a.imag << " i";

	return ost;
}
