//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------
#include "degui.h"
#include "internal.h"
#include <ctype.h>

//----------------------------------------------------------------------------
// Textbox object
//----------------------------------------------------------------------------
// The list procedure to get the text lines
textbox_object *textbox_object::current_textbox;

char* default_textbox_object_text = "This is the default text for a textbox object.  Cool, eh?";

// The number of elements to expand when expanding allocation for line breaks
// Larger number mean faster setup but more memory wasteage
int textbox_object::allocation_step = 10;

//----------------------------------------------------------------------------
// Constructor
textbox_object::textbox_object (char *str, const bool wrap, const bool wwrap)
   :text(NULL),
   length(0),
   self_allocated(false),
   _lines(0),
   wrapping(wrap),
   word_wrapping(wwrap),
   width(9999),
   last_width(-1),
   last_font(NULL),
   places(NULL),
   allocated(0),
   buffered_place(NULL),
   list_grabber_proc(textbox_object::list_procedure)
   {
   set_text(str);

   // Setup the defaults
   set_x(100);
   set_y(100);
   set_h(100);
   set_w(100);
   }
//----------------------------------------------------------------------------
// Destructor
textbox_object::~textbox_object (void)
   {
   // Unbuffer anything and release the memory
   if(buffered_place!=NULL) buffered_place[0] = buffered_char;
   if(places!=NULL) delete [] places;
   if(self_allocated==true) delete text;
   }
//----------------------------------------------------------------------------
// Helper function to determine line lengths
char* textbox_object::find_line_end (char *string)
   {
   int c = 0;
   char *i = string;

   static char temp[5];
   temp[1] = '\0';

   // Find the line length
   while(1)
      {
      // Check to see if we ran out of string
      if(*i=='\0') return NULL;

      // Check for a DOS newline
      if(*i==13)
         {
         if(i[1]==10) i++;
         break;
         }
      // Check for a UNIX ??? newline
      if(*i==10) break;

      // If the line too long for the wrapping
      if((c>width)&&(wrapping==true))
         {
         char *start = i;

         // Line width goes over the end, go back for the whitepsace
         if(word_wrapping==true)
            {
            while(!isspace(*i))
               {
               i--;

               // Check for long lines
               if(i<=string)
                  {
                  i = start;
                  i--;
                  break;
                  }
               }
            i++;
            }
         i--;
         break;
         }

      // Find the width of the character
      temp[0] = *i;
      c += text_length(font,(unsigned char*)temp);

      // Check out the next character
      i++;
      }
   return i;
   }

//----------------------------------------------------------------------------
// Split the text into seperate text portions
void textbox_object::set_text (char *str)
   {
   // Free memory from the last text parse
   if(places!=NULL)
      {
      // Replace any old bufferd chars
      if(buffered_place!=NULL) *buffered_place = buffered_char;
      buffered_place = NULL;

      // Release the old stuff
      if(places!=NULL) delete [] places;
      places = NULL;
      allocated = 0;
      text = NULL;
      length = 0;
      _lines = 0;
      }
   // Make sure we have a reasonable string in our own space
   if(str==NULL)
      {
      text = new char[strlen(default_textbox_object_text)+1];
      if(text==NULL) degui_no_memory();
      strcpy(text,default_textbox_object_text);
      }
   // Or just setup the new text pointers
   else
      {
      text = str;
      }
   length = strlen(text);

   // Make sure we have a reasonable length to deal with
   if(length<2) return;

   // Allocate some initial space for the line breaks
   places = new char*[allocation_step];
   if(places==NULL) degui_no_memory();
   allocated = allocation_step;
   int x;
   for(x=0;x<allocation_step;x++) places[x] = NULL;

   // Get all the line breaks
   char *i = text;
   char *end = text + length;
   while(true)
      {
      // Find the line break
      i = find_line_end(i);

      // If it's a normal line break, remember it
      if(i!=NULL)
         {
         places[_lines] = i+1;
         _lines++;
         }
      // If it's the last line break remember it
      else
         {
         places[_lines] = end;
         _lines++;
         break;
         }

      // Put the stuff in the data arrays
      if(_lines>=allocated)
         {
         // Make new allocation
         char **newplaces = new char*[allocated+allocation_step];
         if(newplaces==NULL) degui_no_memory();

         // Copy the stuff over
         for(x=0;x<allocated;x++)
            {
            newplaces[x] = places[x];
            }
         // Put invalids in the unused but allocated stuff
         for(x=allocated;x<allocated+allocation_step;x++)
            {
            newplaces[x] = NULL;
            }

         // Release the old stuff
         if(places!=NULL) delete [] places;

         // Update the allocation
         places = newplaces;
         allocated += allocation_step;
         }
      // Find the next end point
      i++;
      }
   redraw();

   // We reset the offset when changing text
   offset = 0;
   }
//----------------------------------------------------------------------------
// Getting lines from out of the text string
char* textbox_object::get_line (const int line)
   {
   // Make sure that we have a reasonable request
   if(line>=_lines) return NULL;

   // Replace any old bufferd chars
   if(buffered_place!=NULL) *buffered_place = buffered_char;

   // Buffer the what is under the end of line marker
   buffered_place = places[line];
   buffered_char = *buffered_place;

   // Place the end of line marker
   *buffered_place = '\0';

   // Give up the first line
   if(line==0) return text;

   // Get the line for the user
   return places[line-1];
   }
//---------------------------------------------------------------------------
// Helper function for textbox to get text into lines for listbox
char* textbox_object::list_procedure (int index, int *list_size)
   {
   // Some alais to help readability
   textbox_object *object = textbox_object::current_textbox;

   // Possible readjust the text if the size or font has changed
   // Check if we must reassign the breaks
   if((object->w()!=object->last_width)||(font!=object->last_font))
      {
      // Possibly adjust the width if there is wrapping
      // Assume that there is a scrollbar
      if(object->wrapping==true) object->width = object->w()-25;

      // Reformat the text for this width
      object->set_text(object->text);

      // Check if we don't need a scrollbar
      if(object->lines() < ((object->h()-3)/text_height(font)))
         {
         // Expand the width for the scrollbar thats not there
         object->width += 10;

         // Reformat the text for this width
         object->set_text(object->text);
         }
      object->last_width = object->w();
      object->last_font = font;
      }

   // Get the text lines or the size
   if(index<0)
      {
      *list_size = object->lines();
      return NULL;
      }
   else
      {
      return object->get_line(index);
      }
   }
//----------------------------------------------------------------------------
// Basic message passing functions

// Tell the object to draw itself
void textbox_object::msg_draw (void)
   {
   // Setup the list getter procedure and pass the message to the list
   current_textbox = this;
   list_grabber_proc(-1, &listsize);

   height = (h()-4) / text_height(font);

   int bar = 0;
   if(listsize>height) bar = scrollable_object::width;
   if(bar==0) offset = 0;

   _draw_list_object(list_grabber_proc,
                     &listsize,
                     offset,
                     x(),y(),w()-bar,h(),
                     -1,
                     disabled(),has_focus(),
                     color ? color->fore() : degui_fore_color,
                     color ? color->select() : degui_select_color,
                     color ? color->deselect() : degui_deselect_color,
                     color ? color->disable() : degui_disable_color,
                     color ? color->light_shad() : degui_light_shad_color,
                     color ? color->dark_shad() : degui_dark_shad_color);

   // Possibly draw the scrollbar
   if(bar) _draw_scrollbar(x2()-scrollable_object::width,y(),scrollable_object::width,h(),
                           listsize,
                           offset,
                           height,
                           has_focus(),
                           color ? color->mid() : degui_mid_color,
                           color ? color->back() : degui_back_color,
                           color ? color->deselect() : degui_deselect_color,
                           color ? color->light_shad() : degui_light_shad_color,
                           color ? color->dark_shad() : degui_dark_shad_color);
   _redraw = false;
   }
//----------------------------------------------------------------------------
// Tell the object to deal with a mouse click default is the same as a keypress
void textbox_object::msg_click (void)
   {
   // Figure out if it's on the text or the scrollbar
   int bar = (_lines > height);

   // Clicked on the text area
   if((!bar)||(mouse_x<x2()-scrollable_object::width))
      {
      return;
      }
	// Clicked on the scroll area
   else
      {
      scrollable_object::handle_click();
      }
   }
//----------------------------------------------------------------------------
// Tell the object that a key was pressed while it had the focus
// This should return true if the key was used
bool textbox_object::msg_char (const int c)
   {
   int used_key = true;

   // This code taken from Allegro's GUI.C with some major mods
   int start = offset;
	if(_lines>0)
      {
      int top, bottom,l;

	   if(offset>0) top = offset+1;
	   else top = 0;

      l = (h()-3)/text_height(font);

	   bottom = offset + l - 1;
	   if(bottom >= _lines-1) bottom = _lines-1;
	   else bottom--;

      if((c>>8) == KEY_UP) offset--;
      else if((c>>8) == KEY_DOWN) offset++;
	   else if((c>>8) == KEY_HOME) offset = 0;
	   else if((c>>8) == KEY_END) offset = _lines-l;
	   else if((c>>8) == KEY_PGUP) offset = offset-(bottom-top);
      else if((c>>8) == KEY_PGDN) offset = offset+(bottom-top);
      else used_key = false;

      // Make sure that the list stays in bounds
      if(offset>_lines-l) offset = _lines-l;
      if(offset<0) offset = 0;
	   }
   else return false;

   /* if we changed something, better redraw... */
   if(offset!=start)
      {
      show_mouse(NULL);
      msg_draw();
      show_mouse(screen);
      }
   return used_key;
   }

//----------------------------------------------------------------------------
// Ask the object if they want the focus
bool textbox_object::msg_wantfocus (void)
   {
   // If we don't have a scrollbar we can't do anything with the focus
   if(_lines > height) demand_focus = true;
   else demand_focus = false;

   return demand_focus;
   }
//----------------------------------------------------------------------------

