/****************************************************************************/
/* MSGBOX                                                                   */
/*--------------------------------------------------------------------------*/
/* Opens a message box and waits for the user to press a button             */
/*--------------------------------------------------------------------------*/
/* Author      : DELPRAT Jean-Pierre                                        */
/* Created on  : 20-Apr-95                                                  */
/****************************************************************************/

#ifdef __JPTUI__

  #include "Const.h"
  #include "JPDebug.h"

  #include "Vocab.h"

  #include "JPAppli.h"

  #include "Mouse.h"
  #include "Screen.h"
  #include "Sound.h"
  #include "SpChars.h"
  #include "Strings.h"

  #include "TLabel.h"
  #include "TPushBut.h"
  #include "TWindow.h"

#else

  #include "JPTui.h"
  #include "JPVocab.h"

  #define JPDEBUG_STOP(a)  DEBUG_STOP()

#endif


#include "MsgBox.h"


/*ͻ*/
/*                                CONSTANTS                               */
/*ͼ*/

// Margin between the elements of the box

#define MARGIN_WIDTH      3
#define MARGIN_HEIGHT     2

// Part of the screen that can be used to display the message box

#define MAX_WINDOW_WIDTH  (GetScreenWidth()-6)
#define MAX_WINDOW_HEIGHT (GetScreenHeight()-4)

// Maximum number of buttons in a message box

#define NB_MAX_BUTTONS    3

/*ͻ*/
/*                              STATIC VARIABLES                          */
/*ͼ*/

static int s_button_type;
static int s_button_pressed;

/*ͻ*/
/*                                 FONCTIONS                              */
/*ͼ*/

/****************************************************************************/
/* MsgBoxIconText                                                           */
/*--------------------------------------------------------------------------*/
/* Returns the text of an icon 						    */
/****************************************************************************/

char *MsgBoxIconText(int icon_type)
{
  switch(icon_type)
    {
      case MB_NOICON          : return("");
      case MB_ICONSTOP        : return(SPECIAL_STRING(SST_ICON_STOP));
      case MB_ICONQUESTION    : return(SPECIAL_STRING(SST_ICON_QUESTION));
      case MB_ICONEXCLAMATION : return(SPECIAL_STRING(SST_ICON_EXCLAMATION));
      case MB_ICONINFORMATION : return(SPECIAL_STRING(SST_ICON_INFORMATION));
    }

  JPDEBUG_STOP(DEBUG_ERROR_13);
  return(NULL);
}

/****************************************************************************/
/* MsgBoxButtonText                                                         */
/*--------------------------------------------------------------------------*/
/* Returns the text of a button of the message box 			    */
/****************************************************************************/

char *MsgBoxButtonText(int button_number)
{
  int temp=0;

  switch (button_number)
    {
      case 0 : switch (s_button_type)
		 {
		   case MB_OK               :
		   case MB_OKCANCEL         :
		   case MB_OKQUIT           : temp=VOC_OK_CAPTION;break;
		   case MB_YESNO	    :
		   case MB_YESNOCANCEL      : temp=VOC_YES_CAPTION;break;
		   case MB_ABORTRETRYIGNORE : temp=VOC_ABORT_CAPTION;break;
		   case MB_RETRYCANCEL      : temp=VOC_RETRY_CAPTION;break;
		   default                  : JPDEBUG_STOP(DEBUG_ERROR_19);
		 }
	       break;
      case 1 : switch (s_button_type)
		 {
		   case MB_OKCANCEL         : temp=VOC_CANCEL_CAPTION;break;
		   case MB_OKQUIT           : temp=VOC_QUIT_CAPTION;break;
		   case MB_RETRYCANCEL      : temp=VOC_CANCEL_CAPTION;break;
		   case MB_YESNO	    :
		   case MB_YESNOCANCEL      : temp=VOC_NO_CAPTION;break;
		   case MB_ABORTRETRYIGNORE : temp=VOC_RETRY_CAPTION;break;
		   default                  : JPDEBUG_STOP(DEBUG_ERROR_19);
		 }
	       break;

      case 2 : switch (s_button_type)
		 {
		   case MB_ABORTRETRYIGNORE : temp=VOC_IGNORE_CAPTION;break;
		   case MB_YESNOCANCEL      : temp=VOC_CANCEL_CAPTION;break;
		   default                  : JPDEBUG_STOP(DEBUG_ERROR_19);
		 }
	       break;

      default:JPDEBUG_STOP(DEBUG_ERROR_20);
    }

  return(GetString(temp));
}

/****************************************************************************/
/* MsgBoxNbButtons                                                          */
/*--------------------------------------------------------------------------*/
/* Returns the number of buttons of the message box			    */
/****************************************************************************/

int MsgBoxNbButtons()
{
  switch (s_button_type)
    {
      case MB_OK               : return(1);
      case MB_OKCANCEL         :
      case MB_OKQUIT           :
      case MB_YESNO	       : return(2);
      case MB_ABORTRETRYIGNORE :
      case MB_YESNOCANCEL      :
      case MB_RETRYCANCEL      : return(3);
    }
  JPDEBUG_STOP(DEBUG_ERROR_19);
  return(0);
}

/****************************************************************************/
/* MsgBoxButtonWidth                                                        */
/*--------------------------------------------------------------------------*/
/* Returns the width of the message box buttons 			    */
/****************************************************************************/

int MsgBoxButtonWidth()
{
  register int width=DisplayLength(MsgBoxButtonText(0));
  register int i,w;

  for (i=1;i<MsgBoxNbButtons();i++)
    {
      w=DisplayLength(MsgBoxButtonText(i));
      if (w>width)
	width=w;
    }

  return(width+4);
}

/****************************************************************************/
/* IsMsgBoxCancelButton                                                     */
/*--------------------------------------------------------------------------*/
/* Indicates whether a button can be activated by the <Escape> key          */
/****************************************************************************/

boolean IsMsgBoxCancelButton(int button_number)
{
  switch (button_number)
    {
      case 0 : return(FALSE);

      case 1 : switch (s_button_type)
		 {
		   case MB_OKCANCEL         :
                   case MB_OKQUIT           :
		   case MB_RETRYCANCEL      : return(TRUE);
		 }
	       break;

      case 2 : switch (s_button_type)
		 {
		   case MB_YESNOCANCEL      : return(TRUE);
		 }
	       break;
    }

  return(FALSE);
}

/****************************************************************************/
/* MsgBoxButtonPressedCall                                                  */
/*--------------------------------------------------------------------------*/
/* Function called when the users presses one of the buttons of the box     */
/****************************************************************************/

void MsgBoxButtonPressedCall(PObject /*sender*/,char *button_number)
{
  switch (button_number[0])
    {
      case 0 : switch (s_button_type)
		 {
		   case MB_OK               :
		   case MB_OKCANCEL         :
		   case MB_OKQUIT           : s_button_pressed=OK_BUTTON;
					      break;
		   case MB_ABORTRETRYIGNORE : s_button_pressed=ABORT_BUTTON;
					      break;
		   case MB_YESNOCANCEL      :
		   case MB_YESNO            : s_button_pressed=YES_BUTTON;
					      break;
		   case MB_RETRYCANCEL      : s_button_pressed=RETRY_BUTTON;
					      break;
		   default : JPDEBUG_STOP(DEBUG_ERROR_19);
		 }
	       break;

      case 1 : switch (s_button_type)
		 {
		   case MB_OKCANCEL         : s_button_pressed=CANCEL_BUTTON;
					      break;
		   case MB_OKQUIT           : s_button_pressed=QUIT_BUTTON;
					      break;
		   case MB_ABORTRETRYIGNORE : s_button_pressed=RETRY_BUTTON;
					      break;
		   case MB_YESNOCANCEL      : s_button_pressed=NO_BUTTON;
					      break;
		   case MB_YESNO            : s_button_pressed=NO_BUTTON;
					      break;
		   case MB_RETRYCANCEL      : s_button_pressed=CANCEL_BUTTON;
					      break;
		   default : JPDEBUG_STOP(DEBUG_ERROR_19);
		 }
	       break;

      case 2 : switch (s_button_type)
		 {
		   case MB_ABORTRETRYIGNORE : s_button_pressed=IGNORE_BUTTON;
					      break;
		   case MB_YESNOCANCEL      : s_button_pressed=CANCEL_BUTTON;
					      break;
		   default : JPDEBUG_STOP(DEBUG_ERROR_19);
		 }
	       break;

      default: JPDEBUG_STOP(DEBUG_ERROR_20);
    }


  JPStop();
}

/****************************************************************************/
/* MessageBox                                                               */
/*--------------------------------------------------------------------------*/
/* Opens a message box and waits for the user to press a button             */
/* Returns the button pressed by the user				    */
/****************************************************************************/

int MessageBox(char *title,
	       char *message,
	       int window_style,
	       int type,
	       TSound sound_type)
{
  TMousePointer pointer;

  register
  int      i;

  int      icon_type,
           default_button;

  int      message_height,
           message_width;

  char     *icon_text;
  int      icon_width,
           icon_height;

  int      window_width,
           window_height,
           title_height,
           upper_part_height,
           title_width;

  int      button_width,
           nb_buttons;

  int      margin_width;

  int      temp;

  char     *arg="0";


  PWindow  message_box;
  PLabel   icon_label,
	   message_label;

  PPushButton
           button[NB_MAX_BUTTONS];

  int      max_window_width=MAX_WINDOW_WIDTH;
  int      max_window_height=MAX_WINDOW_HEIGHT;

  // Determine the size of the window and the position of its elements

  s_button_type=type &0xF;
  icon_type=type & 0xF0;
  default_button=(type>>8) & 0xF;

  message_height=GetTextHeight(message);
  message_width=GetTextWidth(message);

  icon_text=MsgBoxIconText(icon_type);
  icon_width=GetTextWidth(icon_text),
  icon_height=GetTextHeight(icon_text);

  window_width=(icon_type==MB_NOICON)?(2*MARGIN_WIDTH+message_width):
					  (3*MARGIN_WIDTH+icon_width+message_width);

  title_width=DisplayLength(title)+2;
  if (title_width>window_width)
    window_width=MIN(max_window_width,title_width);


  title_height=((title[0]==0)?0:1);

  upper_part_height=MAX(icon_height,message_height);

  window_height=3*MARGIN_HEIGHT+upper_part_height+1+title_height;

  button_width=MsgBoxButtonWidth();
  nb_buttons=MsgBoxNbButtons();

  temp=nb_buttons*button_width+(nb_buttons+1)*MARGIN_WIDTH;
  window_width=MAX(temp,window_width);

  margin_width=((icon_type==MB_NOICON)?((window_width-message_width)/2)
				      :((window_width-message_width-icon_width)/3));

  if (window_width>max_window_width)
    {
      message_width-=(window_width-max_window_width);
      window_width=max_window_width;
    }

  if (window_height>max_window_height)
    {
      message_height-=(window_height-max_window_height);
      window_height=max_window_height;
      upper_part_height=MAX(icon_height,message_height);
    }
  // Allocates the elements of the dialog box

  message_box=new TWindow(window_style,
			  ((GetScreenWidth()-window_width)>>1)+1,
			  ((GetScreenHeight()-window_height)>>1)+1,
			  window_width,window_height,title,NO_INFO_BAR,MODAL,MOVABLE);



  icon_label=NULL;
  if (icon_type!=MB_NOICON)
    icon_label=new TLabel(message_box,
			  margin_width,
			  (title_height+MARGIN_HEIGHT+((upper_part_height-icon_height)>>1)),
			  icon_width,icon_height,
			  icon_text);

  message_label=new TLabel(message_box,
			   ((icon_type==MB_NOICON)?(margin_width)
						  :(2*margin_width+icon_width)),
			   (title_height+MARGIN_HEIGHT+((upper_part_height-message_height)>>1)),
			   message_width,message_height,
			   message);

  margin_width=(window_width-(nb_buttons*button_width)) / (nb_buttons+1);


  for (i=0;i<nb_buttons; i++)
    {
      button[i]=new TPushButton(message_box,
			      ((i+1)*margin_width)+(i*button_width),
			      window_height-MARGIN_HEIGHT-1,
			      button_width,
			      MsgBoxButtonText(i),

                               ((default_button==i)?(PB_DEFAULT):(PB_NORMAL))
			      |((IsMsgBoxCancelButton(i))?(PB_CANCEL):(PB_NORMAL)));
      arg[0]=i;

      // Define the function called when the user presses this button

      button[i]->m_set_pressed_callback(MsgBoxButtonPressedCall,arg);
    }

  // Defines the mouse pointer (in case it would have been changed)

  pointer=GetMousePointer();
  SetMousePointer(MP_ARROW);

  // Opens and runs the dialog box

  message_box->m_open();
  PlaySound(sound_type);
  JPRunDialog();

  // Closes the dialog box

  message_box->m_close();
  JPRefresh();

  // Restores the mouse pointer

  SetMousePointer(pointer);

  // Destructs dynamically-allocated objects

  for (i=nb_buttons-1;i>=0;i--)
    delete button[i];

  delete message_label;
  if (icon_label!=NULL)
    delete icon_label;

  delete message_box;

  // Returns the button pressed by the user

  return(s_button_pressed);
}
