// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#ifndef ViewportClass
#define ViewportClass

#include <Lib3d/Vector.H>
#include <Lib3d/Device.H>		// for setDirty inline
#include <Lib3d/internals/ZBuffer.H>	// for setDirty inline

class PipelineData;
class SmoothPipelineData;
class TexturePipelineData;
class ColourRamp;
class Texture;

class Viewport : public Debuggable
{
public:
    Viewport( Exemplar, uint );
    virtual ~Viewport();
    static Viewport *create( Device * );

public:
    virtual void flatPolygonZb(uint nrVertices, 
			       PipelineData * const vertices[], 
			       Colour colour) = 0;

    virtual void flatTriangleZb(PipelineData * const vertices[], 
				Colour colour);

    virtual void smoothPolygonZb(uint nrVertices, 
				 SmoothPipelineData * const vertices[], 
				 const ColourRamp &colour);

    virtual void smoothTriangleZb(SmoothPipelineData * const vertices[], 
				 const ColourRamp &colour);
 
    virtual void wireTriangle(PipelineData * const vertices[],
			      Colour colour);

    virtual void wirePolygon(uint nrVertices, 
			     PipelineData * const vertices[], 
			     Colour colour);

    virtual void texturePolygonZb(uint nrVertices, 
				  TexturePipelineData * const vertices[], 
				  const Texture &texture);

    virtual void textureTriangleZb(TexturePipelineData * const vertices[], 
				   const Texture &texture);

    virtual void lineZb(const DeviceVector &, const DeviceVector &, Colour);


    virtual Colour getColour( uint r, uint g, uint b ) = 0;
    virtual void drawColourSpace() = 0;

    virtual void setDirty( int xmin, int ymin, int xmax, int ymax );
    const Matrix4 &getDeviceTransform() const { return deviceTransform; }

    void swapBuffers();		// Superceded - use the Scene class.

    void notifyResize();
    void clearZBuffer();

    void clipLine(const DeviceVector &, const DeviceVector &, DeviceColour);
			
    uint getWidth() const { return zBuffer->getWidth(); }
    uint getHeight() const { return zBuffer->getHeight(); }
    uint getCoordinateWidth() const { return coordinateWidth; }
    uint getCoordinateHeight() const { return coordinateHeight; }

protected:
    Viewport( Device *, uint, uint );
    virtual Viewport *clone( Device * ) = 0;

    ostream &print( ostream &out ) const;

private:
    static Viewport *children;	// List of exemplar viewports.
    uint quality;		// Ordering criteria for equivalent viewports.
    Viewport *next;		// Next exemplar object.

    uint xscale;		// fixedpoint scale for x and y.
    uint yscale;
    uint coordinateWidth;	// Including scale for fixedpoint math.
    uint coordinateHeight;	// 
    
    void *extensions;		// For future expansion.

protected:  
    Device *device;
    ZBuffer *zBuffer;		// Only ZBuffer rendering is supported, for now.
    Matrix4 deviceTransform;	// cvv to device.    
};


class PipelineData
{
public:
    PipelineData() {}
    DeviceVector device;
};

class SmoothPipelineData : public PipelineData
{
public:
    SmoothPipelineData() : PipelineData() {}
    int intensity;
};

class TexturePipelineData : public SmoothPipelineData
{
public:
    TexturePipelineData() : SmoothPipelineData() {}

    float uw, vw, w;		
};


#endif





 


