// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#include <Lib3d/Model.H>
#include <Lib3d/Pipeline.H>
#include <Lib3d/FlatPipeline.H>
#include <Lib3d/internals/BoundingBox.H>

static Pipeline *defaultPipeline = 0;

Model::Model(uint nrPolygons, 
	     Polygon *polygons,
	     uint nrVertices, 
	     Vertex *vertices,
	     uint nrVertexNormals, 
	     Normal *vertexNormals,
	     uint nrPolygonNormals, 
	     Normal *polygonNormals,
	     uint nrMaterials,
	     Material *materials,
	     Texture *texture,
	     BoundingObject *box)
    : nrPolygons(nrPolygons), 
      nrVertices(nrVertices), 
      nrVertexNormals(nrVertexNormals), 
      nrPolygonNormals(nrPolygonNormals), 
      nrMaterials(nrMaterials), 
      polygons(polygons),
      vertices(vertices),
      vertexNormals(vertexNormals),
      polygonNormals(polygonNormals),
      materials(materials),
      texture(texture),
      box( box ),
      pipeline(0)
{
}

Model::~Model()
{
    delete polygons;
    delete vertices;
    delete vertexNormals;
    delete polygonNormals;
    delete box;
}

void 
Model::render( Viewport &viewport, const Light *lights, uint nrLights, uint flags )
{
    uint clipPlanes;

    if (pipeline == 0) {
	if (defaultPipeline == 0) {
	    defaultPipeline = new FlatPipeline;
	}
	this->usePipeline( *defaultPipeline );
    }

#if 1
    if ( box->testBounds( objectToCvv, D, clipPlanes ) )  {
	pipeline->render(*this, viewport, lights, nrLights, clipPlanes, flags);
    }
#else
    if ( box->testAndDrawBounds(objectToCvv, 
			       D, 
			       clipPlanes, 
			       viewport, 
			       cvvToDevice ) ) {
	pipeline->render(*this, viewport, lights, nrLights, clipPlanes, flags);
    }
#endif
}

ostream &
Model::print( ostream &out ) const
{
    Node::print( out ) << endl;
    out << "\tPolygons: " << nrPolygons << endl;
    out << "\tVertices: " << nrVertices << endl;
    out << "\tPolygon Normals: " << nrPolygonNormals << endl;
    out << "\tVertex Normals: " << nrVertexNormals << endl;
    out << "\tMaterials: " << nrMaterials << endl;
    if (pipeline) {
	out << "\tPipeline: " << *pipeline << endl;
    } else {
	out << "\tNo pipeline." << endl;
    }
    return out;
}

void
Model::recalculateTransforms( const Matrix34 &parentToCvv )
{
    Node::recalculateTransforms( parentToCvv );

    // Much of what occurs below is speculative and may never be used.
    // However, it is executed relatively infrequently.

    objectToDevice.mul(cvvToDevice, objectToCvv);

    Matrix34 cvvToObject;
    cvvToObject.invert(objectToCvv);
    objectViewPos.assign( 0, 0, 0 );
    objectViewPos.mul( cvvToObject );

    objectToCvv_T.transpose(objectToCvv);

    if_debug {
	debug() << "objectToDevice: " << objectToDevice << endlog;
    }
}


void 
Model::usePipeline( Pipeline &p ) {
    pipeline = &p;
    p.registerModel( *this );
}











