/*
 * Copyright (C) 1996, jack
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* $Id: dowprnt.c,v 1.3 1996/08/01 08:22:51 jack Exp $ */

#include <limits.h>
#include <string.h>
#include <stdarg.h>
#include <wchar.h>

#define WC(c) ((wchar_t) (c))
#define INTBUFSIZE (sizeof (unsigned long long) * 8)

/* Integer to Wide Character String */
static wchar_t *
ulltowcs (unsigned long long ullvalue, wchar_t *wbuffer, int base,
	  const wchar_t *wdigits)
{
  wchar_t wtmp[INTBUFSIZE + 1];
  wchar_t *wp, *wq;
  unsigned long ulvalue;
  int mask, shift;

  if (base == 0)
    base = 10;
  if (base < 2 || base > 36)
    {
      *wbuffer = WC ('\0');
      return wbuffer;
    }

  switch (base)
    {
    case 2:
      shift = 1;
      mask = (1 << shift) - 1;
      break;
    case 4:
      shift = 2;
      mask = (1 << shift) - 1;
      break;
    case 8:
      shift = 3;
      mask = (1 << shift) - 1;
      break;
    case 16:
      shift = 4;
      mask = (1 << shift) - 1;
      break;
    case 32:
      shift = 5;
      mask = (1 << shift) - 1;
      break;
    default:
      shift = 0;
      mask = 0;
      break;
    }

  wp = wtmp + sizeof (wtmp);
  wq = wbuffer;
  *--wp = '\0';

  if (ullvalue <= ULONG_MAX)
    {
      ulvalue = (unsigned long) ullvalue;
      if (shift)
	{
	  do
	    {
	      *--wp = wdigits[ulvalue & mask];
	      ulvalue >>= shift;
	    }
	  while (ulvalue > 0);
	}
      else
	{
	  do
	    {
	      *--wp = wdigits[ulvalue % base];
	      ulvalue /= base;
	    }
	  while (ulvalue > 0);
	}
    }
  else
    {
      if (shift)
	{
	  do
	    {
	      *--wp = wdigits[ullvalue & mask];
	      ullvalue >>= shift;
	    }
	  while (ullvalue > 0);
	}
      else
	{
	  do
	    {
	      *--wp = wdigits[ullvalue % base];
	      ullvalue /= base;
	    }
	  while (ullvalue > 0);
	}
    }

  while ((*wq++ = *wp++))
    ;

  return wbuffer;
}

#define FLAG_LEFT	001
#define FLAG_ZERO	002
#define FLAG_ALT	004
#define FLAG_SHORT	010
#define FLAG_LONG	020
#define FLAG_LONGLONG	040
#define ARG(flags,ap) (long long) \
  ((flags) & FLAG_LONGLONG ? va_arg (ap, long long) \
   : (flags) & FLAG_LONG ? va_arg (ap, long) \
   : (flags) & FLAG_SHORT ? (short) va_arg (ap, int) \
   : va_arg (ap, int))
#define UARG(flags,ap) (unsigned long long) \
  ((flags) & FLAG_LONGLONG ? va_arg (ap, unsigned long long) \
   : (flags) & FLAG_LONG ? va_arg (ap, unsigned long) \
   : (flags) & FLAG_SHORT ? (unsigned short) va_arg (ap, unsigned int) \
   : va_arg (ap, unsigned int))

int
_dowprnt (const wchar_t *format, va_list ap, FILE *fp,
	  wint_t (*outfunc) (wint_t, FILE *))
{
  static const wchar_t wUdigits[] = L"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  static const wchar_t wLdigits[] = L"0123456789abcdefghijklmnopqrstuvwxyz";
  wchar_t wbuf[INTBUFSIZE + 1];
  const wchar_t *wdigits;
  const wchar_t *wp = format;
  wint_t wc;
  int rc = 0;

  wdigits = wLdigits;
  while ((wc = *wp++))
    {
      if (wc != WC ('%'))
	{
	  outfunc (wc, fp);
	  rc++;
	}
      else
	{
	  /* control characters... */
	  wint_t wsignc = 0;
	  int flags = 0, length = 0, digit = -1;
	  int base;
	  unsigned long long ullvalue;

	again:
	  switch ((wc = *wp++))
	    {
	    case WC ('\0'):
	      return rc;

	    case WC ('-'):
	      flags |= FLAG_LEFT;
	      flags &= ~FLAG_ZERO;
	      goto again;
	    case WC ('+'):
	      wsignc = WC ('+');
	      goto again;
	    case WC (' '):
	      if (!wsignc)
		wsignc = WC (' ');
	      goto again;
	    case WC ('#'):
	      flags |= FLAG_ALT;
	      goto again;
	    case WC ('0'):
	      if (! (flags & FLAG_LEFT))
		flags |= FLAG_ZERO;
	      goto again;

	    case WC ('.'):
	      if ((wc = *wp++) == WC ('*'))
		digit = va_arg (ap, int);
	      else
		{
		  digit = 0;
		  while (wc >= WC ('0') && wc <= WC ('9'))
		    {
		      digit *= 10;
		      digit += wc - WC ('0');
		      wc = *wp++;
		    }
		  wp--;
		}
	      goto again;

	    case WC ('*'):
	      length = va_arg (ap, int);
	      if (length < 0)
		{
		  length = -length;
		  flags |= FLAG_LEFT;
		}
	      goto again;

	    case WC ('1'):
	    case WC ('2'):
	    case WC ('3'):
	    case WC ('4'):
	    case WC ('5'):
	    case WC ('6'):
	    case WC ('7'):
	    case WC ('8'):
	    case WC ('9'):
	      length = 0;
	      do
		{
		  length *= 10;
		  length += wc - WC ('0');
		  wc = *wp++;
		}
	      while (wc >= WC ('0') && wc <= WC ('9'));
	      wp--;
	      goto again;

	    case WC ('h'):
	      flags |= FLAG_SHORT;
	      goto again;
	    case WC ('l'):
	      if (flags & FLAG_LONG)
		flags |= FLAG_LONGLONG;
	      else
		flags |= FLAG_LONG;
	      goto again;
	    case WC ('L'):
	      flags |= FLAG_LONGLONG;
	      goto again;

	    case WC ('c'):
	      {
		wint_t wivalue;

		wivalue = (wint_t) va_arg (ap, wint_t);
		rc++;
		outfunc (wivalue, fp);
		break;
	      }

	    case WC ('n'):
	      if (flags & FLAG_LONGLONG)
		*va_arg (ap, long long *) = (long long) rc;
	      else if (flags & FLAG_LONG)
		*va_arg (ap, long *) = (long) rc;
	      else if (flags & FLAG_SHORT)
		*va_arg (ap, short *) = (short) rc;
	      else
		*va_arg (ap, int *) = rc;
	      break;

	    case WC ('p'):
	      base = 16;
	      wdigits = wLdigits;
	      flags |= FLAG_ALT;
	      if (digit < 0)
		digit = (sizeof (void *) * 2) + (flags & FLAG_ALT) ? 2 : 0;
	      ullvalue = (unsigned long long) va_arg (ap, void *);
	      ullvalue &= ~(~0LL << (sizeof (void *) * 8));
	      wsignc = 0;
	      goto integer;

	    case WC ('o'):
	      base = 8;
	      flags &= ~FLAG_ALT;
	      goto uinteger;

	    case WC ('x'):
	      base = 16;
	      wdigits = wLdigits;
	      goto uinteger;

	    case WC ('X'):
	      base = 16;
	      wdigits = wUdigits;
	      goto uinteger;

	    case WC ('u'):
	      base = 10;
	      flags &= ~FLAG_ALT;
uinteger:
	      ullvalue = UARG (flags, ap);
	      wsignc = 0;
	      goto integer;

	    case WC ('i'):
	    case WC ('d'):
	      {
		long long llvalue;

		llvalue = ARG (flags, ap);
		if (llvalue < 0)
		  {
		    ullvalue = (unsigned long long) -llvalue;
		    wsignc = WC ('-');
		  }
		else
		  {
		    ullvalue = (unsigned long long) llvalue;
		  }
	      }
	      base = 10;
	      flags &= ~FLAG_ALT;

integer:
	      /* special case */
	      if (ullvalue == 0 && digit == 0)
		break;

	      {
		int len;

		len = (int) wcslen (ulltowcs (ullvalue, wbuf, base, wdigits));
		len += wsignc ? 1 : 0;
		len += (flags & FLAG_ALT) ? 2 : 0;

		if (digit < 0 || (digit -= len) <= 0)
		  digit = 0;
		len += digit;

		if ((length -= len) <= 0)
		  length = 0;
		len += length;
		rc += len;
	      }

	      if (flags & FLAG_ZERO)
		{
		  if (wsignc)
		    outfunc (wsignc, fp);
		  if ((flags & FLAG_ALT) && base == 16)
		    {
		      outfunc (WC ('0'), fp);
		      outfunc (wdigits['x' - 'a' + 10], fp);
		    }
		}

	      if (! (flags & FLAG_LEFT))
		{
		  wchar_t wpadc = (flags & FLAG_ZERO) ? WC ('0') : WC (' ');
		  while (--length >= 0)
		    outfunc (wpadc, fp);
		}

	      if (! (flags & FLAG_ZERO))
		{
		  if (wsignc)
		    outfunc (wsignc, fp);
		  if ((flags & FLAG_ALT) && base == 16)
		    {
		      outfunc (WC ('0'), fp);
		      outfunc (wdigits['x' - 'a' + 10], fp);
		    }
		}

	      {
		while (--digit >= 0)
		  outfunc (WC ('0'), fp);
	      }

	      {
		wchar_t *wbp = wbuf;
		while (*wbp != 0)
		  outfunc (*wbp++, fp);
	      }

	      if (flags & FLAG_LEFT)
		{
		  wchar_t wpadc = (flags & FLAG_ZERO) ? WC ('0') : WC (' ');
		  while (--length >= 0)
		    outfunc (wpadc, fp);
		}
	      break;

	    case WC ('s'):
	      {
		int len;
		wchar_t *wstring;

		wstring = va_arg (ap, wchar_t *);
		if (wstring == NULL)
		  wstring = L"(null)";
		len = (int) wcslen (wstring);

		if (digit < 0 || digit >= len)
		  digit = -1;
		else
		  len = digit;

		if ((length -= len) <= 0)
		  length = 0;
		len += length;
		rc += len;

		if (! (flags & FLAG_LEFT))
		  {
		    while (--length >= 0)
		      outfunc (WC (' '), fp);
		  }

		if (digit < 0)
		  {
		    while (*wstring != 0)
		      outfunc (*wstring++, fp);
		  }
		else
		  {
		    while (--digit >= 0)
		      outfunc (*wstring++, fp);
		  }

		if (flags & FLAG_LEFT)
		  {
		    while (--length >= 0)
		      outfunc (WC (' '), fp);
		  }
		break;
	      }

	    default:
	      rc++;
	      outfunc (wc, fp);
	      break;
	    }
	}
    }

  return rc;
}
