// -----------------------------------------------------------------------
// File lwp.c - pre-emptive LWP C source code.
// Copyright (C) 1997 Paolo De Marino
//
// Original Source Code by Sengan Short (sengan.short@durham.ac.uk)
// and Josh Turpen (snarfy@goodnet.com).
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version,
//  with the only exception that all the people in the THANKS file
//  must receive credit.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public
//  License along with this library; see the file COPYING.LIB.
//  If not, write to the Free Software Foundation, Inc., 675 Mass Ave,
//  Cambridge, MA 02139, USA.
//
//  For contacting the author send electronic mail to
//     paolodemarino@usa.net
//
//  Or paper mail to
//
//     Paolo De Marino
//     Via Donizetti 1/E
//     80127 Naples
//     Italy
//
// History: See history.txt.
// -----------------------------------------------------------------------

#ifdef DEBUG
#define FORTIFY
#include "../../fortify/fortify.h"
#include <stdio.h>              /* Needed for fprintf(stderr,...) */

//#define PARANOID                /* Includes really paranoid tests of coherence */
/* Will dramatically decrease performances */
#endif

#include "lwp.h"
#include <sys/timeb.h>
#include <sys/segments.h>
#include <bios.h>
#include <dos.h>
#include <conio.h>

#define LWP_EXCPTN 0x99         /* arbitrary exception number */

#define IRQ8 0x70               /* used to start the 1khz timer */
#define IRQ0 0x8                /* used to start the variable timer */
#define PIT0 0x40
#define PIT1 0x41
#define PIT2 0x42
#define PITMODE 0x43
#define PITCONST 1193180L
#define PIT0DEF 18.2067597

#ifdef __cplusplus
extern "C"
{
#endif
  void _lwp_top_of_vars ()
  {
  };
#ifdef __cplusplus
}
#endif

/*  Global Variables */
static volatile float tick_per_ms = 0.0182068;
static volatile float ms_per_tick = 54.9246551;
static volatile float freq8h = 18.2067597;
static volatile int counter_8h;
static volatile int counter_reset;
static int _lwp_irq_used = 0;
static int _lwp_speed_used = 0;
static int _lwp_on = 0;

volatile unsigned _lwp_pcount = 1;
/* Priority counter. Used in conjunction with lwp.priority. */
volatile unsigned lwp_dead_threads = 0;
/* Set to 1 when a thread is lwp_kill'ed */

char _fpu_init_state[108];
__dpmi_paddr _lwp_pm_old_handler, _lwp_pm_new_handler;
__dpmi_regs _lwp_regs;
void (*_lwp_old_handler) (int);
void (*_lwp_old_fpu_handler) (int);

lwp *_lwp_cur;
static volatile int _lwp_count;
volatile int lwp_active_threads;
volatile int _lwp_enable;
volatile int _lwp_interrupt_pending;
#ifdef __cplusplus
extern "C"
{
#endif
  void _lwp_bottom_of_vars ()
  {
  };
#ifdef __cplusplus
}
#endif

/* Some function prototypes */
#ifdef __cplusplus
extern "C"
{
#endif
  extern void _lwp_scheduler (int signum);
  extern void _init_fpu (void);
  void _lwp_findnext (void);    // Called by the scheduler, does the real job.

  void _lwp_fpu_handler (int signum);
  void lwp_deinit (void);
  static void lwp_tcp (void);
  static int lwp_install (int irq, int speed);
  static int _lwp_lock_memory (void);
  static int windoze ();
  static void setRTC (int value);
  static void startIRQ8 (void);
  static void stopIRQ8 (void);
  char *get_cmostime (void);
#ifdef __cplusplus
}
#endif


static void
_lwp_top_of_functions ()
{
};
volatile void
_lwp_dead_thread (void)
{
  lwp_kill (lwp_getpid ());
/*
 * Should never come here: lwp_kill(lwp_getpid()) never returns!
 */
  cprintf ("PANIC!\n\r");
}
int
lwp_getactive (void)
{
  return lwp_active_threads;
}
int static
lwp_install (int irq, int speed)
{
  volatile unsigned int pit0_set, pit0_value;

  _lwp_old_handler = signal (SIGILL, _lwp_scheduler);
  if (irq == 8)
    {
      _lwp_pm_new_handler.offset32 = (int) _lwp_pm_irq8_timer_hook;
      _lwp_pm_new_handler.selector = _my_cs ();
      __dpmi_get_protected_mode_interrupt_vector (IRQ8, &_lwp_pm_old_handler);
      __dpmi_set_protected_mode_interrupt_vector (IRQ8, &_lwp_pm_new_handler);
      _lwp_on = 1;
      _lwp_enable = 0;
      setRTC (speed);
      startIRQ8 ();
      return (1);
    }
  else if (irq == 0)
    {
      if (speed < 1)
        return (0);

      _lwp_pm_new_handler.offset32 = (int) _lwp_pm_irq0_timer_hook;
      _lwp_pm_new_handler.selector = _my_cs ();

      __dpmi_get_protected_mode_interrupt_vector (IRQ0, &_lwp_pm_old_handler);
      __dpmi_set_protected_mode_interrupt_vector (IRQ0, &_lwp_pm_new_handler);

      outportb (PITMODE, 0x36);
      pit0_value = PITCONST / speed;
      pit0_set = (pit0_value & 0x00ff);
      outportb (PIT0, pit0_set);
      pit0_set = (pit0_value >> 8);
      outportb (PIT0, pit0_set);
      freq8h = speed;
      counter_8h = 0;
      counter_reset = freq8h / PIT0DEF;
      tick_per_ms = freq8h / 1000;
      ms_per_tick = 1000 / freq8h;
      _lwp_on = 1;
      _lwp_enable = 0;
      return (1);
    }
  else
    return (0);
}

int
lwp_init (int irq, int speed)
{
  _lwp_irq_used = irq;
  _lwp_speed_used = speed;

//  if (atexit (lwp_deinit))
  //    return (0);
  if (!_lwp_lock_memory ())
    return (0);

  if ((_lwp_cur = (lwp *) malloc (sizeof (lwp))) == NULL)
    {
#ifdef DEBUG
      cprintf ("ERROR:  Couldn't malloc _lwp_cur\n\r");
#endif
      return (0);
    }
  if (_lwp_lock_data (&_lwp_cur, sizeof (lwp)))
    {
#ifdef DEBUG
      cprintf ("couldn't lock _lwp_cur\n\r");
#endif
      return (0);
    }
  _init_fpu ();
  _lwp_cur->stack = NULL;
  _lwp_cur->lwpid = LWP_MAIN;
  _lwp_cur->next = _lwp_cur;

  _lwp_cur->priority = 1;       /* DEFAULT PRIORITY. Can be changed later. */

  _lwp_cur->status = LWP_RUNNING;
  _lwp_cur->userptr = (void *) 0;
  _lwp_enable = 1;
  _lwp_interrupt_pending = 0;

/* SIGILL is raised when an unhandled exception is raised */
  _lwp_count = 0;

  lwp_active_threads = 1;       /* One active thread at the moment */

  lwp_spawn (lwp_tcp, 4096, 1); /* Spawn the TCP */

  _lwp_cur->next->lwpid = LWP_TCP;

  _lwp_count = 0;               /* Reset the lwp pid counter */

  return lwp_install (irq, speed);
}
#undef cprintf
#ifdef DEBUG
static void
lwp_generate_exception (void)
  __attribute__ ((noreturn));
     static void lwp_generate_exception (void)
{
  int a = 5, b;
  b = 1 / (a - a);
  printf ("%i\n", b);           /* Force use of b */
}
#endif
#ifdef PARANOID
static void
lwp_paranoid (void)
{
  lwp *curs = _lwp_cur;

  _lwp_enable = 1;

  do
    {
      if (!Fortify_CheckPointer (curs))
        {
          fprintf (stderr, "Invalid pointer detected in thread chain(%p)\n",
                   curs);
          printf ("Invalid pointer detected in thread chain(%p)\n",
                  curs);
          printf ("--Press a key and we will bail out - Use Symify --");
          while (!kbhit ());

          lwp_generate_exception ();
        }
      else
        fprintf (stderr, "Thread %p - PID %i STATUS %i\n",
                 curs, curs->lwpid, curs->status);
      curs = curs->next;
    }
  while (curs != _lwp_cur);

  _lwp_enable = 0;
}
#endif

void
lwp_findnext (void)
{
  int sleepers = 0;             // Are there any sleepers?

  struct timeb buf;

  lwp *old = _lwp_cur;

  _lwp_cur = _lwp_cur->next;

  while (1)
    {
      switch (_lwp_cur->status)
        {
        case LWP_RUNNING:
/*          cprintf("(%i) to run\n\r",_lwp_cur->lwpid); */
          _lwp_pcount = _lwp_cur->priority;
          return;               /* Found! */
        case LWP_SLEEPING:
          sleepers = 1;
          ftime (&buf);
          if (_lwp_cur->waiting.wakeup_time.secs > buf.time)
            break;
          if (_lwp_cur->waiting.wakeup_time.secs == buf.time &&
              _lwp_cur->waiting.wakeup_time.msecs > buf.millitm)
            break;
          _lwp_cur->status = LWP_RUNNING;  /* Else... */
          _lwp_pcount = _lwp_cur->priority;
          return;
        case LWP_WAIT_SEMAPHORE:
          if (_lwp_cur->waiting.what_sema->owned)
            break;
          _lwp_cur->waiting.what_sema->owned = 1;
          _lwp_cur->waiting.what_sema->owner_id = _lwp_cur->lwpid;
          _lwp_cur->status = LWP_RUNNING;
          _lwp_pcount = _lwp_cur->priority;
          return;
        case LWP_WAIT_TRUE:
/*
 *          cprintf("Queue %i waiting for int at %p to become != 0\n\r",
 *                  _lwp_cur->lwpid,_lwp_cur->waiting.what_int);
 */
          if (*(_lwp_cur->waiting.what_int) == 0)
            break;
          _lwp_cur->status = LWP_RUNNING;
          _lwp_pcount = _lwp_cur->priority;
          return;
        case LWP_WAIT_FALSE:
/*
 *          cprintf("Queue %i waiting for int at %p to become == 0\n\r",
 *                  _lwp_cur->lwpid,_lwp_cur->waiting.what_int);
 */
          if (*(_lwp_cur->waiting.what_int) != 0)
            break;
          _lwp_cur->status = LWP_RUNNING;
          _lwp_pcount = _lwp_cur->priority;
          return;
        case LWP_DEAD:
          break;                /* Just skip this thread. */
        default:
          break;
        };
/*
 * Look if the task we've just tried to clean is the current one,
 * and if there are no sleeping tasks.
 * If it were so, it was a grievous fault: deadlock!
 */
      if (!sleepers && _lwp_cur == old)
        {
          cprintf ("Deadlock!\n\r");
          abort ();
        }
      _lwp_cur = _lwp_cur->next;
    };
}

int
lwp_getpid (void)
{
  return (_lwp_cur->lwpid);
}
void
lwp_setuserptr (void *usrdata)
{
  _lwp_cur->userptr = usrdata;  // No need to lock multitasker

}
void *
lwp_getuserptr (void)
{
  return (_lwp_cur->userptr);
}

void
lwp_set_priority (unsigned priority)
{
  if (priority != 0)
    _lwp_cur->priority = priority;
/*
 * Will become effective at next task
 * switch. Otherwise could be used to
 * run endlessly.
 */
}

unsigned
lwp_get_priority (void)
{
  return _lwp_cur->priority;
}

/* Spawns a light weight process.
   Returns the lwpid of the proc or 0 on fail */

unsigned long static
_lwp_flags (void)
{
  long i;

  __asm__ volatile ("pushfl \n"
                    "popl %0"
                    :"=g" (i));
  return i;
}

int
lwp_spawn (void (*proc) (void), int stack_length, unsigned priority)
{
  lwp *tmp;
  int tmp2;

  tmp2 = _lwp_enable;
  _lwp_enable = 1;
  if ((proc == NULL) || (stack_length < 256) || (priority == 0) )
    {
      _lwp_enable = tmp2;
      return (-1);
    }
  if ((tmp = (lwp *) malloc (sizeof (lwp))) == NULL)
    {
      _lwp_enable = tmp2;
      return (-1);
    }
  if ((tmp->stack = (unsigned int *) malloc (stack_length)) == NULL)
    {
      _lwp_enable = tmp2;
      return (-1);
    }
  if (_lwp_lock_data (&tmp, sizeof (lwp)))
    {
      _lwp_enable = tmp2;
      return (-1);
    }
  stack_length = (stack_length + 3) & ~3;  // Make divisible by 4

  tmp->stklen = stack_length;
  if (_lwp_lock_data (&tmp->stack, stack_length))
    {
      _lwp_enable = tmp2;
      return (-1);
    }
  tmp->stackTop = tmp->stack;
/* 8 regs + flags */
  tmp->stack[(stack_length / 4) - 1] = (long) _lwp_dead_thread;
  tmp->stack[(stack_length / 4) - 2] = (long) proc;
  tmp->stack[(stack_length / 4) - 3] =
    tmp->stack[(stack_length / 4) - 4] =  /* regs */
    tmp->stack[(stack_length / 4) - 5] =
    tmp->stack[(stack_length / 4) - 6] =
    tmp->stack[(stack_length / 4) - 7] =
    tmp->stack[(stack_length / 4) - 8] =
    tmp->stack[(stack_length / 4) - 9] = 0;
  tmp->stack[(stack_length / 4) - 10] = (long) _lwp_flags ();
  memcpy (tmp->stack + (stack_length / 4) - 38, _fpu_init_state, 108);

#if 0
  /*8 regs +flags + 27 dwords for the fpu  */
  memcpy (tmp->stack, _fpu_init_state, 108);
  tmp->stack[36] = (int) proc;
  tmp->stack[37] = (int) _lwp_dead_thread;

#endif
  tmp->stack = tmp->stack + (stack_length / 4) - 38;
  tmp->lwpid = ++_lwp_count;
  tmp->next = _lwp_cur->next;
  tmp->priority = priority;
  tmp->status = LWP_RUNNING;
  tmp->userptr = (void *) 0;
  lwp_active_threads++;

  _lwp_cur->next = tmp;

#ifdef PARANOID
  lwp_paranoid ();
#endif

  _lwp_enable = tmp2;
  return (tmp->lwpid);
}

/* Kills a lwp (takes it out of the linked list of lwp's) , or 0 if it fails */
/* Handles suicide perfectly */
int
lwp_kill (int lwpid)
{
  lwp *curs;
  int tmp;

  if (lwpid == LWP_MAIN || lwpid == LWP_TCP)
    return 0;
  /* Main and TCP can't be killed */
  tmp = _lwp_enable;
  _lwp_enable = 1;

/* Are WE going to die? */
  if (_lwp_cur->lwpid == lwpid)
    {
      _lwp_cur->status = LWP_DEAD;  /* This is all */
      lwp_dead_threads++;       /* Wake up lwp_tcp() */
      _lwp_enable = tmp;
      lwp_yield ();

      cprintf ("PANIC!\n\r");   /* Should never get here! */
      return 1;
    }

  curs = _lwp_cur;

  if (curs == NULL)
    return (0);

  while ((curs->next != _lwp_cur) && (curs->lwpid != lwpid))
    curs = curs->next;

  /* went through all of them and wasn't found */
  if (curs->lwpid != lwpid)
    return (0);

  /* now curs IS the right one to kill */

  curs->status = LWP_DEAD;      /* All we have to do is to classify it as dead */
  lwp_dead_threads++;           /* Wake up lwp_tcp() */

  _lwp_enable = tmp;
  return (1);
}

#ifdef DEBUG
static void
printRing (void)
{
  lwp *curs = _lwp_cur;
  printf ("Thread dump----\n");
  do
    {
      printf ("Thread %i	Status %i\n", curs->lwpid, curs->status);
      curs = curs->next;
    }
  while (curs != _lwp_cur);
  printf ("End Thread dump----\n");
}
#endif
/*
 *  This is our TCP, i.e. a never-dying thread. It just waits for the death
 *  of a thread. when such an event occurs, it frees that thread's stack
 *  and resources.
 */
static void
lwp_tcp (void)
{
  lwp *curs, *trailer;
  int tmp;

#ifdef PARANOID
  if (_lwp_cur->lwpid != LWP_TCP || _lwp_cur->status != LWP_RUNNING)
    lwp_generate_exception ();
#endif

  while (1)                     /* Should run until program completion */
    {
      lwp_wait_true (&lwp_dead_threads);  /* When a thread is dead... */
      tmp = _lwp_enable;
      _lwp_enable = 1;
#ifdef DEBUG
      fprintf (stderr, "TCP Started...\n");
      printRing ();
#endif
      trailer = _lwp_cur;
      curs = trailer->next;
      do
        {
#ifdef PARANOID
          lwp_paranoid ();
#endif
#ifdef DEBUG
          if (!Fortify_CheckPointer (curs))
            lwp_generate_exception ();
          fprintf (stderr, "Killing thread pointed by %p\n"
                   "Pointer is %s\n",
                   curs, Fortify_CheckPointer (curs) ? "Valid" : "Invalid");
          if (!Fortify_CheckPointer (curs))
            fprintf (stderr, "Status %i PID %i\n", curs->status, curs->lwpid);
#endif
          if (curs->status == LWP_DEAD)
            {
              trailer->next = curs->next;
              /* Free everything that thread owns */
              _lwp_unlock_data (curs->stackTop, curs->stklen);
              _lwp_unlock_data (&curs, sizeof (lwp));
              free (curs->stackTop);
              free (curs);
              lwp_active_threads--;
              curs = trailer;   /* Back up one unit */
            }
          trailer = curs;
          curs = curs->next;
        }
      while (curs != _lwp_cur);
      lwp_dead_threads = 0;     /* Reset dead threads counter */
#ifdef DEBUG
      printRing ();
      fprintf (stderr, "TCP ended\n");
#endif
      _lwp_enable = tmp;
    }
}
/*
 * It's the user's responsibility to ensure that all the threads have been
 * lwp_kill'ed now!
 */
void
lwp_deinit (void)
{
  volatile unsigned long tick;
  volatile char *cmostime;

  if (!_lwp_on)
    return;

  if (_lwp_cur->lwpid != LWP_MAIN)
    return;
  if (_lwp_irq_used != 8 && _lwp_irq_used != 0)
    return;

/*
 * Let lwp_tcp finish its job (if it has a job to finish, anyhow!).
 * This way, we'll free all the stacks and all the descriptors of all
 * the active threads
 */

  lwp_wait_false (&lwp_dead_threads);

  _lwp_enable = 1;              /* Now, disable multithreading */
#ifdef DEBUG
/*
 * Count active threads
 */
  do
    {
      int count = 0;
      lwp *curs = _lwp_cur;
      cprintf ("Active threads:\n\r");
      do
        {
          count++;
          cprintf ("Thread %i - PID %i\n\r", count, curs->lwpid);
          curs = curs->next;
        }
      while (curs != _lwp_cur);
      cprintf ("%i Threads active (should be 2)\n\r", count);
    }
  while (0);
#endif
/*
 * Now, find and kill lwp_tcp. If we are executing, it can't be running,
 * as it locks multithreading.
 */
  do
    {
      lwp *curs, *trail;

      trail = _lwp_cur;
      curs = _lwp_cur->next;
      for (; curs != _lwp_cur; trail = curs, curs = curs->next)
        if (curs->lwpid == LWP_TCP)
          {
            trail->next = curs->next;
/* Free everything that thread owns */
            _lwp_unlock_data (curs->stackTop, curs->stklen);
            _lwp_unlock_data (&curs, sizeof (lwp));
            free (curs->stackTop);
            free (curs);
#ifdef DEBUG
            cprintf ("TCP Killed\n\r");
#endif
            break;
          }
    }
  while (0);

  if (_lwp_irq_used == 8)
    {
      stopIRQ8 ();
      __dpmi_set_protected_mode_interrupt_vector (IRQ8, &_lwp_pm_old_handler);
      signal (SIGILL, _lwp_old_handler);
      _lwp_unlock_data (&_lwp_cur, sizeof (lwp));
      free (_lwp_cur);

#ifdef DEBUG
      cprintf ("IRQ8 successfully uninstalled.\n\r");
#endif
      _lwp_on = 0;
      return;
    }
  else if (_lwp_irq_used == 0)
    {
      outportb (PITMODE, 0x36);
      outportb (PIT0, 0x00);
      outportb (PIT0, 0x00);
      __dpmi_set_protected_mode_interrupt_vector (IRQ0, &_lwp_pm_old_handler);
      signal (SIGILL, _lwp_old_handler);
      _lwp_unlock_data (&_lwp_cur, sizeof (lwp));
      free (_lwp_cur);
      cmostime = get_cmostime ();
      tick = PIT0DEF *
        (
          (((float) *cmostime) * 3600) +
          (((float) *(cmostime + 1)) * 60) +
          (((float) *(cmostime + 2)))
        );
      biostime (1, tick);

      freq8h = PIT0DEF;
      counter_reset = freq8h / PIT0DEF;
      tick_per_ms = freq8h / 1000;
      ms_per_tick = 1000 / freq8h;
#ifdef DEBUG
      cprintf ("IRQ0 successfully uninstalled.\n\r");
#endif
      _lwp_on = 0;
      return;
    }
  return;
}

char *
get_cmostime (void)
{
  /* used to fix the fast timer */
  char *buff;
  static char buffer[6];
  char ch;

  buff = buffer;
  memset (&_lwp_regs, 0, sizeof (_lwp_regs));
  _lwp_regs.h.ah = 0x02;
  __dpmi_int (0x1a, &_lwp_regs);

  ch = _lwp_regs.h.ch;
  buffer[0] = (char) ((int) (ch & 0x0f) + (int) ((ch >> 4) & 0x0f) * 10);
  ch = _lwp_regs.h.cl;
  buffer[1] = (char) ((int) (ch & 0x0f) + (int) ((ch >> 4) & 0x0f) * 10);
  ch = _lwp_regs.h.dh;
  buffer[2] = (char) ((int) (ch & 0x0f) + (int) ((ch >> 4) & 0x0f) * 10);
  buffer[3] = _lwp_regs.h.dl;
  buffer[4] = (char) (_lwp_regs.x.flags & 0x0001);
  buffer[5] = 0x00;

  return (buff);
}
static void
_lwp_bottom_of_functions ()
{
};

static int
_lwp_lock_memory ()
{
  if (_lwp_lock_code (_lwp_top_of_functions, (long) &_lwp_bottom_of_functions - (long) &_lwp_top_of_functions))
    {
      /* fail */
#ifdef DEBUG
      cprintf ("ERROR:  DPMI error locking code segment functions\n\r");
#endif
      return (0);
    }
  if (_lwp_lock_code (&_lwpasm_start, (long) &_lwpasm_end - (long) _lwpasm_start))
    {
      /* fail */
#ifdef DEBUG
      cprintf ("ERROR:  DPMI error locking code segment IRQ handlers\n\r");
#endif
      return (0);
    }
  if (_lwp_lock_data (_lwp_top_of_vars, (long) &_lwp_bottom_of_vars - (long) &_lwp_top_of_vars))
    {
#ifdef DEBUG
      cprintf ("ERROR:  DPMI error locking data segment variables\n\r");
#endif
      return (0);
    }
  return (1);
}
void
_lwp_fpu_handler (int signum)
{
  int tmp = _lwp_enable;
  _lwp_enable = 1;
  _lwp_old_fpu_handler (signum);
  _lwp_enable = tmp;
}
/* returns true if windoze is running */
static int
windoze ()
{
  __dpmi_regs regs;
  regs.x.ax = 0x1600;
  __dpmi_int (0x2F, &regs);
  return ((regs.x.ax - 0x1600));
}
/* copied from go32 lib funcs */
int
_lwp_lock_data (void *lockaddr, unsigned long locksize)
{
  unsigned long baseaddr;
  __dpmi_meminfo memregion;
  if (windoze ())
    return (0);
  if (__dpmi_get_segment_base_address (_my_ds (), &baseaddr) == -1)
    return (-1);

  memset (&memregion, 0, sizeof (memregion));

  memregion.address = baseaddr + (unsigned long) lockaddr;
  memregion.size = locksize;

  if (__dpmi_lock_linear_region (&memregion) == -1)
    return (-1);

  return (0);
}

int
_lwp_unlock_data (void *lockaddr, unsigned long locksize)
{
  unsigned long baseaddr;
  __dpmi_meminfo memregion;
  if (windoze ())
    return (0);
  if (__dpmi_get_segment_base_address (_my_ds (), &baseaddr) == -1)
    return (-1);

  memset (&memregion, 0, sizeof (memregion));

  memregion.address = baseaddr + (unsigned long) lockaddr;
  memregion.size = locksize;

  if (__dpmi_unlock_linear_region (&memregion) == -1)
    return (-1);

  return (0);
}

int
_lwp_lock_code (void *lockaddr, unsigned long locksize)
{
  unsigned long baseaddr;
  __dpmi_meminfo memregion;
  if (windoze ())
    return (0);
  if (__dpmi_get_segment_base_address (_my_cs (), &baseaddr) == -1)
    return (-1);

  memset (&memregion, 0, sizeof (memregion));

  memregion.address = baseaddr + (unsigned long) lockaddr;
  memregion.size = locksize;

  if (__dpmi_lock_linear_region (&memregion) == -1)
    return (-1);

  return (0);
}

int
_lwp_unlock_code (void *lockaddr, unsigned long locksize)
{
  unsigned long baseaddr;
  __dpmi_meminfo memregion;
  if (windoze ())
    return (0);
  if (__dpmi_get_segment_base_address (_my_cs (), &baseaddr) == -1)
    return (-1);

  memset (&memregion, 0, sizeof (memregion));

  memregion.address = baseaddr + (unsigned long) lockaddr;
  memregion.size = locksize;

  if (__dpmi_unlock_linear_region (&memregion) == -1)
    return (-1);

  return (0);
}

static void
setRTC (int value)
{
  __asm__ __volatile__ ("cli\n\t"
                        "movb $0x0A, %%al\n\t"
                        "outb %%al, $0x70\n\t"  /* output status a   */
                        "inb $0x71, %%al\n\t"  /* get current stats */
                        "addb $0xF0, %%bl\n\t"  /* mask top 4 bits   */
                        "andb %%bl, %%al\n\t"  /* output new div    */
                        "subb $0xF0, %%bl\n\t"  /* only set bottom 4 bits */
                        "orb  %%bl, %%al\n\t"
                        "movb %%al, %%bl\n\t"
                        "movb $0xA, %%al\n\t"
                        "outb %%al, $0x70\n\t"  /* outputing to status a */
                        "movb %%bl, %%al\n\t"
                        "outb %%al, $0x71\n\t"  /* done */
                        "sti\n\t"
                        :       /* none */
                        :"b" (value)
                        :"%eax", "%edx");
}
static void
startIRQ8 (void)
{
  unsigned char shtuff;
  disable ();
  outportb (0x70, 0x0B);        /* status B */
  shtuff = inportb (0x71);      /* get status */
  shtuff = shtuff | 0x40;       /* mask off interrupt enable bit */
  outportb (0x70, 0x0B);        /* status B again */
  outportb (0x71, shtuff);      /* output new value */
  outportb (0x70, 0x0C);        /* status C */
  inportb (0x71);               /* ack interrupt */
  enable ();
}

static void
stopIRQ8 (void)
{
  unsigned char shtuff;
  disable ();
  outportb (0x70, 0x0B);        /* status B */
  shtuff = inportb (0x71);      /* get status */
  shtuff = shtuff & 0xBF;       /* mask off interrupt enable bit */
  outportb (0x70, 0x0B);        /* status B again */
  outportb (0x71, shtuff);      /* output new value */
  outportb (0x70, 0x0C);        /* status C */
  inportb (0x71);               /* ack interrupt */
  enable ();
}
void
lwp_sleep (unsigned int secs, unsigned short msecs)
{
  struct timeb now;
  int tmp = _lwp_enable;

  _lwp_enable = 1;              // Lock multitasking engine

  ftime (&now);

  _lwp_cur->status = LWP_SLEEPING;
  _lwp_cur->waiting.wakeup_time.msecs = (now.millitm + msecs) % 1000;
  _lwp_cur->waiting.wakeup_time.secs = now.time + secs +
    (now.millitm + msecs) / 1000;

  _lwp_enable = tmp;
  lwp_yield ();
}

void
lwp_wait_true (volatile int *what)  // Wait until what != 0
 {
  volatile int tmp = _lwp_enable;
  _lwp_enable = 1;              // Lock multitasking engine

  if (*what == 0)
    {
      _lwp_cur->status = LWP_WAIT_TRUE;
      _lwp_cur->waiting.what_int = what;

      _lwp_enable = tmp;
      lwp_yield ();
    }
  else
    _lwp_enable = tmp;          // Already != 0

}
void
lwp_wait_false (volatile int *what)  // Wait until what == 0
 {
  volatile int tmp = _lwp_enable;
  _lwp_enable = 1;              // Lock multitasking engine

  if (*what != 0)
    {
      _lwp_cur->status = LWP_WAIT_FALSE;
      _lwp_cur->waiting.what_int = what;

      _lwp_enable = tmp;
      lwp_yield ();
    }
  else
    _lwp_enable = tmp;          // Alreay == 0

}
// When a task waits for an integer to become true/false, it can be useful to
// force all the tasks waiting for that integer to pass as if it were become
// true/false.
void
lwp_pulse_true (volatile int *what)
{
  lwp *cursor = _lwp_cur->next;
  volatile int tmp = _lwp_enable;
  _lwp_enable = 1;              // Lock multitasking engine

  while (cursor != _lwp_cur)
    {
      if ((cursor->status == LWP_WAIT_TRUE) &&
          (cursor->waiting.what_int == what))
        cursor->status = LWP_RUNNING;  // If the tasks waits for this integer,
      // let it run.

      cursor = cursor->next;
    }

  _lwp_enable = tmp;
}
void
lwp_pulse_false (volatile int *what)
{
  lwp *cursor = _lwp_cur->next;
  volatile int tmp = _lwp_enable;
  _lwp_enable = 1;              // Lock multitasking engine

  while (cursor != _lwp_cur)
    {
      if ((cursor->status == LWP_WAIT_FALSE) &&
          (cursor->waiting.what_int == what))
        cursor->status = LWP_RUNNING;
      cursor = cursor->next;
    }

  _lwp_enable = tmp;
}

void
lwp_wait_semaphore (lwp_semaphore * sema)
{
  volatile int tmp = _lwp_enable;
  _lwp_enable = 1;              // Lock multitasking engine

  _lwp_cur->status = LWP_WAIT_SEMAPHORE;
  _lwp_cur->waiting.what_sema = sema;

  _lwp_enable = tmp;
  lwp_yield ();
}

void
lwp_init_semaphore (lwp_semaphore * sema)
{
  sema->owned = 0;
}

int
lwp_release_semaphore (lwp_semaphore * sema)
{
  if (sema->owned)
    {
      if (sema->owner_id != _lwp_cur->lwpid)
        return -1;
      sema->owned = 0;
    };
  return 0;
}
