// Associated include file : Common/Error.H

#include "common/common.h"

#define MaxErrorHandlers  50

// Global Vars

int          Error;
PErrorViewer ErrorViewer;

// Local Vars

static PErrorHandler  ErrorHandlers[MaxErrorHandlers];
static int            NbErrorHandlers;
static char          *Messages[10] =
      { "Out of Memory",
        "Impossible to open a disk file",
        "Impossible to create a disk file",
        "Impossible to read in a disk file",
        "Impossible to write in a disk file",
        "Impossible to close a disk file",
        "Impossible to seek in a disk file",
        "Found unexpected End Of File",
        "Line Too long in a disk file",
        "Programmer: No more error handlers"
      };

// ===== Gestion globale des erreurs ========================================

//---------------------------------------------------------------------------
// SafeMAlloc

void *SafeMAlloc(int32 Size)
{ // Tente l'allocation
  void *Temp=malloc((size_t)Size);
  // Vrifie si le rsulat est correct
  if (Temp==NULL) Error=erLowMemory;
  // Renvoie une valeur, NULL si il y a eu un problme
  return Temp;
}

//---------------------------------------------------------------------------
// SafeFree

void SafeFree(void **Block)
{ // Vrifie que le bloc  libr existe bien
  if (*Block==NULL) return;
  // Libre le bloc
  free((char*)*Block);
  // Marque le bloc comme libre
  *Block=NULL;
}

//---------------------------------------------------------------------------
// GetErrorMessage

void GetErrorMessage(char *Message)
{ int i;
  for(i=0;i<NbErrorHandlers;i++)
    if (ErrorHandlers[i](Message)) return;
  sprintf(Message,"Unknown Error n%d",Error);
}

//---------------------------------------------------------------------------
// AddErrorHandler

void AddErrorHandler(PErrorHandler ErrorHandler)
{ // Verifie que la liste des Handlers n'est pas pleine
  if (NbErrorHandlers==MaxErrorHandlers)
  { Error=erNoMoreErrorHandlers;
    ErrorViewer();
  }
  // Ajoute le handler
  NbErrorHandlers++;
  ErrorHandlers[NbErrorHandlers-1]=ErrorHandler;
}

//---------------------------------------------------------------------------
// FatalError

void FatalError(void)
{ if (Error)
  { char Message[100];
    GetErrorMessage(Message);
	 printf("#### FATAL ERROR\n");
    printf("#### %s\n\n",Message);
    exit(1);
  }
}

// ===== Gestion locale des erreurs =========================================

static boolean ErrorMessages(char *Message)
{ int MessageNo;
  switch(Error)
  { case erLowMemory:              MessageNo=0;   break;
    case erDiskAccess_Open:        MessageNo=1;   break;
    case erDiskAccess_Create:      MessageNo=2;   break;
    case erDiskAccess_Read:        MessageNo=3;   break;
    case erDiskAccess_Write:       MessageNo=4;   break;
    case erDiskAccess_Close:       MessageNo=5;   break;
    case erDiskAccess_Seek:        MessageNo=6;   break;
    case erDiskAccess_EOF:         MessageNo=7;   break;
	 case erDiskAccess_LineTooLong: MessageNo=8;   break;
    case erNoMoreErrorHandlers:    MessageNo=9;   break;
    default:
      return FALSE;
  }
  strcpy(Message,Messages[MessageNo]);
  return TRUE;
}

// ===== Init/Done de cette unit ===========================================

void InitError(void)
{ // Initialise le gestionnaire d'erreurs
  Error=0;
  NbErrorHandlers=1;
  ErrorHandlers[0]=ErrorMessages;
  ErrorViewer=FatalError;
}

void DoneError(void)
{ Debug("ERROR...     Closed\n");
}
