/****************************************************************************
*
*				Visual - A 3D Scientific Visualisation Program
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: demoarea.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Member functions for the class DemoArea, a class for
*				performing demos in a double buffered rendering area.
*
*				Note that the rendering area does not use the standard
*				MegaVision color model, but uses the native color model of
*				the current MGL video mode (color mapped or RGB).
*
* $Id: demoarea.cpp 1.2 1994/03/09 11:29:32 kjb release $
*
****************************************************************************/

#include "demo.hpp"

#pragma	hdrstop

#include "demoarea.hpp"
#include <dos.h>
#include <stdio.h>
#include <stdarg.h>

/*----------------------------- Implementation ----------------------------*/

// We need a rather large stack to flood fill high resolution displays!

extern unsigned _stklen = 51200U;

// Define the text displayed in the Explanation window during the demos

char welcomeText[] = "Welcome the the MegaGraph Graphics Library, a high "
					 "performance graphics library for PC's. This program "
					 "will give you an idea of the kind of primitives that "
					 "the library supports, and the sort of things you can "
					 "do with it.";

char lineText1[] =   "The MGL can draw lines - blindingly fast! "
					 "Click to continue...";
char lineText2[] =   "Of course, you are not restricted to drawing lines "
					 "that are a single pixel wide. You can use an "
					 "arbitrary pen width, and an arbitrary pen pattern "
					 "if you so desire. Click to continue...";

char ellipseText[] = "The MGL can draw ellipses any way you like. They can "
					 "be outlined in an arbitrary rectangular pen, or they "
					 "can of course be filled. Click to continue...";

char arcText[] =	 "The MGL can draw elliptical arc's any way you like. "
					 "They can be outlined, or they can of course be filled. "
					 "Click to continue...";

char rectText[] =	 "The MGL can draw rectangles any way you like. They "
					 "can be outlined, or they can of course be filled. "
					 "Click to continue...";

char polyText1[] = 	 "The MGL can draw arbitrary polgyons in any pattern "
					 "you like. Click to continue...";
char polyText2[] =   "However the MGL can also render convex polygons "
					 "- blindingly fast! Click to continue...";
char polyText3[] =   "But that is not all, how about some smooth shaded "
					 "polygons? Click to continue...";

char colorText1[] =  "The MGL works with 24 bit color values internally "
					 "and in color mapped modes like this you can assign "
					 "a color value to any index you like. This mode can "
					 "handle sixteen simulataneous colors at once. ";
char colorText2[] =  "The MGL works with 24 bit color values internally "
					 "and in color mapped modes like this you can assign "
					 "a color value to any index you like. This mode can "
					 "handle 256 simulataneous colors at once. Click to "
					 "continue...";
char colorText3[] =  "The MGL can also do high speed palette rotations. "
					 "Click to continue...";
char colorText4[] =  "Of course you can rotate the other direction also. "
					 "Click to continue...";
char colorText5[] =  "And naturally you can perform the classic palette "
					 "fade in and out!";
char colorText6[] =  "The MGL works with 24 bit color values internally "
					 "and in TrueColor (or direct color) modes like this "
					 "you can specify each color value directly.";

char patText[] = 	 "The MGL can fill with any arbitrary pattern that "
					 "you can think up. Naturally it comes with a good "
					 "selection of patterns that you can use right away.";

char markerText[] =	 "The MGL can also plot markers, which are great for "
					 "creating graphs and charts.";

char floodText1[] =  "The MGL can also perform very fast flood fills, both "
					 "boundary fills and interior fills. Click to perform "
					 "a boundary fill...";
char floodText2[] =  "Click to continue...";
char floodText3[] =  "Click to perform an interior fill...";

char animText1[] = 	 "The MGL can also perform smooth animation using "
					 "double buffering techniques. Click to continue...";
char animText2[] =   "Unfortunately this video mode does not support "
					 "buffering, so the animation is unavailable.";

#define SWAP(a,b)	{ a^=b; b^=a; a^=b; }
#define	MAXNUM	1000

int			backColor = scJade;
int			maxx,maxy,val;
attributes	attr;
color_t		maxcolor;
point		p[MAXNUM+1];

// Array of useful patterns

#define	NUMPATS	(sizeof(bitpat) / sizeof(bitpat[0]))

pattern bitpat[] = {
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
	{0x7F, 0xFF, 0xF7, 0xFF, 0x7F, 0xFF, 0xF7, 0xFF},
	{0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF},
	{0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA},
	{0xEE, 0xDD, 0xBB, 0x77, 0xEE, 0xDD, 0xBB, 0x77},
	{0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88},
	{0xB1, 0x30, 0x03, 0x1B, 0xD8, 0xC0, 0x0C, 0x8D},
	{0x80, 0x10, 0x02, 0x20, 0x01, 0x08, 0x40, 0x04},
	{0xFF, 0x88, 0x88, 0x88, 0xFF, 0x88, 0x88, 0x88},
	{0xFF, 0x80, 0x80, 0x80, 0xFF, 0x08, 0x08, 0x08},
	{0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x40, 0x20, 0x00, 0x02, 0x04, 0x08, 0x00},
	{0x82, 0x44, 0x39, 0x44, 0x82, 0x01, 0x01, 0x01},
	{0xF8, 0x74, 0x22, 0x47, 0x8F, 0x17, 0x22, 0x71},
	{0x55, 0xA0, 0x40, 0x40, 0x55, 0x0A, 0x04, 0x04},
	{0x20, 0x50, 0x88, 0x88, 0x88, 0x88, 0x05, 0x02},
	{0xBF, 0x00, 0xBF, 0xBF, 0xB0, 0xB0, 0xB0, 0xB0},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x08, 0x00, 0x80, 0x00, 0x08, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00},
	{0x11, 0x22, 0x44, 0x88, 0x11, 0x22, 0x44, 0x88},
	{0xFF, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00},
	{0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80},
	{0xAA, 0x00, 0x80, 0x00, 0x88, 0x00, 0x80, 0x00},
	{0xFF, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80},
	{0x08, 0x1C, 0x22, 0xC1, 0x80, 0x01, 0x02, 0x04},
	{0x88, 0x14, 0x22, 0x41, 0x88, 0x00, 0xAA, 0x00},
	{0x40, 0xA0, 0x00, 0x00, 0x04, 0x0A, 0x00, 0x00},
	{0x03, 0x84, 0x48, 0x30, 0x0C, 0x02, 0x01, 0x01},
	{0x80, 0x80, 0x41, 0x3E, 0x08, 0x08, 0x14, 0xE3},
	{0x10, 0x20, 0x54, 0xAA, 0xFF, 0x02, 0x04, 0x08},
	{0x77, 0x89, 0x8F, 0x8F, 0x77, 0x98, 0xF8, 0xF8},
	{0x00, 0x08, 0x14, 0x2A, 0x55, 0x2A, 0x14, 0x08},
	{0x00, 0x08, 0x14, 0x2A, 0x55, 0x2A, 0x14, 0x08},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x02, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x02, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0xA8, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0xA8, 0x00, 0x22, 0x00, 0x8A, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0x22, 0x00, 0x8A, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0x22, 0x00, 0xAA, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0xA2, 0x00, 0xAA, 0x00, 0x22, 0x00},
	{0xAA, 0x00, 0xA2, 0x00, 0xAA, 0x00, 0x2A, 0x00},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0x2A, 0x00},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x04, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x00, 0xAA, 0x04, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x00, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x01},
	{0xAA, 0x44, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x01},
	{0xAA, 0x44, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x54, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x54, 0xAA, 0x11, 0xAA, 0x45, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x11, 0xAA, 0x45, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x11, 0xAA, 0x55, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x51, 0xAA, 0x55, 0xAA, 0x11},
	{0xAA, 0x55, 0xAA, 0x51, 0xAA, 0x55, 0xAA, 0x15},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x15},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xEA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xEA, 0x55, 0xAA, 0x55, 0xAE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xAA, 0x55, 0xAE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xAA, 0x55, 0xEE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xBA, 0x55, 0xEE, 0x55, 0xAA, 0x55},
	{0xEE, 0x55, 0xBA, 0x55, 0xEE, 0x55, 0xAB, 0x55},
	{0xEE, 0x55, 0xBB, 0x55, 0xEE, 0x55, 0xAB, 0x55},
	{0xEE, 0x55, 0xBB, 0x55, 0xEE, 0x55, 0xBB, 0x55},
	{0xFE, 0x55, 0xBB, 0x55, 0xEE, 0x55, 0xBB, 0x55},
	{0xFE, 0x55, 0xBB, 0x55, 0xEF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xBB, 0x55, 0xEF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xBB, 0x55, 0xFF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xFB, 0x55, 0xFF, 0x55, 0xBB, 0x55},
	{0xFF, 0x55, 0xFB, 0x55, 0xFF, 0x55, 0xBF, 0x55},
	{0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xBF, 0x55},
	{0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55},
	{0xFF, 0xD5, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55},
	{0xFF, 0xD5, 0xFF, 0x55, 0xFF, 0x5D, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x55, 0xFF, 0x5D, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x55, 0xFF, 0xDD, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x75, 0xFF, 0xDD, 0xFF, 0x55},
	{0xFF, 0xDD, 0xFF, 0x75, 0xFF, 0xDD, 0xFF, 0x57},
	{0xFF, 0xDD, 0xFF, 0x77, 0xFF, 0xDD, 0xFF, 0x57},
	{0xFF, 0xDD, 0xFF, 0x77, 0xFF, 0xDD, 0xFF, 0x77},
	{0xFF, 0xFD, 0xFF, 0x77, 0xFF, 0xDD, 0xFF, 0x77},
	{0xFF, 0xFD, 0xFF, 0x77, 0xFF, 0xDF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0x77, 0xFF, 0xDF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0x77, 0xFF, 0xFF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x77},
	{0xFF, 0xFF, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x7F},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}
	};

long randoml(long max)
/****************************************************************************
*
* Function:		randoml
* Parameters:	max	- Maximum random number value+1
* Returns:		Random long between 0 and max-1.
*
* Description:	Routine to compute random long numbers. We do this simply
*				by expanding a random integer into the correct range,
*				which means some values will never get generated :-(,
*				but hell, it works for what we want (it will also fail
*				for large numbers).
*
****************************************************************************/
{
	return ((float)random(MAXINT) * max) / (float)MAXINT;
}

bool done(void)
/****************************************************************************
*
* Function:		done
* Returns:		True if a mouse click or key was pressed.
*
****************************************************************************/
{
	TEvent	evt;

	if (eventQueue.getNext(evt) && (evt.what & (evKeyboard | evMouseUp)))
		return true;
	return false;
}

void halt(void)
/****************************************************************************
*
* Function:		halt
*
* Description:	Halts until a key is pressed or the mouse is clicked.
*
****************************************************************************/
{
	eventQueue.flush();
	while (!done());
}

void clearView(void)
{
	MGL_setColor(MGL_realColor(backColor));
	MGL_fillRectCoord(0,0,MGL_maxx()+1,MGL_maxy()+1);
}

void beginDemo(void)
/****************************************************************************
*
* Function:		beginDemo
*
* Description:	Begin the next demo.
*
****************************************************************************/
{
	maxx = MGL_maxx();
	maxy = MGL_maxy();
	maxcolor = MGL_maxColor();
	MGL_getAttributes(&attr);
	TProgram::application->setSystemPalette();
	clearView();
	eventQueue.flush();
}

void endDemo(void)
/****************************************************************************
*
* Function:		endDemo
*
* Description:	Ends the demonstration
*
****************************************************************************/
{
	MGL_restoreAttributes(&attr);
}

DemoArea::DemoArea(const TRect& bounds)
	: TRenderArea(bounds)
/****************************************************************************
*
* Function:		DemoArea::DemoArea
* Parameters:	bounds		- Bounding box for the rendering area
*
* Description:	Constructor for the DemoArea class.
*
****************************************************************************/
{
	cntDemo = -1;
}

void DemoArea::handleEvent(TEvent& event,phaseType)
/****************************************************************************
*
* Function:		DemoArea::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routine for the DemoArea class.
*
****************************************************************************/
{
	if (event.what == evBroadcast) {
		switch (event.message.command) {
			case cmLineDemo:
			case cmEllipseDemo:
			case cmArcDemo:
			case cmPolygonDemo:
			case cmRectangleDemo:
			case cmColorDemo:
			case cmMarkerDemo:
			case cmPatternDemo:
			case cmFloodFillDemo:
			case cmAnimationDemo:
				cntDemo = event.message.command;
				repaint();				// Force repaint to execute the demo
				clearEvent(event);
				break;
			}
		}
}

static int gprintf(char *fmt, ... )
/****************************************************************************
*
* Function:		gprintf
* Parameters:	fmt		- Format string
*				...		- Standard printf style parameters
* Returns:		Number of items converted successfully.
*
* Description:	Simple printf style output routine for sending text to
*				the current viewport. It begins drawing the string at
*				the current CP location, and move the CP to the start of
*				the next logical line.
*
****************************************************************************/
{
	va_list	argptr;					/* Argument list pointer			*/
	char	buf[255];				/* Buffer to build sting into		*/
	int		cnt;					/* Result of SPRINTF for return 	*/
	Point	CP;

	va_start(argptr,fmt);

	cnt = vsprintf(buf,fmt,argptr);
	MGL_getCP(&CP);
	MGL_drawStr(buf);				/* Display the string				*/
	CP.y += MGL_textHeight();		/* Advance to next line				*/
	MGL_moveTo(CP);

	va_end(argptr);

	return cnt;						/* Return the conversion count		*/
}

void DemoArea::displayStatusInfo()
/****************************************************************************
*
* Function:		DemoArea::displayStatusInfo
*
* Description:	Displays status information about the current video mode.
*				This is used to paint the demonstration window when no
*				demonstration is currently running.
*
****************************************************************************/
{
	setDemoTitle("Status Information");
	setInfoText(welcomeText);

	MGL_moveToCoord(10,10);

	/* Get required information */

	MGL_setColor(MGL_realColor(scWhite));

	gprintf("Graphics device: (%d) %s", MGL_getDriver(),
		MGL_driverName(MGL_getDriver()));
	gprintf("Graphics mode: (%d) %s", MGL_getMode(),
		MGL_modeName(MGL_getMode()));
	gprintf("Screen resolution: ( 0, 0, %d, %d )",MGL_sizex(),MGL_sizey());
	gprintf("Colors available: %ld", MGL_maxColor()+1);
	gprintf("Maximum Page Index: %d", MGL_maxPage());

	attributes		attr;
	text_settings	tset;

	MGL_getAttributes(&attr);
	tset = attr.tsettings;		/* Make an alias */
	gprintf("Current font: %s", tset.fnt->name);
}

void lineSpeedDemo()
{
	int		i,j;

	beginDemo();

	for (i = 0; i < MAXNUM; i++) {
		p[i].x = random(maxx);
		p[i].y = random(maxy);
		}

	MGL_beginDrawing();
	for (j = 0; j < 10; j++) {
		MGL_setColor(MGL_realColor(j+1));
		for (i = 0; i < MAXNUM; i++)
			MGL_lineFast(p[i],p[i+1]);
		}
	MGL_endDrawing();
	endDemo();
}

void lineNormalDemo()
{
	point		p1,p2;

	beginDemo();
	MGL_beginDrawing();
	while (!done()) {
		p1.x = random(maxx);
		p1.y = random(maxy);
		p2.x = random(maxx);
		p2.y = random(maxy);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);
		MGL_setPenSize(random(5)+1,random(5)+1);

		if ((val = random(5)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else MGL_setPenStyle(SOLID_PATTERN);

		MGL_lineFast(p1,p2);
		}
	MGL_endDrawing();
	endDemo();
}

void DemoArea::lineDemo()
{
	setDemoTitle("Line Demonstration");
	setInfoText(lineText1);
	halt();
	setInfoText("");
	lineSpeedDemo();
	delay(500);
	setInfoText(lineText2);
	lineNormalDemo();
	cntDemo = -1;
}

void DemoArea::ellipseDemo()
{
	setDemoTitle("Ellipse Demonstration");
	setInfoText(ellipseText);
	beginDemo();

	TRect	r;

	while (!done()) {
		r.left() = random(maxx-100);
		r.top() = random(maxy-100);
		r.right() = r.left() + random(100);
		r.bottom() = r.top() + random(100);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		MGL_setPenSize(random(5)+1,random(5)+1);

		if ((val = random(5)) == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 2) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else MGL_setPenStyle(SOLID_PATTERN);

		if (random(3) == 0)
			MGL_fillEllipse(r);
		else
			MGL_ellipse(r);
		}

	endDemo();
	cntDemo = -1;
}

void DemoArea::arcDemo()
{
	setDemoTitle("Elliptical Arc Demonstration");
	setInfoText(arcText);
	beginDemo();

	int 	startAngle,endAngle;
	TRect	r;

	beginDemo();
	while (!done()) {
		r.left() = random(maxx-100);
		r.top() = random(maxy-100);
		r.right() = r.left() + random(100);
		r.bottom() = r.top() + random(100);
		startAngle = random(360);
		endAngle = random(360);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		MGL_setPenSize(random(5)+1,random(5)+1);

		if ((val = random(5)) == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 2) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else MGL_setPenStyle(SOLID_PATTERN);

		if (random(3) == 0)
			MGL_fillEllipseArc(r,startAngle,endAngle);
		else
			MGL_ellipseArc(r,startAngle,endAngle);
		}
	endDemo();
	cntDemo = -1;
}

void DemoArea::rectangleDemo()
{
	setDemoTitle("Rectangle Demonstration");
	setInfoText(rectText);
	beginDemo();

	TRect	r;

	while (!done()) {
		r.left() = random(maxx);
		r.right() = random(maxx);
		r.top() = random(maxy);
		r.bottom() = random(maxy);

		/* Fix the rectangle so it is not empty */

		if (r.right() < r.left())
			SWAP(r.left(),r.right());
		if (r.bottom() < r.top())
			SWAP(r.top(),r.bottom());

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		MGL_setPenSize(random(5)+1,random(5)+1);

		if ((val = random(5)) == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 2) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else MGL_setPenStyle(SOLID_PATTERN);

		if (random(3) == 0)
			MGL_rect(r);
		else
			MGL_fillRect(r);
		}

	endDemo();
	cntDemo = -1;
}

#define MaxPts		6				// Maximum # of pts in polygon

void polyDemo(void)
{
	int		i;
	point	poly[MaxPts];			// Space to hold polygon data

	beginDemo();
	while (!done()) {
		for (i = 0; i < MaxPts; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			}

		MGL_setColor(randoml(maxcolor)+1);
		MGL_setBackColor(randoml(maxcolor)+1);

		if ((val = random(3)) == 0) {
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else if (val == 1) {
			MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
			MGL_setPenBitmapPattern(&bitpat[random(NUMPATS)+1]);
			}
		else MGL_setPenStyle(SOLID_PATTERN);

		MGL_fillPolygon(MaxPts,poly,0,0);
		}

	endDemo();
}

void convexPolyDemo(void)
{
	point	poly[3];

	beginDemo();
	MGL_beginDrawing();
	MGL_setPolygonType(CONVEX_POLYGON);
	while (!done()) {
		poly[0].x = random(maxx);
		poly[0].y = random(maxy);
		poly[1].x = random(maxx);
		poly[1].y = random(maxy);
		poly[2].x = random(maxx);
		poly[2].y = random(maxy);

		MGL_setColor(randoml(maxcolor)+1);
		MGL_fillPolygon(3,poly,0,0);
		}
	MGL_endDrawing();
	endDemo();
}

void shadedPolyDemo(void)
{
	int		i;
	point	poly[3];
	color_t	colors[3];
	palette	pal[255];

	// Setup the palette to show smooth shading between bright red and
	// bright blue.

	beginDemo();
	if (maxcolor == 255) {
		MGL_getPalette(pal,255,0);
		for (i = scUser; i <= maxcolor; i++) {
			pal[i].red = ((long)PALMAX * (i-scUser)) / (maxcolor-scUser);
			pal[i].green = 0;
			pal[i].blue = ((long)PALMAX * ((maxcolor-scUser)-(i-scUser))) / (maxcolor-scUser);
			}
		MGL_setPalette(pal,255,0);
		}

	MGL_beginDrawing();

	while (!done()) {
		for (i = 0; i < 3; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			colors[i] = scUser + randoml(maxcolor-scUser);
			}
		MGL_fillGouraudPolygon(3,poly,colors,0,0);
		}

	MGL_endDrawing();
	endDemo();
}

void DemoArea::polygonDemo()
{
	setDemoTitle("Polygon Demonstration");
	setInfoText(polyText1);
	polyDemo();
	setInfoText(polyText2);
	convexPolyDemo();
	if (MGL_maxColor() == 255) {
		setInfoText(polyText3);
		shadedPolyDemo();
		}
	cntDemo = -1;
}

void DemoArea::colorDemo()
{
	int		color,width,height,x,y,i,j,top,bottom,start,palsize;
	palette	pal[256],temp[256];

	setDemoTitle("Color Demonstration");

	beginDemo();
	if (maxcolor <= 15) {
		// Simple color demonstration for 16 color displays

		setInfoText(colorText1);
		width = 2 * ((MGL_maxx()+1) / 16);
		height = 2 * ((MGL_maxy()-10)  / 10);

		x = width / 2;
		y = height / 2;
		color = 1;
		for (j = 0; j < 3; j++) {
			for (i = 0; i < 5; i++) {
				MGL_setColor(color++);
				MGL_fillRectCoord(x,y,x+width,y+height);
				x += (width/2) * 3;
				}
			y += (height / 2) * 3;
			x = width / 2;
			}
		}
	else if (maxcolor == 255) {
		// Color demonstration for color mapped displays with 255 colors

		setInfoText(colorText2);
		width = 2 * ((MGL_maxx()+1) / 47);
		height = 2 * ((MGL_maxy()-10)  / 48);
		palsize = maxcolor - scUser + 1;

		x = width / 2;
		y = height / 2;
		color = scUser;
		for (j = 0; j < maxcolor-scUser+1; j++) {
			MGL_setColor(color++);
			MGL_fillRectCoord(x,y,x+width,y+height);
			x += (width/2) * 3;
			if (((j+1) % 16) == 0) {
				y += (height / 2) * 3;
				x = width / 2;
				}
			}

		halt();
		setInfoText(colorText3);

		// Palette rotations

		MGL_getPalette(pal,palsize,scUser);
		while (!done()) {
			MGL_rotatePalette(pal,palsize,PAL_ROTATE_UP);
			MGL_setPalette(pal,palsize,scUser);
			}
		setInfoText(colorText4);
		while (!done()) {
			MGL_rotatePalette(pal,palsize,PAL_ROTATE_DOWN);
			MGL_setPalette(pal,palsize,scUser);
			}
		setInfoText(colorText5);

		// Palette fade out

		MGL_getPalette(pal,MGL_getPaletteSize(),0);
		for (i = 31; i >= 0; i--) {
			MGL_fadePalette(temp,pal,MGL_getPaletteSize(),(uchar)(i*8));
			MGL_setPalette(temp,MGL_getPaletteSize(),0);
			}

		// Palette fade in

		delay(1000);
		for (i = 0; i <= 31; i++) {
			MGL_fadePalette(temp,pal,MGL_getPaletteSize(),(uchar)(i*8));
			MGL_setPalette(temp,MGL_getPaletteSize(),0);
			}
		}
	else {
		// Color demonstration for HiColor and TrueColor modes

		setInfoText(colorText6);

		start = MGL_maxx()/8;
		width = MGL_maxx() - start*2;
		MGL_beginDrawing();

		top = MGL_maxy()/8;
		bottom = MGL_maxy()/8 + MGL_maxy()/5;
		for (x = 0; x < width; x++) {
			MGL_setColor(MGL_packColor((x * 255L) / width,0,0));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}

		top += MGL_maxy()/5;
		bottom += MGL_maxy()/5;
		for (x = 0; x < width; x++) {
			MGL_setColor(MGL_packColor(0,(x * 255L) / width,0));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}

		top += MGL_maxy()/5;
		bottom += MGL_maxy()/5;
		for (x = 0; x < width; x++) {
			MGL_setColor(MGL_packColor(0,0,(x * 255L) / width));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}

		top += MGL_maxy()/5;
		bottom += MGL_maxy()/5;
		for (x = 0; x < width/2; x++) {
			MGL_setColor(MGL_packColor((((width/2)-x) * 255L) / (width/2),
				(x * 255L) / (width/2),0));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}
		for (; x < width; x++) {
			MGL_setColor(MGL_packColor(0,
				( ((width/2) - (x - (width/2))) * 255L) / (width/2),
				((x-width/2) * 255L) / (width/2) ));
			MGL_lineCoordFast(start+x,top,start+x,bottom);
			}
		MGL_endDrawing();
		}
	endDemo();
	cntDemo = -1;
}

#define	POINTS	10

int		data1[POINTS] = { 1, 3, 5, 4, 3, 2, 1, 5, 4, 2 };
int		data2[POINTS] = { 4, 6, 10, 2, 6, 4, 8, 10, 6, 2 };
int		data3[POINTS] = { 1, 3, 2, 5, 7, 9, 5, 4, 5, 8 };

void DemoArea::markerDemo()
{
	int		i,stepx,stepy;
	point	p[POINTS];

	setDemoTitle("Marker Demonstration");
	setInfoText(markerText);

	beginDemo();
	stepx = maxx / 12;
	stepy = maxy / 12;

	// Draw the graph axes

	MGL_setColor(MGL_realColor(scWhite));
	MGL_lineCoord(stepx,maxy - stepy,stepx*11,maxy-stepy);
	MGL_lineCoord(stepx,maxy - stepy,stepx,maxy - stepy*11);

	for (i = stepx*11; i >= stepx; i -= stepx)
		MGL_lineCoord(i,maxy - stepy - 3,i,maxy - stepy + 3);

	for (i = maxy - stepy; i >= maxy-stepy*11; i -= stepy)
		MGL_lineCoord(stepx-3,i,stepx+3,i);

	if (MGL_sizey() >= 349)
		MGL_setPenSize(2,2);

	// Draw the first set of data

	for (i = 0; i < POINTS; i++) {
		p[i].x = stepx + i*stepx;
		p[i].y = maxy - stepy - data1[i]*stepy;
		}

	if (MGL_sizey() >= 599)
		MGL_setMarkerSize(6);
	else MGL_setMarkerSize(4);
	MGL_setColor(MGL_realColor(LIGHTRED));
	MGL_polyLine(POINTS,p);
	MGL_setMarkerColor(MGL_realColor(LIGHTMAGENTA));
	MGL_polyMarker(POINTS,p);

	// Draw the second set of data

	for (i = 0; i < POINTS; i++) {
		p[i].x = stepx + i*stepx;
		p[i].y = maxy - stepy - data2[i]*stepy;
		}

	MGL_setColor(MGL_realColor(LIGHTGREEN));
	MGL_polyLine(POINTS,p);
	MGL_setMarkerColor(MGL_realColor(LIGHTMAGENTA));
	MGL_setMarkerStyle(MARKER_CIRCLE);
	MGL_polyMarker(POINTS,p);

	// Draw the third set of data

	for (i = 0; i < POINTS; i++) {
		p[i].x = stepx + i*stepx;
		p[i].y = maxy - stepy - data3[i]*stepy;
		}

	MGL_setColor(MGL_realColor(LIGHTBLUE));
	MGL_polyLine(POINTS,p);
	MGL_setMarkerColor(MGL_realColor(LIGHTMAGENTA));
	MGL_setMarkerStyle(MARKER_X);
	MGL_polyMarker(POINTS,p);

	endDemo();
	cntDemo = -1;
}

void DemoArea::patternDemo()
{
	int 	i,j,x,y,height,width,space;

	setDemoTitle("Pattern Demonstration");
	setInfoText(patText);

	beginDemo();
	width  = (maxx+1) / 20;
	height = (maxy+1) / 9;
	if (MGL_sizey() < 479)
		space = 3;
	else space = 10;

	x = y = space;
	MGL_setColor(MGL_realColor(scWhite));
	MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);

	for (j = 0; j < 7; j++) {			/* For 7 rows of boxes 		*/
		for (i = 0; i < 15; i++) {		/* For 15 columns of boxes		*/
			MGL_setPenBitmapPattern(&bitpat[j*15 + i]);
			MGL_fillRectCoord(x,y,x+width+1,y+height+1);
			x += width + space;			/* Advance to next col			*/
			}
		x = space;
		y += height + space;
		}
	endDemo();
	cntDemo = -1;
}

// Function prototypes for floodfill routines in 'ffill.c'

extern "C" {
void MGL_interiorFill(int x,int y);
void MGL_boundaryFill(int x,int y,color_t bdr);
}

void boundaryFillDemo(void)
{
	int     i,j;
    color_t color;
	point	poly[6];

	beginDemo();
	srand(200);

	// Draw a set of polygons

	for (j = 0; j < 3; j++) {
		color = random(10);				/* Keep random in sync			*/
		color = (j == 0) ? BLUE : (j == 1) ? RED : YELLOW;

		for (i = 0; i < 6; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			}

		MGL_setColor(MGL_realColor(color));
		MGL_fillPolygon(6,poly,0,0);
		}
	halt();

	MGL_setColor(MGL_realColor(LIGHTGREEN));
	MGL_boundaryFill(10,10,MGL_realColor(RED));
}

void interiorFillDemo(void)
{
	int     i,j;
    color_t color;
	point	poly[6];

	beginDemo();
	srand(200);

	// Draw a set of polygons

	for (j = 0; j < 3; j++) {
		color = random(10);				/* Keep random in sync			*/
		color = (j == 0) ? BLUE : (j == 1) ? RED : YELLOW;

		for (i = 0; i < 6; i++) {
			poly[i].x = random(maxx);
			poly[i].y = random(maxy);
			}

		MGL_setColor(MGL_realColor(color));
		MGL_fillPolygon(6,poly,0,0);
		}
	halt();

	MGL_setColor(MGL_realColor(LIGHTGREEN));
	MGL_interiorFill(10,10);
}

void DemoArea::floodFillDemo()
{
	setDemoTitle("Flood Fill Demonstration");
	setInfoText(floodText1);
	boundaryFillDemo();
	setInfoText(floodText2);
	halt();
	setInfoText(floodText3);
	interiorFillDemo();
	cntDemo = -1;
}

int fixAngle(int angle)
{
	while (angle < 0)
		angle += 360;
	while (angle >= 360)
		angle -= 360;
	return angle;
}

void DemoArea::animationDemo()
{
	int		stepx,stepy,secAngle,minAngle;
	rect	extent,dirtyRect;

	setDemoTitle("Animation Demonstration");
	if (MGL_maxPage() > 0) {
		setInfoText(animText1);
		beginDemo();

		cntDemo = cmEmptyDemo;
		doubleBufferOn();

		extent.left() = extent.top() = 0;
		extent.right() = maxx/5;
		extent.bottom() = ((long)extent.right() * 1000) / MGL_getAspectRatio();
		dirtyRect = extent;

		stepx = 1;
		stepy = 1;
		secAngle = minAngle = 90;

		while (!done()) {
			MGL_setColor(MGL_realColor(backColor));
			MGL_fillRect(dirtyRect);
			MGL_setColor(MGL_realColor(LIGHTRED));
			MGL_fillEllipse(extent);
			MGL_setColor(MGL_realColor(scWhite));
			MGL_fillEllipseArc(extent,secAngle-5,secAngle);
			MGL_fillEllipseArc(extent,minAngle-5,minAngle);

			swapBuffers();			// Swap the display buffers

			// Bounce the clock off the walls

			dirtyRect = extent;
			dirtyRect.inset(-ABS(stepx),-ABS(stepy));
			if (extent.left() + stepx < 0)
				stepx = -stepx;
			if (extent.right() + stepx > maxx)
				stepx = -stepx;

			if (extent.top() + stepy < 0)
				stepy = -stepy;
			if (extent.bottom() + stepy > maxy)
				stepy = -stepy;

			extent.offset(stepx,stepy);

			// Update the hand movement

			secAngle = fixAngle(secAngle - 5);
			if (secAngle == 90)
				minAngle = fixAngle(minAngle - 5);
			}

		doubleBufferOff();
		endDemo();
		}
	else setInfoText(animText2);
	cntDemo = -1;
}

void DemoArea::setDemoTitle(char *title)
/****************************************************************************
*
* Function:		DemoArea::setDemoTitle
* Parameters:	title	- New title for the demonstration
*
* Description:	Sets the demonstration title information, and forces a
*				repaint event.
*
****************************************************************************/
{
	TEvent	e;

	message(TProgram::deskTop,evBroadcast,cmSetDemoTitle,title);
	getEvent(e,evRepaint);
}

void DemoArea::setInfoText(char *text)
/****************************************************************************
*
* Function:		DemoArea::setInfoText
* Parameters:	text	- Text for the new window.
*
* Description:	Sets the explanation information in the information
*				window.
*
****************************************************************************/
{
	TEvent	e;

	message(TProgram::deskTop,evBroadcast,cmSetExplanation,text);
	getEvent(e,evRepaint);
}

void DemoArea::render()
/****************************************************************************
*
* Function:		DemoArea::render
*
* Description:	Renders the data in the visualisation in either the 2D or
*				3D format depending on the current setting of the 2D view
*				flag.
*
****************************************************************************/
{
	MGL_setColor(MGL_realColor(backColor));
	TRect clip(0,0,size.x+1,size.y+1);
	MGL_setClipRect(clip);
	MGL_fillRectCoord(0,0,size.x+1,size.y+1);

	switch (cntDemo) {
		case cmLineDemo:
			lineDemo();
			break;
		case cmEllipseDemo:
			ellipseDemo();
			break;
		case cmArcDemo:
			arcDemo();
			break;
		case cmPolygonDemo:
			polygonDemo();
			break;
		case cmRectangleDemo:
			rectangleDemo();
			break;
		case cmColorDemo:
			colorDemo();
			break;
		case cmMarkerDemo:
			markerDemo();
			break;
		case cmPatternDemo:
			patternDemo();
			break;
		case cmFloodFillDemo:
			floodFillDemo();
			break;
		case cmAnimationDemo:
			animationDemo();
			break;
		case cmEmptyDemo:
			break;
		default:
			displayStatusInfo();
		}
}
