/****************************************************************************
*
*						MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: main.c $
* Version:		$Revision: 1.2 $
*
* Language:		ANSI C
* Environment:	IBM PC (MS DOS)
*
* Description:	Main program for the MegaGraph graphics library test
*				programs. This contains a set of quick and dirty routines
*				to set up the graphics environment in a number of different
*				ways to test the library routines.
*
*				It depends upon the 'getopt.c' and 'getopt.h' files, from
*				my ctools10.zip archive. They should be included in the
*				distribution. It also requires the zen timer library which
*				should also be included in the distribution.
*
* $Id: main.c 1.2 1994/03/10 09:25:52 kjb release $
*
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <conio.h>
#include <dos.h>
#include <string.h>
#include "getopt.h"
#include "mgraph.h"
#include "ztimer.h"

/*--------------------------- Global Variables ----------------------------*/

char	modename[80];				/* Name of active video mode		*/
bool	smallclip;					/* Small clipping mode				*/

/*------------------------- Implementation --------------------------------*/

void four_dots(void)
/****************************************************************************
*
* Function:		four_dots
*
* Description:	Very simple routine to draw 4 dots, one at each corner
*				of the display in the current drawing color.
*
****************************************************************************/
{
	MGL_beginPixel();
	MGL_pixelCoord(0,0);
	MGL_pixelCoord(MGL_maxx(),0);
	MGL_pixelCoord(MGL_maxx(),MGL_maxy());
	MGL_pixelCoord(0,MGL_maxy());
	MGL_endPixel();
}

void ReportTime(ulong count)
/****************************************************************************
*
* Function:		ReportTime
* Parameters:	count	- Number of microseconds to count with.
*
* Description:	Simple routine to convert a count of microseconds into
*				seconds and display it on the standard output device.
*
****************************************************************************/
{
	ulong	secs;

	secs = count / 1000000L;
	count = count - secs * 1000000L;
	printf("Time taken: %lu.%06lu seconds\n",secs,count);
}

void help(int driver,int *modeList)
/****************************************************************************
*
* Function:		help
* Parameters:	driver		- Device driver detected.
*				chipID		- Device driver chip ID.
*				dac			- Video DAC installed
*				memory		- Amount of memory installed
*				modeList	- List of valid video mode numbers.
*
* Description:	Provide command line usage information about the program,
*				and how to modify the options.
*
****************************************************************************/
{
	int		i;

	printf("Options are:\n");
	printf("  -v        Map all output to a small viewport\n");
	printf("  -f<style> Set the fill style (TRANS,OPAQUE or PIX)\n");
	printf("  -h<size>  Set the pen height\n");
	printf("  -w<size>  Set the pen width\n");
	printf("  -x        Scan convert primitives in XOR mode\n");
	printf("  -a        Scan convert primitives in AND mode\n");
	printf("  -o        Scan convert primitives in OR mode\n");
	printf("  -c        Turn clipping off\n");
	printf("  -C        Clip to a small clip rectangle\n");
	printf("  -p        Use page 1 if available\n");
	printf("  -V        Use the non-acclerated VESA VBE drivers\n");
	printf("  -E        Force EGA operation\n");
	printf("  -P<value> Set the number of palette entries set at a time (default 100)\n");
	printf("  -m\"mode\"  Set the video mode. Valid modes are:\n\n");

	i = 0;
	while(modeList[i] != -1) {
		printf("            \"%s\" - %d page\n",
			MGL_modeName(modeList[i]),
			MGL_availablePages(driver,modeList[i]));
		i++;
		}

	printf("\nVideo Card: %s\n",MGL_driverName(driver));
	exit(1);
}

void init_graphics(char *testname,int argc,char *argv[])
/****************************************************************************
*
* Function:		init_graphics
* Parameters:	testname	- Name of the test being run
*				argc,argv	- Parameters to the main() routine.
*
* Description:	Parses the command line and starts the graphics environment
*				using the specified options.
*
****************************************************************************/
{
	int		i,driver,mode,err;
	int		*modeList;
	int		option;
	int		writemode;
	int		fillstyle;
	bool	smallview;
	bool	sep_page;
	int		clipping;
	int		height,width,snowlevel;
	char	*argument;
	rect	r;
	pattern		pat = {0xF8, 0x74, 0x22, 0x47, 0x8F, 0x17, 0x22, 0x71};
	pixpattern	pix = {{0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08},
					   {0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x01},
					   {0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x01, 0x02},
					   {0x04, 0x05, 0x06, 0x07, 0x08, 0x01, 0x02, 0x03},
					   {0x04, 0x05, 0x06, 0x07, 0x08, 0x01, 0x02, 0x03},
					   {0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x01, 0x02},
					   {0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x01},
					   {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08}};

	printf("%s\n\n",testname);
	driver = grDETECT;
	writemode = REPLACE_MODE;
	fillstyle = SOLID_PATTERN;
	clipping = CLIPON;
	smallclip = false;
	smallview = false;
	sep_page = false;
	height = width = 1;
	snowlevel = -1;

#ifdef	USE_LINKED
	MGL_registerDriver(EGANAME,EGA_driver);
	MGL_registerDriver(VGANAME,VGA_driver);
	MGL_registerDriver(VGA256NAME,VGA256_driver);
	MGL_registerDriver(VGAXNAME,VGAX_driver);
	MGL_registerDriver(SVGA256NAME,SVGA256_driver);
	MGL_registerDriver(SVGA32KNAME,SVGA32K_driver);
	MGL_registerDriver(SVGA64KNAME,SVGA64K_driver);
	MGL_registerDriver(SVGA16MNAME,SVGA16M_driver);
	MGL_registerDriver(SVGAS3NAME,SVGAS3_driver);
#endif

	MGL_detectGraph(&driver,&mode);
	modeList = MGL_availableModes(driver);

	do {
		option = getopt(argc,argv,"EIVvf:xaocCm:h:w:pP:",&argument);
		switch(option) {
			case 'V':
				MGL_detectGraph(&driver,&mode);
				if (driver > grSVGA)
					driver = grSVGA;
				modeList = MGL_availableModes(driver);
				break;
			case 'E':
				driver = grEGA;
				mode = grEGA_640x350x16;
				break;
			case 'v':
				smallview = true;
				printf("Mapping to small viewport ...\n");
				break;
			case 'p':
				sep_page = true;
				printf("Using page 1 for all output if available ...\n");
				break;
			case 'P':
				snowlevel = atoi(argument);
				break;
			case 'f':
				strlwr(argument);
				if (strcmp(argument,"trans") == 0) {
					fillstyle = BITMAP_PATTERN_TRANSPARENT;
					printf("Using TRANSPARENT bitmap style ...\n");
					}
				else if (strcmp(argument,"opaque") == 0) {
					fillstyle = BITMAP_PATTERN_OPAQUE;
					printf("Using OPAQUE bitmap style ...\n");
					}
				else if (strcmp(argument,"pix") == 0) {
					fillstyle = PIXMAP_PATTERN;
					printf("Using PIXMAP bitmap style ...\n");
					}
				else {
					printf("invalid argument\a\n");
					exit(1);
					}
				break;
			case 'x':
				writemode = XOR_MODE;
				printf("Using XOR write mode ...\n");
				break;
			case 'a':
				writemode = AND_MODE;
				printf("Using AND write mode ...\n");
				break;
			case 'o':
				writemode = OR_MODE;
				printf("Using OR write mode ...\n");
				break;
			case 'c':
				clipping = CLIPOFF;
				printf("Turning clipping off ...\n");
				break;
			case 'C':
				smallclip = true;
				printf("Clipping to a small rectangle ...\n");
				break;
			case 'h':
				height = atoi(argument);
				printf("Setting pen height to %d ...\n",height);
				break;
			case 'w':
				width = atoi(argument);
				printf("Setting pen width to %d ...\n",width);
				break;
			case 'm':
				/* Look up the mode number given it's name */

				i = 0;
				while (true) {
					if ((mode = modeList[i++]) == -1)
						break;
					if (strcmp(MGL_modeName(mode),argument) == 0)
						break;
					}
				if (mode == -1) {
					printf("Invalid video mode selected\n");
					exit(1);
					}
				break;
			case PARAMETER:
			case INVALID:
				help(driver,modeList);
			}
		} while (option != ALLDONE);

	printf("\nPress a key to start ...");
	getch();

#ifdef	MGL_TEST
	MGL_init(&driver,&mode,"");
#else
	MGL_init(&driver,&mode,"..\\");
#endif
	err = MGL_result();
	if (err != grOK) {
		printf("Graphics error: %s\n",MGL_errorMsg(err));
		printf("Driver: %d, Mode: %d\n",driver,mode);
		exit(1);
		}
	MGL_setWriteMode(writemode);
	MGL_setPenStyle(fillstyle);
	if (fillstyle == BITMAP_PATTERN_OPAQUE)
		MGL_setBackColor(5);
	MGL_setPenBitmapPattern(&pat);
	MGL_setPenPixmapPattern(&pix);
	MGL_setClipMode(clipping);
	MGL_setPenSize(height,width);
	if (sep_page) {
		MGL_setActivePage(1);
		MGL_setVisualPage(1);
		MGL_clearDevice();
		}
	if (smallview) {
		r.left = (MGL_maxx()+1)/4;
		r.right = r.left * 3;
		r.top = (MGL_maxy()+1)/4;
		r.bottom = r.top * 3;

		r.left -= 2;
		r.top -= 2;

		MGL_setColor(1);
		MGL_insetRect(r,-1,-1);
		MGL_rect(r);
		MGL_insetRect(r,1,1);
		MGL_setColor(MGL_maxColor());
		MGL_setViewport(r);
		}
	if (smallclip) {
		r.left = (MGL_maxx()+1)/4;
		r.right = r.left * 3;
		r.top = (MGL_maxy()+1)/4;
		r.bottom = r.top * 3;
		MGL_setColor(1);
		MGL_insetRect(r,-1,-1);
		MGL_rect(r);
		MGL_insetRect(r,1,1);
		MGL_setColor(MGL_maxColor());
		MGL_setClipRect(r);
		}
	if (snowlevel != -1)
		MGL_setPaletteSnowLevel(snowlevel);
	strcpy(modename,MGL_modeName(mode));
}

void exit_graphics(void)
/****************************************************************************
*
* Function:		void exit_graphics(void)
*
* Description:	Pause for a key press before killing graphics modes, and
*				then output the name of the video mode used.
*
****************************************************************************/
{
	getch();
	MGL_exit();
	printf("Name of video mode used: \"%s\"\n",modename);
}
