/****************************************************************************
*
*						MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: testgr.c $
* Version:		$Revision: 1.3 $
*
* Language:		ANSI C
* Environment:	IBM PC (MS DOS)
*
* Description:	Set of test programs to test the MegaGraph graphics library
*				routines. You must define which test to compile for when
*				you compile this source file.
*
*				This could have been factorised into separate source files,
*				but having all the tests in one file makes it easy to
*				find code that tests a part of the library using a global
*				text search.
*
* $Id: testgr.c 1.3 1994/03/10 09:25:52 kjb release $
*
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <conio.h>
#include <dos.h>
#include <string.h>
#include <alloc.h>
#include "mgraph.h"
#include "mgldos.h"
#include "ztimer.h"

extern	bool smallclip;

/* Define the test being compiled. You will need to comment this out when
 * compiling all tests via the makefile.
 */

#ifdef	MGL_TEST
#define TEST15
#endif

/* Routines in main.c */

void four_dots(void);
void init_graphics(char *testname,int argc,char *argv[]);
void exit_graphics(void);
void ReportTime(ulong count);

/* Different mouse cursors */

cursor CHECK = {
	0xFFF0, 0xFFE0, 0xFFC0, 0xFF81,
	0xFF03, 0x0607, 0x000F, 0x001F,
	0x803F, 0xC07F, 0xE0FF, 0xF1FF,
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,

	0x0000, 0x0006, 0x000C, 0x0018,
	0x0030, 0x0060, 0x70C0, 0x3980,
	0x1F00, 0x0E00, 0x0400, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0005, 0x000A };

cursor CROSS = {
	0xF01F, 0xE00F, 0xC007, 0x8003,
	0x0441, 0x0C61, 0x0381, 0x0381,
	0x0381, 0x0C61, 0x0441, 0x8003,
	0xC007, 0xE00F, 0xF01F, 0xFFFF,

	0x0000, 0x07C0, 0x0920, 0x1110,
	0x2108, 0x4004, 0x4004, 0x783C,
	0x4004, 0x4004, 0x2108, 0x1110,
	0x0920, 0x07C0, 0x0000, 0x0000,
	0x0007, 0x0007 };

cursor GLOVE = {
	0xF3FF, 0xE1FF, 0xE1FF, 0xE1FF,
	0xE1FF, 0xE049, 0xE000, 0x8000,
	0x0000, 0x0000, 0x07FC, 0x07F8,
	0x9FF9, 0x8FF1, 0xC003, 0xE007,

	0x0C00, 0x1200, 0x1200, 0x1200,
	0x1200, 0x13B6, 0x1249, 0x7249,
	0x9249, 0x9001, 0x9001, 0x8001,
	0x4002, 0x4002, 0x2004, 0x1FF8,
	0x0004, 0x0000 };

cursor IBEAM = {
	0xF39F, 0xFD7F, 0xFEFF, 0xFEFF,
	0xFEFF, 0xFEFF, 0xFEFF, 0xFEFF,
	0xFEFF, 0xFEFF, 0xFEFF, 0xFEFF,
	0xFEFF, 0xFEFF, 0xFD7F, 0xF39F,

	0x0C60, 0x0280, 0x0100, 0x0100,
	0x0100, 0x0100, 0x0100, 0x0100,
	0x0100, 0x0100, 0x0100, 0x0100,
	0x0100, 0x0100, 0x0280, 0x0C60,
	0x0007, 0x0008 };

ulong	count1,count2,count3,count4;

long randoml(long max)
/****************************************************************************
*
* Function:		randoml
* Parameters:	max	- Maximum random number value+1
* Returns:		Random long between 0 and max-1.
*
* Description:	Routine to compute random long numbers. We do this simply
*				by expanding a random integer into the correct range,
*				which means some values will never get generated :-(,
*				but hell, it works for what we want (it will also fail
*				for large numbers).
*
****************************************************************************/
{
    return ((float)rand() * max) / (float)RAND_MAX;
}

#ifdef	TEST1

/* TEST1	- Initialisation test.
 *
 * Test the initialisation routines. Put system into graphics mode, back
 * into text mode, and then change to a new graphics mode.
 *
 * Note that this routine also tests the functionality of the mouse
 * cursor routines.
 */

void cdecl main(int argc,char *argv[])
{
    int driver,chip,memory,dac,mode,err;

    dac = grDETECTDAC;
    MGL_detectGraph(&driver,&chip,&memory,&dac,&mode);

	_installMouse = true;			/* Install mouse cursor routines	*/
	MS_init();
	init_graphics("Initialisation test",argc,argv);
	four_dots();
	MS_show();
	getch();
	MS_hide();
	getch();
	MS_hide();
	getch();
	MS_hide();
	getch();
	MS_show();
	getch();
	MS_show();
	getch();
	MS_show();
	getch();
	MS_show();
	MS_setCursor(&CROSS);
	getch();
	MS_hide();
	MS_setCursor(&DEF_CURSOR);
	MS_show();
	getch();

	MS_hide();
	MGL_restoreCRTMode();
	MGL_restoreCRTMode();
	printf("Back in text mode\n");
	MS_show();
	getch();

	MS_hide();
	mode = grEGA_640x200x16;
	MGL_exit();
    MGL_init(&driver,&chip,&memory,&dac,&mode,"..\\");
	err = MGL_result();
	if (err != grOK) {
		printf("Graphics error: %s\n",MGL_errorMsg(err));
		printf("Driver: %d, Mode: %d\n",driver,mode);
		exit(1);
		}
	four_dots();
	MS_show();
	getch();

	MS_hide();
	mode = grEGA_320x200x16;
	MGL_exit();
    MGL_init(&driver,&chip,&memory,&dac,&mode,"..\\");
	err = MGL_result();
	if (err != grOK) {
		printf("Graphics error: %s\n",MGL_errorMsg(err));
		printf("Driver: %d, Mode: %d\n",driver,mode);
		exit(1);
		}
	four_dots();
	MS_show();
	exit_graphics();

	MS_exit();
}

#endif

#ifdef	TEST2

/* TEST2	- Raw pixel speed test.
 *
 * Test the raw pixel drawing speed of the library.
 */

void fill_pixel(void)
{
	int		i,j,maxx,maxy;

	maxx = MGL_maxx();
	maxy = MGL_maxy();
	MGL_beginPixel();
	for (j = 0; j <= maxy; j++)
		for (i = 0; i <= maxx; i++)
			MGL_pixelCoord(i,j);
	MGL_endPixel();
}

void cdecl main(int argc,char *argv[])
{
	color_t	i;

	init_graphics("Pixel plotting speed test",argc,argv);

	four_dots();
	getch();
	LZTimerOn();
	fill_pixel();
	MGL_setColor(MGL_defaultColor()/3);
	fill_pixel();
	LZTimerOff();
	i = MGL_getPixelCoord(0,0);

	exit_graphics();

	printf("Value of pixel (0,0) = %ld\n",i);
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST3

/* TEST3	- Screen clearing test
 *
 * Test the screen clearing routines of the library
 */

void cdecl main(int argc,char *argv[])
{
	int		i;

	init_graphics("Screen clearing test",argc,argv);

	LZTimerOn();
	for (i = 0; i < 200; i++) {
		MGL_setBackColor(MGL_realColor(i & 15));
		MGL_clearDevice();
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST4

/* TEST4	- Simple line drawing/clipping test
 *
 * This simple test draws a simple pattern on the display, and tests the
 * clipping and drawing routines.
 */

void cdecl main(int argc,char *argv[])
{
	int		i,maxx,maxy;
	rect	clip2;

	init_graphics("Line drawing/clipping test",argc,argv);

	LZTimerOn();
	maxx = MGL_maxx();
	maxy = MGL_maxy();
	for (i = 0; i < maxx; i += 10) {
		MGL_lineCoord(maxx/2,maxy/2,i,0);
		MGL_lineCoord(maxx/2,maxy/2,i,maxy);
		}
	for (i = 0; i < maxy; i += 10) {
		MGL_lineCoord(maxx/2,maxy/2,0,i);
		MGL_lineCoord(maxx/2,maxy/2,maxx,i);
		}

	clip2.left = maxx/8;
	clip2.top = maxy/8;
	clip2.right = clip2.left * 7;
	clip2.bottom = clip2.top * 7;
	MGL_insetRect(clip2,-1,-1);
	MGL_setColor(MGL_realColor(BLUE));
	MGL_rect(clip2);
	MGL_insetRect(clip2,1,1);
	MGL_setClipRect(clip2);

	MGL_setColor(MGL_realColor(YELLOW));
	for (i = 0; i < maxx; i += 10) {
		MGL_lineCoord(maxx/2,maxy/2,i,0);
		MGL_lineCoord(maxx/2,maxy/2,i,maxy);
		}
	for (i = 0; i < maxy; i += 10) {
		MGL_lineCoord(maxx/2,maxy/2,0,i);
		MGL_lineCoord(maxx/2,maxy/2,maxx,i);
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST5

/* TEST5	- Line drawing speed test
 *
 * This routine draws a number of random lines at full speed to time
 * how fast the routines are.
 */

#define	MAXNUM	1000
#define	LOOPS	5

point	p[MAXNUM+1];

void cdecl main(int argc,char *argv[])
{
	int		i,j,maxx,maxy;

	init_graphics("Line drawing speed test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(1000);
	for (i = 0; i < MAXNUM; i++) {
		p[i].x = random(maxx);
		p[i].y = random(maxy);
		}

	LZTimerOn();
	for (j = 0; j < LOOPS; j++) {
		MGL_setColor(MGL_realColor(j+1));
		for (i = 0; i < MAXNUM; i++)
			MGL_line(p[i],p[i+1]);
		}
	LZTimerOff();
	count1 = LZTimerCount();
	getch();

	MGL_clearDevice();

	MGL_beginDrawing();
	LZTimerOn();
	for (j = 0; j < LOOPS; j++) {
		MGL_setColor(MGL_realColor(j+1));
		for (i = 0; i < MAXNUM; i++)
			MGL_lineFast(p[i],p[i+1]);
		}
	LZTimerOff();
	MGL_endDrawing();
	count2 = LZTimerCount();

	exit_graphics();
	printf("Normal line drawing:   ");	ReportTime(count1);
	printf("Fast line drawing:     ");	ReportTime(count2);
}

#endif

#ifdef	TEST6

/* TEST6	- Horizontal/Vertical line drawing test
 *
 * This routine tests the drawing of horizontal and vertical lines, which
 * are faster to draw than general lines.
 */

void cdecl main(int argc,char *argv[])
{
	int		i,j,maxx,maxy;

	init_graphics("Horizontal/Vertical line test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	LZTimerOn();
	for (j = 1; j < 20; j++) {
		MGL_setColor(MGL_realColor(j & 15));
		for (i = 0; i <= maxy; i += 2) {
			MGL_lineCoord(0,i,maxx,i);
			MGL_lineCoord(maxx,i+1,0,i+1);
			}
		}
	LZTimerOff();
	count1 = LZTimerCount();

	getch();

	LZTimerOn();
	for (j = 1; j < 20; j++) {
		MGL_setColor(MGL_realColor(j & 15));
		for (i = 0; i <= maxx; i += 2) {
			MGL_lineCoord(i,0,i,maxy);
			MGL_lineCoord(i+1,maxy,i+1,0);
			}
		}
	LZTimerOff();
	count2 = LZTimerCount();

	exit_graphics();
	printf("Horizontal lines: ");	ReportTime(count1);
	printf("Vertical lines:   ");	ReportTime(count2);
}

#endif

#ifdef	TEST7

/* TEST7	- Filled rectangle test.
 *
 * This routine tests and times the drawing of filled rectangles.
 */

void cdecl main(int argc,char *argv[])
{
	int		j,maxx,maxy;

	init_graphics("Filled rectangle test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	MGL_setBackColor(MGL_realColor(MAGENTA));
	MGL_clearDevice();
	MGL_setBackColor(MGL_realColor(BLUE));

	LZTimerOn();
	for (j = 1; j < 100; j++) {
		MGL_setColor(MGL_realColor(j & 15));
		MGL_fillRectCoord(5,5,maxx-3,maxy-3);
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST8

/* TEST8	- Patterned drawing test.
 *
 * This routine draws a number of pre-defined bitmaps on the display screen.
 */

#include "bitmaps.c"

void cdecl main(int argc,char *argv[])
{
	int		i;
	char	buf[20];

	init_graphics("Bitmap pattern test",argc,argv);

	MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);

	MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
	for (i = 0; i < sizeof(bitpat) / 8; i++) {
		MGL_setPenBitmapPattern(&bitpat[i]);
		MGL_setColor(7);
		MGL_fillRectCoord(0,0,MGL_maxx()+1,MGL_maxy()+1);
		sprintf(buf,"Bitmap number %d",i);
		MGL_setColor(WHITE);
		MGL_drawStrXY(MGL_maxx()/2,MGL_maxy()/2,buf);
		getch();
		}

	exit_graphics();
}

#endif

#ifdef	TEST9

/* TEST9	- Polyline drawing test.
 *
 * Tests the polyLine, polyPoint and polyMarker routines.
 */

void cdecl main(int argc,char *argv[])
{
	point	vArray[4];

	init_graphics("Polyline test",argc,argv);

	MGL_setColor(GREEN);

	vArray[0].x = 20;	vArray[0].y = 175;
	vArray[1].x = 620;	vArray[1].y = 20;
	vArray[2].x = 590;	vArray[2].y = 330;
	vArray[3].x = 320;	vArray[3].y = 10;
	MGL_polyPoint(4,vArray);
	getch();
	MGL_setColor(RED);
	MGL_polyLine(4,vArray);
	MGL_setColor(BLUE);
	MGL_polyMarker(4,vArray);

	exit_graphics();
}

#endif

#ifdef	TEST10

/* TEST10	- Circle drawing speed test.
 *
 * This routine draws a number of random ellipses as fast as possible,
 * timing the result.
 */

#define	MAXNUM	1000
#define	LOOPS	5

rect	r[MAXNUM];

void cdecl main(int argc,char *argv[])
{
	int		i,j,maxx,maxy;

	init_graphics("Ellipse speed test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(1000);
	for (i = 0; i < MAXNUM; i++) {
		r[i].left = random(maxx-100);
		r[i].top = random(maxy-100);
		r[i].right = r[i].left + random(100);
		r[i].bottom = r[i].top + random(100);
		}

	LZTimerOn();
	for (j = 0; j < LOOPS; j++) {
		MGL_setColor(MGL_realColor(j+1));
		for (i = 0; i < MAXNUM; i++) {
			MGL_ellipse(r[i]);
			}
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST11

/* TEST11	- Filled circle drawing speed test.
 *
 * This routine draws a number of random filled ellipses as fast as
 * possible, timing the result.
 */

#define	MAXNUM	1000
#define	LOOPS	5

rect	r[MAXNUM];

void cdecl main(int argc,char *argv[])
{
	int		i,j,maxx,maxy,maxcolor;

	init_graphics("Filled ellipse speed test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();
	maxcolor = MGL_maxColor();

	srand(1000);
	for (i = 0; i < MAXNUM; i++) {
		r[i].left = random(maxx-100);
		r[i].top = random(maxy-100);
		r[i].right = r[i].left + random(100);
		r[i].bottom = r[i].top + random(100);
		}

	LZTimerOn();
	for (j = 0; j < LOOPS; j++) {
		for (i = 0; i < MAXNUM; i++) {
			MGL_setColor(i % maxcolor + 1);
			MGL_fillEllipse(r[i]);
			}
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST12

/* TEST12	- Elliptical arc speed test.
 *
 * This routine draws a number of random elliptical arcs as fast as
 * possible, timing the result.
 */

#define	MAXNUM	1000
#define	LOOPS	5

rect	r[MAXNUM];
int		startAng[MAXNUM];
int		endAng[MAXNUM];

void cdecl main(int argc,char *argv[])
{
	int		i,j,maxx,maxy;

	init_graphics("Elliptical arc speed test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(100);
	for (i = 0; i < MAXNUM; i++) {
		r[i].left = random(maxx-100);
		r[i].top = random(maxy-100);
		r[i].right = r[i].left + random(100);
		r[i].bottom = r[i].top + random(100);
		startAng[i] = random(360);
		endAng[i] = random(360);
		}

	LZTimerOn();
	for (j = 0; j < LOOPS; j++) {
		MGL_setColor(MGL_realColor(j+1));
		for (i = 0; i < MAXNUM; i++)
			MGL_ellipseArc(r[i],startAng[i],endAng[i]);
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST13

/* TEST13	- Filled elliptical arc speed test.
*
* This routine draws a number of random elliptical arcs as fast as
* possible, timing the result.
*/

#define	MAXNUM	1000
#define	LOOPS	5

rect	r[MAXNUM];
int		startAng[MAXNUM];
int		endAng[MAXNUM];

void cdecl main(int argc,char *argv[])
{
   int		i,j,maxx,maxy,maxcolor;

   init_graphics("Filled elliptical arc speed test",argc,argv);

   maxx = MGL_maxx();
   maxy = MGL_maxy();
   maxcolor = MGL_maxColor();

   srand(100);
   for (i = 0; i < MAXNUM; i++) {
	   r[i].left = random(maxx-100);
	   r[i].top = random(maxy-100);
	   r[i].right = r[i].left + random(100);
	   r[i].bottom = r[i].top + random(100);
	   startAng[i] = random(360);
	   endAng[i] = random(360);
	   }

   LZTimerOn();
   for (j = 0; j < LOOPS; j++) {
	   for (i = 0; i < MAXNUM; i++) {
		   MGL_setColor(i % maxcolor + 1);
		   MGL_fillEllipseArc(r[i],startAng[i],endAng[i]);
		   }
	   }
   LZTimerOff();

   exit_graphics();
   ReportTime(LZTimerCount());
}

#endif

#ifdef TEST14

/* TEST14	- Palette programming test routine.
 *
 * This routine draws a number of colors rectangles on the display, and
 * tests the palette programming routines.
 */

void cdecl main(int argc,char *argv[])
{
	int		color,maxcolor,width,height,x,y,i,j,palsize;
	palette	pal[256],temppal[256];
	palette	oldpal[256];

	init_graphics("Color demonstration",argc,argv);

	maxcolor = MGL_maxColor();
	palsize = MGL_getPaletteSize();

	if (maxcolor <= 15) {
		/* Simple color demonstration for 16 color displays */

		width = 2 * ((MGL_maxx()+1) / 16);
		height = 2 * ((MGL_maxy()-10)  / 10);

		x = width / 2;
		y = height / 2;
		color = 1;
		for (j = 0; j < 3; j++) {
			for (i = 0; i < 5; i++) {
				MGL_setColor(color++);
				MGL_fillRectCoord(x,y,x+width,y+height);
				x += (width/2) * 3;
				}
			y += (height / 2) * 3;
			x = width / 2;
			}
		}
	else {
		/* Color demonstration for large numbers of colors */

		width = 2 * ((MGL_maxx()+1) / 46);
		height = 2 * ((MGL_maxy()-10)  / 47);

		x = width / 2;
		y = height / 2;
		color = 1;
		for (j = 0; j < 16; j++) {
			for (i = 0; i < 16; i++) {
				MGL_setColor(color++);
				MGL_fillRectCoord(x,y,x+width,y+height);
				x += (width/2) * 3;
				}
			y += (height / 2) * 3;
			x = width / 2;
			}
		}

	getch();

	MGL_getPalette(oldpal,palsize,0);
	MGL_getPalette(pal,palsize,0);

	pal[0].red = 0;		pal[0].green = 0;	pal[0].blue = 0;	// Black
	pal[1].red = 255;	pal[1].green = 255;	pal[1].blue = 255;	// White
	pal[2].red = 204;	pal[2].green = 204;	pal[2].blue = 204;	// LightGray
	pal[3].red = 128;	pal[3].green = 128;	pal[3].blue = 128;	// DarkGray
	pal[4].red = 0;		pal[4].green = 0;	pal[4].blue = 128;	// Blue
	pal[5].red = 0;		pal[5].green = 128;	pal[5].blue = 128;	// Jade
	pal[6].red = 255;	pal[6].green = 255;	pal[6].blue = 128;	// Yellow
	pal[7].red = 255;	pal[7].green = 255;	pal[7].blue = 204;	// LightYellow

	MGL_setPalette(pal,palsize,0);
/*
	maxcolor++;
	for (i = 0; i < maxcolor; i++) {
		pal[i].red = i * ((PALMAX+1) / maxcolor);
		pal[i].green = 0;
		pal[i].blue = 0;
		MGL_setPaletteEntry(i,pal[i].red,pal[i].green,pal[i].blue);
		}

	getch();
	for (i = 0; i < maxcolor; i++) {
		pal[i].red = 0;
		pal[i].green = i * ((PALMAX+1) / maxcolor);
		pal[i].blue = 0;
		}
	MGL_setPalette(pal,palsize,0);

	getch();
	for (i = 0; i < maxcolor; i++) {
		pal[i].red = 0;
		pal[i].green = 0;
		pal[i].blue = i * ((PALMAX+1) / maxcolor);
		}
	MGL_setPalette(pal,palsize,0);

	getch();
	for (i = 0; i < maxcolor; i++) {
		pal[i].red = (i * ((PALMAX+1) / maxcolor)) / 2;
		pal[i].green = (i * ((PALMAX+1) / maxcolor)) / 2;
		pal[i].blue = i * ((PALMAX+1) / maxcolor);
		}
	MGL_setPalette(pal,palsize,0);
	getch();

	MGL_setPalette(oldpal,palsize,0);
	MGL_getPalette(pal,palsize,0);

	if (MGL_maxColor() > 15) {
		// Palette rotations

		while (!kbhit()) {
			MGL_rotatePalette(pal+1,palsize-1,PAL_ROTATE_UP);
			MGL_setPalette(pal+1,palsize-1,1);
			}
		getch();

		while (!kbhit()) {
			MGL_rotatePalette(pal+1,palsize-1,PAL_ROTATE_DOWN);
			MGL_setPalette(pal+1,palsize-1,1);
			}
		getch();

		// Palette fade out

		temppal[0] = pal[0];
		for (i = 63; i >= 0; i--) {
			MGL_fadePalette(temppal+1,pal+1,palsize-1,i*4);
			MGL_setPalette(temppal+1,palsize-1,1);
			}

		for (i = 0; i <= 63; i++) {
			MGL_fadePalette(temppal+1,pal+1,palsize-1,i*4);
			MGL_setPalette(temppal+1,palsize-1,1);
			}
		getch();
		}

	for (i = 0; i < 200; i++)
		MGL_setPalette(oldpal,palsize,0);
*/
	exit_graphics();
}

#endif

#ifdef	TEST15

/* TEST15	- Polygon drawing speed test.
 *
 * This routine tests the speed of the polygon drawing routines. It tests
 * both the complex and convex polygon drawing routines.
 */

#define MaxPts		6		/* Maximum # of pts in polygon	*/
#define	NUM			40

typedef	struct {
	point	poly[MaxPts];
	int		color;
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	color_t maxcolor;
	int		maxx,maxy,backcolor;
	point	tri1[3];
	int 	i,j;
	rect	clip,view;

	init_graphics("Polygon speed test",argc,argv);

	maxcolor = MGL_maxColor();
	maxx = MGL_maxx();
	maxy = MGL_maxy();

	MGL_getClipRect(&clip);
	srand(100);
#if 0
	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + randoml(maxcolor-1);

		for (i = 0; i < (MaxPts-1); i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	MGL_setPolygonType(COMPLEX_POLYGON);
	LZTimerOn();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}
	LZTimerOff();
	count1 = LZTimerCount();
	getch();
#endif
//	MGL_setActivePage(1);
//	MGL_setVisualPage(1);

/*
	backcolor = MGL_getBackColor();
	MGL_setBackColor(BLACK);
	MGL_clearDevice();
	MGL_setBackColor(backcolor);
	MGL_getViewport(&view);
	MGL_setClipRect(view);
	MGL_setColor(1);
	MGL_insetRect(clip,-1,-1);
	MGL_rect(clip);
	MGL_insetRect(clip,1,1);
	MGL_setClipRect(clip);
*/

	/* Create the array of triangular convex polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + random(maxcolor-1);

		for (i = 0; i < 3; i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	for (i = 0; i < maxx/2; i++) {
		clip.left = i;
		clip.right = maxx-i;
		MGL_setClipRect(clip);
		MGL_setPolygonType(CONVEX_POLYGON);
		MGL_clearDevice();
		for (j = 0; j < NUM; j++) {
			MGL_setColor(table[j].color);
			MGL_fillPolygon(3,table[j].poly,0,0);
			}
		}
#if 0
	MGL_setPolygonType(CONVEX_POLYGON);
	LZTimerOn();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(3,table[j].poly,0,0);
		}
	LZTimerOff();
	count2 = LZTimerCount();
#endif
#if 0
	getch();
	backcolor = MGL_getBackColor();
	MGL_setBackColor(BLACK);
	MGL_clearDevice();
	MGL_setBackColor(backcolor);
	MGL_getViewport(&view);
	MGL_setClipRect(view);
	MGL_setColor(1);
	MGL_insetRect(clip,-1,-1);
	MGL_rect(clip);
	MGL_insetRect(clip,1,1);
	MGL_setClipRect(clip);

	/* Create a single 24x24 pixel triangle */

	tri1[0].x = 10;	tri1[0].y = 0;
	tri1[1].x = 1;	tri1[1].y = 24;
	tri1[2].x = 23;	tri1[2].y = 24;

	/* Now draw 10,000 of them. */

	MGL_beginDrawing();
	LZTimerOn();
	for (i = 0; i < 100; i++)
		for (j = 0; j < 100; j++) {
			MGL_setColor(((i+j)%15)+1);
			MGL_fillPolygonFast(3,tri1,i,j);
			}
	LZTimerOff();
	MGL_endDrawing();
	count3 = LZTimerCount();

	/* Create a single sliver triangles */

	tri1[0].x = 10;	tri1[0].y = 0;
	tri1[1].x = 0;	tri1[1].y = 4;
	tri1[2].x = 5;	tri1[2].y = 4;

	/* Now draw 10,000 of them. */

	MGL_clearDevice();
	MGL_beginDrawing();
	LZTimerOn();
	for (i = 0; i < 100; i++)
		for (j = 0; j < 100; j++) {
			MGL_setColor(((i+j)%15)+1);
			MGL_fillPolygonFast(3,tri1,i,j);
			}
	LZTimerOff();
	MGL_endDrawing();
	count4 = LZTimerCount();
#endif
	exit_graphics();
	printf("Complex polygons with %d vertices:\n    ",MaxPts);
	ReportTime(count1);
	printf("Convex Triangular polygons:\n    ");
	ReportTime(count2);
	printf("10,000 24x24 triangular polygons:\n    ");
	ReportTime(count3);
	printf("    Speed:      %.0f polys/sec\n", (10000000000.0) / count3);
	printf("10,000 sliver triangular polygons:\n    ");
	ReportTime(count4);
	printf("    Speed:      %.0f polys/sec\n", (10000000000.0) / count4);
}

#endif

#ifdef	TEST16

/* TEST16	- Page flip animation test
 *
 * This routine tests the page flipping routines, but drawing polygons
 * into the off-screen image and then displaying it.
 */

#define MaxPts		6		/* Maximum # of pts in polygon	*/
#define	NUM			100

typedef	struct {
	point	poly[MaxPts];
	int		color;
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	int 	maxcolor,maxx,maxy;
	int 	i,j;
	int		vpage,apage,maxPage;

	init_graphics("Page flip animation test",argc,argv);

	maxcolor = MGL_maxColor();
	maxx = MGL_maxx();
	maxy = MGL_maxy();
	maxPage = MGL_maxPage() + 1;
	if (maxPage == 1) {
		MGL_exit();
		printf("Mode does not support page flipping...\n");
		exit(1);
		}

	srand(100);

	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + random(maxcolor-1);

		for (i = 0; i < (MaxPts-1); i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	vpage = 0;
	apage = 1;

	/* Make sure we initially clear the display, as the top few pages
	 * of SuperVGA's will not be cleared when the video mode is set.
	 */

	MGL_setActivePage(apage);
	MGL_clearDevice();

	LZTimerOn();
	for (j = 0; j < NUM; j++) {
		MGL_setActivePage(apage);
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		vpage = ++vpage % maxPage;
		MGL_setVisualPage(vpage);
		apage = ++apage % maxPage;
		MGL_copyPage(apage,vpage);
		}
	LZTimerOff();

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST17

/* TEST17	- Bit block transfer test.
 *
 * Tests the MGL_getImage and MGL_putImage bit block transfer routines.
 */

#define MaxPts		6		/* Maximum # of pts in polygon	*/
#define	NUM			10

typedef	struct {
	point	poly[MaxPts];
	int		color;
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	int		maxx,maxy;
	int 	i,j;
	long	size;
	rect	r;
	char	*image;

	init_graphics("MGL_getImage/MGL_putImage test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(100);

	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + random(15);

		for (i = 0; i < (MaxPts-1); i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	for (j = 0; j < NUM; j++) {
		MGL_setColor(MGL_realColor(table[j].color));
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	getch();

	r.left = 50;
	r.top = 50;
	r.right = maxx - 50;
	r.bottom = maxy/2 - 50;
	size = MGL_imageSize(r);

	image = farmalloc(size);

	MGL_getImage(r,image);
	MGL_setColor(MGL_realColor(BLACK));
	MGL_fillRectCoord(r.left-1,r.top-1,r.right+1,r.bottom+1);

	LZTimerOn();
	MGL_putImage(r,image,REPLACE_MODE);
	LZTimerOff();

	getch();

	LZTimerOn();
	MGL_putImage(r,image, XOR_MODE);
	LZTimerOff();
	getch();
	MGL_putImage(r,image, OR_MODE);
	getch();
	MGL_putImage(r,image, AND_MODE);

	getch();

	for (i = 0; i < 100; i += 2) {
		MGL_putImageCoord(r.left,i,r.right,r.bottom+i,image,REPLACE_MODE);
		}

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST18

/* TEST18	- Divot save/restore test.
 *
 * This routine tests the byte aligned divot save/restore routines.
 */

#define MaxPts		6		/* Maximum # of pts in polygon	*/
#define	NUM			10

typedef	struct {
	point	poly[MaxPts];
	int		color;
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	int 	maxx,maxy;
	int 	i,j;
	long	size;
	rect	r;
	void	far *divot;

	init_graphics("Divot save/restore test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(100);

	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + random(15);

		for (i = 0; i < (MaxPts-1); i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	for (j = 0; j < NUM; j++) {
		MGL_setColor(MGL_realColor(table[j].color));
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	r.left = 101;
	r.top = 100;
	r.right = r.left + 200;
	r.bottom = r.top + 100;
	size = MGL_divotSize(r);

	divot = malloc(size);

	LZTimerOn();
	MGL_getDivot(r,divot);
	LZTimerOff();
	count1 = LZTimerCount();
	MGL_setColor(MGL_realColor(BLACK));
	MGL_fillRectCoord(r.left-1,r.top-1,r.right+1,r.bottom+1);

	getch();

	MGL_setActivePage(1);
	MGL_setVisualPage(1);

	LZTimerOn();
	MGL_putDivot(divot);
	LZTimerOff();
	count2 = LZTimerCount();

	exit_graphics();
	printf("Save:    ");	ReportTime(count1);
	printf("Restore: ");	ReportTime(count2);
}

#endif

#ifdef	TEST19

/* TEST19	- Mouse interface test.
 *
 * This routines test the integrity of the mouse interface routines.
 */

#define MaxPts		6		/* Maximum # of pts in polygon	*/
#define	NUM			10

typedef	struct {
	point	poly[MaxPts];
	int		color;
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	int 		maxx,maxy;
	int 		i,j;
	color_t		oldColor;
	void		*state;
	ms_status	pos;

	_installMouse = true;
	init_graphics("Mouse interface test",argc,argv);
	MS_init();

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	srand(100);

	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + random(15);

		for (i = 0; i < (MaxPts-1); i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	for (j = 0; j < NUM; j++) {
		MGL_setColor(MGL_realColor(table[j].color));
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}
	MGL_setColor(MGL_realColor(BLUE));
	MGL_rectCoord(0,0,maxx+1,maxy+1);

	MGL_setActivePage(0);			/* Active page possibly different to
									 * visual page
									 */
	MS_show();
	MS_moveTo(maxx/4,3*maxy/4);
	getch();

	MS_pos(&pos);
	printf("(%d,%d) ",pos.x,pos.y);

	state = malloc(i = MS_stateSize());
	MS_saveState(state);
	MS_setCursor(&CROSS);
	getch();

	MS_restoreState(state);
	getch();

	oldColor = MGL_getCursorColor();
	MGL_setCursorColor(MGL_realColor(BLUE));
	MS_hide();	MS_show();
	getch();

	MGL_setCursorColor(oldColor);
	MS_hide();	MS_show();
	getch();

	exit_graphics();
	MS_exit();
}

#endif

#ifdef	TEST20

/* TEST20	- Text direction drawing test.
 *
 * This routine tests the text drawing routines for a specified font.
 */

void cdecl main(int argc,char *argv[])
{
	int 	maxx,maxy;
	int 	horiz,vert,err;
	char	*str,fontfile[80];
	font	far *font = NULL;

	char	buf[80];
	int		i;

	fontfile[0] = '\0';
	if (argc > 1 && argv[1][0] != '-') {
		strcpy(fontfile,argv[1]);
		argc--;
		argv++;
		}
	init_graphics("Text direction drawing test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	if (strlen(fontfile) != 0) {
		MGL_availableFont(fontfile);
		if ((err = MGL_result()) != grOK) {
			MGL_exit();
			printf("Graphics error: %s\n",MGL_errorMsg(err));
			exit(1);
			}
		font = MGL_loadFont(fontfile);
		}
	else
		font = DEFAULT_FONT;

	MGL_setTextSize(1,1,1,1);

	for (horiz = LEFT_TEXT; horiz <= RIGHT_TEXT; horiz++)
		for (vert = TOP_TEXT; vert <= BASELINE_TEXT; vert++) {
			MGL_clearDevice();
			MGL_setColor(RED);
			MGL_lineCoord(0,maxy/2,maxx,maxy/2);
			MGL_lineCoord(maxx/2,0,maxx/2,maxy);
			MGL_setTextJustify(LEFT_TEXT,TOP_TEXT);
			MGL_setTextDirection(RIGHT_DIR);
			MGL_useFont(DEFAULT_FONT);
			MGL_setColor(GREEN);
			MGL_moveToCoord(0,0);
			MGL_drawStr("horiz: ");
			switch (horiz) {
				case LEFT_TEXT:
					str = "LEFT";
					break;
				case CENTER_TEXT:
					str = "CENTER";
					break;
				case RIGHT_TEXT:
					str = "RIGHT";
				}
			MGL_drawStr(str);
			MGL_moveToCoord(0,MGL_textHeight());
			MGL_drawStr("vert:  ");
			switch (vert) {
				case TOP_TEXT:
					str = "TOP";
					break;
				case CENTER_TEXT:
					str = "CENTER";
					break;
				case BOTTOM_TEXT:
					str = "BOTTOM";
					break;
				case BASELINE_TEXT:
					str = "BASELINE";
					break;
				}
			MGL_drawStr(str);

			MGL_setTextJustify(horiz,vert);
			MGL_useFont(font);

			MGL_setColor(BLUE);
			MGL_lineCoord(0,maxy/2+MGL_textHeight()-1,maxx,maxy/2+MGL_textHeight()-1);
			MGL_lineCoord(maxx/2+MGL_textHeight()-1,0,maxx/2+MGL_textHeight()-1,maxy);

			MGL_setColor(WHITE);
			MGL_setTextDirection(LEFT_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes left");

			MGL_setTextDirection(DOWN_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes down");

			MGL_setTextDirection(UP_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes up");

			MGL_setTextDirection(RIGHT_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes right");

			getch();
			}

	MGL_clearDevice();

	MGL_useFont(font);
	MGL_setColor(RED);
	MGL_lineCoord(0,maxy/2,maxx,maxy/2);
	MGL_setColor(BLUE);
	MGL_lineCoord(0,maxy/2-font->ascent,maxx,maxy/2-font->ascent);
	MGL_lineCoord(0,maxy/2-font->descent,maxx,maxy/2-font->descent);
	MGL_setColor(GREEN);
	MGL_lineCoord(0,maxy/2-font->descent+font->leading,maxx,maxy/2-font->descent+font->leading);
	MGL_lineCoord(0,maxy/2+MGL_textHeight()-1,maxx,maxy/2+MGL_textHeight()-1);

	MGL_setTextJustify(CENTER_TEXT,TOP_TEXT);
	MGL_setTextDirection(RIGHT_DIR);
	MGL_setColor(WHITE);

	strcpy(buf,"ABCDefgj");
	for (i = 140; i < 150; i++)
		buf[i-132] = i;
	buf[i-132] = 0;
	MGL_drawStrXY(maxx/2,maxy/2,buf);

	if (strlen(fontfile) != 0)
		MGL_unloadFont(font);

	exit_graphics();
	ReportTime(count1);
}

#endif

#ifdef	TEST21

/* TEST21	- Font drawing test.
 *
 * This routine displays all of the characters in a specified font file.
 */

void cdecl main(int argc,char *argv[])
{
	int 	i,x,maxx,err;
	char	*str;
	font	far *font;
	uchar	buf[10];
	char	fontfile[80];
	rect	r;

	fontfile[0] = '\0';
	if (argc > 1 && argv[1][0] != '-') {
		strcpy(fontfile,argv[1]);
		argc--;
		argv++;
		}
	init_graphics("Font drawing test",argc,argv);

	maxx = MGL_maxx();

	if (strlen(fontfile) != 0) {
		MGL_availableFont(fontfile);
		if ((err = MGL_result()) != grOK) {
			MGL_exit();
			printf("Graphics error: %s\n",MGL_errorMsg(err));
			exit(1);
			}
		font = MGL_loadFont(fontfile);
		}
	else
		font = DEFAULT_FONT;

	MGL_useFont(font);
	MGL_setTextSize(1,1,1,1);

	buf[1] = '\0';
	str = (char*)buf;
	LZTimerOn();
	x = 1;
	MGL_moveToCoord(0,0);
	for (i = 1; i < 256; i++) {
		buf[0] = i;
		MGL_drawStr(str);
		if (MGL_getX() + MGL_maxCharWidth() > maxx)
			MGL_moveToCoord(x++,MGL_getY() + MGL_textHeight());
		}
	LZTimerOff();

	if (smallclip) {
		r.left = (MGL_maxx()+1)/4;
		r.right = r.left * 3;
		r.top = (MGL_maxy()+1)/4;
		r.bottom = r.top * 3;
		MGL_setColor(1);
		MGL_insetRect(r,-1,-1);
		MGL_rect(r);
		MGL_insetRect(r,1,1);
		MGL_setColor(YELLOW);
		MGL_setClipRect(r);
		MGL_setClipMode(CLIPON);

		x = 1;
		MGL_moveToCoord(0,0);
		for (i = 1; i < 256; i++) {
			buf[0] = i;
			MGL_drawStr(str);
			if (MGL_getX() + MGL_maxCharWidth() > maxx)
				MGL_moveToCoord(x++,MGL_getY() + MGL_textHeight());
			}
		}

	if (strlen(fontfile) != 0)
		MGL_unloadFont(font);

	exit_graphics();
	ReportTime(LZTimerCount());
}

#endif

#ifdef	TEST22

/* TEST22	- Border drawing tests.
 *
 * This routine tests the border drawing routines, which allow you to
 * draw Window and OS/2 2.0 style borders and buttons.
 */

#define	THICKNESS	2

void cdecl main(int argc,char *argv[])
{
	int 	maxx,maxy;
	rect	r;

	_installMouse = true;
	init_graphics("Border drawing tests",argc,argv);
	MS_init();

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	MGL_setBackColor(MGL_realColor(LIGHTGRAY));
	MGL_clearDevice();
	MGL_setColor(MGL_realColor(RED));

	r.left = 10;
	r.top = 10;
	r.right = maxx/2 - 10;
	r.bottom = maxy/2 - 10;
	MGL_drawBorder(r,BDR_INSET,THICKNESS);

	r.left = maxx/2 + 10;
	r.top = 10;
	r.right = maxx - 10;
	r.bottom = maxy/2 - 10;
	MGL_drawBorder(r,BDR_OUTSET,THICKNESS);

	r.left = 10;
	r.top = maxy/2 + 10;
	r.right = maxx/2 - 10;
	r.bottom = maxy - 10;
	MGL_drawBorder(r,BDR_OUTLINE,THICKNESS);

	r.left = maxx/2 + 10;
	r.top = maxy/2 + 10;
	r.right = maxx - 10;
	r.bottom = maxy - 10;
	MGL_drawBorder(r,BDR_OUTLINE,THICKNESS);

	MGL_drawVDivider((r.right+r.left)/2,r.top+3,(r.top+r.bottom)/2 - 2);
	MGL_drawHDivider((r.top+r.bottom)/2,r.left+3,r.right - 4);

	MS_show();

	exit_graphics();
	MS_exit();
}

#endif

#ifdef	TEST23

/* TEST23	- Gouraud shaded polygon drawing speed test.
 *
 * This routine tests the speed of the shaded polygon drawing routines.
 */

#define MaxPts		3		/* Maximum # of pts in polygon	*/
#define	NUM			100

typedef	struct {
	point	poly[MaxPts];
	color_t	colors[MaxPts];
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	int 	maxcolor,maxx,maxy,backcolor;
	int 	i,j;
	rect	view,clip;
	point	tri1[3];
	color_t	colors[3];
	palette	pal[256];
	char	buf[40];

	init_graphics("Gouraud shaded polygon speed test",argc,argv);

	maxcolor = MGL_maxColor();
	maxx = MGL_maxx();
	maxy = MGL_maxy();
	MGL_getClipRect(&clip);

/*	if (maxcolor < 255) {
		MGL_exit();
		printf("Gouraud shading not supported in this video mode\n");
		exit(1);
		}*/

	/* Setup the palette to show smooth shading between bright red and
	 * bright blue. Note that we leave color 0 as black.
	 */

	pal[0].red = pal[0].green = pal[0].blue = 0;
	for (i = 1; i <= maxcolor; i++) {
		pal[i].red = ((long)PALMAX * i) / maxcolor;
		pal[i].green = 0;
		pal[i].blue = ((long)PALMAX * (maxcolor - i)) / maxcolor;
		}
	MGL_setPalette(pal,MGL_getPaletteSize(),0);

	srand(100);

	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		for (i = 0; i < MaxPts; i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			table[j].colors[i] = 1 + random(maxcolor-1);
			}
		}

	LZTimerOn();
	for (j = 4; j < NUM; j++) {
		MGL_fillGouraudPolygon(MaxPts,table[j].poly,table[j].colors,0,0);

		// Label the vertice for the polygon

		MGL_setTextJustify(CENTER_TEXT,CENTER_TEXT);
		for (i = 0; i < MaxPts; i++) {
			sprintf(buf,"%d",table[j].colors[i]);
			MGL_drawStrXY(table[j].poly[i].x,table[j].poly[i].y,buf);
			}
		getch();
		}
	LZTimerOff();
	count1 = LZTimerCount();
	getch();

	backcolor = MGL_getBackColor();
	MGL_setBackColor(BLACK);
	MGL_clearDevice();
	MGL_setBackColor(backcolor);
	MGL_getViewport(&view);
	MGL_setClipRect(view);
	MGL_setColor(1);
	MGL_insetRect(clip,-1,-1);
	MGL_rect(clip);
	MGL_insetRect(clip,1,1);
	MGL_setClipRect(clip);

	/* Create a single 24x24 pixel triangle */

	tri1[0].x = 10;	tri1[0].y = 0;		colors[0] = 1;
	tri1[1].x = 1;	tri1[1].y = 24;		colors[1] = maxcolor;
	tri1[2].x = 23;	tri1[2].y = 24;		colors[2] = maxcolor;

	/* Now draw 49,300 of them. */

	MGL_beginDrawing();
	LZTimerOn();
	for (i = 0; i < 290; i++)
		for (j = 0; j < 170; j++) {
			colors[0] = ((i+j)%15)+1;
			MGL_fillGouraudPolygon(3,tri1,colors,i,j);
			}
	LZTimerOff();
	MGL_endDrawing();
	count2 = LZTimerCount();

	exit_graphics();
	printf("Shaded polygons:\n    ");
	ReportTime(count1);
	printf("49,300 24x24 polygons:\n    ");
	ReportTime(count2);
}

#endif

#ifdef	TEST24

/* TEST24	- Bit Block copy test
 *
 * Tests the MGL_copyImage routine for scrolling the display.
 */

#define MaxPts		6		/* Maximum # of pts in polygon	*/
#define	NUM			10

typedef	struct {
	point	poly[MaxPts];
	int		color;
	} entry;

entry	table[NUM];

void cdecl main(int argc,char *argv[])
{
	int 	maxcolor,maxx,maxy,maxpage;
	int 	i,j,page;
	rect	r;

	init_graphics("MGL_copyImage test",argc,argv);

	maxcolor = MGL_maxColor();
	maxx = MGL_maxx();
	maxy = MGL_maxy();
	maxpage = MGL_maxPage();
	page = MGL_getActivePage();

	srand(100);

	/* Create the array of polygons */

	for (j = 0; j < NUM; j++) {
		table[j].color = 1 + random(maxcolor-1);

		for (i = 0; i < (MaxPts-1); i++) {
			table[j].poly[i].x = random(maxx);
			table[j].poly[i].y = random(maxy);
			}
		table[j].poly[i] = table[j].poly[0];
		}

	/* Scroll upwards */

	MGL_clearDevice();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	r.left = 0;
	r.top = 5;
	r.right = maxx+1;
	r.bottom = maxy+1;

	getch();
	LZTimerOn();
	for (i = 0; i < maxy; i += 8)
		MGL_copyImage(r,0,0,page,page);
	LZTimerOff();
	count1 = LZTimerCount();

	getch();

	/* Scroll downwards */

	MGL_clearDevice();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	r.left = 0;
	r.top = 0;
	r.right = maxx+1;
	r.bottom = maxy-4;

	getch();
	LZTimerOn();
	for (i = 0; i < maxy; i += 8)
		MGL_copyImage(r,0,5,page,page);
	LZTimerOff();
	count2 = LZTimerCount();

	getch();

	/* Scroll right */

	MGL_clearDevice();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	r.left = 0;
	r.top = 0;
	r.right = maxx-3;
	r.bottom = maxy+1;

	getch();
	LZTimerOn();
	for (i = 0; i < maxx; i += 8)
		MGL_copyImage(r,4,0,page,page);
	LZTimerOff();
	count3 = LZTimerCount();

	getch();

	/* Scroll left */

	MGL_clearDevice();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	r.left = 4;
	r.top = 0;
	r.right = maxx+1;
	r.bottom = maxy+1;

	getch();
	LZTimerOn();
	for (i = 0; i < maxx; i += 8)
		MGL_copyImage(r,0,0,page,page);
	LZTimerOff();
	count4 = LZTimerCount();

	getch();

	/* Scroll right and down */

	MGL_clearDevice();
	for (j = 0; j < NUM; j++) {
		MGL_setColor(table[j].color);
		MGL_fillPolygon(MaxPts,table[j].poly,0,0);
		}

	r.left = 0;
	r.top = 0;
	r.right = maxx-4;
	r.bottom = maxy-4;
	for (i = 0; i < maxy; i += 8)
		MGL_copyImage(r,5,5,page,page);

	/* Copy between video pages */

	if (maxpage > 0) {
		MGL_setActivePage(1);
		MGL_setVisualPage(1);
		MGL_clearDevice();
		for (j = 0; j < NUM; j++) {
			MGL_setColor(table[j].color);
			MGL_fillPolygon(MaxPts,table[j].poly,0,0);
			}

		getch();
		MGL_setActivePage(0);
		MGL_setVisualPage(0);
		MGL_clearDevice();
		getch();
		MGL_copyImageCoord(0,0,
						   maxx+1,maxy+1,
						   0,0,
						   1,0);
		getch();
		MGL_setActivePage(1);
		MGL_setVisualPage(1);
		MGL_clearDevice();
		getch();
		MGL_copyImageCoord(maxx/2-100,0,
						   maxx/2+100,maxy+1,
						   maxx/2-100,0,
						   0,1);
		}

	exit_graphics();
	printf("Up:    ");	ReportTime(count1);
	printf("Down:  ");	ReportTime(count2);
	printf("Right: ");	ReportTime(count3);
	printf("Left:  ");	ReportTime(count4);
}

#endif

#ifdef	TEST25

/* TEST25	- Text underline test
 *
 * This routine tests the text underline calculation routines.
 */

void cdecl main(int argc,char *argv[])
{
	int 	maxx,maxy,x,y;
	int 	horiz,vert,err;
	char	*str,fontfile[80];
	font	far *font = NULL;

	fontfile[0] = '\0';
	if (argc > 1 && argv[1][0] != '-') {
		strcpy(fontfile,argv[1]);
		argc--;
		argv++;
		}

	init_graphics("Text direction drawing test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();

	if (strlen(fontfile) != 0) {
		MGL_availableFont(fontfile);
		if ((err = MGL_result()) != grOK) {
			MGL_exit();
			printf("Graphics error: %s\n",MGL_errorMsg(err));
			exit(1);
			}
		font = MGL_loadFont(fontfile);
		}
	else
		font = DEFAULT_FONT;

	MGL_setTextSize(1,1,1,1);

	for (horiz = LEFT_TEXT; horiz <= RIGHT_TEXT; horiz++)
		for (vert = TOP_TEXT; vert <= BASELINE_TEXT; vert++) {
			MGL_clearDevice();
			MGL_setColor(RED);
			MGL_lineCoord(0,maxy/2,maxx,maxy/2);
			MGL_lineCoord(maxx/2,0,maxx/2,maxy);
			MGL_setTextJustify(LEFT_TEXT,TOP_TEXT);
			MGL_setTextDirection(RIGHT_DIR);
			MGL_useFont(DEFAULT_FONT);
			MGL_setColor(GREEN);
			MGL_moveToCoord(0,0);
			MGL_drawStr("horiz: ");
			switch (horiz) {
				case LEFT_TEXT:
					str = "LEFT";
					break;
				case CENTER_TEXT:
					str = "CENTER";
					break;
				case RIGHT_TEXT:
					str = "RIGHT";
				}
			MGL_drawStr(str);
			MGL_moveToCoord(0,MGL_textHeight());
			MGL_drawStr("vert:  ");
			switch (vert) {
				case TOP_TEXT:
					str = "TOP";
					break;
				case CENTER_TEXT:
					str = "CENTER";
					break;
				case BOTTOM_TEXT:
					str = "BOTTOM";
				}
			MGL_drawStr(str);

			MGL_setTextJustify(horiz,vert);
			MGL_useFont(font);

			MGL_setColor(WHITE);
			MGL_setTextDirection(LEFT_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes left");
			x = maxx/2; y = maxy/2;
			MGL_underScoreLocation(&x,&y,"This text goes left");
			MGL_setColor(BLUE);
			MGL_lineCoord(x,y,x-MGL_textWidth("This text goes left"),y);

			MGL_setColor(WHITE);
			MGL_setTextDirection(DOWN_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes down");
			x = maxx/2; y = maxy/2;
			MGL_underScoreLocation(&x,&y,"This text goes down");
			MGL_setColor(BLUE);
			MGL_lineCoord(x,y,x,y+MGL_textWidth("This text goes down"));

			MGL_setColor(WHITE);
			MGL_setTextDirection(UP_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes up");
			x = maxx/2; y = maxy/2;
			MGL_underScoreLocation(&x,&y,"This text goes up");
			MGL_setColor(BLUE);
			MGL_lineCoord(x,y,x,y-MGL_textWidth("This text goes up"));

			MGL_setColor(WHITE);
			MGL_setTextDirection(RIGHT_DIR);
			MGL_drawStrXY(maxx/2,maxy/2,"This text goes right");
			x = maxx/2; y = maxy/2;
			MGL_underScoreLocation(&x,&y,"This text goes right");
			MGL_setColor(BLUE);
			MGL_lineCoord(x,y,x+MGL_textWidth("This text goes right"),y);

			getch();
			}

	if (strlen(fontfile) != 0)
		MGL_unloadFont(font);

	exit_graphics();
	ReportTime(count1);
}

#endif

#ifdef	TEST26

/* TEST26	- Monochrome image drawing routines
 *
 * Tests the MGL_putMonoImage routine for drawing monochrome bitmaps.
 */

uchar image[] = {
	0xFF, 0xDE, 0x7B, 0xFE,		/* 11111111 11011110 01111011 11111110	*/
	0xFF, 0xCC, 0x31, 0xFE,		/* 11111111 11001100 00110001 11111110	*/
	0x8C, 0x4C, 0x31, 0x82,		/* 10001100 01001100 00110001 10000010	*/
	0x0C, 0x0C, 0x31, 0x80,		/* 00001100 00001100 00110001 10000000	*/
	0x0C, 0x0C, 0x31, 0x90,		/* 00001100 00001100 00110001 10010000	*/
	0x0C, 0x0F, 0xF1, 0xF0,		/* 00001100 00001111 11110001 11110000	*/
	0x0C, 0x0F, 0xF1, 0xF0,		/* 00001100 00001111 11110001 11110000	*/
	0x0C, 0x0C, 0x31, 0x90,		/* 00001100 00001100 00110001 10010000	*/
	0x0C, 0x0C, 0x31, 0x80,		/* 00001100 00001100 00110001 10000000	*/
	0x0C, 0x0C, 0x31, 0x82,		/* 00001100 00001100 00110001 10000010	*/
	0x0C, 0x0C, 0x31, 0xFE,		/* 00001100 00001100 00110001 11111110	*/
	0x1E, 0x1E, 0x7B, 0xFE,		/* 00011110 00011110 01111011 11111110	*/
	};

void cdecl main(int argc,char *argv[])
{
	int		maxx,maxy,x,y;
	color_t	maxcolor;

	init_graphics("MGL_putMonoImage test",argc,argv);

	maxx = MGL_maxx();
	maxy = MGL_maxy();
	maxcolor = MGL_maxColor();

	for (x = 0; x < maxx; x += 40)
		for (y = 0; y < maxy; y += sizeof(image)/4 + 2) {
			MGL_setColor((x + y) % (maxcolor+1));
			MGL_putMonoImage(x,y,4,sizeof(image)/4,image);
			}

	exit_graphics();
}

#endif
