/****************************************************************************
*
*						  Protected Mode Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: pmode.c $
* Version:		$Revision: 1.1 $
*
* Language:		ANSI C
* Environment:	IBM PC (MSDOS) Real mode and 16/32 bit Protected Mode
*
* Description:	Module implementing DOS extender independant protected
*				mode programming. This module will need to be included in
*				all programs that use SciTech Software's products that
*				are to be compiled in protected mode, and will need to be
*				compile with the correct defines for the DOS extender that
*				you will be using (or with no defines for real mode
*				emulation of these routines).
*
* $Id: pmode.c 1.1 1994/03/10 09:05:43 kjb release $
*
****************************************************************************/

#include <stdlib.h>
#include "pmode.h"

/*----------------------------- Implementation ----------------------------*/

/* Set the type target being compiled for so that we can do switching at
 * runtime as well as compile time.
 */

#if	defined(REALMODE)
int	_PMODE_modeType = PMODE_realMode;
#elif defined(PMODE286)
int _PMODE_modeType = PMODE_286;
#elif defined(PMODE386)
int _PMODE_modeType = PMODE_386;
#endif

static void *bios = NULL;

void * PMODE_getBIOSPointer(void)
{
	if (bios == NULL)
		bios = PMODE_mapLinearPointer(MK_PHYS(0x40, 0x00), 0xFFFF);
	return bios;
}

/*-------------------------------------------------------------------------*/
/* DOS Real Mode support.												   */
/*-------------------------------------------------------------------------*/

#ifdef REALMODE

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	/* Simply create a far pointer to the base address and ignore limit */
	int seg = base >> 4;
	int offset = base - (seg << 4);
	limit = limit;
    return MK_FP(seg, offset);
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* No need to do anything here */
	ptr = ptr;
}

void * PMODE_allocRealSeg(unsigned short size, unsigned long *segid,
	unsigned *segment, unsigned *offset)
{
	/* Call malloc() to allocate the memory for us */
	void *p = malloc(size);
	*segid = (unsigned long)p;
	*segment = FP_SEG(p);
	*offset = FP_OFF(p);
	return p;
}

void PMODE_freeRealSeg(unsigned long segid)
{
	free((void *)segid);
}

void * PMODE_createCSAlias(void *dataptr)
{
	return dataptr;
}

void PMODE_freeCSAlias(void * alias)
{
	alias = alias;
}

void far * PMODE_createDSAlias(void (*codeptr)())
{
    return (void far *)codeptr;
}

void PMODE_freeDSAlias(void far * alias)
{
	alias = alias;
}

int PMODE_int86(int intno, RMREGS *in, RMREGS *out)
{
	return int86(intno,in,out);
}

int PMODE_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	return int86x(intno,in,out,sregs);
}

int PMODE_installMouseHandler(int mask, void (cdecl far * handler)())
{
	union REGS		regs;
	struct SREGS	sregs;

	sregs.es = FP_SEG(handler);
	regs.x.dx = FP_OFF(handler);
	regs.x.cx = mask;
	regs.x.ax = 20;
	int86x(0x33, &regs, &regs, &sregs);
	return 1;
}

void far * PMODE_getISR(int intno, SAVEINT *save)
{
	save->real_isr = (unsigned long)_dos_getvect(intno);
	return (void far *)save->real_isr;
}

typedef void (interrupt far *intfp)();

void PMODE_restoreISR(int intno, SAVEINT *save)
{
	_dos_setvect(intno, (intfp)save->real_isr);
}

void PMODE_setISR(int intno, void far *isr)
{
	_dos_setvect(intno, (intfp)isr);
}

#endif

/*-------------------------------------------------------------------------*/
/* Generic DPMI support.												   */
/* UNTESTED																   */
/*-------------------------------------------------------------------------*/

#ifdef DPMI

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	int			sel;
	union REGS	r;

	/* Under DPMI the pointer that we allocate will have a maximum
	 * limit of 64k
	 */

	/* Allocate 1 descriptor */
	r.x.ax = 0;
	r.x.cx = 1;
	int86(0x31, &r, &r);
	if (r.x.cflag) return NULL;

	/* Set base address */
	sel = r.x.bx = r.x.ax;
	r.x.ax = 7;
	r.x.cx = base >> 16;
	r.x.dx = base & 0xFFFF;
	int86(0x31, &r, &r);
	if (r.x.cflag) return NULL;

	/* Set limit */
	r.x.ax = 8;
	r.x.cx = 0;
	r.x.dx = limit;
	int86(0x31, &r, &r);
	if (r.x.cflag) return NULL;

	return MK_FP(sel, 0);
}

void PMODE_freeLinearPointer(void *ptr)
{
	union REGS	r;

	r.x.ax = 1;
	r.x.bx = FP_SEG(ptr);
	int86(0x31, &r, &r);
}

/* This needs to be filled in with the remaining functions */

#endif

/*-------------------------------------------------------------------------*/
/* Phar Lap 286|DOS Extender support.									   */
/* No Interrupt support at this stage                                      */
/*-------------------------------------------------------------------------*/

#ifdef PHARLAP286

#include <phapi.h>

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	int seg,off;
	SEL	sel;

	/* Compute segment and offset of address */
	seg = base >> 4;
	off = base - (seg << 4);

	/* Make segment */
	if (DosMapRealSeg(seg, (ULONG)limit+off+1, &sel)) return NULL;
	return MK_FP(sel, off);
}

void PMODE_freeLinearPointer(void *ptr)
{
	DosFreeSeg(FP_SEG(ptr));
}

void * PMODE_allocRealSeg(unsigned short size, unsigned long *segid,
	unsigned *segment, unsigned *offset)
{
	SEL	sel;

	if (DosAllocRealSeg(size, (PUSHORT)segment, segid))
		return NULL;
	*offset = 0;
	return MK_FP(*segid, 0);
}

void PMODE_freeRealSeg(unsigned long segid)
{
	DosFreeSeg(segid);
}

/* The following CS alias routines do not seem to work with blocks of
 * memory that have been allocate with malloc(), but only with memory in
 * in the DGROUP data segment. Maybe there is an alternative method
 * of doing this with Phar Lap that will solve this problem?
 */

void * PMODE_createCSAlias(void *dataptr)
{
	SEL sel;

	if (DosCreateCSAlias(FP_SEG(dataptr), &sel))
		return NULL;
	return MK_FP(sel, FP_OFF(dataptr));
}

void PMODE_freeCSAlias(void *alias)
{
	DosFreeSeg(FP_SEG(alias));
}

void far * PMODE_createDSAlias(void (*codeptr)())
{
	SEL sel;

	if (DosCreateDSAlias(FP_SEG(codeptr), &sel))
		return NULL;
	return MK_FP(sel, FP_OFF(codeptr));
}

void PMODE_freeDSAlias(void far *alias)
{
	DosFreeSeg(FP_SEG(alias));
}

#define	IN(reg)		r.reg = in->x.reg
#define	OUT(reg)	out->x.reg = r.reg

int PMODE_int86(int intno, RMREGS *in, RMREGS *out)
{
	REGS16	r;

	r.flags = 0;
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	DosRealIntr(intno, &r, 0, 0);
	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	return r.ax;
}

int PMODE_int86x(int intno, RMREGS *in, RMREGS *out, RMSREGS *sregs)
{
	REGS16	r;

	r.cs = sregs->cs; r.ds = sregs->ds; r.es = sregs->es;
	r.flags = 0;
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	DosRealIntr(intno, &r, 0, 0);
	sregs->cs = r.cs; sregs->ds = r.ds; sregs->es = r.es;
	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	return r.ax;
}

#endif

/*-------------------------------------------------------------------------*/
/* Ergo DPM support.													   */
/* UNTESTED																   */
/*-------------------------------------------------------------------------*/

#ifdef ERGODPM

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	int	sel,seg,off;

	/* Compute segment */
	seg = base >> 4;
	off = base - (seg << 4);
	limit = limit;

	/* Make segment */
	ParaToSel(seg, &sel);

	/* Construct pointer */
	return MK_FP(sel, off);
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* Freeing selector from ParaToSel is unwise since they are a global
	 * resource
	 */
	ptr = ptr;
}

/* This needs to be filled in with the remaining functions */

#endif

/*-------------------------------------------------------------------------*/
/* Phar Lap 386|DOS Extender support.									   */
/* UNTESTED																   */
/*-------------------------------------------------------------------------*/

#ifdef PHARLAP386

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
#if 0
	union REGS 		r;
	struct SREGS 	s;

	/* Init segment registers */
	segread(&s);

	/* allocate 0 length segment */
	r.h.ah = 0x48;
	r.x.ebx = 0;
	int86(0x21, &r, &r);
	if (r.x.cflag) return NULL;

	/* Modify base/length of segment */
	s.es = r.x.ax;
	r.x.ax = 0x250A;
	r.x.ebx = base;						/* Base address */
	r.x.ecx = ((limit+1)+4095)/4096;	/* Map 4K pages */
	int86x(0x21, &r, &r, &s);
	if (r.x.cflag) return NULL;

	return MK_FP(s.es, r.x.eax);
#endif
	/* This needs to be re-written to return a near pointer */
	return NULL;
}

void PMODE_freeLinearPointer(void *ptr)
{
#if 0
	union REGS 		r;
	struct SREGS 	s;

	segread(&s);
	s.es = FP_SEG(ptr);
	r.h.ah = 0x49;
	int86x(0x21, &r, &r, &s);
#endif
}

/* This needs to be filled in with the remaining functions */

#endif

/*-------------------------------------------------------------------------*/
/* Symantec C++ DOSX support (also X32-VM support).						   */
/*-------------------------------------------------------------------------*/

#ifdef	DOSX

/* Maintain a near pointer to the low 1Mb memory area */

static char *lowmemory = NULL;

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	if (lowmemory == NULL)
		lowmemory = _x386_map_physical_address(0, 1024L*1024L);
	return (void*)(lowmemory + base);
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* No need to do anything here */
	ptr = ptr;
}

void * PMODE_allocRealSeg(unsigned short size, unsigned long *segid,
	unsigned *segment, unsigned *offset)
{
	union REGS	r;

	r.h.ah = 0x48;					/* DOS function 48h - allocate mem	*/
	r.x.bx = (size + 0xF) >> 4;		/* Number of paragraphs to allocate	*/
	int86(0x21, &r, &r);			/* Call DOS extender				*/
	if (r.x.cflag)
		return NULL;				/* Could not allocate the memory	*/
	*segment = r.x.ax;
	*offset = 0;
	*segid = -1;
    return (void*)r.e.ebx;
}

void PMODE_freeRealSeg(unsigned long segid)
{
	/* Cannot de-allocate this memory */
	segid = segid;
}

void * PMODE_createCSAlias(void *dataptr)
{
	return dataptr;
}

void PMODE_freeCSAlias(void * alias)
{
	alias = alias;
}

void far * PMODE_createDSAlias(void (*codeptr)())
{
	union REGS	r;
	unsigned	sel;

	r.x.ax = 0x3501;			/* Allocate protected mode selector		*/
	int86(0x21, &r, &r);
	if (r.x.cflag) return NULL;	/* No selectors left! */
	sel = r.x.bx;

	r.x.ax = 0x3504;			/* Get Selector base address			*/
	r.x.bx = FP_SEG(codeptr);	/* of current code segment				*/
	int86(0x21, &r, &r);
	if (r.x.cflag) return NULL;

	r.x.ax = 0x3503;			/* Set Selector base address			*/
	r.x.bx = sel;
	int86(0x21, &r, &r);
	if (r.x.cflag) return NULL;

	r.x.ax = 0x3505;			/* Set Selector limit					*/
	r.x.bx = sel;
	r.e.ecx = 0xFFFFFFFF;
	int86(0x21, &r, &r);
	if (r.x.cflag) return NULL;

	return MK_FP(sel, FP_OFF(codeptr));
}

void PMODE_freeDSAlias(void far *alias)
{
	union REGS	r;

	r.x.ax = 0x3502;			/* De-allocate protected mode selector	*/
	r.x.bx = FP_SEG(alias);
	int86(0x21, &r, &r);
}

int PMODE_int86(int intno, RMREGS *in, RMREGS *out)
{
	return int86_real(intno,in,out);
}

int PMODE_int86x(int intno, RMREGS *in, RMREGS *out, RMSREGS *sregs)
{
    return int86x_real(intno,in,out,sregs);
}

#include <msmouse.h>

static void (cdecl far *_handler)() = NULL;
static char _ms_stack[1024];

void _ms_handler(unsigned mask, unsigned state, unsigned curposx,
	unsigned curposy)
{
asm {	mov eax,mask;		/* Load params back into registers	*/
		mov	ebx,state;
		mov	ecx,curposx;
        mov edx,curposy; }
    _handler();             /* Call event handling routine      */
}

int PMODE_installMouseHandler(int mask, void (cdecl far * handler)())
{
    /* For some reason Symantec C++ does not seem to correctly set the
     * segment values for external assemly routines assembled with TASM.
     * Hence since we know the routine is in the same code segment, we
     * force the code segment value here.
     */
    _handler = MK_FP(FP_SEG(_ms_handler), (long)handler);
    msm_signal(0xFFFF, _ms_handler, &_ms_stack[1024]);
	return 1;
}

void far * PMODE_getISR(int intno, SAVEINT *save)
{
	union REGS      r;
	struct SREGS    s;

	segread(&s);
	r.x.ax = 0x2502;        /* Get pmode interrupt vector		*/
	r.x.cx = intno;
	int86x(0x21, &r, &r, &s);
	save->pmode_isr = MK_FP(s.es, r.e.ebx);

	r.x.ax = 0x2503;		/* Get real mode interrupt vector	*/
	r.x.cx = intno;
	int86(0x21, &r, &r);
	save->real_isr = r.e.ebx;

	return save->pmode_isr;
}

void PMODE_restoreISR(int intno, SAVEINT *save)
{
	union REGS		r;
	struct SREGS	s;

	segread(&s);
	r.x.ax = 0x2507;		/* Set real and pmode vectors		*/
	r.x.cx = intno;
	s.ds = FP_SEG(save->pmode_isr);
	r.e.edx = (long)save->pmode_isr;
	r.e.ebx = save->real_isr;
	int86x(0x21, &r, &r, &s);
}

void PMODE_setISR(int intno, void far *isr)
{
    union REGS      r;
    struct SREGS    s;

    /* Once again in the code below, if the pointer to the ISR comes from
     * some TASM'ed code it gets the segment value all wrong, so we force
     * it to be the code segment here. Could well be a bug in FP_SEG() or
     * something :-(
     */

    segread(&s);
    r.x.ax = 0x2506;        /* Hook real and protected vectors  */
    r.x.cx = intno;
    s.ds = FP_SEG(_ms_handler);
    r.e.edx = (long)isr;
    int86x(0x21, &r, &r, &s);
}

#endif

/*-------------------------------------------------------------------------*/
/* Intel Code Builder support.											   */
/* UNTESTED																   */
/*-------------------------------------------------------------------------*/

#if	defined(CODEBUILDER)

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	limit = limit;
	return (void *)base;		/* Use linear address */
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* No free pointer */
	ptr = ptr;
}

/* This needs to be filled in with the remaining functions */

#endif

/*-------------------------------------------------------------------------*/
/* Ergo OS/386 support.													   */
/* UNTESTED 															   */
/*-------------------------------------------------------------------------*/

#ifdef ERGO386

/* Size must be <64K for this function */
void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	/* Limit will be restricted to less than 64k with this routine */

	union REGS r;

	/* Use create real window function */
	r.x.ax = 0xE803;
	r.x.cx = 0;					/* cx:dx is limit */
	r.x.dx = limit;
	r.x.si = base >> 16;		/* si:bx is address */
	r.x.bx = base & 0xFFFF;
	int86(0x21, &r, &r);

	/* Construct pointer */
	return MK_FP(r.x.ax, 0);
}

void PMODE_freeLinearPointer(void *ptr)
{
	union REGS 		r;
	struct SREGS 	s;
	segread(&s);
	s.es = FP_SEG(ptr);
	r.h.ah = 0x49;
	int86x(0x21, &r, &r, &s);
}

/* This needs to be filled in with the remaining functions */

#endif

/*-------------------------------------------------------------------------*/
/* Watcom C/C++ with Rational DOS/4GW support.							   */
/*-------------------------------------------------------------------------*/

#ifdef	DOS4GW

#include <mem.h>

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	limit = limit;
	return (void *)base;		/* Use linear address */
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* No need to do anything here */
	ptr = ptr;
}

void * PMODE_allocRealSeg(unsigned short size, unsigned long *segid,
	unsigned *segment, unsigned *offset)
{
	union REGS		r;

	r.w.ax = 0x100;					/* DPMI allocate DOS memory		*/
	r.w.bx = (size + 0xF) >> 4;		/* number of paragraphs 		*/
	int386(0x31, &r, &r);
	if (r.w.cflag) return NULL;		/* DPMI call failed				*/

	*segment = r.w.ax;				/* Real mode segment			*/
	*offset = 0;
	*segid = r.w.dx;				/* Protected mode selector		*/

    /* Return a linear pointer to the memory */
    return (void*)MK_PHYS(*segment,*offset);
}

void PMODE_freeRealSeg(unsigned long segid)
{
	union REGS	r;

	r.w.ax = 0x101;					/* DPMI free DOS memory			*/
	r.w.dx = segid;					/* DX := selector from 0x100	*/
	int386(0x31, &r, &r);
}

void * PMODE_createCSAlias(void *dataptr)
{
	return dataptr;
}

void PMODE_freeCSAlias(void * alias)
{
	alias = alias;
}

void far * PMODE_createDSAlias(void (*codeptr)())
{
	union REGS	r;

	r.w.ax = 0x0A;					/* DPMI create alias descriptor	*/
	r.w.bx = FP_SEG(codeptr);
	int386(0x31, &r, &r);
	if (r.w.cflag) return NULL;		/* DPMI call failed				*/

	return MK_FP(r.w.ax, FP_OFF(codeptr));
}

void PMODE_freeDSAlias(void far * alias)
{
	union REGS	r;

	r.w.ax = 0x01;					/* DPMI free descriptor			*/
	r.w.bx = FP_SEG(alias);			/* Selector to free				*/
	int386(0x31, &r, &r);
}

typedef struct {
	long	edi;
	long	esi;
	long	ebp;
	long	reserved;
	long	ebx;
	long	edx;
	long	ecx;
	long	eax;
	short	flags;
	short	es,ds,fs,gs,ip,cs,sp,ss;
	} _RMREGS;

#define IN(reg)     rmregs.e##reg = in->x.reg
#define OUT(reg)    out->x.reg = rmregs.e##reg

int PMODE_int86(int intno, RMREGS *in, RMREGS *out)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);

	memset(&sr, 0, sizeof(sr));
	r.w.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = FP_SEG(&rmregs);
    r.x.edi = FP_OFF(&rmregs);
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt			*/

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PMODE_int86x(int intno, RMREGS *in, RMREGS *out,
	RMSREGS *sregs)
{
	_RMREGS			rmregs;
	union REGS		r;
	struct SREGS	sr;

	memset(&rmregs, 0, sizeof(rmregs));
	IN(ax); IN(bx); IN(cx); IN(dx); IN(si); IN(di);
	rmregs.es = sregs->es;
	rmregs.ds = sregs->ds;

	memset(&sr, 0, sizeof(sr));
	r.w.ax = 0x300;					/* DPMI issue real interrupt	*/
	r.h.bl = intno;
	r.h.bh = 0;
	r.w.cx = 0;
	sr.es = FP_SEG(&rmregs);
	r.x.edi = FP_OFF(&rmregs);
	int386x(0x31, &r, &r, &sr);		/* Issue the interrupt */

	OUT(ax); OUT(bx); OUT(cx); OUT(dx); OUT(si); OUT(di);
	sregs->es = rmregs.es;
	sregs->cs = rmregs.cs;
	sregs->ss = rmregs.ss;
	sregs->ds = rmregs.ds;
	out->x.flags = rmregs.flags;
	out->x.cflag = rmregs.flags & 0x1;
	return out->x.ax;
}

int PMODE_installMouseHandler(int mask, void (cdecl far * handler)())
{
	union REGS		regs;
	struct SREGS	sregs;

	segread(&sregs);
	sregs.es = FP_SEG(handler);
	regs.x.edx = FP_OFF(handler);
	regs.x.ecx = mask;
	regs.x.eax = 20;
	int386x(0x33, &regs, &regs, &sregs);
	return 1;
}

/* No need to save the real mode ISR as DOS4GW always restores this for
 * us.
 */

void far * PMODE_getISR(int intno, SAVEINT *save)
{
	save->pmode_isr = (void far *)_dos_getvect(intno);
	return save->pmode_isr;
}

typedef void (interrupt far *intfp)();

void PMODE_restoreISR(int intno, SAVEINT *save)
{
	_dos_setvect(intno, (intfp)save->pmode_isr);
}

void PMODE_setISR(int intno, void far *isr)
{
	_dos_setvect(intno, (intfp)isr);
}

#endif

/*-------------------------------------------------------------------------*/
/* DJGPP port of GNU C++ support.										   */
/* UNTESTED																   */
/*-------------------------------------------------------------------------*/

#ifdef DJGPP

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	limit = limit;
	return (void *)(base + 0xE0000000);
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* No free pointer */
	ptr = ptr;
}

/* This needs to be filled in with the remaining functions */

#endif

/*-------------------------------------------------------------------------*/
/* EMX port of GNU C++ support.											   */
/* UNTESTED																   */
/*-------------------------------------------------------------------------*/

#ifdef	EMX

void * PMODE_mapLinearPointer(long base, unsigned limit)
{
	limit = limit;
	return (void *)base;
}

void PMODE_freeLinearPointer(void *ptr)
{
	/* No free pointer */
	ptr = ptr;
}

/* This needs to be filled in with the remaining functions */

#endif
