/*===========================================================================
SOLAR slrreply v0.94 :: Module process.c

This source code has been placed into the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
03-30-93 KJH  Started history.
              Changed tons of code to accomidate the new extract.c module.
              Added extensive error checking system to packet handling.
04-14-93 KJH  Modified spacing for news logging.
06-13-93 KJH  Tweaked code for decompressing packet. Was still assuming
              ZIP format as a default and leaving files open on errors.
              Cleaned up code and tightened lost+found processing.
07-15-94 KJH  Replaced all printf() with fprintf()
===========================================================================*/

/* Header files */
#include <stdio.h>
#include <process.h>
#include <string.h>
#include <time.h>
#include <dir.h>
#include <stdlib.h>
#include <spawno.h>   /* A product of Ralph Brown */

/* Local definitions */
#define BATCH_EXT               "MSG"
#define REPLY_NAME          "REPLIES"
#define NEWS_PROGRAM      "RNEWS.EXE"
#define MAIL_PROGRAM      "RMAIL.EXE"
#define BUFSIZE                  1024    /* Size of buffer          */
#define REPLY_TMP      "SOLARREP.TMP"    /* Temporary reply message */
#define REPLY_FINAL    "SOLARTMP.MAI"    /* Final mail message file */
#define NO                          1
#define YES                         0

/* External Data */
extern char temp_path[MAXPATH];          /* From slrreply.c */
extern char config_path[MAXPATH];        /* From slrreply.c */
extern char solar_path[MAXPATH];         /* From slrreply.c */
extern char lost_path[MAXPATH];          /* From slrreply.c */
extern char log_path[MAXPATH];           /* From slrreply.c */
extern char domain_name[40];             /* From slrreply.c */
extern char uucp_name[10];               /* From slrreply.c */
extern char username[10];                /* From slrreply.c */
extern char _slrerr[80];                 /* From slrreply.c */
extern char logbuf[60];                  /* From slrreply.c */
extern int  force_mode;                  /* From slrreply.c */

/* External Functions */
extern long extract_msg(FILE *msg_file); /* From extract.c  */
extern int  logit();                     /* From slrreply.c */
extern char *rfctime();									 /* From rfctime.c  */
extern int  process_list();              /* From list.c     */

/* Local Functions */
int  set_directory(char *path);
int  decompress_packet();
int  load_decompress_parms();
int  read_reply_file(FILE *reply_file);
int  exec_news();
int  exec_mail();
int  lost_message(char *msg_path);
int  lost_packet();
int  copyfile(char *path1, char *path2);
char *extract_destination(char *address);

/* Local Data */
char decompress[4];
char program[MAXPATH];
char options[40];
char reply_kind[10];
char prefix[9];
char msg_type;

/*
Function: int process_packet()
Purpose : Change into solarwork directory, decompress packet, and
          process the MSG files as directed by REPLIES. This is a
          big, hairy function.
Return  : Number of MSG files processed successfully.
*/

int process_packet()
{
	FILE *reply_file = NULL;
  FILE *msg_file   = NULL;
  int  news_count  = 0;
  int  mail_count  = 0;
  long byte_count  = 0L;
  long msg_size    = 0L;
  char msg_path[MAXPATH];

  /* Initialize the SPAWNO routine to swap
     slrreply.exe to disk in current directory on spawn */

  init_SPAWNO(".",SWAP_DISK);

  /* Initialize global variables */

  strcpy(decompress,"NON");
  strcpy(program,"NONE");
  options[0] = '\0';
  strcpy(reply_kind,"NONE");
	strcpy(prefix,"NONE");

  /* Set directory to solarwork temp directory */

  if (set_directory(temp_path) != 0)
    goto BadExit;

  /* Decompress the reply packet */

  if (decompress_packet() != 0)
    fprintf(stdout,"<warning> : %s\n",_slrerr);

  /* Process LIST file first */

  if (process_list() != 0)
    goto BadExit;

  /* Open the REPLIES file for processing */

  if ((reply_file = fopen(REPLY_NAME,"rt")) == NULL)
	{
		sprintf(_slrerr,"%s file not found in packet",REPLY_NAME);
		goto BadExit;
	}

  /* Process the reply packet MSG files
     as directed by REPLIES file */

  fprintf(stdout,"\nProcessing message files...\n");
  while (read_reply_file(reply_file) == 0)
	{

    /* Build filename for MSG file */

    strcpy(msg_path,prefix);
    strcat(msg_path,".");
    strcat(msg_path,BATCH_EXT);
    fprintf(stdout,"\nMsg file : %s : ",msg_path);

    /* Check to make sure MSG file is a supported HDPF
       format; either news or mail */

    if ((strcmp(reply_kind,"mail") != 0) && \
        (strcmp(reply_kind,"news") != 0))
    {

      /* Unsupported HDPF MSG file, move into lost+found */

      fprintf(stdout,"Unknown reply kind=%s : not processed\n",reply_kind);
      if (lost_message(msg_path) == 0)
        fprintf(stdout,"-Saved %s to lost+found\n",msg_path);
      else
        fprintf(stdout,"-Message %s lost: %s\n",msg_path,_slrerr);

    }
    else   /* The MSG format is supported */
    {

      /* Open the MSG file for processing. If we can't,
         that ain't good. Try to save to lost+found and
         life goes on. */

      fprintf(stdout,"processing as %s\n",reply_kind);
      if ((msg_file = fopen(msg_path,"rb")) == NULL)
      {
        fprintf(stdout,"process: Message %s is child-proof, cannot open\n",msg_path);
        if (lost_message(msg_path) == 0)
          fprintf(stdout,"-Saved %s to lost+found\n",msg_path);
        else
          fprintf(stdout,"-Message %s lost: %s\n",msg_path,_slrerr);
        goto NextBatch;
			}

      /* Call the extract function to do it's thing. This
         pulls one or more messages from MSG file, so the
         msg_file must remain open. */

      while ((msg_size = extract_msg(msg_file)) != 0L)
			{
        if (msg_size > 0L)
        {

          /* Figure out what program to pass the message to
             based on it's HDPF msg type */

          switch (msg_type) {

            /* Binary news message. */

            case 'B'  : if (exec_news() != 0)
                        {
                          if (msg_file) fclose(msg_file);
                          fprintf(stderr,"exec_news(): %s\n",_slrerr);
                          if (lost_message(msg_path) == 0)
                            fprintf(stdout,"-Saved %s to lost+found\n\n",msg_path);
													else
                            fprintf(stdout,"-Message %s lost: %s\n\n",msg_path,_slrerr);
                          unlink(REPLY_TMP);
                          goto NextBatch;
                        }
												else
												{
                          news_count++;
													byte_count += msg_size;
                          fprintf(stdout,"News message posted: %lu bytes\n",msg_size);
												}
												break;

            /* Binary mail message, call mail program */

						case 'b'  : if (exec_mail() != 0)
												{
                          if (msg_file) fclose(msg_file);
                          fprintf(stderr,"exec_mail(): %s\n",_slrerr);
                          if (lost_message(msg_path) == 0)
                            fprintf(stdout,"-Saved %s to lost+found\n\n",msg_path);
                          else
                            fprintf(stdout,"-Message %s lost: %s\n\n",msg_path,_slrerr);
                          unlink(REPLY_TMP);
                          unlink(REPLY_FINAL);
                          goto NextBatch;
                        }
												else
												{
                          mail_count++;
													byte_count += msg_size;
                        }
                        break;

            /* Unsupported message type found */

            default   : fprintf(stdout,"Unsupported message type: %c\n",msg_type);
                        if (msg_file) fclose(msg_file);
                        if (lost_message(msg_path) == 0)
                          fprintf(stdout,"-Saved %s to lost+found\n\n",msg_path);
                        else
                          fprintf(stdout,"-Message %s lost: %s\n\n",msg_path,_slrerr);
                        goto NextBatch;
          }
        }
        else  /* The message file extract function puked */
        {
          if (msg_size == -1L)
					{
            if (msg_file) fclose(msg_file);
            fprintf(stdout,"extract: %s\n",_slrerr);
            if (msg_file) fclose(msg_file);
            if (lost_message(msg_path) != 0)
              fprintf(stdout,"-Message lost: %s\n",_slrerr);
            else
              fprintf(stdout,"-Saved %s to lost+found\n\n",msg_path);
            goto NextBatch;
          }
          else  /* Puked because of a stupid zero byte MSG file. */
          {
            fprintf(stdout,"Skipping msg in %s : %s\n",msg_path,_slrerr);
          }
        }
      }

      /* Reply packet processed 'normally', close the msg_file */

      if (msg_file) fclose(msg_file);
    }

    /* Delete MSG file and move on to the next MSG file */

NextBatch:
    unlink(msg_path);
	}

  /* All finished, close up the REPLIES file */

  fclose(reply_file);

GoodExit:

  /* Tell everyone what we accomplished */

  fprintf(stdout,"\nPacket processing complete: %u news, %u mail, %lu bytes\n",news_count,mail_count,byte_count);
  sprintf(logbuf," %-8s | R  %4u news %3u mail  %8lu bytes",username,news_count,mail_count,byte_count);
	logit();
	return (mail_count + news_count);

BadExit:

  /* Something bad happened. Make sure the files are
     closed and try to save the packet to lost+found */

  if (reply_file) fclose(reply_file);
	if (msg_file)   fclose(msg_file);
  fprintf(stderr,"<slrreply> process: %s\n",_slrerr);
  if (lost_packet() != 0)
    fprintf(stdout,"Packet lost: %s\n",_slrerr);
  else
    fprintf(stdout,"Packet saved in lost+found\n");
  return 0;
}

/*
Function: int set_directory(char *path)
Purpose : perform a chdir to given directory
Params  : path = full path to new directory
Return  : zero on success, non-zero on error
*/

int set_directory(char *path)
{
  char disk_drive = path[0];
  char new_path[MAXPATH];
	int disknum = 0;
  int x = 0;

	disk_drive = toupper(disk_drive);
	disknum = disk_drive - 65;
	if ((disknum >= 0) && (disknum <=25))
	{
		setdisk(disknum);
	}
	else
	{
    sprintf(_slrerr,"invalid disk: %c",disk_drive);
    goto ErrorExit;
	}

  new_path[0] = '\0';
	for (x = 2; x < strlen(path); x++)
	{
		new_path[x - 2] = path[x];
	}
	new_path[x - 2] = '\0';
	if (chdir(new_path) != 0)
	{
    sprintf(_slrerr,"directory %s does not exist",new_path);
		goto ErrorExit;
	}
GoodExit:
	return 0;
ErrorExit:
	return 1;
}

/*
Function: decompress_packet()
Purpose : spawn decompression program to decompress packet.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int decompress_packet()
{
	struct ffblk ffblk;
  char packetpath[MAXPATH];
  char newpath[MAXPATH];
  char drive[MAXDRIVE];
	char dir[MAXDIR];
	char file[MAXFILE];
  char ext[MAXEXT];
  int flags, x, ext_len, done;

  /* Scan solarwork directory for first file, assume
     it is the packet unless filename is STATUS.TMP */

  done = findfirst("*.*",&ffblk,0);
  if (strcmp(ffblk.ff_name,"STATUS.TMP") == 0)
  {
    unlink(ffblk.ff_name);
    done = findnext(&ffblk);
  }

  /* Check to make sure there was a packet found */

  if (done != 0)
  {

    /* No packet found. See if force mode is in effect */

    if (force_mode == YES)
    {

      /* Interactively accept full path to reply packet */

      fprintf(stdout,"Enter full path to packet file: ");
      gets(packetpath);
			if (packetpath[0] == '\0') exit(0);
      fprintf(stdout,"\n");

      /* Check to make sure named packet file exists */

      if (findfirst(packetpath,&ffblk,0) == 0)
      {

        /* copy packet into solarwork */

        strcpy(newpath,temp_path);
        strcat(newpath,"\\");
        strcat(newpath,ffblk.ff_name);
        fprintf(stdout,"copying %s --> ",packetpath);
        if (copyfile(packetpath,newpath) != 0)
        {
          fprintf(stdout,"\n");
          goto ErrorExit;
        }
        fprintf(stdout,"%s\n",newpath);
      }
      else  /* didn't find the named packet */
      {
        sprintf(_slrerr,"reply packet %s not found", packetpath);
        goto ErrorExit;
      }
    }
    else  /* no force mode, no reply packet */
    {
      strcpy(_slrerr,"reply packet not found in solarwork");
      goto ErrorExit;
    }
	}

  /* Parse the packet filename to get extention */

  flags = fnsplit(ffblk.ff_name,drive,dir,file,ext);

  /* make sure packet filename _has_ an extention */

  if (flags & EXTENSION)
	{
    /* strip off the leading '.' and save
       extention as compression format */

    ext_len = strlen(ext);
		for (x = 0; x < ext_len; x++)
		{
			decompress[x] = ext[x+1];
		}
		decompress[x] = '\0';

    /* load the decompression program command line
       from config file */

    if (load_decompress_parms() != 0)
      goto ErrorExit;

    /* Spawn the decompression program */

    fprintf(stdout,"\nDecompressing %s format packet...\n",decompress);
    if (spawnlp(P_WAIT,program,program,options,ffblk.ff_name,NULL) != 0)
    {
      sprintf(_slrerr,"error spawning %s",program);
      goto ErrorExit;
    }
  }
  else /* No extension, unknown packet archive format */
  {
    strcpy(_slrerr,"packet filename has no extension, format unknown");
    goto ErrorExit;
  }

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: load_decompress_parms()
Purpose : Load the appropriate decompress parameters from the Solar
          configuration file.
Return  : 0 on success, non-zero on error.
*/

int load_decompress_parms()
{
	FILE *config_file = fopen(config_path,"rt");

	int idx  = 0;

	char buf[128];
	char search_string[20];
	char *p = NULL;

  /* Set some values */
  strcpy(program,"NONE");
  options[0] = '\0';
  strcpy(search_string,"decompress-");
  strcat(search_string,decompress);

  /* Make sure the config file is open */
  if (!config_file)
  {
    sprintf(_slrerr,"config file %s not found",config_path);
    goto ErrorExit;
  }

  /* Read throught the config file */
  while (fgets(buf,128,config_file) != NULL)
	{
    /* Look for a decompress parameter */
    if (strnicmp(buf,"decompress",10) == 0)
		{
      /* Get everything before the '=' seperator */
      if ((p = strtok(buf,"=")) != NULL)
			{
        /* Replace trailing LF with NULL */
        idx = 0;
        while (p[idx] != '\n')
        {
          p[idx] = p[idx++];
        }
        p[idx] = '\0';

        /* Check to see if this line has archive type we want */
        if (stricmp(p,search_string) == 0)
        {
          /* Get everything after the '=' separator */
          if ((p = strtok(NULL,"=")) == NULL)
          {
            sprintf(_slrerr,"incomplete configuration: %s",buf);
            goto ErrorExit;
          }
          /* Again, replace trailing LF with NULL */
          idx = 0;
          while (p[idx] != '\n')
          {
            buf[idx] = p[idx++];
          }
          buf[idx] = '\0';

          /* Get the first word, which is the decompress program */
          if ((p = strtok(buf," ")) != NULL)
          {
            /* Maybe I need a fgets function which does this.. */
            idx = 0;
            while (p[idx] != '\n')
            {
              program[idx] = p[idx++];
            }
            program[idx] = '\0';

            /* Get any program options there might be. Better not use '|' */
            if ((p = strtok(NULL,"|")) != NULL)
            {
              /* La dee da dee da */
              idx = 0;
              while (p[idx] != '\n')
              {
                options[idx] = p[idx++];
              }
              options[idx] = '\0';
            }
            else
            {
              options[0] = '\0';
            }
          }
          /* Well, this was the one we wanted, so exit loop */
          break;
        }
      }
    }
  }
  /* Make sure the program got set to something */
  if (strcmp(program,"NONE") == 0)
  {
    sprintf(_slrerr,"unknown archive file format");
    goto ErrorExit;
  }
GoodExit:
  if (config_file) fclose(config_file);
  return 0;
ErrorExit:
  if (config_file) fclose(config_file);
  return 1;
}

/*
Function: read_reply_file()
Purpose : Read the REPLIES file and store values.
Return  : 0 on success, non-zero on error. Do not set _slrerr,
          as error conditions are not trapped from above.
*/

int read_reply_file(FILE *reply_file)
{
  char buf[128];
	char *p;

  /* Read a line from reply file */

  if (fgets(buf,128,reply_file) == NULL)
    goto ErrorExit;

  /* Parse out first value, prefix */

  if ((p = strtok(buf,"\t")) == NULL)
  {
    fprintf(stderr,"%s: Unknown file format.\n",REPLY_NAME);
    goto ErrorExit;
  }
  strcpy(prefix,p);

  /* Parse out second value, reply kind */

  if ((p = strtok(NULL,"\t")) == NULL)
	{
    fprintf(stderr,"%s: Unknown file format.\n",REPLY_NAME);
    goto ErrorExit;
  }
  strcpy(reply_kind,p);

  /* Parse out third value, message type */

  if ((p = strtok(NULL,"\t")) == NULL)
  {
    fprintf(stderr,"%s: Unknown file format.\n",REPLY_NAME);
    goto ErrorExit;
  }
  msg_type = p[0];

GoodExit:
  return 0;
ErrorExit:
  return 1;
}

/*
Function: exec_news()
Purpose : Feed message to news program.
Return  : 0 on success, non-zero on error.
*/

int exec_news()
{
  char full_path[MAXPATH];

  strcpy(full_path,temp_path);
  strcat(full_path,"\\");
  strcat(full_path,REPLY_TMP);

  return(spawnlp(P_WAIT,NEWS_PROGRAM,NEWS_PROGRAM,full_path,NULL));
}

/*
Function: exec_mail()
Purpose : Build final mail message and pass to mail program.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int exec_mail()
{
  FILE *mail_file  = NULL;
  FILE *final_file = NULL;
  char destination[128];
  char execbuf[BUFSIZE];
  char date_time[35];
  char command[512];
  char to[256];
  char cc[256];
  static char *p = NULL;

  /* Set To: and Cc: headers to NONE */

  strcpy(to,"NONE");
  strcpy(cc,"NONE");

  /* Open the temporary mail file for read */

  if ((mail_file = fopen(REPLY_TMP,"rb")) == NULL)
  {
    sprintf(_slrerr,"error opening %s",REPLY_TMP);
    goto ErrorExit;
  }

  /* Get To: and Cc: headers from
     temporary mail file */

  while (fgets(execbuf,BUFSIZE,mail_file) != NULL)
  {
    if (strlen(execbuf) < 3) break;
    if (strnicmp(execbuf,"To:",3) == 0)
    {
      strcpy(to,execbuf);
		}
		if (strnicmp(execbuf,"Cc:",3) == 0)
		{
      strcpy(cc,execbuf);
    }
  }
  fclose(mail_file);

  /* Make sure there was a To: header */

  if (strcmp(to,"NONE") == 0)
  {
    strcpy(_slrerr,"no destination for mail");
    goto ErrorExit;
  }

  /* Parse multiple addresses out of To: header
     and build/send message to each */

  p = strtok(to,",");
  while (p)
  {
    /* Get the destination for this copy of mail message */

    strcpy(destination,extract_destination(p));
    fprintf(stdout,"Sending to: %s\n",destination);

    /* Re-open temporary mail file for read */

    if ((mail_file = fopen(REPLY_TMP,"rb")) == NULL)
    {
      sprintf(_slrerr,"error opening %s",REPLY_TMP);
      goto ErrorExit;
    }

    /* Open final mail file for write */

    if ((final_file = fopen(REPLY_FINAL,"wb")) == NULL)
    {
      sprintf(_slrerr,"error opening %s",REPLY_FINAL);
      goto ErrorExit;
    }

    /* Get time stamp in RFC 822 format */

    if (strcpy(date_time,rfctime()) == NULL)
    {
      strcpy(_slrerr,"error creating rfctime");
      goto ErrorExit;
    }

    /* Write the From_ and leading Received: headers to
       final mail message */

    fprintf(final_file,"From %s@%s  %s remote from %s\n",username,domain_name,date_time,uucp_name);
    fprintf(final_file,"Received: by %s (Solar)\n",domain_name);
    fprintf(final_file,"       via HDPF; %s\n",date_time);
    fprintf(final_file,"       for %s\n",destination);

    /* Copy the temporary mail file into
       final mail file */

    while (fgets(execbuf,BUFSIZE,mail_file) != NULL)
    {
      fprintf(final_file,"%s",execbuf);
    }
    fclose(mail_file);
    fclose(final_file);

    /* Build mail program command and execute. Cannot
       use a spawn because of the < redirection */

    strcpy(command,MAIL_PROGRAM);
    strcat(command," ");
    strcat(command,destination);
    strcat(command," < ");
    strcat(command,REPLY_FINAL);
    system(command);

    /* Get the next destination from To: header */

    p = strtok(NULL,",");
  }

  /* Perform the same processing for the Cc:
     header, if it exists */

  if (strcmp(cc,"NONE") != 0)
  {
    p = strtok(cc,",");
    while (p)
    {
      strcpy(destination,extract_destination(p));
      fprintf(stdout,"Sending Cc to: %s\n",destination);

      if ((mail_file = fopen(REPLY_TMP,"rb")) == NULL)
      {
        sprintf(_slrerr,"error opening %s",REPLY_TMP);
        goto ErrorExit;
      }
      if ((final_file = fopen(REPLY_FINAL,"wb")) == NULL)
      {
        sprintf(_slrerr,"error opening %s",REPLY_FINAL);
        goto ErrorExit;
      }
      if (strcpy(date_time,rfctime()) == NULL)
      {
        strcpy(_slrerr,"error creating rfctime");
        goto ErrorExit;
      }
      fprintf(final_file,"From %s@%s  %s remote from %s\n",username,domain_name,date_time,uucp_name);
      fprintf(final_file,"Received: by %s (Solar)\n",domain_name);
      fprintf(final_file,"       via HDPF; %s\n",date_time);
      fprintf(final_file,"       for %s\n",destination);

      while (fgets(execbuf,BUFSIZE,mail_file) != NULL)
      {
        fprintf(final_file,"%s",execbuf);
      }
      fclose(mail_file);
      fclose(final_file);

      strcpy(command,MAIL_PROGRAM);
      strcat(command," ");
      strcat(command,destination);
      strcat(command," < ");
      strcat(command,REPLY_FINAL);
      system(command);

      p = strtok(NULL,",");
    }
  }

GoodExit:
  return 0;
ErrorExit:
  if (mail_file) fclose(mail_file);
  if (final_file) fclose(final_file);
  return -1;
}

/*
Function: extract_destination()
Purpose : Pull single address out of To: header. The address can
          be free-standing, or enclosed in < >'s, according to
          RFC 822.
Return  : Single destination address.
*/

char *extract_destination(char *address)
{
	int  idx  = 0;
	int  idx2 = 0;
  static char final_add[128];

	if ((strnicmp(address,"To:",3) == 0) || (strnicmp(address,"Cc:",3) == 0))
	{
    while (address[idx++] != ':');
    while (address[idx] == ' ') idx++;
  }
  if (strstr(address,"<") != NULL)
	{
    while (address[idx++] != '<');
	}
  while (address[idx] == ' ') idx++;
  while ((address[idx] != '\0') && \
				 (address[idx] != '\n') && \
				 (address[idx] != '>')  && \
				 (address[idx] != ' '))
	{
		final_add[idx2++] = address[idx++];
	}
	final_add[idx2] = '\0';
	return final_add;
}

/*
Function: int lost_packet()
Purpose : A fatal error occurred and the entire packet could have
          been lost. Move the packet to the lost directory.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int lost_packet()
{
  struct ffblk ffblk;
  char newpath[MAXPATH];
  int done;

  done = findfirst("*.*",&ffblk,0);
  while (!done)
  {
    strcpy(newpath,lost_path);
    strcat(newpath,"\\");
    strcat(newpath,ffblk.ff_name);
    if (copyfile(ffblk.ff_name,newpath) != 0)
    {
      goto ErrorExit;
    }
    unlink(ffblk.ff_name);
    done = findnext(&ffblk);
  }

GoodExit:
  return 0;
ErrorExit:
  return -1;
}

/*
Function: int lost_message(char *msg_path)
Purpose : A fatal error occurred and all or part of a message file has
          been lost. Move the message to the lost directory.
Return  : 0 on success, non-zero on error and set _slrerr.
*/

int lost_message(char *msg_path)
{
  char newpath[MAXPATH];

  strcpy(newpath,lost_path);
  strcat(newpath,"\\");
  strcat(newpath,msg_path);
  if (copyfile(msg_path,newpath) != 0)
  {
    goto ErrorExit;
  }
GoodExit:
  return 0;
ErrorExit:
  return -1;
}

/*
Function: int copyfile(char *path1, char *path2)
Purpose : Moves file path1 to path2.
Return  : zero on success, non-zero on error and set _slrerr.
*/

int copyfile(char *path1, char *path2)
{
	FILE *file1 = NULL;
	FILE *file2 = NULL;
	char copybuf;

	if ((file1 = fopen(path1,"rb")) == NULL)
	{
    sprintf(_slrerr,"file %s not found",path1);
		goto ErrorExit;
  }
  if ((file2 = fopen(path2,"wb")) == NULL)
  {
    sprintf(_slrerr,"error opening %s for write",path2);
		goto ErrorExit;
  }

  copybuf = fgetc(file1);
  while (!feof(file1))
  {
    fprintf(file2,"%c",copybuf);
    copybuf = fgetc(file1);
  }
  fclose(file1);
  fclose(file2);

GoodExit:
  return 0;
ErrorExit:
  if (file1) fclose(file1);
  if (file2) fclose(file2);
  return 1;
}

