/* 1.0  04-27-84 */
/************************************************************************
 *			Robert C. Tausworthe				*
 *			Jet Propulsion Laboratory			*
 *			Pasadena, CA 91009		1984		*
 ************************************************************************/

#include "defs.h"
#include "stdtyp.h"
#include "errno.h"
#include "mathcons.h"

/*   Coefficients from J. L. Schonfelder, Math. Comp. Vol. 32, No. 144,
 *   Oct. 1978, pp. 1232-1240.  Converted from Chebyshev coefficients
 *   to polynomial coefficients.
 */

#define bDEGREE 19
static double b[bDEGREE+1] =
{	/* b[0] */	 6.74933236039655050263e-01,
	/* b[1] */	-2.61111860931245377287e-01,
	/* b[2] */	 1.19479138609851943329e-01,
	/* b[3] */	-4.86627774491551332176e-02,
	/* b[4] */	 1.71283445718166884035e-02,
	/* b[5] */	-5.23487583615739131346e-03,
	/* b[6] */	 1.40509142365203239024e-03,
	/* b[7] */	-3.35143533535934984684e-04,
	/* b[8] */	 7.18010084384964546189e-05,
	/* b[9] */	-1.39462736935665598139e-05,
	/* b[10] */	 2.47580204969781916589e-06,
	/* b[11] */	-4.04509593030131189153e-07,
	/* b[12] */	 6.11956771414692047983e-08,
	/* b[13] */	-8.61749835843079723418e-09,
	/* b[14] */	 1.13482488917751004919e-09,
	/* b[15] */	-1.40320352244567126036e-10,
	/* b[16] */	 1.63314151643890654668e-11,
	/* b[17] */	-1.79944661052623065189e-12,
	/* b[18] */	 1.97235204103604890406e-13,
	/* b[19] */	-1.95488364156685071066e-14
};

#define cDEGREE 21
static double c[cDEGREE+1] = 
{	/* c[0] */	 5.40464821348814759403e-01,
	/* c[1] */	-2.61515522487415932119e-02,
	/* c[2] */	-2.88573438386340392753e-03,
	/* c[3] */	-5.29353396945768734440e-04,
	/* c[4] */	-8.54344544297635788098e-05,
	/* c[5] */	-1.74905253140942193568e-05,
	/* c[6] */	-3.06142196558938128874e-06,
	/* c[7] */	-6.87177932914162287489e-07,
	/* c[8] */	-1.18400244002269115299e-07,
	/* c[9] */	-3.04457664850508328527e-08,
	/* c[10] */	-4.49026736274657305330e-09,
	/* c[11] */	-1.55646576715015340596e-09,
	/* c[12] */	-1.22012057640254963189e-10,
	/* c[13] */	-1.01539218332404736429e-10,
	/* c[14] */	 4.97278086303190793842e-12,
	/* c[15] */	-8.76177272209771675989e-12,
	/* c[16] */	 3.33713910876554204151e-12,
	/* c[17] */	-1.94148792669164948165e-12,
	/* c[18] */	-2.99117077089093858376e-13,
	/* c[19] */	 2.10106516619397560135e-13,
	/* c[20] */	 3.40434284957405645400e-13,
	/* c[21] */	-1.73433792163219535723e-13,
};

/************************************************************************/
	double
erf(x)		/* return the error function of x, as defined in
		   Handbook of Mathematical Functions, National Bureau of
		   Standards AMS-55, pp. 297.				*/
/*----------------------------------------------------------------------*/
double x;
{
	int minus, k;
	double val, t, erfc();

	minus = (x < 0 ? 1 : 0);
	t = (minus ? -x : x);
	if (t > 2.0)
	{	t = 1.0 - erfc(t);
		return (minus ? -t : t);
	}
	else
	{	t = x * x / 2.0 - 1.0;
		val = b[bDEGREE];
		for (k = bDEGREE-1; k >= 0; k--)
			val = val * t + b[k];
		return (x * val);
	}
}

/*\p*********************************************************************/
	double
erfc(x)			/* complementary error function, 1-erf(x)	*/

/*----------------------------------------------------------------------*/
double x;
{
	int k, minus, chrstc();
	double t, val, y, z, erf(), exp();

	minus = (x < 0.0 ? 1 : 0);
	z = (minus ? -x : x);
	if (z < 2.0)
		return (1.0 - erf(x));

	y = x * x;
	t = (10.5 - y)/(2.5 + y);
	val = c[cDEGREE];
	for (k = cDEGREE-1; k >= 0; k--)
		val = val * t + c[k];
	val = exp(-y) * val / z;
	return (minus ? 2.0 - val : val);
}
