/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	kprintf;
include	sound;
include	hardware, list;
include	error;
include	object;
include	process;
include	_startup;
include	arena;

NMESSAGES:	const	int = 64;		// must be a power of two

MessageTable:	[NMESSAGES] message;
FreeMsg:	queue;
FreeControl:	semaphore;

message:	public	type	inherit queue {
	signaling:	boolean;
	sendBuf:	vaddr_t;
	replyBuf:	vaddr_t;
	destination:	ref object;
	source:		ref object;
	control:	semaphore;

	visible:

	cumReply:	size_t;
	errorCode:	int;
	state:		messageState;

	public:

	dumpOnAbort:	boolean;
	header:		messageHeader;

display:	() =
	{
	kprintf("%s %d -> %d cum %d error %d ", MessageState[state],
			source->me, destination->me, cumReply, errorCode);
	kprintf("%d: %d bytes\n", header.func, header.len);
	source->where dumpHex(sendBuf, header.len);
	}

create:	factory	(dest: ref object, r: unsigned[16], sid: ref far external, 
					func: unsigned[32],
					buf: vaddr_t,
					len: size_t,
					rbuf: vaddr_t,
					rlen: size_t) ref message =
	{
	n:	threadLock;

	if	(!FreeControl down(TRUE))
		return 0;
	n lock();
	self = ref message(FreeMsg dequeue());
	n unlock();
	header.rights = r;
	header.sender = sid;
	header.expected = rlen;
	header.len = len;
	header.func = func;
	header.sequence = dest->sequence++;
	destination = dest;
	source = CurProc->animates;
	cumReply = 0;
	errorCode = 0;
	sendBuf = buf;
	replyBuf = rbuf;
	state = MS_SENT;
	control = [ 0 ];
	signaling = FALSE;
	dumpOnAbort = FALSE;
	return self;
	}

createIntReply:	factory	(dest: ref object, r: unsigned[16], 
					sid: ref far external, 
					func: unsigned[32],
					buf: vaddr_t,
					len: size_t) ref message =
	{
	n:	threadLock;

	if	(!FreeControl down(TRUE))
		return 0;
	n lock();
	self = ref message(FreeMsg dequeue());
	n unlock();
	header.rights = r;
	header.sender = sid;
	header.expected = sizeof int;
	header.len = len;
	header.func = func;
	header.sequence = dest->sequence++;
	destination = dest;
	source = CurProc->animates;
	cumReply = 0;
	errorCode = 0;
	sendBuf = buf;
	replyBuf = 0;
	state = MS_SENT;
	control = [ 0 ];
	signaling = FALSE;
	dumpOnAbort = FALSE;
	return self;
	}

createSelfGenerated:	factory	(dest: ref object, func: unsigned[32],
					buf: vaddr_t,
					len: unsigned[32]) ref message =
	{
	n:	threadLock;

	if	(!FreeControl down(TRUE))
		return 0;
	n lock();
	self = ref message(FreeMsg dequeue());
	n unlock();
	header.rights = AR_ANY;
	header.sender = 0;
	header.expected = 0;
	header.len = len;
	header.func = func;
	header.sequence = dest->sequence++;
	destination = dest;
	source = dest;
	cumReply = 0;
	errorCode = 0;
	sendBuf = buf;
	replyBuf = 0;
	state = MS_SENT;
	control = [ 0 ];
	signaling = TRUE;
	dumpOnAbort = FALSE;
	return self;
	}

kernelSend:	(dest: ref object, func: int, r: unsigned[16], 
				src: ref object, buf: vaddr_t, len: int,
				rbuf: vaddr_t, rlen: int) =
	{
	memSet(self, 0, sizeof *self);
	header.id = messageId(self);
	header.rights = r;
	if	(src)
		header.sender = src->me;
	header.expected = rlen;
	header.len = len;
	header.func = func;
	destination = dest;
	source = src;
	sendBuf = buf;
	replyBuf = rbuf;
	state = MS_SENT;
	control = [ 0 ];
	}

waitForDone:	(src: ref object) int =
	{
	retn:	int;
	s:	messageState;
	e:	int;

	if	(!control down(TRUE)){
		src rejectedSend(ERRINTERRUPTED);
		retn = 0;		// possibly reject
		}
	else if	(sendBuf)
		retn = cumReply;
	else
		retn = errorCode;
//	kprintf("waitForDone retn %d\n", retn);
//	display();
	wrapup();
	s = state;
	e = errorCode;
	close();
	if	(s == MS_REJECTED)
		src rejectedSend(e);
	return retn;
	}
/*
	This code is just a safeguard to make sure that a message has
	processed a reply or a reject.
 */
wrapup:	() =
	{
	if	(state == MS_SENT ||
		 state == MS_RECEIVED)
		reply_(0, 0);
	}

reply_:	(buf: vaddr_t, len: size_t) =
	{
	n:	threadLock;

	if	(header.func == int(&external.childExit)){
//		kprintf("childExit - %d -> %d ", source->me, destination->me);
		n lock();
		extract();
		state = MS_REPLIED;
		n unlock();
		source processTermination();
//		kprintf("terminated ");
		close();
//		kprintf("message closed\n");
		}
	else if	(source == destination){	// a self generated message
		n lock();
		extract();
		state = MS_REPLIED;
		n unlock();
		close();
		}
	else	{
		replyPartial(buf, len);
		n lock();
		extract();
		state = MS_REPLIED;
		n unlock();
		control up();
		}
	}

reject:	(code: int) =
	{
	n:	threadLock;

	n lock();
	extract();
	state = MS_REJECTED;
	errorCode = code;
	n unlock();
	if	(header.func == int(&external.childExit)){
		source processTermination();
		close();
		}
	else if	(source == destination)		// a self generated message
		close();
	else
		control up();
	}

readText:	(offs: vaddr_t, buf: pointer, len: size_t) int =
	{

		// Can't read past the message text

	if	(offs >= header.len)
		return 0;
	if	(offs + len > header.len)
		len = header.len - offs;
	return source->where read(sendBuf + offs, buf, len);
	}

discardText:	() =
	{
	}

replyPartial:	(buf: vaddr_t, len: size_t) =
	{
	if	(len && cumReply < header.expected){
		if	(cumReply + len > header.expected)
			len = header.expected - cumReply;
		if	(replyBuf == 0){
			if	(len <= sizeof int)
				destination->where read(buf, &errorCode, len);
			}
		else	{
			replyPtr:	pointer;

			replyPtr = source->where obtainWrite(replyBuf, len);
			replyBuf += len;
			cumReply += len;
			if	(replyPtr == 0)
				return;
			destination->where read(buf, replyPtr, len);
			source->where unlock();
			}
		}
	}

close:	() =
	{
	if	(source == destination &&
		 sendBuf)
		free(pointer(sendBuf));
	if	(self >= MessageTable &&
		 self < &MessageTable[NMESSAGES]){
		n:	threadLock;

		n lock();
		state = MS_FREE;
		FreeMsg enqueue(self);
		n unlock();
		FreeControl up();
		header.id += NMESSAGES;
		}
	}

	};

messageState:	public	type	byte = {
	MS_FREE,
	MS_SENT,
	MS_RECEIVED,
	MS_REPLIED,
	MS_REJECTED,
	};

displayMessages:	public	() =
	{
	i:	int;
	m:	ref message;
	q:	ref queue;

	for	(i = 0; i < NMESSAGES; i++){
		m = &MessageTable[i];
		if	(m->state != MS_FREE)
			m display();
		}
	kprintf("%24d count\r", FreeControl.count);
	for	(q = FreeMsg.next, i = 0; q != &FreeMsg; q = q->next, i++)
		kprintf("%4d: %8p\r", i, q);
	kprintf("\n");
	}

MessageState:	public	[] ref char = [
	"MS_FREE",
	"MS_SENT",
	"MS_RECEIVED",
	"MS_REPLIED",
	"MS_REJECTED",
	];

messageClass:	public	type	byte = {
	MC_LOCAL,
	MC_KERNEL,
	MC_INCOMING,
	MC_OUTGOING
	};

m_reject:	public	(seq: _sequence_t, code: int) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m reject(code);
	}

m_readText:	public	(seq: _sequence_t, offs: vaddr_t, buf: vaddr_t, len: size_t) int =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m == 0)
		return ERRINVALIDARGUMENT;

		// Can't read past the message text

	bufp:	pointer;

	bufp = CurArena obtainWrite(buf, len);
	if	(bufp == 0)
		return ERRINVALIDDATA;

	i:	int;
	i = m readText(offs, bufp, len);
	CurArena unlock();
	return i;
	}

m_discardText:	public	(seq: _sequence_t) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m discardText();
	}

m_reply:	public	(seq: _sequence_t, buf: vaddr_t, len: size_t) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m reply_(buf, len);
	}

m_replyPartial:	public	(seq: _sequence_t, buf: vaddr_t, len: size_t) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m replyPartial(buf, len);
	}

local_receive:	public	(hdr: ref messageHeader) int =
	{
	return CurProc->animates receive(vaddr_t(hdr));
	}

local_reject:	public	(seq: _sequence_t, code: int) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m reject(code);
	}

local_readText:	public	(seq: _sequence_t, offs: unsigned, buf: pointer, len: int) int =
	{
	m:	ref message;

	if	(len < 0)
		return ERRINVALIDARGUMENT;
	m = getMessage(seq);
	if	(m == 0)
		return ERRINVALIDARGUMENT;

		// Can't read past the message text

	return m readText(offs, buf, len);
	}

local_discardText:	public	(seq: _sequence_t) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m discardText();
	}
/*
	This function handles replies from kernel objects.  In this
	case the messageId is actually a pointer to the message object
	itself.
 */
local_reply:	public	(seq: _sequence_t, buf: pointer, len: int) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m reply_(vaddr_t(buf), len);
	}

local_replyPartial:	public	(seq: _sequence_t, buf: pointer, len: int) =
	{
	m:	ref message;

	m = getMessage(seq);
	if	(m)
		m replyPartial(vaddr_t(buf), len);
	}

getMessage:	(seq: _sequence_t) ref message =
	{
	m:	ref message;

	m = pointer(CurProc->animates->msgs.next);
	if	(m == &CurProc->animates->msgs ||
		 m->state != MS_RECEIVED ||
		 m->header.sequence != seq)
		return 0;
	else
		return m;
	}

startup:	entry	() =
	{
	i:	int;

	FreeMsg = [];
	for	(i = 0; i < NMESSAGES; i++){
		MessageTable[i].header.id = i;
		FreeMsg enqueue(&MessageTable[i]);
		}
	FreeControl = [ NMESSAGES ];
	P_receive = &local_receive;
	P_reject = &local_reject;
	P_readText = &local_readText;
	P_discardText = &local_discardText;
	P_reply = &local_reply;
	P_replyPartial = &local_replyPartial;
	}
